/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#include <gnome.h>
#include "lib/minmax.h"
#include "lib/cpudetect.h"
#include <config.h>
#include "pref.h"

extern CpuCaps gCpuCaps;

int
pref_verify_sample_width(struct pref *pref,
                         union pref_value *value) {
    if(value->i != 1 ||
       value->i != 2 ||
       value->i != 4) 
        return 1;
    return 0;

}

/* 
 * NOTE: You're probably better off editing this stuff in the
 * ~/.gnome/gnusound configuration file.
 */

struct pref prefs[] = { 

    /*
     * The number of seconds between two mouse wheel movements that
     * make them a separate event.
     */

    { "wheel_time_treshold", PREF_FLOAT, 0,
      { .f = 1 }, { .f = 1 }, 0, 500, NULL } ,
    
    /*
     * The number of pixels between two mouse wheel event locations
     * that make them separate events.
     */
    
    { "wheel_x_treshold", PREF_INT, 0,
      { .i = 5 }, { .i = 5 }, 0, 1000, NULL } ,

    /* 
     * Number of undo's. 
     */

    { "undo_depth_max", PREF_INT, 0,
      { .i = 40 }, { .i = 40 }, 0, 1000, NULL } ,

    { "playback_device", PREF_STRING, 0,
      { .s = NULL }, { .s = "/dev/dsp" }, 0, 0, NULL } ,

    /* 
     * Number of frames to send to the audio device per mixer
     * iteration.  Small values = faster response, large values =
     * fewer audio dropouts.
     */
    
    { "playback_buffer_size", PREF_INT, 0,
      { .i = 4096 }, { .i = 4096 }, 32, 65536, NULL },

    /*
     * Whether to convert signed 8 bit samples to unsigned 8 bit
     * samples on playback. For soundcards where AFMT_S8 doesn't work.
     */

    { "playback_signed_int8_to_unsigned_int8", PREF_INT, 0,
      { .i = 0 }, { .i = 0 }, 0, 1, NULL },

    { "playback_channels", PREF_INT, 0, 
      { .i = 2 }, { .i = 2 }, 1, 32, NULL },

    { "record_device", PREF_STRING, 0, 
      { .s = NULL }, { .s = "/dev/dsp" }, 0, 0, NULL }, 

    /* 
     * If your recording is always "too late" then you can use this
     * parameter to throw away the first few frames of your recording
     * and get better sync. Not very elegant but it works.
     */

    { "record_discard_frames", PREF_INT, 0,
      { .i = 0 }, { .i = 0 }, 0, 100000, NULL }, 
    { "default_bpm", PREF_FLOAT, 0, 
      { .f = 120 }, { .f = 120 }, 1, 10000, NULL },
    { "default_sample_rate", PREF_FLOAT, 0, 
      { .f = 44100 }, { .f = 44100 }, 1, 1000000, NULL },
    { "default_sample_width", PREF_INT, 0,
      { .i = 2 }, { .i = 2 }, 1, 4, pref_verify_sample_width },
    { "ladspa_path", PREF_STRING, 0, 
      { .s = NULL }, { .s = "/usr/local/lib/ladspa" }, 0, 0, NULL }, 
    { "ladspa_segment_time", PREF_FLOAT, 0,
      { .f = 0.01 }, { .f = 0.01 }, 0.00001, 10000, NULL }, 
    { "max_tracks", PREF_INT, 0, 
      { .i = 16 }, { .i = 16 }, 1, MAX_TRACKS, NULL },
    { "snap_to_grid", PREF_INT, 0,
      { .i = 0 }, { .i = 0 }, 0, 1, NULL },
    { "snap_to_cuepoints", PREF_INT, 0,
      { .i = 0 }, { .i = 0 }, 0, 1, NULL },
    { "show_grid", PREF_INT, 0, 
      { .i = 0 }, { .i = 0 }, 0, 1, NULL },
    { "show_zero", PREF_INT, 0,
      { .i = 0 }, { .i = 0 }, 0, 1, NULL },
    { "show_envelope", PREF_INT, 0, 
      { .i = 0 }, { .i = 0 }, 0, 1, NULL },
    { "record_replace", PREF_INT, 0, 
      { .i = 0 }, { .i = 0 }, 0, 1, NULL },
    { "follow_playback", PREF_INT, 0,
      { .i = 0 }, { .i = 0 }, 0, 1, NULL },
    
    { "invariant_playback_channels", PREF_INT, PREF_WRITEONCE, { .i = 2 }, { .i = 2 }, 1, 32, NULL },
    { "invariant_max_tracks", PREF_INT, PREF_WRITEONCE, { .i = 16 }, { .i = 16 }, 1, 32, NULL },

    { "colors:background", PREF_STRING, 0, { .s           = NULL }, { .s = "0x000000" }, 0, 0, NULL },
    { "colors:point_record", PREF_STRING, 0, { .s         = NULL }, { .s = "0xFF0000" }, 0, 0, NULL },
    { "colors:point_play", PREF_STRING, 0, { .s           = NULL }, { .s = "0x00FF00" }, 0, 0, NULL },
    { "colors:wave", PREF_STRING, 0, { .s                 = NULL }, { .s = "0x00CC44" }, 0, 0, NULL },
    { "colors:wave_lighttone", PREF_STRING, 0, { .s       = NULL }, { .s = "0x00CC44" }, 0, 0, NULL },
    { "colors:wave_darktone", PREF_STRING, 0, { .s        = NULL }, { .s = "0xFF0002" }, 0, 0, NULL },
    { "colors:zero", PREF_STRING, 0, { .s                 = NULL }, { .s = "0x00CC44" }, 0, 0, NULL },
    { "colors:selection_wave", PREF_STRING, 0, { .s       = NULL }, { .s = "0x008833" }, 0, 0, NULL },
    { "colors:selection", PREF_STRING, 0, { .s            = NULL }, { .s = "0x005522" }, 0, 0, NULL },
    { "colors:selection_background", PREF_STRING, 0, { .s = NULL }, { .s = "0xE5E5E5" }, 0, 0, NULL },
    { "colors:block", PREF_STRING, 0, { .s                = NULL }, { .s = "0x005522" }, 0, 0, NULL },
    { "colors:mark", PREF_STRING, 0, { .s                 = NULL }, { .s = "0xFFFF00" }, 0, 0, NULL },
    { "colors:grid", PREF_STRING, 0, { .s                 = NULL }, { .s = "0x5A00AF" }, 0, 0, NULL },
    { "colors:grid_font", PREF_STRING, 0, { .s            = NULL }, { .s = "0xE5E5E5" }, 0, 0, NULL },
    { "colors:info_font", PREF_STRING, 0, { .s            = NULL }, { .s = "0xE5E5E5" }, 0, 0, NULL },
    { "colors:info_font_record", PREF_STRING, 0, { .s     = NULL }, { .s = "0xFF0000" }, 0, 0, NULL },
    { "colors:info_font_play", PREF_STRING, 0, { .s       = NULL }, { .s = "0x00FF00" }, 0, 0, NULL },
    { "colors:marker_slope_main", PREF_STRING, 0, { .s    = NULL }, { .s = "0x6BF0F9" }, 0, 0, NULL },
    { "colors:marker_slope_aux", PREF_STRING, 0, { .s     = NULL }, { .s = "0x489FA5" }, 0, 0, NULL },
    { "colors:marker_text_background", PREF_STRING, 0, { .s = NULL }, { .s = "0xFF0000" }, 0, 0, NULL },
    { "colors:marker_text", PREF_STRING, 0, { .s          = NULL }, { .s = "0xFFFFFF" }, 0, 0, NULL },
    { "colors:toggles_mute", PREF_STRING, 0, { .s          = NULL }, { .s = "0x0000FF" }, 0, 0, NULL },
    { "colors:toggles_solo", PREF_STRING, 0, { .s          = NULL }, { .s = "0xFFFF00" }, 0, 0, NULL },

    { NULL, 0, 0, { .i = 0 }, { .i = 0 }, 0, 0, NULL }
};

struct pref *
pref_find(const char *name) {
    int i, found = 0;
    if(name[0] == '\0')
        return NULL;
    for(i = 0; prefs[i].name; i++) {
        if(prefs[i].name[0] != name[0] &&
           prefs[i].name[1] != name[1])
            continue;
        if(strcmp(prefs[i].name, name) == 0) {
            found = 1;
            break;
        }
    }
    if(!found)
        return NULL;
    return &(prefs[i]);
}

int
pref_get_as_int(const char *name) {
    struct pref *p = pref_find(name);
    if(p)
        return p->value.i;
    FAIL("warning: could not locate int pref %s, returning 0.\n",
         name);
    return 0;
}

float
pref_get_as_float(const char *name) {
    struct pref *p = pref_find(name);
    if(p) 
        return p->value.f;
    FAIL("warning: could not locate float pref %s, returning 0.\n",
         name);
    return 0;
}

const char *
pref_get_as_string(const char *name) {
    struct pref *p = pref_find(name);
    if(p) 
        return p->value.s;
    FAIL("warning: could not locate string pref %s, returning 0.\n",
         name);
    return 0;
}

struct pref *
pref_get(const char *name) {
    return pref_find(name);
}

int 
pref_set_float(const char *name,
               float f) {
    union pref_value v;
    struct pref *p = pref_find(name);
    if(!p) 
        return 0;
    if(p->type != PREF_FLOAT) {
        FAIL("attempt to set float value %f on non-float type %s.\n",
             f, name);
        return 0;
    }
    v.f = f;
    return pref_set(name, &v);
}

int 
pref_set_int(const char *name,
             int i) {
    union pref_value v;
    struct pref *p = pref_find(name);
    if(!p) 
        return 0;
    if(p->type != PREF_INT) {
        FAIL("attempt to set int value %d on non-int type %s.\n",
             i, name);
        return 0;
    }
    v.i = i;
    return pref_set(name, &v);
}

int 
pref_set_string(const char *name,
                const char *s) {
    union pref_value v;
    struct pref *p = pref_find(name);
    if(!p) 
        return 0;
    if(p->type != PREF_STRING) {
        FAIL("attempt to set string value %s on non-string type %s.\n",
             s, name);
        return 0;
    }
    v.s = (char *)s;
    return pref_set(name, &v);
}

int
pref_set(const char *name,
         union pref_value *value) {
    char *s;
    struct pref *p = pref_find(name);
    if(!p) 
        return 0;
    
    if((p->flags & PREF_WRITEONCE) &&
       (p->flags & PREF_HASBEENSET)) {
        FAIL("cannot set write-once value %s.\n",
             p->name);
        return 0;
    }
    
    if(p->callback)
        if(!p->callback(p, value))
            return 0;

    switch(p->type) {
    case PREF_FLOAT:
        if(p->min != p->max) {
            if(value->f < p->min) {
                DEBUG("value for %s clamped to lower bound %f.\n", 
                      name, p->min);
                value->f = p->min;
            }
            if(value->f > p->max) {
                DEBUG("value for %s clamped to upper bound %f.\n", 
                      name, p->max);
                value->f = p->max;
            }
        }
        p->value.f = value->f;
        break;
    case PREF_INT:
        if(p->min != p->max) {
            if(value->i < p->min) {
                DEBUG("value for %s clamped to lower bound %f.\n", 
                      name, p->min);
                value->i = p->min;
            }
            if(value->i > p->max) {
                DEBUG("value for %s clamped to upper bound %f.\n", 
                      name, p->max);
                value->i = p->max;
            }
        }
        p->value.i = value->i;
        break;
    case PREF_STRING:
        if(value->s == NULL) {
            FAIL("NULL value for %s, setting default %s.\n", name, p->def.s);
            s = strdup(p->def.s);
            if(s == NULL) {
                FAIL("default also NULL, not changing value for %s.\n", name);
            } else {
                p->value.s = s;
            }
        } else {
            if(p->value.s)
                free(p->value.s);
            s = strdup(value->s);
            if(s == NULL)
                FAIL("could not set %s, could not copy string %s\n", name, value->s);
            else
                p->value.s = s;
        }
        break;
    }
    p->flags |= PREF_HASBEENSET;
    return 1;
}

int
pref_init() {
    int i, r = 0;
    char path[512], val[256], *s;

    for(i = 0; prefs[i].name; i++) {
        if(strncmp(prefs[i].name, "invariant_", 10) == 0)
            continue;
        switch(prefs[i].type) {
        case PREF_FLOAT:
            snprintf(val, 255, "%f", prefs[i].def.f);
            snprintf(path, 512, "/gnusound/preferences/%s=%s", prefs[i].name, val);
            r |= pref_set_float(prefs[i].name, gnome_config_get_float(path));
            break;
        case PREF_INT:
            snprintf(val, 255, "%d", prefs[i].def.i);
            snprintf(path, 512, "/gnusound/preferences/%s=%s", prefs[i].name, val);
            r |= pref_set_int(prefs[i].name, gnome_config_get_int(path));
            break;
        case PREF_STRING:
            snprintf(val, 255, "%s", prefs[i].def.s);
            snprintf(path, 512, "/gnusound/preferences/%s=%s", prefs[i].name, val);
            s = gnome_config_get_string(path);
            r |= pref_set_string(prefs[i].name, s);
            g_free(s);
            break;
        }
    }

    pref_set_int("invariant_max_tracks", pref_get_as_int("max_tracks"));
    pref_set_int("invariant_playback_channels", pref_get_as_int("playback_channels"));

    return !r;
}

void
pref_sync() {
    int i;
    char path[512];
    for(i = 0; prefs[i].name; i++) {
        if(strncmp(prefs[i].name, "invariant_", 10) == 0)
            continue;
        switch(prefs[i].type) {
        case PREF_FLOAT:
            snprintf(path, 512, "/gnusound/preferences/%s", prefs[i].name);
            gnome_config_set_float(path, prefs[i].value.f);
            break;
        case PREF_INT:
            snprintf(path, 512, "/gnusound/preferences/%s", prefs[i].name);
            gnome_config_set_int(path, prefs[i].value.i);
            break;
        case PREF_STRING:
            snprintf(path, 512, "/gnusound/preferences/%s", prefs[i].name);
            gnome_config_set_string(path, prefs[i].value.s);
            break;
        }
    }
    gnome_config_sync();
}

void
pref_exit() {
    int i;
    pref_sync();
    for(i = 0; prefs[i].name; i++) 
        if(prefs[i].type == PREF_STRING)
            if(prefs[i].value.s)
                free(prefs[i].value.s);
    
}
