/* 
   EOSchemaGeneration.h

   Copyright (C) 2000 Free Software Foundation, Inc.

   Author: Mirko Viviani <mirko.viviani@rccr.cremona.it>
   Date: October 2000

   This file is part of the GNUstep Database Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef __EOSchemaGeneration_h__
#define __EOSchemaGeneration_h__

#include <EOAccess/EOSQLExpression.h>


@interface EOSQLExpression (EOSchemaGeneration)

+ (NSArray *)foreignKeyConstraintStatementsForRelationship:(EORelationship *)relationship;
    // default implementation verifies that relationship joins on foreign key
    // of destination and calls
    // prepareConstraintStatementForRelationship:sourceColumns:destinationColumns:

+ (NSArray *)createTableStatementsForEntityGroup:(NSArray *)entityGroup;
+ (NSArray *)dropTableStatementsForEntityGroup:(NSArray *)entityGroup;
+ (NSArray *)primaryKeyConstraintStatementsForEntityGroup:(NSArray *)entityGroup;
+ (NSArray *)primaryKeySupportStatementsForEntityGroup:(NSArray *)entityGroup;
+ (NSArray *)dropPrimaryKeySupportStatementsForEntityGroup:(NSArray *)entityGroup;
    // These methods generate the SQL statements necessary to implement the
    // schema generation feature for one entityGroup. An entityGroup
    // is an array that contains entities which have the same externalNames.
    // For example, an implementation of the createTableStatementsForEntityGroup:
    // method might return an array containing one EOSQLExpression object that
    // holds the necessary SQL to create a table in an RDBMS.

+ (NSArray *)createTableStatementsForEntityGroups:(NSArray *)entityGroups;
+ (NSArray *)dropTableStatementsForEntityGroups:(NSArray *)entityGroups;
+ (NSArray *)primaryKeyConstraintStatementsForEntityGroups:(NSArray *)entityGroups;
+ (NSArray *)primaryKeySupportStatementsForEntityGroups:(NSArray *)entityGroups;
+ (NSArray *)dropPrimaryKeySupportStatementsForEntityGroups:(NSArray *)entityGroups;
    // These methods generate all of the SQL statements necessary to implement the
    // schema generation feature for a list of entityGroups. They loop through each
    // of the entityGroups and invoke their more primitive method with each one.
    // Subclasses can override these methods and add SQL statements to precede or
    // follow the statements for the individual groups.

+ (void)appendExpression:(EOSQLExpression *)expression toScript:(NSMutableString *)script;
    // Should add expression to executable SQL script.  For Oracle and Informix this
    // adds the trailing semi-colon.  For Sybase "go" is added.

+ (NSString *)schemaCreationScriptForEntities:(NSArray *)entities
				      options:(NSDictionary *)options;
    // Returns a full script suitable to create the schema for the
    // given entities, in a form appropriate for the target database.
    // Options are the same as below.

+ (NSArray *)schemaCreationStatementsForEntities:(NSArray *)entities
					 options:(NSDictionary *)options;
    // Returns an array of EOSQLExpressions suitable to create the schema
    // for the given entities, in a form appropriate for the target database.
    // Options dictionary describes options for creation.  Possible values include:
    //    createTables:            YES/NO [YES]
    //    dropTables:              YES/NO [YES]
    //    createPrimaryKeySupport  YES/NO [YES]
    //    dropPrimaryKeySupport    YES/NO [YES]
    //    primaryKeyConstraints:   YES/NO [YES]
    //    foreignKeyConstraints:   YES/NO [NO]
    //    createDatabase           YES/NO [NO]
    //    dropDatabase             YES/NO [NO]


- (NSString *)columnTypeStringForAttribute:(EOAttribute *)attribute;
    // Assembles an adaptor specific type string suitable for use in a create
    // table statement. Subclassers should override this method for specific behavior.
    // The default implementation does the following:
    // 	    if (precision != 0) typeString generated is "externalType(precision, scale)"
    // 	    else if (width != 0) typeString generated is "externalType(width)"
    //	    else typeString generated is "externalType"

- (NSString *)allowsNullClauseForConstraint:(BOOL)allowsNull;
    // Generates an adaptor specific string for use in a create table statement that
    // indicates whether this column allows null.

- (void)addCreateClauseForAttribute:(EOAttribute *)attribute;
    // Assembles the part of the create table statement for this attribute using the
    // previous two methods.

- (void)prepareConstraintStatementForRelationship:(EORelationship *)relationship
				    sourceColumns:(NSArray *)sourceColumns
			       destinationColumns:(NSArray *)destinationColumns;
    // Assembles an adaptor specific constraint statement for relationship.

+ (NSArray *)createDatabaseStatementsForConnectionDictionary:(NSDictionary *)connectionDictionary administrativeConnectionDictionary:(NSDictionary *)administrativeConnectionDictionary;
    // Generate the SQL Statements that will create a database (or user for
    // Oracle) that can be accessed by the provided connectionDictionary.

+ (NSArray *)dropDatabaseStatementsForConnectionDictionary:(NSDictionary *)connectionDictionary administrativeConnectionDictionary:(NSDictionary *)administrativeConnectionDictionary;
    // Generate the SQL statements to drop the database (or user for Oracle)

+ (EOSQLExpression *)selectStatementForContainerOptions;
    // Generate the SQL SELECT statement that provides a list of available
    // containers for a new database (or user for Oracle).

@end


//
// Keys to use the options dictionary for
//   schemaCreationScriptForEntities:options: and 
//   schemaCreationStatementsForEntities:options:
//

GDL2ACCESS_EXPORT NSString *EOCreateTablesKey;
GDL2ACCESS_EXPORT NSString *EODropTablesKey;
GDL2ACCESS_EXPORT NSString *EOCreatePrimaryKeySupportKey;
GDL2ACCESS_EXPORT NSString *EODropPrimaryKeySupportKey;
GDL2ACCESS_EXPORT NSString *EOPrimaryKeyConstraintsKey;
GDL2ACCESS_EXPORT NSString *EOForeignKeyConstraintsKey;
GDL2ACCESS_EXPORT NSString *EOCreateDatabaseKey;
GDL2ACCESS_EXPORT NSString *EODropDatabaseKey;

#endif
