/* 
   gnuyahoo.c: main initialization file
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <guile/gh.h>
#include <argp.h>
#include <ctype.h>
#include <errno.h>
#include <config.h>
#include <glib.h>

#include "messenger.h"
#include "extension.h"
#include "gy-utils.h"
#include "yahoo-wrapper.h"
#include "interpreter.h"
#include "gnuyahoo.h"


/* private members REALLY. extern'ing these variables are strongly
   discouraged. instead use exported methods to access these variables
*/
const char *argp_program_version = PACKAGE " " VERSION;

const char *argp_program_bug_address =
"<gnuyahoo-bugs@lists.sourceforge.net>";

char *default_login_id;
char *default_password;
char *default_prompt = GY_PROMPT_STRING;
int setup_mode = 0;

typedef void (*sighandler_t) (int);

/* Program documentation. */
static char doc[] = "GNUYahoo! is a freely available GNU messenger for Yahoo. \
GNUYahoo! is a purely console based application which has a READLINE and GUILE \
interfaces.";

/* A description of the arguments we accept. */
static char args_doc[] = " ";

/* The options we understand. */
static struct argp_option options[] = {
  {"user",  'u', "YAHOOID", 0,  "User login account name" },
  {"status", 's', "STATUS", 0,  "Default login status number" },
  { 0 }
};

/* Used by `main' to communicate with `parse_opt'. */
struct arguments
{
  char *args[2];                
};


char *
get_default_login_id (void)
{
  return default_login_id;
}

void
set_default_login_id (char *default_login_id_value)
{
  if (default_login_id == NULL)
    default_login_id = default_login_id_value;
}

char *
get_default_password (void)
{
  return default_password;
}

void
set_default_password (char *default_password_value)
{
  if (default_password == NULL)
    default_password = default_password_value;
}

char *
get_default_prompt (void)
{
  return default_prompt;
}

void
set_default_prompt (char *default_prompt_value)
{
  // if (default_prompt == NULL)
  default_prompt = default_prompt_value;
}

int
get_setup_mode (void)
{
  return setup_mode;
}

void
set_setup_mode (int setup_mode_value)
{
  setup_mode = setup_mode_value;
}

/* Parse a single option. */
static error_t
parse_opt (int key, char *arg, struct argp_state *state)
{

  char *status_str = NULL;
  char *user;
  char status;

  /* Get the INPUT argument from `argp_parse', which we
     know is a pointer to our arguments structure. */
  switch(key)
    {
    case 'u' :
      user = strdup (arg);
      if (!user)
	{
	  fprintf (stderr, "Insufficient memory\n");
	  exit (EXIT_FAILURE);
	}
      set_default_login_id (user);
      break;
      
    case 's':
      status = *arg;
      if(isalpha(status))
	{
	  fprintf (stderr, "Invalid option: --%c\n", status);
	  argp_usage (state);
	}
      
      status_str = strdup (arg);
      if (!status_str) 
	{
	  fputs ("Insufficient memory\n", stderr);
	  exit (EXIT_FAILURE);
	} 
      set_current_status (atoi (status_str));
      break;
      
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

static struct argp argp = { options, parse_opt, args_doc, doc };
  

void
running_for_first_time (void)
{
  struct stat *config_file_stat = NULL;
  int config_file_fd;
  char *config_directory = get_config_directory ();
  char *config_filename = get_config_filename ();
  char *downloads_directory = get_downloads_directory ();

  if (mkdir (config_directory, GY_CONFIG_DIRECTORY_MODE) != 0)
    {
      // mkdir  succeeded
      // means config dir is there already
      return;
    }
  else
    {
      printf ("\nRunning " PACKAGE " for first time!!\n");
      printf ("[%s] created\n", config_directory);
    }

  // see if config file is there
  if (stat (config_filename, config_file_stat) != 0)
    {
      // config file missing, so touch it
      config_file_fd = open (config_filename, O_CREAT | O_RDWR);
      if (config_file_fd == -1)
	{
	  perror ("open[config_file]");
	  exit (EXIT_FAILURE);
	}
      if (fchmod (config_file_fd, GY_CONFIG_DIRECTORY_MODE) != 0)
	{
	  perror ("fchmod[" GY_CONFIG_FILE "]");
	  exit (EXIT_FAILURE);
	}
      if (write (config_file_fd, GY_DEFAULT_CONFIG_CONTENTS, 
		 strlen (GY_DEFAULT_CONFIG_CONTENTS)) <= 0)
	{
	  perror ("write[config_file]");
	  exit (EXIT_FAILURE);
	}
      close (config_file_fd);
      printf ("[%s] created\n\n", config_filename);
      //create downloads directory

      if(mkdir (downloads_directory, 0755) !=0)
	{
	  perror ("creating download directory");
	  exit (EXIT_FAILURE);
	}
    }

  // this is required for delayed, higher level setup procedures
  // that needs to be executed after guile initialization
  set_setup_mode (1);

  free (config_directory);
  free (config_filename);
  free (downloads_directory);
}


void
inner_main (int argc, char **argv)
{
  /* your guile environment is ready :)
     you are free to eval any guile code */

  /* >>> GUILE INITIALIZATION STARTS */

  /* interface our gnu messenger env to guile by hooks and new
     procedures */
  guile_env_init ();

  /* gy guile env is ready. you can now call gnu messenger routines
     from with in scheme */

  /* load and evaluate init.scm from well known path */
  gy_load (GY_INIT_FILE);

  // load and evaluate your config file, which contains scheme code
  gh_eval_file_with_standard_handler (get_config_filename ());

  /* <<< GUILE INITIALIZATION ENDS */


  // ACTUAL MAIN STARTS FROM HERE
  messenger_main (argc, argv);

  /* control can never reach here. messenger_main calls exits
     internally */
}

int
main (int argc, char *argv[])
{
  struct arguments argum;
  struct rlimit resource_limit;
  /* generate core dump on seg-fault */
  resource_limit.rlim_cur =
    resource_limit.rlim_max = RLIM_INFINITY;
  if (setrlimit (RLIMIT_CORE, &resource_limit) != 0)
    {
      perror ("setrlimit");
      exit (EXIT_FAILURE);
    }

  running_for_first_time ();

  get_terminal_attributes ();

  /* command line arguments overrides all default settings, including
     the config file */
  argp_parse (&argp, argc, argv, 0, 0, &argum);

  gh_enter (argc, argv, inner_main);

  /* control can never reach here. gh_enter calls exit internally */
  return 0;
}
