/*
   messenger.c : main file and heart for the messenger

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <guile/gh.h>
#include <config.h>

#include <glib.h>
#include <yahoo2.h>
#include <yahoo2_callbacks.h>

#include "interpreter.h"
#include "gy-utils.h"
#include "gnuyahoo.h"
#include "yahoo-wrapper.h"
#include "extension.h"
#include "messenger.h"

extern int errno;
extern int rl_point;
int auto_insert_mode = 0;
int quit_flag = 0;

/* For libyahoo2 */
char pager_host[] = "scs.yahoo.com";
char pager_port[] = "5050";
char filetransfer_host[] = "filetransfer.msg.yahoo.com";
char filetransfer_port[] = "80";

struct yahoo_session gnuyahoo_session;

static char gnuyahoo_logo[] =
  "  +-----------------------------------------------------+ \n"
  "  |                                                     |+\n"
  "  |                 ...   ...                           ||\n"
  "  |               ..         ..                         ||\n"
  "  |              ..          ..                         ||\n"
  "  |               ...~~`'~~...                          ||\n"
  "  |                '(O##(O).***                         ||\n"
  "  |                  |##......*******......-_           ||\n"
  "  |                  |##......................          ||\n"
  "  |                  ##./ `....GNU Yahoo........        ||\n"
  "  |                 (--)  `.................   ..       ||\n"
  "  |                  ##   `.................     **     ||\n"
  "  |                        .............. .       **    ||\n"
  "  | Thank You for choosing .....    v .. ..        `*   ||\n"
  "  | Free Software          `. ..     ......             ||\n"
  "  | Please tell your        ....      .. ..             ||\n"
  "  | friends about GNU       ....      .....             ||\n"
  "  | <www.gnu.org>           WW WW      WW WW            ||\n"
  "  |                                                     ||\n"
  "  +-----------------------------------------------------+|\n"
  "   +-----------------------------------------------------+\n";

int 
get_auto_insert_mode (void)
{
  return auto_insert_mode;
}

void 
set_auto_insert_mode (int auto_insert_mode_value)
{
  auto_insert_mode = auto_insert_mode_value;
}

int 
get_quit_flag (void)
{
  return quit_flag;
}

void 
set_quit_flag (int quit_flag_value)
{
  quit_flag = quit_flag_value;
}

int 
get_yahoo_session_id ()
{
  return gnuyahoo_session.session_id;
}

struct yahoo_session *
get_gnuyahoo_session ()
{
  return &gnuyahoo_session;
}

/* handle SIGTERM or like signals and do a proper logout */
void 
signal_handler (int signum)
{
  /* set the quit flag for proper logout and freeing of resources */
  PRINTF_MESSAGE ("signal [%d] caught!! better logout safely\n",
		  signum);
  set_quit_flag (1);
}

// heart of gnuyahoo
void 
main_loop ()
{
  fd_set fds;
  quit_flag = 0;
  
  /* register quit handler for SIGTERM */
  if (signal (SIGTERM, signal_handler) == SIG_ERR)
    perror ("signal(SIGTERM)");
  
  while (!quit_flag)
    {
      FD_ZERO (&fds);
      FD_SET (fileno (stdin), &fds);
      FD_SET (gnuyahoo_session.socket_fd, &fds);
      
      while (1)
	{
	  errno = 0;
	  // toggle_session_mode - AUTO-INSERT / VANILLA
	  if (get_session_mode ())
	    {
	      if (auto_insert_mode == 1)
		{
		  if (*(get_current_target_buddy ()) != '\0')
		    {
		      rl_insert_text (get_current_target_buddy ());
		      rl_insert_text (" ");
		      rl_redisplay ();
		    }
		}
	      
	      auto_insert_mode = 0;
	    }
	  
	  if (select (FD_SETSIZE, &fds, NULL, NULL, NULL) < 0)
	    {
	      if (quit_flag)
		{
		  /* go down and see, there is another check for
		     breaking the main loop */
		  break;
		}
	      if (errno == EINTR)
		{
		  /* resuming from interrupt, so not an error situatuon,
		     this generally happens when you suspend your
		     messenger with "C-z" and then "fg". This is allowed "
		  */
		  rl_reset_line_state ();
		  rl_forced_update_display ();
		  
		  continue;
		}
	      perror ("select()");
	      exit (EXIT_FAILURE);
	    }
	  else
	    break;
	}
      
      if (quit_flag)
	{
	  /* this is second check, select might have returned after
	     interrupt due to SIGTERM signal */
	  break;
	}
      
      if (FD_ISSET (fileno (stdin), &fds))
	{
	
// 	  if (get_current_target_buddy ())
// 	    {
// 	      // send typing notifications
// 	      yahoo_send_typing (gnuyahoo_session.session_id,
// 				 get_current_target_buddy (),
// 				 TRUE);
// 	    }
	  rl_callback_read_char ();
	}
      
      if (FD_ISSET (gnuyahoo_session.socket_fd, &fds))
	{
// 	  if (get_current_target_buddy ())
// 	    {
// 	      // send typing-finished notifications
// 	      yahoo_send_typing (gnuyahoo_session.session_id,
// 				 get_current_target_buddy (),
// 				 FALSE);
// 	    }
 	  if (yahoo_read_ready (gnuyahoo_session.session_id, 
				gnuyahoo_session.socket_fd) <= 0)
	    quit_flag = 1;
	}
    }
}


// messenger starts from here
void
messenger_main (int argc, char **argv)
{
  int size = 0;
  char *login_id = get_default_login_id ();
  char *password = get_default_password ();
  
  if (!login_id)
    {
      printf ("Yahoo ID: ");
      if (getline (&login_id, &size, stdin) == -1)
	{
	  perror ("getline()");
	  exit (EXIT_FAILURE);
	}
      // Removing newline character
      login_id[strlen (login_id) - 1] = '\0';
      set_default_login_id (login_id);
    }
  
  if (!password)
    {
      password = (char *) strdup (getpass ("Password: "));
      set_default_password (password);
      
      if (!password)
	{
	  fprintf (stderr, "Insufficient memory\n");
	  exit (EXIT_FAILURE);
	}
    }

  register_callbacks ();
  yahoo_set_log_level(1);
  
  gnuyahoo_session.login_id = login_id;
  gnuyahoo_session.password = password;

  /* nothing set thru --status or 
     (gy-set-default-status! staus-number) */
  if (get_current_status () == -1)
    set_current_status (GY_DEFAULT_STATUS);

  gnuyahoo_session.status = get_current_status ();

  gnuyahoo_session.session_id = 0;
  gnuyahoo_session.socket_fd = 0;
  gnuyahoo_session.refresh_time.tv_sec = 10 * 60; // 10 mins.
  
  printf ("Logging in as [%s] in [%s] mode\n",
	  gnuyahoo_session.login_id,
	  yahoo_status_code (gnuyahoo_session.status));
  fflush (stdin);
  
  gnuyahoo_session.session_id =
    yahoo_login (gnuyahoo_session.login_id, 
		 gnuyahoo_session.password, 
		 gnuyahoo_session.status);
  if (gnuyahoo_session.session_id <= 0)
      exit (EXIT_FAILURE);
  
   rl_callback_handler_install (get_default_prompt (), interpreter);
  rl_attempted_completion_function = (CPPFunction *) complete_text;

  main_loop ();
  
  fprintf (stderr, "\nLogging out ... ");
  yahoo_logoff (gnuyahoo_session.session_id);
  fprintf (stderr, "done\n");
  
  puts (gnuyahoo_logo);
  
  if (login_id)
    free (login_id);
  if (password)
    free (password);
  
  set_terminal_attributes ();
  exit (EXIT_SUCCESS);
}
