package jvn

import (
	"encoding/xml"
	"fmt"
	"io/ioutil"
	"net/http"
	"os"
	"time"

	"github.com/cheggaaa/pb"
	c "github.com/kotakanbe/go-cve-dictionary/config"
	log "github.com/kotakanbe/go-cve-dictionary/log"
	"github.com/kotakanbe/go-cve-dictionary/util"
	"github.com/parnurzeal/gorequest"
)

type rdf struct {
	Items []Item `xml:"item"`
}

// Item ... http://jvndb.jvn.jp/apis/getVulnOverviewList_api.html
type Item struct {
	About       string       `xml:"about,attr"`
	Title       string       `xml:"title"`
	Link        string       `xml:"link"`
	Description string       `xml:"description"`
	Publisher   string       `xml:"publisher"`
	Identifier  string       `xml:"identifier"`
	References  []references `xml:"references"`
	Cpes        []cpe        `xml:"cpe"`
	Cvsses      []Cvss       `xml:"cvss"`
	Date        string       `xml:"date"`
	Issued      string       `xml:"issued"`
	Modified    string       `xml:"modified"`
}

type cpe struct {
	Version string `xml:"version,attr"` // cpe:/a:mysql:mysql
	Vendor  string `xml:"vendor,attr"`
	Product string `xml:"product,attr"`
	Value   string `xml:",chardata"`
}

type references struct {
	ID     string `xml:"id,attr"`
	Source string `xml:"source,attr"`
	Title  string `xml:"title,attr"`
	URL    string `xml:",chardata"`
}

// Cvss ... CVSS
type Cvss struct {
	Score    string `xml:"score,attr"`
	Severity string `xml:"severity,attr"`
	Vector   string `xml:"vector,attr"`
	Version  string `xml:"version,attr"`
}

// FetchJvn fetches vulnerability information from JVN
func FetchJvn(years []int) (items []Item, err error) {
	urls := makeJvnURLs(years)
	items, err = fetchJvnConcurrently(urls)
	if err != nil {
		return items, fmt.Errorf(
			"Failed to fetch cve data from JVN. err: %s", err)
	}
	return
}

func makeJvnURLs(years []int) (urls []string) {
	latestFeeds := []string{
		"http://jvndb.jvn.jp/ja/rss/jvndb_new.rdf",
		"http://jvndb.jvn.jp/ja/rss/jvndb.rdf",
	}

	if len(years) == 0 {
		return latestFeeds
	}

	urlFormat := "http://jvndb.jvn.jp/ja/rss/years/jvndb_%d.rdf"
	for _, year := range years {
		urls = append(urls, fmt.Sprintf(urlFormat, year))

		thisYear := time.Now().Year()
		if year == thisYear {
			urls = append(urls, latestFeeds...)
		}
	}
	return
}

func fetchJvnConcurrently(urls []string) (allItems []Item, err error) {
	reqChan := make(chan string, len(urls))
	resChan := make(chan []Item, len(urls))
	errChan := make(chan error, len(urls))
	defer close(reqChan)
	defer close(resChan)
	defer close(errChan)

	go func() {
		for _, url := range urls {
			reqChan <- url
		}
	}()

	concurrency := len(urls)
	tasks := util.GenWorkers(concurrency)
	for range urls {
		tasks <- func() {
			select {
			case url := <-reqChan:
				log.Infof("Fetching... %s", url)
				items, err := fetchJvn(url)
				if err != nil {
					errChan <- err
					return
				}
				resChan <- items
			}
		}
	}

	errs := []error{}
	bar := pb.New(len(urls))
	if c.Conf.Quiet {
		bar.Output = ioutil.Discard
	} else {
		bar.Output = os.Stderr
	}
	bar.Start()
	timeout := time.After(10 * 60 * time.Second)
	for range urls {
		select {
		case items := <-resChan:
			allItems = append(allItems, items...)
		case err := <-errChan:
			errs = append(errs, err)
		case <-timeout:
			return allItems, fmt.Errorf("Timeout Fetching Jvn")
		}
		bar.Increment()
	}
	bar.Finish()
	//  bar.FinishPrint("Finished to fetch CVE information from JVN.")
	if 0 < len(errs) {
		return allItems, fmt.Errorf("%s", errs)
	}
	return allItems, nil
}

func fetchJvn(url string) (items []Item, err error) {
	var body string
	var errs []error
	var resp *http.Response

	resp, body, errs = gorequest.New().Proxy(c.Conf.HTTPProxy).Get(url).End()
	if len(errs) > 0 || resp == nil || resp.StatusCode != 200 {
		return items, fmt.Errorf(
			"HTTP error. errs: %v, url: %s", errs, url)
	}

	var rdf rdf
	if err = xml.Unmarshal([]byte(body), &rdf); err != nil {
		return items, fmt.Errorf(
			"Failed to unmarshal. url: %s, err: %s", url, err)
	}
	items = append(items, rdf.Items...)
	return
}
