indexing

	description:

		"Sparse tables, implemented with arrays. Ancestor of hash tables %
		%which should supply their hashing mechanisms."

	library: "Gobo Eiffel Structure Library"
	copyright: "Copyright (c) 2000-2003, Eric Bezault and others"
	license: "Eiffel Forum License v2 (see forum.txt)"
	date: "$Date: 2003/06/01 16:50:21 $"
	revision: "$Revision: 1.22 $"

deferred class DS_SPARSE_TABLE [G, K]

inherit

	DS_TABLE [G, K]
		rename
			put as force,
			put_new as force_new
		end

	DS_SPARSE_CONTAINER [G, K]
		rename
			make as make_sparse_container,
			has as has_item
		redefine
			search, new_cursor
		end

feature {NONE} -- Initialization

	make (n: INTEGER) is
			-- Create an empty table and allocate
			-- memory space for at least `n' items.
			-- Use `=' as comparison criterion for items.
			-- Use `equal' as comparison criterion for keys.
		require
			positive_n: n >= 0
		do
			create key_equality_tester
			make_with_equality_testers (n, Void, key_equality_tester)
		ensure
			empty: is_empty
			capacity_set: capacity = n
			before: before
		end

	make_equal (n: INTEGER) is
			-- Create an empty table and allocate
			-- memory space for at least `n' items.
			-- Use `equal' as comparison criterion for items.
			-- Use `equal' as comparison criterion for keys.
		require
			positive_n: n >= 0
		do
			create equality_tester
			create key_equality_tester
			make_with_equality_testers (n, equality_tester, key_equality_tester)
		ensure
			empty: is_empty
			capacity_set: capacity = n
			before: before
		end

	make_default is
			-- Create an empty table and allocate memory
			-- space for at least `default_capacity' items.
			-- Use `=' as comparison criterion for items.
			-- Use `equal' as comparison criterion for keys.
		do
			make (default_capacity)
		ensure then
			before: before
		end

	make_map (n: INTEGER) is
			-- Create an empty table and allocate
			-- memory space for at least `n' items.
			-- Use `=' as comparison criterion for items.
			-- Use `=' as comparison criterion for keys.
		require
			positive_n: n >= 0
		do
			make_with_equality_testers (n, Void, Void)
		ensure
			empty: is_empty
			capacity_set: capacity = n
			before: before
		end

	make_map_equal (n: INTEGER) is
			-- Create an empty table and allocate
			-- memory space for at least `n' items.
			-- Use `equal' as comparison criterion for items.
			-- Use `=' as comparison criterion for keys.
		require
			positive_n: n >= 0
		do
			create equality_tester
			make_with_equality_testers (n, equality_tester, Void)
		ensure
			empty: is_empty
			capacity_set: capacity = n
			before: before
		end

	make_map_default is
			-- Create an empty table and allocate memory
			-- space for at least `default_capacity' items.
			-- Use `=' as comparison criterion for items.
			-- Use `=' as comparison criterion for keys.
		do
			make_map (default_capacity)
		ensure
			empty: is_empty
			capacity_set: capacity = default_capacity
			before: before
		end

	make_with_equality_testers (n: INTEGER;
		an_item_tester: like equality_tester;
		a_key_tester: like key_equality_tester) is
			-- Create an empty table and allocate
			-- memory space for at least `n' items.
			-- Use `an_item_tester' as comparison criterion for items.
			-- Use `a_key_tester' as comparison criterion for keys.
		require
			positive_n: n >= 0
		do
			equality_tester := an_item_tester
			key_equality_tester := a_key_tester
			make_sparse_container (n)
		ensure
			empty: is_empty
			capacity_set: capacity = n
			before: before
			equality_tester_set: equality_tester = an_item_tester
			key_equality_tester_set: key_equality_tester = a_key_tester
		end

feature -- Access

	infix "@", item (k: K): G is
			-- Item associated with `k'
		do
			search_position (k)
			check hash_k: position /= No_position end
			Result := items_item (position)
		end

	key (k: K): K is
			-- Key associated with `k'
		require
			has_k: has (k)
		do
			search_position (k)
			check hash_k: position /= No_position end
			Result := keys_item (position)
		end

	found_key: K is
			-- Key of item found by last call to `search'
		require
			key_found: found
		do
			Result := keys_item (found_position)
		end

	key_for_iteration: K is
			-- Key at internal cursor position
		require
			not_off: not off
		do
			Result := cursor_key (internal_cursor)
		end

	new_cursor: DS_SPARSE_TABLE_CURSOR [G, K] is
			-- New external cursor for traversal
		do
			create Result.make (Current)
		end

	key_equality_tester: KL_EQUALITY_TESTER [K]
			-- Equality tester for keys;
			-- A void equality tester means that `='
			-- will be used as comparison criterion.

feature -- Status report

	has (k: K): BOOLEAN is
			-- Is there an item associated with `k'?
		do
			search_position (k)
			Result := position /= No_position
		end

	valid_key (k: K): BOOLEAN is
			-- Is `k' a valid key?
		do
			Result := True
		ensure then
			defintion: Result = True
		end

	key_equality_tester_settable (a_tester: like key_equality_tester): BOOLEAN is
			-- Can `set_key_equality_tester' be called with `a_tester'
			-- as argument in current state of container?
		do
			Result := is_empty
		end

feature -- Search

	search (k: K) is
			-- Search for item at key `k'.
			-- If found, set `found' to true, and set
			-- `found_item' to item associated with `k'.
		do
			search_position (k)
			found_position := position
		ensure then
			found_set: found = has (k)
			found_item_set: found implies (found_item = item (k))
		end

feature -- Comparison

	is_equal (other: like Current): BOOLEAN is
			-- Is table equal to `other'?
			-- Do not take cursor positions, capacity
			-- nor `equality_tester' into account.
		local
			a_key: K
			i: INTEGER
		do
			if Current = other then
				Result := True
			elseif same_type (other) and count = other.count then
				from
					i := last_position
					Result := True
				until
					not Result or i < 1
				loop
					if clashes_item (i) > Free_watermark then
						a_key := keys_item (i)
						Result := other.has (a_key) and then
							other.item (a_key) = items_item (i)
					end
					i := i - 1
				end
			end
		end

feature -- Setting

	set_key_equality_tester (a_tester: like key_equality_tester) is
			-- Set `key_equality_tester' to `a_tester'.
			-- A void key equality tester means that `='
			-- will be used as comparison criterion.
		require
			key_equality_tester_settable: key_equality_tester_settable (a_tester)
		do
			key_equality_tester := a_tester
		ensure
			key_equality_tester_set: key_equality_tester = a_tester
		end

feature -- Element change

	replace (v: G; k: K) is
			-- Replace item associated with `k' by `v'.
			-- Do not move cursors.
		do
			unset_found_item
			search_position (k)
			check has_k: position /= No_position end
			items_put (v, position)
		end

	replace_found_item (v: G) is
			-- Replace item associated with
			-- the key of `found_item' by `v'.
			-- Do not move cursors.
		require
			item_found: found
		do
			items_put (v, found_position)
		ensure
			replaced: found_item = v
			same_count: count = old count
		end

	put (v: G; k: K) is
			-- Associate `v' with key `k'.
			-- Do not move cursors.
		require
			not_full: not is_full
		local
			i, h: INTEGER
		do
			unset_found_item
			search_position (k)
			if position /= No_position then
				items_put (v, position)
			else
				i := free_slot
				if i = No_position then
					last_position := last_position + 1
					i := last_position
				else
					free_slot := Free_offset - clashes_item (i)
				end
				h := slots_position
				clashes_put (slots_item (h), i)
				slots_put (i, h)
				items_put (v, i)
				keys_put (k, i)
				count := count + 1
			end
		ensure
			same_count: (old has (k)) implies (count = old count)
			one_more: (not old has (k)) implies (count = old count + 1)
			inserted: has (k) and then item (k) = v
		end

	put_new (v: G; k: K) is
			-- Associate `v' with key `k'.
			-- Do not move cursors.
		require
			not_full: not is_full
			new_item: not has (k)
		local
			i, h: INTEGER
		do
			unset_found_item
			i := free_slot
			if i = No_position then
				last_position := last_position + 1
				i := last_position
			else
				free_slot := Free_offset - clashes_item (i)
			end
			h := hash_position (k)
			clashes_put (slots_item (h), i)
			slots_put (i, h)
			items_put (v, i)
			keys_put (k, i)
			count := count + 1
		ensure
			one_more: count = old count + 1
			inserted: has (k) and then item (k) = v
		end

	put_last (v: G; k: K) is
			-- Associate `v' with key `k'. Put `v' at the end of table
			-- if no item was already associated with `k', or replace
			-- existing item otherwise.
			-- Do not move cursors.
		require
			not_full: not is_full
		local
			i, h: INTEGER
		do
			unset_found_item
			search_position (k)
			if position /= No_position then
				items_put (v, position)
			else
				i := last_position + 1
				if i > capacity then
					compress
					i := last_position + 1
				end
				h := slots_position
				clashes_put (slots_item (h), i)
				slots_put (i, h)
				items_put (v, i)
				keys_put (k, i)
				last_position := i
				count := count + 1
			end
		ensure
			same_count: (old has (k)) implies (count = old count)
			one_more: (not old has (k)) implies (count = old count + 1)
			inserted: has (k) and then item (k) = v
			last: (not old has (k)) implies last = v
		end

	force (v: G; k: K) is
			-- Associate `v' with key `k'.
			-- Resize table if necessary.
			-- Do not move cursors.
		local
			i, h: INTEGER
		do
			unset_found_item
			search_position (k)
			if position /= No_position then
				items_put (v, position)
			else
				if count = capacity then
					resize (new_capacity (count + 1))
					h := hash_position (k)
				else
					h := slots_position
				end
				i := free_slot
				if i = No_position then
					last_position := last_position + 1
					i := last_position
				else
					free_slot := Free_offset - clashes_item (i)
				end
				clashes_put (slots_item (h), i)
				slots_put (i, h)
				items_put (v, i)
				keys_put (k, i)
				count := count + 1
			end
		end

	force_new (v: G; k: K) is
			-- Associate `v' with key `k'.
			-- Resize table if necessary.
			-- Do not move cursors.
		local
			i, h: INTEGER
		do
			unset_found_item
			if count = capacity then
				resize (new_capacity (count + 1))
			end
			i := free_slot
			if i = No_position then
				last_position := last_position + 1
				i := last_position
			else
				free_slot := Free_offset - clashes_item (i)
			end
			h := hash_position (k)
			clashes_put (slots_item (h), i)
			slots_put (i, h)
			items_put (v, i)
			keys_put (k, i)
			count := count + 1
		end

	force_last (v: G; k: K) is
			-- Associate `v' with key `k'. Put `v' at the end of table
			-- if no item was already associated with `k', or replace
			-- existing item otherwise.
			-- Resize table if necessary.
			-- Do not move cursors.
		local
			i, h: INTEGER
		do
			unset_found_item
			search_position (k)
			if position /= No_position then
				items_put (v, position)
			else
				i := last_position + 1
				if i > capacity then
					resize (new_capacity (i))
					h := hash_position (k)
				else
					h := slots_position
				end
				clashes_put (slots_item (h), i)
				slots_put (i, h)
				items_put (v, i)
				keys_put (k, i)
				last_position := i
				count := count + 1
			end
		ensure
			same_count: (old has (k)) implies (count = old count)
			one_more: (not old has (k)) implies (count = old count + 1)
			inserted: has (k) and then item (k) = v
			last: (not old has (k)) implies last = v
		end

feature {DS_SPARSE_TABLE_CURSOR} -- Cursor implementation

	cursor_key (a_cursor: like new_cursor): K is
			-- Key at `a_cursor' position
		require
			a_cursor_not_void: a_cursor /= Void
			a_cursor_valid: valid_cursor (a_cursor)
			a_cursor_not_off: not cursor_off (a_cursor)
		do
			Result := keys_item (a_cursor.position)
		end

end
