/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasellipse.c - ellipse item.
 */

/**
 * SECTION:goocanvasellipse
 * @Title: GooCanvasEllipse
 * @Short_Description: an ellipse item.
 *
 * GooCanvasEllipse represents an ellipse item.
 *
 * It is a subclass of #GooCanvasItemSimple and so inherits all of the style
 * properties such as "stroke-color", "fill-color" and "line-width".
 *
 * It also implements the #GooCanvasItem interface, so you can use the
 * #GooCanvasItem functions such as goo_canvas_item_raise() and
 * goo_canvas_item_rotate().
 *
 * To create a #GooCanvasEllipse use goo_canvas_ellipse_new().
 *
 * To get or set the properties of an existing #GooCanvasEllipse, use
 * g_object_get() and g_object_set().
 *
 * To respond to events such as mouse clicks on the ellipse you must connect
 * to the signal handlers of the corresponding #GooCanvasEllipseView objects.
 * (See goo_canvas_view_get_item_view() and #GooCanvasView::item-view-created.)
 */
#include <config.h>
#include <math.h>
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include "goocanvasellipse.h"
#include "goocanvasellipseview.h"


enum {
  PROP_0,

  PROP_CENTER_X,
  PROP_CENTER_Y,
  PROP_RADIUS_X,
  PROP_RADIUS_Y
};


static void item_interface_init (GooCanvasItemIface *iface);
static void goo_canvas_ellipse_get_property (GObject            *object,
					     guint               param_id,
					     GValue             *value,
					     GParamSpec         *pspec);
static void goo_canvas_ellipse_set_property (GObject            *object,
					     guint               param_id,
					     const GValue       *value,
					     GParamSpec         *pspec);

G_DEFINE_TYPE_WITH_CODE (GooCanvasEllipse, goo_canvas_ellipse,
			 GOO_TYPE_CANVAS_ITEM_SIMPLE,
			 G_IMPLEMENT_INTERFACE (GOO_TYPE_CANVAS_ITEM,
						item_interface_init))


static void
goo_canvas_ellipse_class_init (GooCanvasEllipseClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass*) klass;

  gobject_class->get_property = goo_canvas_ellipse_get_property;
  gobject_class->set_property = goo_canvas_ellipse_set_property;

  g_object_class_install_property (gobject_class, PROP_CENTER_X,
				   g_param_spec_double ("center-x",
							_("Center X"),
							_("The x coordinate of the center of the ellipse"),
							-G_MAXDOUBLE,
							G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_CENTER_Y,
				   g_param_spec_double ("center-y",
							_("Center Y"),
							_("The y coordinate of the center of the ellipse"),
							-G_MAXDOUBLE,
							G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_RADIUS_X,
				   g_param_spec_double ("radius-x",
							_("Radius X"),
							_("The horizontal radius of the ellipse"),
							0.0, G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_RADIUS_Y,
				   g_param_spec_double ("radius-y",
							_("Radius Y"),
							_("The vertical radius of the ellipse"),
							0.0, G_MAXDOUBLE, 0.0,
							G_PARAM_READWRITE));
}


static void
goo_canvas_ellipse_init (GooCanvasEllipse *ellipse)
{

}


/**
 * goo_canvas_ellipse_new:
 * @parent: the parent item, or %NULL. If a parent is specified, it will assume
 *  ownership of the item, and the item will automatically be freed when it is
 *  removed from the parent. Otherwise call g_object_unref() to free it.
 * @center_x: the x coordinate of the center of the ellipse.
 * @center_y: the y coordinate of the center of the ellipse.
 * @radius_x: the horizontal radius of the ellipse.
 * @radius_y: the vertical radius of the ellipse.
 * @first_property: the name of the first property to set, or %NULL.
 * @...: the remaining property names and values to set, terminated with a
 *  %NULL.
 * 
 * Creates a new ellipse item.
 *
 * <!--PARAMETERS-->
 *
 * Here's an example showing how to create an ellipse centered at (100.0,
 * 100.0), with a horizontal radius of 50.0 and a vertical radius of 30.0.
 * It is drawn with a red outline with a width of 5.0 and filled with blue:
 *
 * <informalexample><programlisting>
 *  GooCanvasItem *ellipse = goo_canvas_ellipse_new (mygroup, 100.0, 100.0, 50.0, 30.0,
 *                                                   "stroke-color", "red",
 *                                                   "line-width", 5.0,
 *                                                   "fill-color", "blue",
 *                                                   NULL);
 * </programlisting></informalexample>
 * 
 * Returns: a new ellipse item.
 **/
GooCanvasItem*
goo_canvas_ellipse_new (GooCanvasItem *parent,
			gdouble        center_x,
			gdouble        center_y,
			gdouble        radius_x,
			gdouble        radius_y,
			const gchar   *first_property,
			...)
{
  GooCanvasItem *item;
  GooCanvasEllipse *ellipse;
  va_list args;

  item = g_object_new (GOO_TYPE_CANVAS_ELLIPSE, NULL);
  ellipse = GOO_CANVAS_ELLIPSE (item);

  ellipse->center_x = center_x;
  ellipse->center_y = center_y;
  ellipse->radius_x = radius_x;
  ellipse->radius_y = radius_y;

  va_start (args, first_property);
  g_object_set_valist (G_OBJECT (item), first_property, args);
  va_end (args);

  if (parent)
    {
      goo_canvas_item_add_child (parent, item, -1);
      g_object_unref (item);
    }

  return item;
}


static void
goo_canvas_ellipse_get_property (GObject              *object,
				 guint                 prop_id,
				 GValue               *value,
				 GParamSpec           *pspec)
{
  GooCanvasEllipse *ellipse = (GooCanvasEllipse*) object;

  switch (prop_id)
    {
    case PROP_CENTER_X:
      g_value_set_double (value, ellipse->center_x);
      break;
    case PROP_CENTER_Y:
      g_value_set_double (value, ellipse->center_y);
      break;
    case PROP_RADIUS_X:
      g_value_set_double (value, ellipse->radius_x);
      break;
    case PROP_RADIUS_Y:
      g_value_set_double (value, ellipse->radius_y);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}


static void
goo_canvas_ellipse_set_property (GObject              *object,
				 guint                 prop_id,
				 const GValue         *value,
				 GParamSpec           *pspec)
{
  GooCanvasEllipse *ellipse = (GooCanvasEllipse*) object;

  switch (prop_id)
    {
    case PROP_CENTER_X:
      ellipse->center_x = g_value_get_double (value);
      break;
    case PROP_CENTER_Y:
      ellipse->center_y = g_value_get_double (value);
      break;
    case PROP_RADIUS_X:
      ellipse->radius_x = g_value_get_double (value);
      break;
    case PROP_RADIUS_Y:
      ellipse->radius_y = g_value_get_double (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }

  g_signal_emit_by_name (ellipse, "changed", TRUE);
}


static GooCanvasItemView*
goo_canvas_ellipse_create_view (GooCanvasItem     *item,
				GooCanvasView     *canvas_view,
				GooCanvasItemView *parent_view)
{
  return goo_canvas_ellipse_view_new (canvas_view, parent_view,
				      (GooCanvasEllipse*) item);
}


static void
item_interface_init (GooCanvasItemIface *iface)
{
  iface->create_view = goo_canvas_ellipse_create_view;
}


