<?php
/*
  This code is part of GOsa (https://gosa.gonicus.de)
  Copyright (C) 2003  Cajus Pollmeier

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*! \brief   The plugin base class
    \author  Cajus Pollmeier <pollmeier@gonicus.de>
    \version 2.00
    \date    24.07.2003

    This is the base class for all plugins. It can be used standalone or
    can be included by the tabs class. All management should be done 
    within this class. Extend your plugins from this class.
 */

class plugin
{
  /*!
      \brief Reference to parent object

      This variable is used when the plugin is included in tabs
      and keeps reference to the tab class. Communication to other
      tabs is possible by 'name'. So the 'fax' plugin can ask the
      'userinfo' plugin for the fax number.

      \sa tab
  */
  var $parent= NULL;

  /*!
     \brief Configuration container

     Access to global configuration
  */
  var $config= NULL;

  /*!
      \brief Mark plugin as account
      
      Defines whether this plugin is defined as an account or not.
      This has consequences for the plugin to be saved from tab
      mode. If it is set to 'FALSE' the tab will call the delete
      function, else the save function. Should be set to 'TRUE' if
      the construtor detects a valid LDAP object.

      \sa plugin::plugin()
  */
  var $is_account= FALSE;

  /*!
     \brief Mark plugin as template
     
     Defines whether we are creating a template or a normal object.
     Has conseqences on the way execute() shows the formular and how
     save() puts the data to LDAP.

     \sa plugin::save() plugin::execute()
  */
  var $is_template= FALSE;

  /*!
     \brief Represent temporary LDAP data

     This is only used internally.
  */
  var $attrs= array();

  /* Sort ranges for most list dialogs */
  var $sorting= "all";
  var $sortranges= array( "all" => "/^.*/",
			"0-9" => "/^[0-9]/",
			"a-f" => "/^[a-f]/i",
			"g-l" => "/^[g-l]/i",
			"m-s" => "/^[m-s]/i",
			"t-z" => "/^[t-z]/i");

  /*!
     \brief Used standard values

     dn
  */
  var $dn= "";
  var $uid= "";
  var $sn= "";
  var $givenName= "";
  var $acl= "*none*";
  var $dialog= FALSE;

  /* attribute list for save action */
  var $attributes= array();
  var $objectclasses= array();
  var $new= TRUE;

  /*! \brief plugin constructor
 
      If 'dn' is set, the node loads the given 'dn' from LDAP

     \param dn Distinguished name to initialize plugin from
     \sa plugin()
  */
  function plugin ($config, $dn= NULL)
  {
	/* Configuration is fine, allways */
	$this->config= $config;	
  	$this->dn= $dn;

	/* Handle new accounts, don't read information from LDAP */
	if ($dn == "new"){
		return;
	}

	/* Get LDAP descriptor */
	$ldap= $this->config->get_ldap_link();
	if ($dn != NULL){

	  	/* Load data to 'attrs' and save 'dn' */
	  	$ldap->cat ($dn);
	  	$this->attrs= $ldap->fetch();
	
	  	/* Convert all attributs from utf8 */
	  	foreach ($this->attributes as $val){
	  		if (isset($this->attrs["$val"][0])){
	  			$this->$val= utf8_decode($this->attrs["$val"][0]);
	  		}
	  	}

		/* Set the template flag according to the existence of objectClass
		   gosaUserTemplate */
		if (isset($this->attrs['objectClass'])){
			if (in_array ("gosaUserTemplate", $this->attrs['objectClass'])){
				$this->is_template= TRUE;
				@DEBUG (DEBUG_TRACE, __LINE__, __FUNCTION__, __FILE__,
					"found", "Template check");
			}
		}

		/* Is Account? */
		error_reporting(0);
		$found= TRUE;
		foreach ($this->objectclasses as $obj){
			if (!in_array ($obj, $this->attrs['objectClass'])){
				$found= FALSE;
				break;
			}
		}
		error_reporting(E_ALL);
		if ($found){
			$this->is_account= TRUE;
			@DEBUG (DEBUG_TRACE, __LINE__, __FUNCTION__, __FILE__,
				"found", "Object check");
		}
	}
  }

  /*! \brief execute plugin
 
      Generates the html output for this node
  */
  function execute()
  {
	/* Do we represent a valid account? */
	if (!$this->is_account){
		echo "<img src=\"images/stop.png\" align=center>&nbsp;<b>".
		_("This 'dn' has no account extensions.")."</b>";
		return;
	}

	/* Show dummy message */
	echo _("This is an empty plugin.");
  }

  /* remove object from parent */
  function remove_from_parent()
  {
	/* include global link_info */
	$ldap= $this->config->get_ldap_link();

	/* Get current objectClasses in order to add the required ones */
	$ldap->cat($this->dn);
	$tmp= $ldap->fetch ();
	if (isset($tmp['objectClass'])){
		$oc= $tmp['objectClass'];
	} else {
		$oc= array("count" => 0);
	}

	/* Remove objectClasses from entry */
	$ldap->cd($this->dn);
	$this->attrs= array();
	$this->attrs['objectClass']= array();
	for ($i= 0; $i<$oc["count"]; $i++){
		if (!in_array($oc[$i], $this->objectclasses)){
			$this->attrs['objectClass'][]= $oc[$i];
		}
	}

	/* Unset attributes from entry */
	foreach ($this->attributes as $val){
		$this->attrs["$val"]= array();
	}

	/* Unset account info */
	$this->is_account= FALSE;

	/* Do not write in plugin base class, this must be done by
	   children, since there are normally additional attribs,
	   lists, etc. */
	/*
	$ldap->modify($this->attrs);
	*/
  }


  /* Save data to object */
  function save_object()
  {
	/* Save values to object */
	foreach ($this->attributes as $val){
		if (chkacl ($this->acl, "$val") == "" && isset ($_POST["$val"])){
			$this->$val= stripcslashes($_POST["$val"]);
		}
	}
  }


  /* Save data to LDAP, depending on is_account we save or delete */
  function save()
  {
	/* include global link_info */
	$ldap= $this->config->get_ldap_link();

	/* Start with empty array */
	$this->attrs= array();

	/* Get current objectClasses in order to add the required ones */
	$ldap->cat($this->dn);
	$tmp= $ldap->fetch ();
	if (isset($tmp['objectClass'])){
		$oc= $tmp["objectClass"];
		$this->new= FALSE;
	} else {
		$oc= array("count" => 0);
		$this->new= TRUE;
	}

	/* Load (minimum) attributes, add missing ones */
	$this->attrs['objectClass']= $this->objectclasses;
	for ($i= 0; $i<$oc["count"]; $i++){
		if (!in_array($oc[$i], $this->objectclasses)){
			$this->attrs['objectClass'][]= $oc[$i];
		}
	}

	/* Copy standard attributes */
	foreach ($this->attributes as $val){
		if ($this->$val != ""){
			$this->attrs["$val"]= utf8_encode($this->$val);
		} elseif (!$this->new) {
			$this->attrs["$val"]= array();
		}
	}

	/* Do not write in plugin base class, this must be done by
	   children, since there are normally additional attribs,
	   lists, etc. */
	/*
	$ldap->modify($this->attrs);
	*/
  }

  /* Check formular input */
  function check()
  {
	$message= array();
	return ($message);
  }

  /* Adapt from template, using 'dn' */
  function adapt_from_template($dn)
  {
	/* Include global link_info */
	$ldap= $this->config->get_ldap_link();

	/* Load requested 'dn' to 'attrs' */
	$ldap->cat ($dn);
	$this->attrs= $ldap->fetch();

	/* Walk through attributes */
	foreach ($this->attributes as $val){

		if (isset($this->attrs["$val"][0])){

			/* If attribute is set, decode from utf8 and replace
			   dynamic parts: %sn, %givenName and %uid. Fill these
			   in our local variables. */
			$value= utf8_decode($this->attrs["$val"][0]);

			foreach (array("sn", "givenName", "uid") as $repl){
				if (preg_match("/%$repl/i", $value)){
					$value= preg_replace ("/%$repl/i", $this->parent->$repl, $value);
				}
			}
			$this->$val= $value;
		}
	}

	/* Is Account? */
	$found= TRUE;
	foreach ($this->objectclasses as $obj){
		if (!in_array_ics ($obj, $this->attrs['objectClass'])){
			$found= FALSE;
			break;
		}
	}
	if ($found){
		$this->is_account= TRUE;
	}
  }

  /* Indicate whether a password change is needed or not */
  function password_change_needed()
  {
	return FALSE;
  }

  /* Check sorting variable */
  function check_sortranges()
  {
	if (isset($_GET['sorting'])){
		$this->sorting= $_GET['sorting'];
		if (!array_key_exists($this->sorting, $this->sortranges)){
			$this->sorting= "all";
		}
	}
  }

  /* Show header message for tab dialogs */
  function show_header($button_text, $text, $disabled= FALSE)
  {
	if ($disabled == TRUE){
		$state= "disabled";
	} else {
		$state= "";
	}
	echo "<table width=100%><tr>\n<td colspan=2><p><b>$text</b></p>\n";
	echo "<input type=submit value=\"$button_text\" name=\"modify_state\" ".
		chkacl($this->acl, "all")." ".$state.
		"><p class=\"seperator\">&nbsp;</p></td></tr></table>";
  }

  function postcreate()
  {
	/* Find postcreate entries for this class */
	$command= $this->search($this->config->data['MENU'], get_class($this), "POSTCREATE");
	if ($command == ""){
		$command= $this->search($this->config->data['SERVICE'], get_class($this), "POSTCREATE");
	}

	if ($command != ""){
		/* Walk through attribute list */
		foreach ($this->attributes as $attr){
			$command= preg_replace("/%$attr/", $this->$attr, $command);
		}
		if (check_command($command)){
			@DEBUG (DEBUG_SHELL, __LINE__, __FUNCTION__, __FILE__,
				$command, "Execute");

			shell_exec($command);
		} else {
			$message= sprintf(_("Command '%s', specified as POSTCREATE for plugin '%s' doesn't seem to exist."), $command, get_class($this));
			print_red ($message);
		}
	}
  }

  function postremove()
  {
	/* Find postremove entries for this class */
	$command= $this->search($this->config->data['MENU'], get_class($this), "POSTREMOVE");
	if ($command == ""){
		$command= $this->search($this->config->data['SERVICE'], get_class($this), "POSTCREATE");
	}

	if ($command != ""){
		/* Walk through attribute list */
		foreach ($this->attributes as $attr){
			$command= preg_replace("/%$attr/", $this->$attr, $command);
		}
		if (check_command($command)){
			@DEBUG (DEBUG_SHELL, __LINE__, __FUNCTION__, __FILE__,
				$command, "Execute");

			shell_exec($command);
		} else {
			$message= sprintf(_("Command '%s', specified as POSTREMOVE for plugin '%s' doesn't seem to exist."), $command, get_class($this));
			print_red ($message);
		}
	}
  }

  function search($arr, $name, $return)
  {
	if (is_array($arr)){
		foreach ($arr as $a){
			if (isset($a['CLASS']) &&
				strtolower($a['CLASS']) == strtolower($name)){

				if (isset($a[$return])){
					return ($a[$return]);
				} else {
					return ("");
				}
			} else {
				$res= $this->search ($a, $name, $return);
				if ($res != ""){
					return $res;
				}
			}
		}
	}
	return ("");
  }

}

?>
