%{
/*
Copyright (C) 2000-2013  The PARI group.

This file is part of the GP2C package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.*/

#include "config.h"
#include <unistd.h>
#include <string.h>
#include "header.h"
#include "parse.h"

#define MAX_INCLUDE_DEPTH 10
YY_BUFFER_STATE include_stack[MAX_INCLUDE_DEPTH];
int depth = 0;
int linecount_stack[MAX_INCLUDE_DEPTH];
const char *nameparse_stack[MAX_INCLUDE_DEPTH];

int linecount=1;
int yydebug;
static enum {Mread_lines,Min_braces} mode=Mread_lines;
static int start=0;
static int lastcom=-1;
static void countline(char *text,int len)
{
  int i;
  for (i=0; i<len; i++)
    linecount+=(text[i]=='\n');
}
/* yytext is not global on OSF1 flex */
static void copytoken(char *yytext)
{
  yylval.str.s=strdup(yytext);
  if (yydebug) fprintf(stderr,"[%s]",yytext);
  yylval.str.c=lastcom; lastcom=-1;
}
static void copyentry(char *yytext)
{
  if (strchr(yytext,' '))
    die(err_parse,"Space inside identifier name `%s'",yytext);
  copytoken(yytext);
}
/* similarly for yyleng */
static void copystring(char *yytext, int yyleng)
{
  yylval.str.s=strdup(yytext+1);
  yylval.str.s[yyleng-2]=0;
  if (yydebug) fprintf(stderr,"[%s]",yytext);
  countline(yytext,yyleng);
  yylval.str.c=lastcom; lastcom=-1;
}
static int yywrap(void ) {return 1;}

/* There are two start conditions:
   -- toplevel
   -- INITIAL

We start at toplevel. As soon as we see a non toplevel command, we go to
INITIAL. When the command is completed, we revert to toplevel. It is better to
stay at toplevel when we encounter non-command tokens (comments, white spaces).

The mode can be
Mread_lines: line by line mode
Min_braces : inside braces
*/
/* REAL_NOEXPO can be confused with member function to integer
 * so we let parse.y handle them
 */
%}
INTEGER [0-9]([0-9 ]*[0-9])?
EXPO [eE][+-]?{INTEGER}
REAL_NODOT {INTEGER}{EXPO}
REAL_TAIL  {INTEGER}?"."{INTEGER}{EXPO}?
REAL_EXPO  {INTEGER}?"."{EXPO}
REAL_NOEXPO {INTEGER}"."
REAL {REAL_NODOT}|{REAL_TAIL}|{REAL_EXPO}
STRING \"([^"\\]|(\\.))*\"
ENTRY [A-Za-z]([A-Za-z0-9_ ]*[A-Za-z0-9_])?
%x toplevel
%%
        if (!start) {start=1; BEGIN(toplevel);}
{ENTRY}         copyentry(yytext); return KENTRY;
{INTEGER}       copytoken(yytext); return KINTEGER;
{REAL}          copytoken(yytext); return KREAL;
{REAL_NOEXPO}/[^_A-Za-z.]  copytoken(yytext); return KREAL;
{STRING}        copystring(yytext,yyleng); return KSTRING;

<<EOF>>  {
  if ( --depth < 0 )
    yyterminate();
  else
  {
    yy_delete_buffer( YY_CURRENT_BUFFER );
    yy_switch_to_buffer(include_stack[depth]);
    linecount = linecount_stack[depth];
    free((void*)nameparse);
    nameparse = nameparse_stack[depth];
  }
}

<toplevel>\\r[^\n]+ { /*With credit to flex manpage*/
  char *s=yytext+2;
  while(*s==' ' || *s=='\t') s++;
  if ( depth >= MAX_INCLUDE_DEPTH )
    die(-1,"\\r includes nested too deeply" );
  linecount_stack[depth] = linecount;
  nameparse_stack[depth] = nameparse;
  include_stack[depth++] = YY_CURRENT_BUFFER;
  linecount=1;
  yyin = fopen(s,"r");
  if (!yyin)
  {
    char *t=(char*)malloc(strlen(s)+4);
    sprintf(t,"%s.gp",s);
    yyin = fopen(t,"r");
    if (!yyin)
      die(-1,"cannot open file `%s'",s);
    nameparse=strdup(t);
    free(t);
  }
  else nameparse=strdup(s);
  yy_switch_to_buffer( yy_create_buffer( yyin, YY_BUF_SIZE ) );
}


<toplevel>\\[a-qs-z][^\n]*|\?[^\n]* {
        fprintf(stderr,"Warning:%s:%d: meta commands not implemented\n%s\n",
                        nameparse,linecount,yytext);
                        }

<toplevel>##?[ \t\r]*$ { /*We cannot recognize #<EOF> */
        fprintf(stderr,"Warning:%s:%d: meta commands not implemented\n%s\n",
                        nameparse,linecount,yytext);
                        }

".."            return KDOTDOT;
"->"            return KARROW;
\)[ \t\r]*->    return KPARROW;
"\\/"           return KDR;
">>"            return KSR;
"<<"            return KSL;
"<="            return KLE;
">="            return KGE;
"!="|"<>"       return KNE;
"==="           return KID;
"=="            return KEQ;
"&&"            return KAND;
"||"            return KOR;

"++"            return KPP;
"--"            return KSS;
"+="            return KPE;
"-="            return KSE;
"*="            return KME;
"/="            return KDE;
"\\="           return KEUCE;
"\\/="          return KDRE;
"%="            return KMODE;
">>="           return KSRE;
"<<="           return KSLE;

"\n"            {linecount++;if (mode==Mread_lines) {BEGIN(toplevel);return '\n';}}
<toplevel>"\n"  {linecount++;return '\n';}
"{"             {mode=Min_braces;BEGIN(INITIAL);}
"}"             {mode=Mread_lines;BEGIN(toplevel);}

<INITIAL,toplevel>"\\\n"   {linecount++;}

"=" { for( ; ; )
      {
        int c, nc, star=0;
        for( ; ; )
        {
          c = input();
          if ( c == EOF  )
            return c;
          if ( c == '\n' )
            linecount++;
          else if ( c!=' ' && c!='\t' && c!='\r')
            break;
        }
        /*Since only one char of look-ahead is available, we have to
          handle the / or \ by ourself. Fortunately:
          after =/, only * is valid,
          after =\, only \ and \\n are valid.
          Thus we handle them and reject the rest.
         */
        if (c!='/' && c!='\\')
        {
          unput(c);
          return '=';
        }
        nc=input();
        if (c=='\\' && nc=='\n')
        {
          linecount++;
          continue;
        }
        if ((c=='/' && nc!='*') || (c=='\\' && nc!='\\'))
        {
          fprintf(stderr,"%s:%d: parse error after =%c%c\n"
           ,nameparse,linecount,(char)c,(char)nc);
          fprintf(stderr,"Errors found: aborting...\n");
          exit(1);
        }
        if (c=='/')
        {
          for ( ; ; )
          {
            int c = input();
            if ( c == EOF )
              die(-1,"Unfinished comment");
            if ( c == '/' && star) break;
            if ( c == '*' ) star=1;
            else star=0;
            if ( c == '\n' ) linecount++;
          }
        }
        else
        {
          for ( ; ; )
          {
            int c = input();
            if ( c == EOF )
              die(-1,"Unfinished comment");
            if ( c == '/' && star) c='\\';
            if ( c == '*' ) star=1;
            else star=0;
            if ( c == '\n' )
            {
              linecount++;
              break;
            }
            if ( c == '\r' )
              continue;
          }
        }
     }
  }


<INITIAL,toplevel>[ \t\r]+
<INITIAL,toplevel>"/*"|"\\\\" {
        int star=0;
        int n;
        int nl=0;
        int lc=linecount;
        if (lastcom==-1)
          lastcom=newcomment();
        n=lastcom;
        pushcomment(n,'/');
        pushcomment(n,'*');
        if (yytext[0]=='/')
        {
          for ( ; ; )
          {
            int c = input();
            if ( c == EOF )
              die(-1,"Unfinished comment");
            pushcomment(n,c);
            if ( c == '/' && star) break;
            if ( c == '*' ) star=1;
            else star=0;
            if ( c == '\n' ) linecount++;
          }
        }
        else
        {
          nl=1;
          for ( ; ; )
          {
            int c = input();
            if ( c == EOF )
              die(-1,"Unfinished comment");
            if ( c == '/' && star) c='\\';
            if ( c == '*' ) star=1;
            else star=0;
            if ( c == '\n' )
            {
              linecount++;
              break;
            }
            if ( c == '\r' )
              continue;
            pushcomment(n,c);
          }
          pushcomment(n,' ');
          pushcomment(n,'*');
          pushcomment(n,'/');
          pushcomment(n,'\n');
        }
        /*We hack: we put white spaces after comments in the comments
          This makes output looking better in most cases.
          Sadly we can't add white spaces that are before comments.
        */
        for( ; ; )
        {
          int c = input();
          if ( c == EOF  )
            return '\n';
          if ( c == '\n' )
          {
            linecount++;
            pushcomment(n,c);
            nl=1;
          }
          else if ( c == ' ' || c == '\t')
          {
            if (nl==0)
              pushcomment(n,c);
              /*We only add spaces on the same line*/
          }
          else if ( c != '\r' )
          {
            unput(c);
            break;
          }
        }
        if (lc<linecount && mode==Mread_lines)
        {
          BEGIN(toplevel);
          return '\n';
        }
      }
<toplevel>. {BEGIN(INITIAL); unput(yytext[0]);}
. return yytext[0];
%%
