/***************************************************************************
 *            qof-expenses.h
 *
 *  Thu Oct 21 08:18:44 2004-2005
 *  Copyright  2004,2005,2006  Neil Williams  <linux@codehelp.co.uk>
 ****************************************************************************/
/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/** @addtogroup QOF
	@{ */
/** @addtogroup Exp Expenses

	Like ::DB, Expenses uses UTC time within QOF and sets localtime when
	calling pilot-link code.

	@{ */
/** @file  qof-expenses.h
	@brief QOF expense definitions for pilot-link
	@author Copyright (c) 2004-2005 Neil Williams <linux@codehelp.co.uk>
*/

#ifndef _QOF_EXPENSES_H
#define _QOF_EXPENSES_H

#include <glib.h>
#include <qof.h>

#define GPE_MOD_EXP "gpe-expenses-module"
#define GPE_QOF_EXPENSES "gpe_expenses"
#define QOF_EXPENSES_DESC "GPE QOF expenses"
#define EXP_DATE      "expense_date"
#define EXP_TYPE      "type_of_expense"
#define EXP_PAYMENT   "form_of_payment"
#define EXP_CURRENCY  "currency_code"
#define EXP_AMOUNT    "expense_amount"
#define EXP_VENDOR    "expense_vendor"
#define EXP_CITY      "expense_city"
#define EXP_ATTENDEES "expense_attendees"
#define EXP_NOTE      "expense_note"
#define EXP_CATEGORY  "category"
#define EXP_DISTANCE  "distance_unit"
#define EXP_KVP       "currency_lookup"

/** type:  KVP_TYPE_STRING */
#define PQ_CURRENCY_MNEMONIC "expense/currency/mnemonic"  
/** type:  KVP_TYPE_STRING */
#define PQ_CURRENCY_SYMBOL   "expense/currency/symbol"
/** type:  KVP_TYPE_GINT64 */
#define PQ_CURRENCY_FRACTION "expense/currency/fraction"

typedef struct Expense Expense_t;

/** \brief Wraps QOF around Expense_t

This allows QOF to build as an option rather than adding QofInstance to
Expense_t and requiring QOF for all builds.

*/
typedef struct QofExp_s QofExp;

typedef void (*GpeCurrencyCB) (gpointer key, gpointer value, gpointer user_data);

void gpe_currency_foreach(GpeCurrencyCB cb, gpointer user_data);

/** \brief gpe-expenses currency data wrapper.

Uses the same format as the Palm Default Currency Table:

http://pilot-qof.sourceforge.net/manual.html#currency
*/
typedef struct 
{
	/** pilot-link integer ID */
	gint            pq_code;
	/** currency fraction, pennies/cents etc. */
	gint            fraction;
	/** locale symbol. */
	const gchar* 	symbol;
	/** gnucash/cashutil mnemonic. */
	const gchar*	mnemonic;
	/** TRUE if symbol is not UTF-8. (Currently,
	this is only useful if the symbol can still be shown
	as iso-8859-1.) */	
	gboolean        non_utf8;
}GpeCurrency;

/** \brief Set details of a GpeCurrency from a currency_code

Lookup details of a GpeCurrency from a short identifier. The
QofInstance will have a KvpFrame set (replacing any existing
values for the same keys) containing:
 -::PQ_CURRENCY_MNEMONIC
 -::PQ_CURRENCY_FRACTION
 -::PQ_CURRENCY_SYMBOL

\param inst The QofInstance to hold the KvpFrame containing the
	details of the currency.
\param currency_code The identifier of this currency.

\return a pointer to the GpeCurrency identified by the
	currency_code or NULL on error.

*/
GpeCurrency *
gpe_currency_lookup (QofInstance* inst, gint currency_code);

/** \brief Lookup a currency by name

Slower than ::gpe_currency_lookup because the entire currency
hash must be searched. Otherwise, equivalent.

*/
GpeCurrency *
gpe_currency_lookup_name (QofInstance* inst, gchar * mnemonic);

#define DISTANCE_LIST(_) \
 _(Miles, = 0) \
 _(Kilometers,)

#define PAYMENT_TYPE_LIST(_) \
 _(AmericanExpress, = 0) \
 _(Cash, = 1 ) \
 _(Cheque, ) \
 _(CreditCard, ) \
 _(MasterCard, ) \
 _(Prepaid, ) \
 _(VISA, ) \
 _(Unfiled, )

#define EXPENSE_TYPE_LIST(_) \
 _(Airfare, = 0) \
 _(Breakfast, = 1) \
 _(Bus, = 2) \
 _(BusinessMeals, ) \
 _(CarRental, ) \
 _(Dinner, ) \
 _(Entertainment, ) \
 _(Fax, ) \
 _(Gas, ) \
 _(Gifts, ) \
 _(Hotel, ) \
 _(Incidentals, ) \
 _(Laundry, ) \
 _(Limo, ) \
 _(Lodging, ) \
 _(Lunch, ) \
 _(Mileage, ) \
 _(Other, ) \
 _(Parking, ) \
 _(Postage, ) \
 _(Snack, ) \
 _(Subway, ) \
 _(Supplies, ) \
 _(Taxi, ) \
 _(Telephone, ) \
 _(Tips, ) \
 _(Tolls, ) \
 _(Train, )

DEFINE_ENUM(ExpenseDistance, DISTANCE_LIST)
DEFINE_ENUM(ExpensePayment, PAYMENT_TYPE_LIST)
DEFINE_ENUM(ExpenseType, EXPENSE_TYPE_LIST)

AS_STRING_DEC  (ExpenseDistance, DISTANCE_LIST)
FROM_STRING_DEC(ExpenseDistance, DISTANCE_LIST)
AS_STRING_DEC  (ExpensePayment, PAYMENT_TYPE_LIST)
FROM_STRING_DEC(ExpensePayment, PAYMENT_TYPE_LIST)
AS_STRING_DEC  (ExpenseType, EXPENSE_TYPE_LIST)
FROM_STRING_DEC(ExpenseType, EXPENSE_TYPE_LIST)

/** \brief Register this object with QOF
*/
gboolean ExpensesRegister (void);

/** @} */
/** @} */
#endif				/* _QOF_EXPENSES_H */
