/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satdata.h:  Data structures and functions to store and retrieve     
              satellite data.                                         

  Copyright (C)  2001-2003  Alexandru Csete.

  Authors:   Alexandru Csete <csete@users.sourceforge.net>
             John Magliacane (original author of tracking code).

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#ifndef SATDATA_H
#define SATDATA_H

/* When to monitor/autotrack satellite.
   Note that satellites are only monitored
   when up.
*/
enum sat_monitor_level_e {
	MON_NEVER = 0,       /* never monitor satellite */
	MON_VISIBLE,         /* monitor when visible from groundstation */
	MON_SUNLIGHT,        /* monitor when sat is in sunlight */
	MON_ALWAYS           /* always monitor this satellite */
};

typedef enum sat_monitor_level_e sat_monitor_level_t;

/* The priority of the satellite - does matter if
   two satellites has to be monitored at the same
   time.
*/
enum sat_monitor_prio_e {
	MON_PRIO_NONE = 0,
	MON_PRIO_VERYLOW,
	MON_PRIO_LOW,
	MON_PRIO_MEDIUM,
	MON_PRIO_HIGH,
	MON_PRIO_VERYHIGH,
	MON_PRIO_MUST
};

typedef enum sat_monitor_prio_e sat_monitor_prio_t;

/* Satellite status (bad and misleading word). Will be handy... */
enum sat_status_e {
	SAT_STATUS_NONE = 0,     /* unknown (fx. el = 0) */
	SAT_STATUS_DECAYED,      /* satellite has decayed */
	SAT_STATUS_GEOSTAT,      /* geostationary sat */
	SAT_STATUS_NOAOS,        /* sat never comes above horizon */
	SAT_STATUS_BHOR,         /* sat is below horizon ie. el. < 0 */
	SAT_STATUS_CAMEUP,       /* sat just came up */
	SAT_STATUS_AHOR,         /* sat is above horizon ie el. > 0 */
	SAT_STATUS_WENTDOWN      /* sat just went down */
};

typedef enum sat_status_e sat_status_t;


/* Enumeration indicating whether the current channel has
   uplink, downlink or both.
*/
enum sat_link_type_e {
	SAT_LINK_TYPE_NONE = 0,
	SAT_LINK_TYPE_UPLINK,
	SAT_LINK_TYPE_DOWNLINK,
	SAT_LINK_TYPE_TWOWAYS
};

typedef enum sat_link_type_e sat_link_type_t;

/* Satellite frequency range. For a single frequency
   (eg. beacon) set upper = 0
*/
struct sat_link_struc {
	long long lower;      /* lower limit in Hz (hamlib compatibility) */
	long long upper;      /* upper limit in Hz (hamlib compatibility) */
	GSList modes;         /* list of integers indicating available modes (hamlib) */
};

typedef struct sat_link_struc sat_link_t;


struct sat_channel_struc {
	gchar *name;
	sat_link_type_t *type;
	sat_link_t *uplink;
	sat_link_t *downlink;
};

typedef struct sat_channel_struc sat_channel_t;



/* Satellite data structure. Each satellite is represented
   by a pointer to this kind of structure.
*/
struct sat_struc {

	/* data are obtained from the element set */
	gchar    *name;          /* Satellite name 24 chars + ending \0 */
	guint32  catnum;         /* Catalog number, used as indentifier key */
	gchar    class;          /* Classification, U = unclassified */
	guint16  lyear;          /* Launch year */
	guint32  lnum;           /* Launch number */
	gchar    *piece;         /* Piece of launch */
	guint16  epyear;         /* Epoch Year (last two digits) */
	gdouble  epday;          /* Epoch (day of the year and fractional portion of year) */
	gchar    ephem;          /* Ephermis type, 1=SGP, 2=SGP4, 3=SDP4, 4=SGP8, 5=SDP8 */
	                         /* but usually 0 for distributed element sets */
	guint32  elnum;          /* Element set number - not always reliable */
	gdouble  incl;           /* Inclination [deg] */
	gdouble  raan;           /* Right Ascension of the Ascending Node [deg] */
	gdouble  eccn;           /* Eccentricity */
	gdouble  argper;         /* Argument of perigee [deg] */
	gdouble  meanan;         /* Mean anomaly [deg] */
	gdouble  meanmo;         /* Mean motion [rev/day] */
	gdouble  drag;           /* First time derivative of mean motion [rev/day/day] */
	gdouble  decac;          /* Second time der. of mean motion, not used in SGP4/SDP4 models */
	gdouble  bstar;          /* BSTAR drag term [(earth radii)^(-1)] */
	gulong   eporb;          /* Orbit number at epoch [rev] */

	/* data obtained from tracking */
	gdouble  perigee;
	gdouble  apogee;
	gdouble  lon;            /* Longitude */
	gdouble  lat;            /* Lattitude */
	gdouble  az;             /* Azimuth [deg] */
	gdouble  el;             /* Elevation [deg] */
	gdouble  oldel;          /* Previous elevation used for status */
	gdouble  fp;             /* Footprint [km]? */
	gdouble  range;          /* Range [km] */
	gdouble  alt;            /* Altitude [km] */
	gdouble  vel;            /* Velocity [km/hr] */
	gulong   orbit;          /* orbit number [rev] */
	gchar    vis;            /* visibility: _V_isible,_E_clipse,_D_aylight or _ _unknown */
	gdouble  aos;            /* next AOS */
	gdouble  los;            /* next LOS */
	gboolean geostat;        /* TRUE if geostationary */
	gboolean decayed;        /* TRUE if decayed */
	gboolean hasaos;         /* TRUE if sat ever comes above horizon */
	gdouble  dop100;         /* Doppler at 100MHz */
	gdouble  phase;

	/* specific data used while tracking */
	gdouble	age, daynum, epoch, sma, t1, se, e1, e2, n0,
		k2, s1, c1, l8, s9, c9, s8, c8, r9, z9, x9, y9, o, w,
		s0, s2, c2, q0, m, e,  s3, c3, c0, r3, m1, m5, x0,
		yzero, x1, yone, r, z1, g7, s7, c7, x,  y,  z, x5, y5,
		z5, z8, x8, y8, df, yr, range2;

	/* other data */
	/*gchar               group[25];*/ /* which group the sat belongs to */
	sat_monitor_level_t monitor;   /* monitor the sat? */
	sat_monitor_prio_t  prio;      /* monitoring priority */
	sat_status_t        status;    /* AOS/LOS status */

	GSList *channels;              /* List of pointers to sat_channel_t structures */
	gchar *file;                   /* file name from which the data was loaded */
};

typedef struct sat_struc sat_t;


/* type casting macros */
#define SAT_DATA (x) ((sat_t *) x)


/* function prototypes */
gint satdata_init (void);
gint satdata_add_sat (sat_t *);
sat_t *satdata_get_sat (guint);
gint satdata_get_sat_index (guint32, gchar *);
guint satdata_count_sat (void);
void satdata_remove_sat (guint);
gint satdata_close (void);


#endif
