/* gpsusi.cc
 */
#include "gpsshogi/revision.h"
#include "benchmark.h"

// Note: -DMINIMAL の場合(i.e., gpsusione) の特別動作
// - setoption Thread を無視 (-N x は可)
// - setoption USI_Hash を無視 (常に最大を利用)

#include "osl/record/usi.h"
#include "osl/record/opening/openingBook.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaIOError.h"
#include "osl/state/numEffectState.h"
#include "osl/game_playing/speculativeSearchPlayer.h"
#include "osl/game_playing/alphaBetaPlayer.h"
#include "osl/game_playing/gameState.h"
#include "osl/game_playing/weightTracer.h"
#include "osl/game_playing/bookPlayer.h"
#include "osl/game_playing/csaLogger.h"
#include "osl/checkmate/dfpn.h"
#include "osl/checkmate/dfpnParallel.h"
#include "osl/eval/ml/openMidEndingEval.h"
#include "osl/eval/progressEval.h"
#include "osl/rating/featureSet.h"
#include "osl/misc/pointerQueue.h"
#include "osl/misc/ctime.h"
#include "osl/misc/ncores.h"
#include "osl/progress/ml/newProgress.h"
#include "osl/move_generator/legalMoves.h"
#include "osl/move_probability/featureSet.h"
#include "osl/rating/ratingEnv.h"
#include "osl/oslConfig.h"
#include "osl/sennichite.h"
#include "osl/enter_king/enterKing.h"
#include <boost/scoped_ptr.hpp>
#include <boost/thread/thread.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/program_options.hpp>
#include <boost/foreach.hpp>
namespace po = boost::program_options;
#include <string>
#include <sstream>
#include <iostream>
#ifdef _WIN32
#  include <malloc.h>
#endif
#ifndef NDEBUG
const bool debug=true;
#else
const bool debug=false;
#endif
bool read_from_file = false;
volatile bool suspend_reading = false;

using namespace osl;
CArray<int,2> bwtime;
MoveVector ignore_moves;
int byoyomi, byoyomi_extension;
bool ponder_enabled = false;
int limit_depth = 10, book_depth = 30, multi_pv_width = 0;
std::string csa_file = "";
std::string input_log_file = "";
std::string error_log_file = "";
int verbose = 0;
// set option 対応にしても良いけどあまり需要ないかも
#ifdef MINIMAL
int book_width_root = 16, book_width = 3;
#else
int book_width_root = 16, book_width = 10;
#endif
uint64_t node_count_hard_limit = 0;
bool new_move_probability = true;

bool is_verbose()
{
  return debug || verbose || ponder_enabled || error_log_file != "";
}

volatile int go_count = 0;
volatile int bestmove_count = 0;
volatile int search_count = 0;
volatile bool last_go_is_checkmate = false;
volatile bool stop_checkmate = false;
boost::mutex go_stop_mutex;
boost::condition go_stop_condition;

struct UsiState
{
  SimpleState initial_state;
  vector<Move> moves;
  volatile bool aborted;
  UsiState() : aborted(false) {}

  bool isSuccessorOf(const UsiState& parent)
  {
    return ! aborted && ! parent.aborted
      && initial_state == parent.initial_state
      && moves.size() == parent.moves.size()+1
      && std::equal(parent.moves.begin(), parent.moves.end(), moves.begin());
  }
  const NumEffectState currentState() const
  {
    NumEffectState state(initial_state);
    BOOST_FOREACH(Move m, moves)
      state.makeMove(m);
    return state;
  }
};

boost::scoped_ptr<std::ofstream> csa_stream;
boost::scoped_ptr<game_playing::CsaLogger> csa_logger;
struct StopWatch
{
  MilliSeconds prev;
  void init()
  {
    prev = MilliSeconds::now();
  }
  int elapsedCSA() 
  {
    MilliSeconds now = MilliSeconds::now();
    int result = std::max(1, (int)floor((now - prev).toSeconds()));
    prev = now;
    return result;
  }
} stop_watch;

class StatefulPlayer
{
  UsiState usi_state;
  game_playing::GameState state;
  boost::scoped_ptr<game_playing::ComputerPlayer> player;
  CArray<int,2> initial_time;
  Player my_turn;
public:
  mutable boost::mutex mutex;
  StatefulPlayer(const UsiState& initial_state,
		 game_playing::ComputerPlayer *p,
		 Player turn)
    : usi_state(initial_state), state(initial_state.initial_state),
      player(p), my_turn(turn)
  {
    initial_time.fill(-1);
    if (csa_logger) {
      if (turn == BLACK)
	csa_logger->init("gpsshogi", "human", state.state());
      else
	csa_logger->init("human", "gpsshogi", state.state());
    }
    BOOST_FOREACH(Move m, usi_state.moves) {
      state.pushMove(m);
      player->pushMove(m);
      if (csa_logger)
	csa_logger->pushMove(m, 0);
    }
    stop_watch.init();
  }
  void search()
  {
    if (is_verbose())
      std::cerr << state.state() << "\n";
    assert(state.state().turn() == my_turn);
    if (initial_time[BLACK] < 0) initial_time[BLACK] = bwtime[BLACK];
    if (initial_time[WHITE] < 0) initial_time[WHITE] = bwtime[WHITE];
    MoveWithComment ret;
    try
    {
      if (! ignore_moves.empty())
	player->setRootIgnoreMoves(&ignore_moves, false);
      {
	boost::mutex::scoped_lock lk(go_stop_mutex);
	++search_count;
	go_stop_condition.notify_all();
      }
      if (byoyomi_extension > 0) {
	assert(byoyomi > 0);
	const osl::MilliSeconds::Interval b(byoyomi);
	const osl::MilliSeconds::Interval bmax(byoyomi_extension);
	osl::search::TimeAssigned msec(b, bmax);
	if (osl::game_playing::ComputerPlayerSelectBestMoveInTime *p
	    = dynamic_cast<osl::game_playing::ComputerPlayerSelectBestMoveInTime *>(player.get()))
	  ret = p->selectBestMoveInTime(state, msec);
	else
	  throw std::runtime_error("type error in byoyomi_extension");
      }
      else
	ret = player->selectBestMove(state, initial_time[my_turn]/1000, 
				     (initial_time[my_turn]-bwtime[my_turn])/1000, 
				     byoyomi/1000);
    }
    catch (std::exception& e) {
      std::cerr << "search failed by exception: " << e.what() << "\n";
      throw;
    }
    catch (...) {
      std::cerr << "search failed\n";
      throw;
    }
    if (ret.move.isNormal()) {
      boost::mutex::scoped_lock lk(mutex);
      usi_state.moves.push_back(ret.move);
      state.pushMove(ret.move);
      player->pushMove(ret.move);
      if (csa_logger)
	csa_logger->pushMove(ret, stop_watch.elapsedCSA());
    } else {
      boost::mutex::scoped_lock lk(mutex);
      usi_state.aborted = true;
    }
    player->setRootIgnoreMoves(0, false);
    ignore_moves.clear();
    OslConfig::resetRootWindow();
    std::string result = "bestmove " + record::usi::show(ret.move);
    {
      boost::mutex::scoped_lock lk(OslConfig::lock_io);
      std::cout << result << "\n" << std::flush;
    }
    if (is_verbose())
      std::cerr << "sent: " << result << "\n";
    suspend_reading = false;
    boost::mutex::scoped_lock lk(go_stop_mutex);
    ++bestmove_count;
    go_stop_condition.notify_all();
  }
  void newMove(Move move)
  {
    if (csa_logger)
      csa_logger->pushMove(move, stop_watch.elapsedCSA());
    {
      boost::mutex::scoped_lock lk(mutex);
      usi_state.moves.push_back(move);
      state.pushMove(move);
      player->pushMove(move);
    }
    assert(move.player() != my_turn);
    search();
  }
  const UsiState current() const
  {
    boost::mutex::scoped_lock lk(mutex);
    return usi_state;
  }
  void stopSearchNow()
  {
    player->stopSearchNow();
  }
  void needRestartNext()
  {
    usi_state.aborted = true;
  }
};
boost::scoped_ptr<StatefulPlayer> player;

struct Queue : osl::misc::PointerQueue<Move>
{
  void pop(Move& move) 
  {
    move = *pop_front();
  }
  bool pop_if_present(Move& move) 
  {
    boost::shared_ptr<Move> ret = pop_front_non_block();
    if (ret)
      move = *ret;
    return ret;
  }
  void push(Move move)
  {
    boost::shared_ptr<Move> ptr(new Move(move));
    push_back(ptr);
  }
};

Queue queue;

struct ThreadRun
{
  ThreadRun()
  {
  }
  void operator()() const
    __attribute__((noinline))
#ifdef _WIN32
    __attribute__((force_align_arg_pointer))
#endif
  {
    if (! goodAlignment())
      return;

    player->search(); 
    while (true) {
      if (debug)
	std::cerr << "wait for opponent\n";
      Move move;
      queue.pop(move);
      if (debug)
	std::cerr << "got " << move << "\n";
      if (! move.isNormal())
	return;
      player->newMove(move);
    }
  }
  static bool goodAlignment();
};

boost::scoped_ptr<UsiState> input_state;
boost::scoped_ptr<boost::thread> search_thread;

game_playing::ComputerPlayer *makePlayer(Player turn, const SimpleState&);
Player input_turn() 
{
  Player turn = input_state->initial_state.turn();
  if (! input_state->moves.empty())
    turn = alt(input_state->moves.back().player());
  return turn;
}
void do_search()
{
  if (search_thread && input_state->isSuccessorOf(player->current())) {
    if (debug)
      std::cerr << "game continued\n";
    queue.push(input_state->moves.back());
    return;
  }
  if (search_thread) {
    queue.push(Move());
    search_thread->join();
    Move dummy;
    while (queue.pop_if_present(dummy))
      ;
  }
  const Player turn = input_turn();
  if (debug)
    std::cerr << "new game " << turn << "\n";
  if (csa_file != "") {
    csa_stream.reset(new std::ofstream(csa_file.c_str()));
    csa_logger.reset(new game_playing::CsaLogger(*csa_stream));
  } 
  game_playing::ComputerPlayer *searcher = makePlayer(turn, input_state->initial_state);
  player.reset(new StatefulPlayer(*input_state, searcher, turn));
  search_thread.reset(new boost::thread(ThreadRun()));  

  boost::mutex::scoped_lock lk(go_stop_mutex);
  while (go_count > search_count)
    go_stop_condition.wait(lk);
}

#ifdef OSL_DFPN_SMP
struct GoMate
{
  checkmate::DfpnParallel& dfpn;
  NumEffectState& state;
  PathEncoding path;
  Move& checkmate_move;
  vector<Move>& pv;
  ProofDisproof& result;
  GoMate(checkmate::DfpnParallel& d, NumEffectState& s, PathEncoding p, 
	 Move& c, vector<Move>& v, ProofDisproof& r)
    : dfpn(d), state(s), path(p), checkmate_move(c), pv(v), result(r)
  {
  }
  void operator()()
  {
    result = dfpn.
      hasCheckmateMove(state, HashKey(state), path, 
		       std::numeric_limits<size_t>::max(), checkmate_move, Move(), &pv);
  }
};
#endif

struct GoMateThread
{
  struct NotifyLock
  {
    NotifyLock()
    {
      boost::mutex::scoped_lock lk(go_stop_mutex);
      ++search_count;
      go_stop_condition.notify_all();
    }
    ~NotifyLock()
    {
      boost::mutex::scoped_lock lk(go_stop_mutex);
      ++bestmove_count;
      go_stop_condition.notify_all();
      stop_checkmate = false;
    }
  };
  double seconds;
  explicit GoMateThread(double s) : seconds(s)
  {
  }
  void operator()();
};

void GoMateThread::operator()() 
{
  NotifyLock lock;
  NumEffectState state(input_state->currentState());
#if (! defined ALLOW_KING_ABSENCE)
  if (state.kingSquare(state.turn()).isPieceStand()) {
    boost::mutex::scoped_lock lk(OslConfig::lock_io);
    std::cout << "checkmate notimplemented\n";
    return;
  }
#endif  
  checkmate::DfpnTable table(state.turn());
  const PathEncoding path(state.turn());
  Move checkmate_move;
  vector<Move> pv;
  ProofDisproof result;
  if (debug)
    std::cerr << state;
#ifdef OSL_DFPN_SMP
  checkmate::DfpnParallel dfpn(std::min(OslConfig::numCPUs(), 8));
#else
  checkmate::Dfpn dfpn;
#endif
  dfpn.setTable(&table);
  MilliSeconds start = MilliSeconds::now();
#ifdef OSL_DFPN_SMP
  boost::thread thread(GoMate(dfpn, state, path, checkmate_move, pv, result));
  double wait = 450;
  for (int i=0; true; ++i) {
    if (thread.timed_join(boost::posix_time::milliseconds((size_t)wait)))
      break;
    double elapsed = start.elapsedSeconds();
    size_t node_count = dfpn.nodeCount();
    if (i % 4 == 0) {
      double memory = OslConfig::memoryUseRatio();
      boost::mutex::scoped_lock lk(OslConfig::lock_io);
      std::cout << "info time " << static_cast<int>(elapsed*1000)
		<< " nodes " << node_count << " nps " << static_cast<int>(node_count/elapsed)
		<< " hashfull " << static_cast<int>(memory*1000) << "\n";
    }
    if ((seconds*0.94-elapsed)*1000 < 110 || stop_checkmate)
      break;
    wait = std::min((seconds*0.94-elapsed)*1000 - 100, 250.0);
  }
  dfpn.stopNow();
  thread.join();
#else  
  size_t step = 100000, total = 0;
  double scale = 1.0; 
  for (size_t limit = step; true; limit = static_cast<size_t>(step*scale)) {
    result = dfpn.
      hasCheckmateMove(state, HashKey(state), path, limit, checkmate_move, Move(), &pv);
    double elapsed = start.elapsedSeconds();
    double memory = OslConfig::memoryUseRatio();
    uint64_t node_count = dfpn.nodeCount();
    total += limit;
    boost::mutex::scoped_lock lk(OslConfig::lock_io);
    std::cout << "info time " << static_cast<int>(elapsed*1000)
	      << " nodes " << node_count << " nps " << static_cast<int>(node_count/elapsed)
	      << " hashfull " << static_cast<int>(memory*1000) << "\n";
    if (result.isFinal() || elapsed >= seconds || memory > 0.9 || stop_checkmate)
      break;
    // estimate: total * std::min(seconds/elapsed, 1.0/memory)
    // next: (estimate - total) / 2 + total
    scale = (total * std::min(seconds/elapsed, 1.0/memory) - total) / 2.0 / step;
    scale = std::max(std::min(16.0, scale), 0.1);
  }
#endif
  if (is_verbose())
    std::cerr << "elapsed " << start.elapsedSeconds() << "\n";
  if (! result.isFinal()) {
    boost::mutex::scoped_lock lk(OslConfig::lock_io);
    std::cout << "checkmate timeout\n" << std::flush;
    return;
  }
  if (! result.isCheckmateSuccess()) {
    boost::mutex::scoped_lock lk(OslConfig::lock_io);
    std::cout << "checkmate nomate\n" << std::flush;
;
    return;
  }
  boost::mutex::scoped_lock lk(OslConfig::lock_io);
  std::cout << "checkmate";
  for (size_t i=0; i<pv.size(); ++i)
    std::cout << " " << record::usi::show(pv[i]);
  std::cout << "\n" << std::flush;
}

void do_checkmate(double seconds)
{
  boost::shared_ptr<boost::thread> (new boost::thread(GoMateThread(seconds)));
}

void play_by_book(int depth_limit)
{
  static osl::record::opening::WeightedBook book(OslConfig::openingBook());
  if (HashKey(input_state->initial_state)
      == HashKey(SimpleState(HIRATE))
      && (int)input_state->moves.size() < depth_limit)
  {
    game_playing::GameState state(input_state->initial_state);
    game_playing::WeightTracer tracer
      (book, is_verbose(), book_width_root, book_width);
    BOOST_FOREACH(Move m, input_state->moves) {
      tracer.update(m);
      state.pushMove(m);
    }
    if (! tracer.isOutOfBook()) {
      const Move best_move = tracer.selectMove();
      if (best_move.isNormal()
	  && (! state.isIllegal(best_move))) {
	boost::mutex::scoped_lock lk(OslConfig::lock_io);
	std::string result = "bestmove " + record::usi::show(best_move);
	std::cout << result << "\n" << std::flush;
	if (is_verbose())
	  std::cerr << "sent: " << result << "\n";
	return;
      }
    }
  }
  boost::mutex::scoped_lock lk(OslConfig::lock_io);
  std::cout << "bestmove pass\n" << std::flush;
  if (is_verbose())
    std::cerr << "sent: " << "bestmove pass\n";
}

void play_declare_win()
{
  const bool can_declare_win
    = EnterKing::canDeclareWin(input_state->currentState());
  const std::string msg = std::string("bestmove ")
    + (can_declare_win ? "win" : "pass") + "\n" ;
  boost::mutex::scoped_lock lk(OslConfig::lock_io);
  std::cout << msg << std::flush;
  if (is_verbose())
    std::cerr << "sent: " << msg;
}

MoveVector generate_good_moves()
{
  game_playing::GameState state(input_state->initial_state);
  BOOST_FOREACH(Move m, input_state->moves) 
    state.pushMove(m);

  MoveVector normal_or_win_or_draw, loss;
  state.generateNotLosingMoves(normal_or_win_or_draw, loss);
  if (is_verbose() && ! loss.empty()) {
    std::cerr << "  removed losing move ";
    BOOST_FOREACH(Move m, loss)
      std::cerr << record::csa::show(m);
    std::cerr << "\n";
  }
  return normal_or_win_or_draw;
}

void genmove()
{
  MoveVector moves = generate_good_moves();
  boost::mutex::scoped_lock lk(OslConfig::lock_io);
  std::cout << "genmove";
  BOOST_FOREACH(Move move, moves)
    std::cout << " " << record::usi::show(move);
  std::cout << "\n";
}

void genmove_probability(int limit)
{
  game_playing::GameState gstate(input_state->initial_state);
  BOOST_FOREACH(Move m, input_state->moves) 
    gstate.pushMove(m);
  const NumEffectState& state= gstate.state();
  const MoveStack& history = gstate.moveHistory();
  progress::ml::NewProgress progress(state);
  MoveLogProbVector moves;
  if (new_move_probability) {
    const move_probability::StandardFeatureSet& feature_set
      = move_probability::StandardFeatureSet::instance();
    Move threatmate = move_probability::StateInfo::findShortThreatmate
      (state, history.lastMove());
    move_probability::StateInfo info(state, progress.progress16(),
				     history, threatmate);
    feature_set.generateLogProb(info, moves);
  } else {
    const rating::StandardFeatureSet& feature_set
      = rating::StandardFeatureSet::instance();
    rating::RatingEnv env;
    env.history = history;
    env.make(state, state.pin(state.turn()), state.pin(alt(state.turn())),
	     progress.progress16());
    feature_set.generateLogProb(state, env, limit, moves);
  }
  for (size_t i=1; i<moves.size(); ++i)
    if (moves[i].logProb() <= moves[i-1].logProb() && moves[i-1].logProb()+1<=limit)
      moves[i].setLogProb(moves[i-1].logProb()+1);
  MoveVector good_moves = generate_good_moves();
  boost::mutex::scoped_lock lk(OslConfig::lock_io);
  std::cout << "genmove_probability";
  BOOST_FOREACH(MoveLogProb move, moves)
    if (good_moves.isMember(move.move()))
      std::cout << " " << record::usi::show(move.move()) << " " << move.logProb();
  std::cout << "\n";
}

void csashow()
{
  NumEffectState state(input_state->currentState());
  boost::mutex::scoped_lock lk(OslConfig::lock_io);
  std::cout << state;
  std::cout << "csashowok\n";
}

void csamove(std::string str)
{
  const NumEffectState state(input_state->currentState());
  const Move move = record::usi::strToMove(str, state);
  boost::mutex::scoped_lock lk(OslConfig::lock_io);
  std::cout << "csamove " << record::csa::show(move) << "\n";
}

std::string make_filename()
{
  return OslConfig::gpsusiConf();
}

void read_config_file()
{
  std::string filename = make_filename();
  if (debug)
    std::cerr << "reading " << filename << "\n";
  std::ifstream fs(filename.c_str());
  std::string key;
  std::string line;
  while (std::getline(fs, line)) {
    std::istringstream is(line);
    is >> key;
    if (key == "LimitDepth") {
      int value;
      if (is >> value)
	limit_depth = value;
      continue;
    }
    if (key == "BookDepth") {
      int value;
      if (is >> value)
	book_depth = value;
      continue;
    }
    if (key == "MultiPVWidth") {
      int value;
      if (is >> value)
	multi_pv_width = value;
      continue;
    }
    if (key == "CSAFile") {
      std::string value;
      if (is >> value)
	csa_file = value;
      continue;
    }
    if (key == "InputLogFile") {
      std::string value;
      if (is >> value)
	input_log_file = value;
      continue;
    }
    if (key == "ErrorLogFile") {
      std::string value;
      if (is >> value)
	error_log_file = value;
      continue;
    }
    if (key == "Verbose") {
      int value;
      if (is >> value)
	verbose = value;
      continue;
    }
    if (key == "UsiOutputPawnValue") {
      int value;
      if (is >> value)
	OslConfig::setUsiOutputPawnValue(value);
      continue;
    }
#ifdef OSL_SMP
    if (key == "Thread") {
      int value;
      if (is >> value) {
#  ifndef MINIMAL
	OslConfig::setNumCPUs(value);
#  else
	if (value != OslConfig::numCPUs())
	  std::cerr << "ignored Thread config " << value << ' ' << OslConfig::numCPUs()
		    << "\n";
#  endif
      }
      continue;
    }
#endif
  }
  
}
void write_config_file()
{
  std::string filename = make_filename();
  std::ofstream os(filename.c_str());
  os << "LimitDepth " << limit_depth << "\n";
  os << "BookDepth " << book_depth << "\n";
  os << "MultiPVWidth " << multi_pv_width << "\n";
  os << "CSAFile " << csa_file << "\n";
  os << "InputLogFile " << input_log_file << "\n";
  os << "ErrorLogFile " << error_log_file << "\n";
  os << "Verbose " << verbose << "\n";
  os << "UsiOutputPawnValue " << OslConfig::usiOutputPawnValue() << "\n";
  os << "Thread " << OslConfig::numCPUs() << "\n";
}

void setOption(std::string& line) 
{
  std::istringstream ss(line);
  std::string set_option, name_str, name, value_str;
  ss >> set_option >> name_str >> name;
#ifndef MINIMAL
  // for safety against human errors, ignore options that can be fixed in advance.
  if (name == "USI_Hash") {
    size_t value;
    if (ss >> value_str >> value) {
      value = std::max(value, (size_t)300);
      value *= 1024*1024;
      value = std::min(value, OslConfig::memoryUseLimit());
      OslConfig::setMemoryUseLimit(value);
      return;
    } else {
      std::cerr << "error setoption " << line << "\n";
    }
  }
#endif
  if (name == "USI_Ponder") {
    std::string value;
    ss >> value_str >> value;
    if (ponder_enabled != (value == "true")
	&& player)
      player->needRestartNext();
    ponder_enabled = (value == "true");
    return;
  }
  if (name == "LimitDepth") {
    int value;
    if (ss >> value_str >> value) {
      if (limit_depth != value && player)
	player->needRestartNext();
      limit_depth = value;
    } else {
      std::cerr << "error setoption " << line << "\n";
    }
  }
  else if (name == "BookDepth") {
    int value;
    if (ss >> value_str >> value) {
      if (book_depth != value && player)
	player->needRestartNext();
      book_depth = value;
    } else {
      std::cerr << "error setoption " << line << "\n";
    }
  }
  else if (name == "MultiPVWidth") {
    int value;
    if (ss >> value_str >> value) {
      if (multi_pv_width != value && player)
	player->needRestartNext();
      multi_pv_width = value;
    } else {
      std::cerr << "error setoption " << line << "\n";
    }
  }
  else if (name == "CSAFile") {
    std::string value;
    ss >> value_str >> value;
    csa_file = value;
  }
  else if (name == "InputLogFile") {
    std::string value;
    ss >> value_str >> value;
    input_log_file = value;
  }
  else if (name == "ErrorLogFile") {
    std::string value;
    ss >> value_str >> value;
    error_log_file = value;
  }
  else if (name == "Verbose") {
    int value;
    if (ss >> value_str >> value) {
      if (verbose != value && player)
	player->needRestartNext();
      verbose = value;
    } else {
      std::cerr << "error setoption " << line << "\n";
    }
  }
  else if (name == "UsiOutputPawnValue") {
    int value;
    if (ss >> value_str >> value) {
      OslConfig::setUsiOutputPawnValue(value);
    } else {
      std::cerr << "error setoption " << line << "\n";
    }
  }
  else if (name == "Thread") {
    int value;
    if (ss >> value_str >> value) {
#ifndef MINIMAL
      OslConfig::setNumCPUs(value);
#else
      if (value != OslConfig::numCPUs())
	std::cerr << "ignored Thread config " << value 
		  << " v.s. " << OslConfig::numCPUs()
		  << "\n";
#endif
    } else {
      std::cerr << "error setoption " << line << "\n";
    }
  }
  write_config_file();
}

#ifdef _WIN32
void play(std::istream& is, std::ostream& os) __attribute__((noinline, force_align_arg_pointer));
#endif
void play(std::istream& is, std::ostream& os)
{
  std::string line;
  std::getline(is, line);
#ifdef GPSUSIONE
  std::string name = "id name gpsusione ";
#else
  std::string name = "id name gpsshogi ";
#endif
#ifdef OSL_SMP
  name += "(smp) ";
#endif
  os << name << gpsshogi::gpsshogi_revision << "\n";
  os << "id author teamgps\n";
  os << "option name LimitDepth type spin default " << limit_depth << " min 4 max 10\n";
  os << "option name BookDepth type spin default " << book_depth << " min 0 max 100\n";
  os << "option name MultiPVWidth type spin default " << multi_pv_width << " min 0 max 1000\n";
  os << "option name CSAFile type string default "
     << ((csa_file != "") ? csa_file : std::string("<empty>")) << "\n";
  os << "option name InputLogFile type string default "
     << ((input_log_file != "") ? input_log_file : std::string("<empty>")) << "\n";
  os << "option name ErrorLogFile type string default "
     << ((error_log_file != "") ? error_log_file : std::string("<empty>")) << "\n";
  os << "option name Verbose type spin default " << verbose << " min 0 max 1\n";
  os << "option name UsiOutputPawnValue type spin default " << OslConfig::usiOutputPawnValue() << " min 100 max 10000\n";
#  ifdef OSL_SMP
  os << "option name Thread type spin default " << OslConfig::numCPUs() << " min 1 max "
     << std::max(1, osl::misc::ncores()) << "\n";
#  endif
  os << "usiok\n" << std::flush;
  boost::scoped_ptr<std::ofstream> input_log_stream;
  if (input_log_file != "") {
    input_log_stream.reset(new std::ofstream(input_log_file.c_str()));
    *input_log_stream << line << "\n" << std::flush;
  }
    
  while (true) {
    if (debug)
      std::cerr << "readyinput\n";
    if (! std::getline(is, line))
      break;
    if (debug) {
      const time_t now = time(0);
      char ctime_buf[64];
      std::cerr << osl::ctime_r(&now, ctime_buf);
      std::cerr << "recv: " << line << "\n" << std::flush;
    }
    if (input_log_stream)
      *input_log_stream << line << "\n" << std::flush;
    if (line == "isready") {
      static bool initalized = false;
      if (! initalized) {
	if (error_log_file != "")
	  freopen(error_log_file.c_str(), "w", stderr);
	eval::ml::OpenMidEndingEval::setUp();
	progress::ml::NewProgress::setUp();
	rating::StandardFeatureSet::instance();
	initalized = true;
      }
      os << "readyok\n";
      if (debug)
	std::cerr << "readyok\n";
      continue;
    }
    if (line.find("setoption") == 0) {
      setOption(line);
      continue;
    }
    if (line.find("usinewgame") == 0)
      continue;
    if (line.find("stop") == 0) {
      {
	boost::mutex::scoped_lock lk(go_stop_mutex);
	if (go_count <= bestmove_count) {
	  std::cerr << "warning stop ignored (go " << go_count
		    << " times, bestmove " << bestmove_count << " times)\n";
	  continue;
	}
      }
      if (last_go_is_checkmate)
	stop_checkmate = true;
      else {
	player->stopSearchNow();
	player->needRestartNext();
      }
      boost::mutex::scoped_lock lk(go_stop_mutex);
      while (go_count > bestmove_count)
	go_stop_condition.wait(lk);
      continue;
    }
    if (line.find("quit") == 0) {
      if (search_thread) {
	queue.push(Move());
	search_thread->join();
      }
      return;
    }

    // extended commands
    if (line.find("sleep") == 0) {
      boost::this_thread::sleep(boost::posix_time::seconds(60));
      continue;
    }
    if (line.find("genmove_probability") == 0) {
      int limit = 2000, value;
      std::istringstream is(line.substr(std::string("genmove_probability").size()));
      if (is >> value)
	limit = value;
      genmove_probability(limit);
      continue;
    }
    if (line.find("genmove") == 0) {
      genmove();
      continue;
    }
    if (line.find("csashow") == 0) {
      csashow();
      continue;
    }
    if (line.find("csamove ") == 0) {
      csamove(line.substr(8));
      continue;
    }
    if (line.find("echo ") == 0) {
      boost::mutex::scoped_lock lk(OslConfig::lock_io);
      std::cout << line.substr(5) << "\n";
      std::cerr << line.substr(5) << "\n";
      continue;
    }

    // searc related commands 
    // do not read next search command while previous search is in progress
    while (read_from_file && suspend_reading)
      boost::this_thread::sleep(boost::posix_time::milliseconds(200));

    if (line.find("position") == 0) {
      if (! input_state)
	input_state.reset(new UsiState);
      try {
	record::usi::parse(line.substr(8), input_state->initial_state, input_state->moves);
	if (is_verbose())
	  std::cerr << "new position " << input_state->moves.size()
		    << " " << line << "\n";
      }
      catch (std::exception& e) {
	std::cerr << "usi parse error " << e.what() << "\n";
	throw e;
      }
      continue;
    }
    if (line.find("go") == 0) {
      boost::mutex::scoped_lock lk(go_stop_mutex);
      while (go_count > bestmove_count) {
	std::cerr << "delay go (go " << go_count
		  << " times, bestmove " << bestmove_count << " times)\n";
	go_stop_condition.wait(lk);
      }
      last_go_is_checkmate = false;
    }
    if (line.find("go mate") == 0) {
      double seconds = 60, millisec;
      std::string option;
      std::istringstream is(line.substr(std::string("go mate").size()));
      if (is >> millisec)
	seconds = std::max(std::min(seconds, millisec/1000.0), 0.5);
      {
	boost::mutex::scoped_lock lk(go_stop_mutex);
	++go_count;
	last_go_is_checkmate = true;
      }
      do_checkmate(seconds);
      continue;
    }
    if (line.find("go book") == 0) {
      int limit = book_depth, value;
      std::istringstream is(line.substr(std::string("go book").size()));
      if (is >> value)
	limit = value;
      play_by_book(limit);
      continue;
    }
    if (line.find("go declare_win") == 0) {
      play_declare_win();
      continue;
    }
    if (line.find("go window ") == 0) {
      if (ponder_enabled) {
	std::string err
	  = "go window not supported if ponder enabled";
	std::cerr << err << "\n";
	  throw std::runtime_error(err);
      }
      bwtime.fill(1000);
      byoyomi = 1000;
      std::string option;
      std::istringstream is(line.substr(strlen("go window ")));
      int alpha, beta;
      if (! (is >> alpha >> beta)) {
	std::string err
	  = "go window: window error ";
	std::cerr << err << "\n";
	  throw std::runtime_error(err);
      }
      std::cerr <<"window is " << alpha << ' ' << beta << " "
		<< (bool)is << ' ' << input_turn() << "\n";
      if (input_turn() == WHITE) {
	std::swap(alpha, beta);
	alpha = -alpha;
	beta = -beta;
	std::cerr <<"window adjustment " << alpha << ' ' << beta << " " << (bool)is << "\n";
      }
      while (is >> option) {
	if (option == "byoyomi")
	  is >> byoyomi;
	else {
	  std::string err
	    = "unknown option in go window " + option;
	  std::cerr << err << "\n";
	  throw std::runtime_error(err);
	}
      }
      byoyomi_extension = byoyomi;
      OslConfig::setRootWindow(alpha, beta);
      suspend_reading = true;
      do_search();
      boost::mutex::scoped_lock lk(go_stop_mutex);
      ++go_count;
      continue;
    }
    if (line.find("go ") == 0) {
      bwtime.fill(1000);
      byoyomi = 0; byoyomi_extension = 0;
      std::string option;
      std::istringstream is(line.substr(3));
      while (is >> option) {
	if (option == "btime")
	  is >> bwtime[BLACK];
	else if (option == "wtime")
	  is >> bwtime[WHITE];
	else if (option == "byoyomi")
	  is >> byoyomi;
	else if (option == "byoyomi_extension")
	  is >> byoyomi_extension;
	else if (option == "infinite")
	  byoyomi = 60*60*1000; // 1hour
	else {
	  std::string err
	    = "unknown option in go " + option;
	  std::cerr << err << "\n";
	  throw std::runtime_error(err);
	}
      }
      if (byoyomi_extension && (byoyomi == 0 || ponder_enabled)) {
	std::cerr << "warning ignored byoyomi_extension\n";
	byoyomi_extension = 0;
      }
      suspend_reading = true;
      do_search();
      boost::mutex::scoped_lock lk(go_stop_mutex);
      ++go_count;
      continue;
    }
    // ponderhit
    if (line.find("gameover") == 0)
      continue;
    // non-standard commands
    if (line.find("ignore_moves") == 0) {
      assert(ignore_moves.empty());
      assert(input_state);
      if (! input_state) {
	std::cerr << "error ignore_moves without position\n";
	continue;
      }
      if (book_depth > 0) {
	std::cerr << "error ignore_moves not supported with book\n";
	continue;
      }
      std::istringstream is(line);
      std::string word;
      is >> word;
      NumEffectState state(input_state->currentState());
      while (is >> word) {
 	ignore_moves.push_back(record::usi::strToMove(word, state));
      }
      std::cerr << "accept ignore_moves\n";
      if (player)
	player->needRestartNext();
      continue;
    }
    if (line.find("query progress") == 0) {
      assert(input_state);
      if (input_state) {
	NumEffectState state(input_state->currentState());
	const osl::progress::Effect5x3 progress(state);
	boost::mutex::scoped_lock lk(OslConfig::lock_io);
	os << "answer progress " << progress.progress16().value()
	   << ' ' << progress.progress16(BLACK).value()
	   << ' ' << progress.progress16(WHITE).value()
	   << "\n";
      }
      else {
	boost::mutex::scoped_lock lk(OslConfig::lock_io);
	os << "error position not specified\n";
      }
      continue;
    }
    if (line.find("query eval") == 0) {
      assert(input_state);
      if (input_state) {
	NumEffectState state(input_state->currentState());
	const osl::eval::ml::OpenMidEndingEval eval(state);
	boost::mutex::scoped_lock lk(OslConfig::lock_io);
	os << "answer eval " << eval.value()
	   << ' ' << eval.captureValue(osl::newPtypeO(osl::WHITE,osl::PAWN))/2
	   << ' ' << eval.progressIndependentValue()
	   << ' ' << eval.openingValue()
	   << ' ' << eval.midgameValue()
	   << ' ' << eval.midgame2Value()
	   << ' ' << eval.endgameValue()
	   << "\n";
      }
      else {
	boost::mutex::scoped_lock lk(OslConfig::lock_io);
	os << "error position not specified\n";
      }
      continue;
    }
    if (line.find("query searchtime ") == 0) {
      assert(input_state);
      if (input_state) {
	std::istringstream is(line);
	std::string dummy;
	int amount, byoyomi, elapsed;
	if (is >> dummy >> dummy >> amount >> byoyomi >> elapsed) {
	  osl::game_playing::GameState state(input_state->initial_state);
	  BOOST_FOREACH(Move m, input_state->moves) 
	    state.pushMove(m);
	  const osl::search::TimeAssigned assigned
	    = osl::game_playing::SearchPlayer::assignTime(state, amount, elapsed, byoyomi, 0);
	  boost::mutex::scoped_lock lk(OslConfig::lock_io);
	  os << "answer searchtime " << assigned.standard.value()
	     << ' ' << assigned.max.value() << "\n";
	}
	else {
	  boost::mutex::scoped_lock lk(OslConfig::lock_io);
	  os << "error in arguments";
	}
      }
      else {
	os << "error position not specified\n";
      }
      continue;      
    }
    if (line.find("query") == 0) {
      boost::mutex::scoped_lock lk(OslConfig::lock_io);
      os << "error unknown query\n";
      continue;      
    }    
    if (line.find("open ") == 0) {
      if (! input_state)
	input_state.reset(new UsiState);
      std::istringstream is(line);
      std::string dummy, filename;
      bool ok = true;
      ok &= static_cast<bool>(is >> dummy >> filename);
      if (ok) {
	try {
	  CsaFile csa(filename.c_str());
	  input_state->initial_state = csa.getInitialState();
	  input_state->moves = csa.getRecord().getMoves();
	}
	catch (record::csa::CsaIOError& e) {
	  std::cerr << e.what() << "\n";
	  ok = false;
	}
      }
      if (ok) {
	boost::mutex::scoped_lock lk(OslConfig::lock_io);
	std::cout << "position "
		  << record::usi::show(input_state->initial_state)
		  << " moves";
	BOOST_FOREACH(Move move, input_state->moves)
	  std::cout << " " << record::usi::show(move);
	std::cout << "\n";
	std::cout << "openok\n";	
      }
      if (! ok)
	std::cerr << "open failed " << line << "\n";
      continue;
    }
    std::cerr << "unknown command " << line << "\n";
  }
  if (read_from_file) {
    std::cerr << "end of input\n";
    boost::this_thread::sleep(boost::posix_time::seconds(60));
    if (search_thread) {
      queue.push(Move());
      search_thread->join();
    }
  } else {
    std::cerr << "stdin closed\n";
  }
}

int main(int argc, char **argv)
{
  int num_benchmark, benchmark_seconds;
  po::options_description options("Options");
  std::string input_file;
#ifndef MINIMAL
  unsigned int eval_random;
#endif
#ifdef OSL_SMP
  int num_cpus;
#endif
  double memory_use_percent;
  options.add_options()
    ("benchmark", "test search performance")
    ("benchmark-single", "test search performance")
    ("benchmark-more", 
     po::value<int>(&num_benchmark)->default_value(0),
     "number of problems for benchmark")
    ("benchmark-seconds", 
     po::value<int>(&benchmark_seconds)->default_value(30),
     "seconds for benchmark")
    ("profile", "profile mode (run in 1 thread even when OSL_SMP is defined)")
    ("read-from-file", po::value<std::string>(&input_file)->default_value(""), "input for debug")
    ("book-width", 
     po::value<int>(&book_width)->default_value(book_width),
     "relative width of book")
    ("book-width-root", 
     po::value<int>(&book_width_root)->default_value(book_width_root),
     "relative width of book at the initial position")
#ifndef MINIMAL
    ("eval-randomness",
     po::value<unsigned int>(&eval_random)->default_value(0),
     "add random value generated by normal distribution of a given standard deviation, to evaluation values")
    ("node-count-hard-limit",
     po::value<uint64_t>(&node_count_hard_limit)->default_value(0),
     "nodes limit for each search (0 for infinity)")
#endif
#ifdef OSL_SMP
    ("num-cpus,N",
     po::value<int>(&num_cpus)->default_value(-1),
     "num cpus for parallel search")
#endif
    ("new-move-probability-in-genmove-logprobability",
     po::value<bool>(&new_move_probability)->default_value(true),
     "use new (experimental) move probability for genmove_probability")
    ("memory-use-percent",
     po::value<double>(&memory_use_percent)->default_value(100.0),
     "percentage for memory use (normally 100)")
    ("help,h", "produce help message")
    ("version", "show version info")
    ;
  po::variables_map vm;
  try
  {
    po::store(po::parse_command_line(argc, argv, options), vm);
    po::notify(vm);
  }
  catch (std::exception& e)
  {
    std::cerr << "error in parsing options" << std::endl
	      << e.what() << std::endl;
    std::cerr << options << std::endl;
    throw;
  }
  if (vm.count("help")) {
    std::cout << options << std::endl;
    return 0;
  }
  if (vm.count("version")) {
    std::cout << "gpsusi " << gpsshogi::gpsshogi_revision << "\n\n"
      << "Copyright (C) 2003-2011 Team GPS.\n";
    return 0;
  }
  osl::OslConfig::setMemoryUsePercent(memory_use_percent);
#ifndef MINIMAL
  if (eval_random)
    OslConfig::setEvalRandom(eval_random);
#endif
  if (vm.count("benchmark")
      || vm.count("benchmark-single")
      || num_benchmark) {
#ifdef OSL_SMP
    if (num_cpus > 0 && num_cpus != OslConfig::numCPUs()
	&& ! vm.count("profile")) {
      std::cerr << "set num-cpus " << OslConfig::numCPUs() << " => " << num_cpus << "\n";
      OslConfig::setNumCPUs(num_cpus);
    }
#endif
    eval::ml::OpenMidEndingEval::setUp();
    progress::ml::NewProgress::setUp();
    rating::StandardFeatureSet::instance();
  }
  if (vm.count("benchmark") || vm.count("benchmark-single")) {
    game_playing::AlphaBeta2OpenMidEndingEvalPlayer player;
    player.setDepthLimit(2000, 400, 200);
    player.setNodeLimit(std::numeric_limits<size_t>::max());
    player.setTableLimit(std::numeric_limits<size_t>::max(), 200);
    benchmark(player, "", benchmark_seconds);
    return 0;
  }
  if (num_benchmark) {
#ifdef OSL_SMP
    if (vm.count("profile"))
      osl::OslConfig::setNumCPUs(1);
#endif
    game_playing::AlphaBeta2OpenMidEndingEvalPlayer player;
    player.setDepthLimit(2000, 400, 200);
    player.setNodeLimit(std::numeric_limits<size_t>::max());
    player.setTableLimit(std::numeric_limits<size_t>::max(), 200);
    benchmark_more(player, num_benchmark, benchmark_seconds);
    return 0;
  }

  OslConfig::setUsiMode();
  OslConfig::setSearchExactValueInOneReply(true); // todo: option
  read_config_file();
#ifdef OSL_SMP
  if (num_cpus > 0 && num_cpus != OslConfig::numCPUs()) {
    std::cerr << "set num-cpus " << OslConfig::numCPUs() << " => " << num_cpus << "\n";
    OslConfig::setNumCPUs(num_cpus);
  }
#endif
  setvbuf(stdout, NULL, _IONBF, 0);
  setvbuf(stdin, NULL, _IONBF, 0);
  if (input_file != "") 
  {
    read_from_file = true;
    std::ifstream is(input_file.c_str());
    play(is, std::cout);
  }
  else 
  {
    play(std::cin, std::cout);
  }
}

// 
bool ThreadRun::goodAlignment()
{
  int dummy __attribute__ ((aligned (16)));
  int offset = (reinterpret_cast<size_t>(&dummy) % 16);
  if (offset) {
    std::cerr << "stack adjust error " << offset << "\n";
    return false;
  }
  return true;
}

game_playing::ComputerPlayer *makePlayer(Player turn, const SimpleState& initial_state)
{
  static osl::record::opening::WeightedBook book(OslConfig::openingBook());
  game_playing::SearchPlayer *search_player = new game_playing::AlphaBeta2OpenMidEndingEvalPlayer;
  dynamic_cast<game_playing::AlphaBeta2OpenMidEndingEvalPlayer&>(*search_player)
    .enableMultiPV(multi_pv_width);
  search_player->setNextIterationCoefficient(byoyomi > 0 ? 1.0 : 1.7);
  if (OslConfig::isMemoryLimitEffective()) 
  {
    search_player->setTableLimit(std::numeric_limits<size_t>::max(), 200);
    search_player->setNodeLimit(std::numeric_limits<size_t>::max());
  }
  else
  {
    search_player->setTableLimit(3000000, 200);
  }
  search_player->setVerbose(is_verbose() ? 2 : 0);
  search_player->setDepthLimit(limit_depth*200, 400, 200);
#ifndef MINIMAL
  if (node_count_hard_limit)
    search_player->setNodeCountHardLimit(node_count_hard_limit);
#endif
  bool use_book = book_depth > 0;
  {
    const SimpleState usual(HIRATE);
    if (! (record::CompactBoard(initial_state) == record::CompactBoard(usual)))
      use_book = false;
  }
  game_playing::ComputerPlayer *result = search_player;
  if (ponder_enabled) {
    result = new game_playing::SpeculativeSearchPlayer(turn, search_player);
    result->allowSpeculativeSearch(true);
  }
  if (use_book) {
    game_playing::WeightTracer *tracer = new game_playing::WeightTracer
      (book, is_verbose(), book_width_root, book_width);
    game_playing::BookPlayer *player
      = new game_playing::BookPlayer(tracer, result);
    player->setBookLimit(book_depth);
    return player;
  }
  else
    return result;
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:

