#include "config.h"

#include <libpq/libpq-fs.h>

#include "connection.h"
#include "statement.h"
#include "result-set.h"
#include "database-metadata.h"
#include "object.h"

#include "gql++/exception.h"

namespace GQL
{

namespace PG
{

PGConnection::PGConnection(PGDriver *driver, const map<string, string>& info)
    : driver_(driver), metadata_(0), read_only_(false), auto_commit_(true), 
      info_(info)
{
  string connect_str;
  map<string, string>::const_iterator it;
  
  for (it = info_.begin(); it != info_.end(); ++it)
  {
    // TODO: escape & quote value
    connect_str += it->first + "=" + it->second; 
    if (it != info.end())
      connect_str += ' ';
  }

  conn_ = PQconnectdb(connect_str.c_str());

  if (PQstatus(conn_) != CONNECTION_OK)
    throw SQLException(string("Connection failed: ") + PQerrorMessage(conn_));

  driver_->reference();
}

PGConnection::~PGConnection()
{
  if (!auto_commit_)
    commit();
  
  PQfinish(conn_);
  
  driver_->unreference();
  
  if (metadata_)
    metadata_->unreference();
}

Statement *PGConnection::create_statement()
{
  return manage(new PGStatement(this));;
}

PreparedStatement *PGConnection::prepare_statement(const string& url)
{
  return 0;
}

CallableStatement * PGConnection::prepare_call(const string& url)
{
  return 0;
}

void PGConnection::set_auto_commit(bool auto_commit)
{
  if (auto_commit_ == auto_commit)
    return;

  if (auto_commit == false)
  {
    // PostgreSQL is always in autocommit mode, one has to "BEGIN" a
    // transaction explicitly
    PQclear(exec_sql("BEGIN TRANSACTION"));
  }
  
  auto_commit_ = auto_commit;
}

void PGConnection::commit()
{
  PQclear(exec_sql("COMMIT"));
  if (auto_commit_ == false)
    PQclear(exec_sql("BEGIN TRANSACTION"));
}

void PGConnection::rollback()
{
  PQclear(exec_sql("ROLLBACK"));
  if (auto_commit_ == false)
    PQclear(exec_sql("BEGIN TRANSACTION"));
}

string PGConnection::get_catalog() const
{
  string empty;
  
  return empty;
}

string PGConnection::native_sql(const string& sql) const
{
  return sql;
}

DatabaseMetaData *PGConnection::get_meta_data()
{
  if (!metadata_)
  {
    metadata_ = manage(new PGDatabaseMetaData(this));
    metadata_->reference();
  }
  return metadata_;
}

SQLObject *PGConnection::create_object()
{
  return manage(new PGSQLObject(this, conn_));
}

SQLObject *PGConnection::create_blob()
{
  Oid  oid = lo_creat(conn_, INV_READ | INV_WRITE);
  
  PGSQLObject *result = manage(new PGSQLObject(this, conn_));
  result->from_int(oid);
  
  return result;
}

void PGConnection::destroy_blob(SQLObject *blob)
{
  int status;
  
  Oid oid = blob->to_int();
  status = lo_unlink(conn_, oid);
  
  if (status == -1)
    throw SQLException(PQerrorMessage(conn_));

  blob->set_null();
}

PGresult *PGConnection::exec_sql(const string& sql)
{
  PGresult *res;
  int status;
  
  if (sql.length() > 8192)
    throw SQLException("SQL Statement too long: " + sql);
  
  res = PQexec(conn_, sql.c_str());
  
  if (!res)
    throw SQLException(PQerrorMessage(conn_));
  
  
  status = PQresultStatus(res);
  if (status != PGRES_COMMAND_OK && status != PGRES_TUPLES_OK)
  {
    throw SQLException(PQerrorMessage(conn_));
  }
  
  return res;
}

}
}
