#include <libpq/libpq-fs.h>

#include "object.h"
#include "connection.h"

namespace GQL
{

namespace PG
{

Blob *PGSQLObject::to_blob() const
{
  Oid oid = to_int();

  return manage(new PGBlob(oid, conn_, pg_conn_));
}

void PGSQLObject::from_blob(const Blob *blob)
{
  Oid oid = InvalidOid;

  if (blob == 0)
    oid = lo_creat(pg_conn_, INV_READ | INV_WRITE);
  else
  {
    const PGBlob *pg_blob = dynamic_cast<const PGBlob *>(blob);
    if (pg_blob != 0)
      oid = pg_blob->oid();
  }
  
  if (oid == InvalidOid)
    throw SQLException("invalid BLOB encountered");
    
  from_int(oid);
}


PGBlob::PGBlob(Oid oid, PGConnection *conn, PGconn *pg_conn)
{
  oid_ = oid;
  conn_ = conn;
  pg_conn_ = pg_conn;
  fd_ = -1;
}

PGBlob::~PGBlob()
{
  close();
}

void PGBlob::open(openmode om)
{
  int mode = 0;
  
  if (om & in)
    mode |= INV_READ;
  if (om & out)
    mode |= INV_WRITE;
  
  if (fd_ >= 0)
    close();
  
  fd_ = lo_open(pg_conn_, oid_, mode);
}

void PGBlob::close()
{
  if (fd_ < 0)
    return;
  
  lo_close(pg_conn_, fd_);
  fd_ = -1;
}

int PGBlob::write(const void *data, int len)
{
  if (fd_ < 0)
    return -1;

  return lo_write(pg_conn_, fd_, (char *)data, len);
}

int PGBlob::read(void *data, int len)
{
  if (fd_ < 0)
    return -1;

  return lo_read(pg_conn_, fd_, (char *)data, len);
}

int PGBlob::seek(int offset, seek_dir w)
{
  int whence = (w == cur ? SEEK_CUR : w == beg ? SEEK_SET : SEEK_END);
  
  if (fd_ < 0)
    return -1;
  
  return lo_lseek(pg_conn_, fd_, offset, whence);
}

int PGBlob::tell() const
{
  if (fd_ < 0)
    return -1;

  return lo_tell(pg_conn_, fd_);
}

}
}


