### Copyright (C) 2010 Peter Williams <peter_ono@users.sourceforge.net>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk, gobject, collections, os

from gquilt_pkg import tlview, dialogue, actions, gutils, ifce, utils
from gquilt_pkg import console, cmd_result, ws_event, icons, diff
from gquilt_pkg import text_edit

Row = collections.namedtuple('Row',
    ['name', 'is_dir', 'style', 'foreground', 'icon', 'status', 'origin'])

_MODEL_TEMPLATE = Row(
    name=gobject.TYPE_STRING,
    is_dir=gobject.TYPE_BOOLEAN,
    style=gobject.TYPE_INT,
    foreground=gobject.TYPE_STRING,
    icon=gobject.TYPE_STRING,
    status=gobject.TYPE_STRING,
    origin=gobject.TYPE_STRING
)

_NAME = tlview.model_col(_MODEL_TEMPLATE, 'name')
_IS_DIR = tlview.model_col(_MODEL_TEMPLATE, 'is_dir')
_STYLE = tlview.model_col(_MODEL_TEMPLATE, 'style')
_FOREGROUND = tlview.model_col(_MODEL_TEMPLATE, 'foreground')
_ICON = tlview.model_col(_MODEL_TEMPLATE, 'icon')
_STATUS = tlview.model_col(_MODEL_TEMPLATE, 'status')
_ORIGIN = tlview.model_col(_MODEL_TEMPLATE, 'origin')

_FILE_ICON = {True : gtk.STOCK_DIRECTORY, False : gtk.STOCK_FILE}

def _get_status_deco(status=None):
    try:
        return ifce.ifce.status_deco_map[status]
    except KeyError:
        return ifce.ifce.status_deco_map[None]

def _generate_row_tuple(data, isdir=None):
    deco = _get_status_deco(data.status)
    row = Row(
        name=data.name,
        is_dir=isdir,
        icon=_FILE_ICON[isdir],
        status=data.status,
        origin=data.origin,
        style=deco.style,
        foreground=deco.foreground
    )
    return row

class Store(tlview.TreeStore):
    def __init__(self, show_hidden=False, populate_all=False, auto_expand=False):
        tlview.TreeStore.__init__(self, _MODEL_TEMPLATE)
        self._file_db = None
        # If 'view' this isn't set explicitly it will be set automatically
        # when any row is expanded
        self.view = None
        self._populate_all = populate_all
        self._auto_expand = auto_expand
        self.show_hidden_action = gtk.ToggleAction('show_hidden_files', 'Show Hidden Files',
                                                   'Show/hide ignored files and those beginning with "."', None)
        self.show_hidden_action.set_active(show_hidden)
        self.show_hidden_action.connect('toggled', self._toggle_show_hidden_cb)
        self.show_hidden_action.set_menu_item_type(gtk.CheckMenuItem)
        self.show_hidden_action.set_tool_item_type(gtk.ToggleToolButton)
    def set_view(self, view):
        self.view = view
        if self._expand_new_rows():
            self.view.expand_all()
    def _expand_new_rows(self):
        return self._auto_expand and self.view is not None
    def _row_expanded(self, dir_iter):
        # if view isn't set then assume that we aren't connexted to a view
        # so the row can't be expanded
        if self.view is None:
            return False
        else:
            return self.view.row_expanded(self.get_path(dir_iter))
    def _update_iter_row_tuple(self, fsobj_iter, to_tuple):
        for index in [_STYLE, _FOREGROUND, _STATUS, _ORIGIN]:
            self.set_value(fsobj_iter, index, to_tuple[index])
    def _toggle_show_hidden_cb(self, toggleaction):
        self._update_dir('', None)
    def fs_path(self, fsobj_iter):
        if fsobj_iter is None:
            return None
        parent_iter = self.iter_parent(fsobj_iter)
        name = self.get_value(fsobj_iter, _NAME)
        if parent_iter is None:
            return name
        else:
            if name is None:
                return os.path.join(self.fs_path(parent_iter), '')
            return os.path.join(self.fs_path(parent_iter), name)
    def _get_file_paths(self, fsobj_iter, path_list):
        while fsobj_iter != None:
            if not self.get_value(fsobj_iter, _IS_DIR):
                path_list.append(self.fs_path(fsobj_iter))
            else:
                child_iter = self.iter_children(fsobj_iter)
                if child_iter != None:
                    self._get_file_paths(child_iter, path_list)
            fsobj_iter = self.iter_next(fsobj_iter)
    def get_file_paths(self):
        path_list = []
        self._get_file_paths(self.get_iter_first(), path_list)
        return path_list
    def _recursive_remove(self, fsobj_iter):
        child_iter = self.iter_children(fsobj_iter)
        if child_iter != None:
            while self._recursive_remove(child_iter):
                pass
        return self.remove(fsobj_iter)
    def _remove_place_holder(self, dir_iter):
        child_iter = self.iter_children(dir_iter)
        if child_iter and self.get_value(child_iter, _NAME) is None:
            self.remove(child_iter)
    def _insert_place_holder(self, dir_iter):
        self.append(dir_iter)
    def _insert_place_holder_if_needed(self, dir_iter):
        if self.iter_n_children(dir_iter) == 0:
            self._insert_place_holder(dir_iter)
    def _populate(self, dirpath, parent_iter):
        dirs, files = self._file_db.dir_contents(dirpath, self.show_hidden_action.get_active())
        for dirdata in dirs:
            row_tuple = _generate_row_tuple(dirdata, True)
            dir_iter = self.append(parent_iter, row_tuple)
            if self._populate_all:
                self._populate(os.path.join(dirpath, dirdata.name), dir_iter)
                if self._expand_new_rows():
                    self.view.expand_row(self.get_path(dir_iter), True)
            else:
                self._insert_place_holder(dir_iter)
        for filedata in files:
            row_tuple = _generate_row_tuple(filedata, False)
            dummy = self.append(parent_iter, row_tuple)
        if parent_iter is not None:
            self._insert_place_holder_if_needed(parent_iter)
    def _update_dir(self, dirpath, parent_iter=None):
        if parent_iter is None:
            child_iter = self.get_iter_first()
        else:
            child_iter = self.iter_children(parent_iter)
            if child_iter:
                if self.get_value(child_iter, _NAME) is None:
                    child_iter = self.iter_next(child_iter)
        dirs, files = self._file_db.dir_contents(dirpath, self.show_hidden_action.get_active())
        dead_entries = []
        for dirdata in dirs:
            row_tuple = _generate_row_tuple(dirdata, True)
            while (child_iter is not None) and self.get_value(child_iter, _IS_DIR) and (self.get_value(child_iter, _NAME) < dirdata.name):
                dead_entries.append(child_iter)
                child_iter = self.iter_next(child_iter)
            if child_iter is None:
                dir_iter = self.append(parent_iter, row_tuple)
                if self._populate_all:
                    self._update_dir(os.path.join(dirpath, dirdata.name), dir_iter)
                    if self._expand_new_rows():
                        self.view.expand_row(self.get_path(dir_iter), True)
                else:
                    self._insert_place_holder(dir_iter)
                continue
            name = self.get_value(child_iter, _NAME)
            if (not self.get_value(child_iter, _IS_DIR)) or (name > dirdata.name):
                dir_iter = self.insert_before(parent_iter, child_iter, row_tuple)
                if self._populate_all:
                    self._update_dir(os.path.join(dirpath, dirdata.name), dir_iter)
                    if self._expand_new_rows():
                        self.view.expand_row(self.get_path(dir_iter), True)
                else:
                    self._insert_place_holder(dir_iter)
                continue
            self._update_iter_row_tuple(child_iter, row_tuple)
            if self._populate_all or self._row_expanded(child_iter):
                self._update_dir(os.path.join(dirpath, name), child_iter)
            child_iter = self.iter_next(child_iter)
        while (child_iter is not None) and self.get_value(child_iter, _IS_DIR):
            dead_entries.append(child_iter)
            child_iter = self.iter_next(child_iter)
        for filedata in files:
            row_tuple = _generate_row_tuple(filedata, False)
            while (child_iter is not None) and (self.get_value(child_iter, _NAME) < filedata.name):
                dead_entries.append(child_iter)
                child_iter = self.iter_next(child_iter)
            if child_iter is None:
                dummy = self.append(parent_iter, row_tuple)
                continue
            if self.get_value(child_iter, _NAME) > filedata.name:
                dummy = self.insert_before(parent_iter, child_iter, row_tuple)
                continue
            self._update_iter_row_tuple(child_iter, row_tuple)
            child_iter = self.iter_next(child_iter)
        while child_iter is not None:
            dead_entries.append(child_iter)
            child_iter = self.iter_next(child_iter)
        for dead_entry in dead_entries:
            self._recursive_remove(dead_entry)
        if parent_iter is not None:
            self._insert_place_holder_if_needed(parent_iter)
    def _get_file_db(self):
        assert 0, '_get_file_db() must be defined in descendants'
    def repopulate(self):
        self._file_db = self._get_file_db()
        self.clear()
        self._populate('', self.get_iter_first())
    def update(self):
        self._file_db = self._get_file_db()
        self._update_dir('', None)
    def on_row_expanded_cb(self, view, dir_iter, dummy):
        self.view = view
        if not self._populate_all:
            self._update_dir(self.fs_path(dir_iter), dir_iter)
            if self.iter_n_children(dir_iter) > 1:
                self._remove_place_holder(dir_iter)
    def on_row_collapsed_cb(self, view, dir_iter, dummy):
        self._insert_place_holder_if_needed(dir_iter)

def _format_file_name_crcb(_column, cell_renderer, store, tree_iter, _arg=None):
    name = store.get_value(tree_iter, _NAME)
    xinfo = store.get_value(tree_iter, _ORIGIN)
    if xinfo:
        name += ' <- %s' % xinfo
    cell_renderer.set_property('text', name)

_VIEW_TEMPLATE = tlview.ViewTemplate(
    properties={'headers-visible' : False},
    selection_mode=gtk.SELECTION_MULTIPLE,
    columns=[
        tlview.Column(
            title='File Name',
            properties={},
            cells=[
                tlview.Cell(
                    creator=tlview.CellCreator(
                        function=gtk.CellRendererPixbuf,
                        expand=False,
                        start=True
                    ),
                    properties={},
                    renderer=None,
                    attributes={'stock-id' : _ICON}
                ),
                tlview.Cell(
                    creator=tlview.CellCreator(
                        function=gtk.CellRendererText,
                        expand=False,
                        start=True
                    ),
                    properties={},
                    renderer=None,
                    attributes={'text' : _STATUS, 'style' : _STYLE, 'foreground' : _FOREGROUND}
                ),
                tlview.Cell(
                    creator=tlview.CellCreator(
                        function=gtk.CellRendererText,
                        expand=False,
                        start=True
                    ),
                    properties={},
                    renderer=tlview.Renderer(function=_format_file_name_crcb, user_data=None),
                    attributes={'style' : _STYLE, 'foreground' : _FOREGROUND}
                )
            ]
        )
    ]
)

UI_DESCR = \
'''
<ui>
  <menubar name='files_left_menubar'>
    <menu name='files_menu' action='menu_files'>
        <placeholder name='tail_end'/>
        <menuitem action='refresh_files'/>
        <menuitem action='auto_refresh_files'/>
    </menu>
  </menubar>
  <menubar name='files_right_menubar'/>
  <popup name='files_popup'>
    <placeholder name='selection_indifferent'/>
    <separator/>
    <placeholder name='selection'/>
    <separator/>
    <placeholder name='selection_not_patched'/>
    <separator/>
    <placeholder name='unique_selection'/>
    <separator/>
    <placeholder name='no_selection'/>
    <separator/>
    <placeholder name='no_selection_not_patched'/>
    <separator/>
    <placeholder name='popup_tail_end'/>
        <menuitem action='refresh_files'/>
  </popup>
  <toolbar name='files_housekeeping_toolbar'>
    <toolitem action='refresh_files'/>
    <separator/>
    <toolitem action='show_hidden_files'/>
  </toolbar>
  <toolbar name='files_refresh_toolbar'>
    <toolitem action='refresh_files'/>
  </toolbar>
</ui>
'''

_KEYVAL_c = gtk.gdk.keyval_from_name('c')
_KEYVAL_C = gtk.gdk.keyval_from_name('C')
_KEYVAL_ESCAPE = gtk.gdk.keyval_from_name('Escape')

class Tree(gtk.VBox, dialogue.BusyIndicatorUser, actions.AGandUIManager, ws_event.Listener):
    def __init__(self, model, busy_indicator=None, auto_refresh=False, show_hidden=False):
        gtk.VBox.__init__(self)
        if model:
            self.model = model
            self.model.show_hidden_action.set_active(show_hidden)
        else:
            self.model = Store(show_hidden=show_hidden)
        self.view = tlview.View(_VIEW_TEMPLATE, self.model)
        self.model.set_view(self.view)
        self.seln = self.view.get_selection()
        dialogue.BusyIndicatorUser.__init__(self, busy_indicator)
        actions.AGandUIManager.__init__(self, self.seln)
        ws_event.Listener.__init__(self)
        self._refresh_interval = 60000 # milliseconds
        self.auto_refresh_action = gtk.ToggleAction('auto_refresh_files', 'Auto Refresh',
                                                   'Automatically/periodically refresh file display', None)
        self.auto_refresh_action.set_active(auto_refresh)
        self.auto_refresh_action.connect('toggled', self._toggle_auto_refresh_cb)
        self.auto_refresh_action.set_menu_item_type(gtk.CheckMenuItem)
        self.auto_refresh_action.set_tool_item_type(gtk.ToggleToolButton)
        self.add_conditional_action(actions.ON_PGND_INDEP_SELN_INDEP, self.auto_refresh_action)
        self.add_conditional_actions(actions.ON_PGND_INDEP_SELN_INDEP,
            [
                ('menu_files', None, '_Files'),
                ('refresh_files', gtk.STOCK_REFRESH, '_Refresh', None,
                 'Refresh/update the file tree display', self._update_tree_cb),
            ]
        )
        self.add_conditional_action(actions.ON_PGND_INDEP_SELN_INDEP, self.model.show_hidden_action)
        self.ui_manager.add_ui_from_string(UI_DESCR)
        hbox = gtk.HBox()
        hbox.pack_start(self.ui_manager.get_widget('/files_left_menubar'))
        hbox.pack_end(self.ui_manager.get_widget('/files_right_menubar'))
        self.pack_start(hbox, False, False)
        self.pack_start(gutils.wrap_in_scrolled_window(self.view))
        self.view.connect('row-expanded', self.model.on_row_expanded_cb)
        self.view.connect('row-collapsed', self.model.on_row_collapsed_cb)
        self.view.connect('button_press_event', self._handle_button_press_cb)
        self.view.connect('key_press_event', self._key_press_cb)
        self.show_all()
        self.model.repopulate()
    def _do_auto_refresh(self):
        if self.auto_refresh_action.get_active():
            self.model.update()
            return True
        else:
            return False
    def _toggle_auto_refresh_cb(self, action=None):
        if self.auto_refresh_action.get_active():
            gobject.timeout_add(self._refresh_interval, self._do_auto_refresh)
    def _update_action_visibility(self):
        pass
    def _handle_button_press_cb(self, widget, event):
        if event.type == gtk.gdk.BUTTON_PRESS:
            if event.button == 3:
                self._update_action_visibility()
                menu = self.ui_manager.get_widget('/files_popup')
                menu.popup(None, None, None, event.button, event.time)
                return True
            elif event.button == 2:
                self.get_selection().unselect_all()
                return True
        elif event.type == gtk.gdk._2BUTTON_PRESS:
            if event.button == 1:
                self._handle_double_click()
                return True
        return False
    def _handle_double_click(self):
        pass
    def _key_press_cb(self, widget, event):
        if gtk.gdk.CONTROL_MASK & event.state:
            if event.keyval in [_KEYVAL_c, _KEYVAL_C]:
                self.add_selected_files_to_clipboard()
                return True
        elif event.keyval == _KEYVAL_ESCAPE:
            self.seln.unselect_all()
            return True
    def get_selected_files(self):
        store, selection = self.seln.get_selected_rows()
        return [store.fs_path(store.get_iter(x)) for x in selection]
    def add_selected_files_to_clipboard(self, clipboard=None):
        if not clipboard:
            clipboard = gtk.clipboard_get(gtk.gdk.SELECTION_CLIPBOARD)
        sel = utils.file_list_to_string(self.get_selected_files())
        clipboard.set_text(sel)
        return False
    def _update_tree_cb(self, _arg=None):
        self.show_busy()
        self.model.update()
        self.unshow_busy()
    def _repopulate_tree_cb(self, _arg=None):
        self.show_busy()
        self.model.repopulate()
        self.unshow_busy()

WS_FILES_UI_DESCR = \
'''
<ui>
  <menubar name='files_left_menubar'>
    <menu name='files_menu' action='menu_files'>
      <placeholder name='tail_end'/>
      <menuitem action='refresh_files'/>
      <menuitem action='auto_refresh_files'/>
    </menu>
  </menubar>
  <menubar name='files_right_menubar'/>
  <popup name='files_popup'>
    <placeholder name='selection_indifferent'/>
      <menuitem action='new_file'/>
    <separator/>
    <placeholder name='selection'/>
      <menuitem action='edit_files'/>
      <menuitem action='peruse_files'/>
      <menuitem action='add_files'/>
      <menuitem action='copy_files'/>
      <menuitem action='move_files'/>
    <separator/>
    <placeholder name='selection_not_patched'/>
    <separator/>
    <placeholder name='unique_selection'/>
    <separator/>
    <placeholder name='no_selection'/>
    <separator/>
    <placeholder name='no_selection_not_patched'/>
    <separator/>
    <placeholder name='popup_tail_end'/>
        <menuitem action='refresh_files'/>
  </popup>
  <toolbar name='files_housekeeping_toolbar'>
    <toolitem action='refresh_files'/>
    <separator/>
    <toolitem action='show_hidden_files'/>
  </toolbar>
  <toolbar name='files_refresh_toolbar'>
    <toolitem action='refresh_files'/>
  </toolbar>
</ui>
'''

class WSStore(Store):
    def __init__(self):
        Store.__init__(self)
    def _get_file_db(self):
        return ifce.ifce.get_ws_file_db()

class WSTree(Tree):
    def __init__(self):
        Tree.__init__(self, WSStore())
        hbox = gtk.HBox()
        button = gtk.CheckButton()
        action = self.get_conditional_action('show_hidden_files')
        action.connect_proxy(button)
        gutils.set_widget_tooltip_text(button, action.get_property('tooltip'))
        hbox.show_all()
        hbox.pack_start(button)
        self.pack_start(hbox, expand=False)
        self.add_conditional_actions(actions.ON_IN_PGND_PMIC_SELN_INDEP,
            [
                ('new_file', gtk.STOCK_NEW, '_New', None,
                 'Add a new file to the top patch', self._add_new_file_to_top_patch),
            ]
        )
        self.add_conditional_actions(actions.ON_IN_PGND_PMIC_SELN,
            [
                ('add_files', gtk.STOCK_ADD, '_Add', None,
                 'Add the selected files to the top patch', self._add_selection_to_top_patch),
                ('copy_files', gtk.STOCK_COPY, '_Copy', None,
                 'Make copies of the selected files in the top patch',  self._copy_selection_to_top_patch),
                ('move_files', gtk.STOCK_DND, '_Move/Rename', None,
                 'Move/rename the selected files in the top patch', self._move_selection_to_top_patch),
                ('edit_files', gtk.STOCK_EDIT, '_Edit', None,
                 'Edit the selected files after adding them to the top patch', self._edit_selection_in_top_patch),
            ]
        )
        self.add_conditional_actions(actions.ON_PGND_INDEP_SELN,
            [
                ('peruse_files', gtk.STOCK_FILE, '_Peruse', None,
                 'Peruse the selected files', self._peruse_selection),
            ]
        )
        self.ui_manager.add_ui_from_string(WS_FILES_UI_DESCR)
        self.add_notification_cb(ws_event.CHANGE_WD, self._repopulate_tree_cb)
        self.add_notification_cb(ws_event.FILE_CHANGES|ws_event.PATCH_REFRESH|ws_event.PATCH_POP|ws_event.PATCH_PUSH, self._update_tree_cb)
    def _handle_double_click(self):
        self._edit_selection_in_top_patch()
    def _update_action_visibility(self):
        self.get_conditional_action('add_files').set_visible(ifce.ifce.has_add_files())
    def _add_new_file_to_top_patch(self, _action=None):
        _add_new_file_to_patch(self)
    def _add_selection_to_top_patch(self, _action=None):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        filelist = []
        dirs = ""
        for f in files:
            if os.path.isdir(f):
                dirs += os.linesep + f
            else:
                filelist.append(f)
        if dirs != "":
            emsg = os.linesep.join(["The directories:", dirs, "",
                                    "have been removed from addition"])
            if not dialogue.ask_ok_cancel(emsg, parent=gutils.get_gtk_window(self)):
                return
        if len(filelist) > 0:
            self.show_busy()
            res, sout, serr = ifce.ifce.do_add_files_to_patch(filelist)
            self.unshow_busy()
            if res != cmd_result.OK:
                dialogue.inform_user(os.linesep.join([sout, serr]))
    def _copy_selection_to_top_patch(self, _action=None):
        _copy_files_to_top_patch(self, self.get_selected_files(), move=False)
    def _move_selection_to_top_patch(self, _action=None):
        _copy_files_to_top_patch(self, self.get_selected_files(), move=True)
    def _edit_selection_in_top_patch(self, _action=None):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        filelist = []
        dirs = ""
        for f in files:
            if os.path.isdir(f):
                dirs += os.linesep + f
            else:
                filelist.append(f)
        if dirs != "":
            emsg = os.linesep.join(["The directories:", dirs, "",
                                    "have been removed from selection"])
            if not dialogue.ask_ok_cancel(emsg, parent=gutils.get_gtk_window(self)):
                return
        if len(filelist) > 0:
            res, sout, serr = ifce.ifce.get_patch_files(None, False)
            if res != cmd_result.OK:
                dialogue.inform_user(os.linesep.join([sout, serr]))
                return
            added_files = []
            for f in filelist:
                if not f in sout:
                    added_files.append(f)
            if len(added_files) > 0:
                res, sout, serr = ifce.ifce.do_add_files_to_patch(added_files)
                if res is not cmd_result.OK:
                    dialogue.inform_user(os.linesep.join([sout, serr]))
            text_edit.edit_files_extern(filelist)
    def _peruse_selection(self, _action=None):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        text_edit.peruse_files_extern(files)

def _add_new_file_to_patch(caller):
    name = dialogue.ask_file_name('New file ...', existing=False)
    if name is None:
        return
    if os.path.isdir(name):
        dialogue.inform_user("Cannot add directories to patch", parent=gutils.get_gtk_window(caller))
        return
    lname = utils.path_relative_to_playground(name)
    if lname == None:
        dialogue.inform_user("File is outside playground", parent=gutils.get_gtk_window(caller))
        return
    if not os.path.exists(lname):
        created = True
        if console.LOG:
            console.LOG.exec_cmd("touch " + lname)
        else:
            utils.run_cmd("touch " + lname)
    else:
        created = False
    caller.show_busy()
    res, sout, serr = ifce.ifce.do_add_files_to_patch([lname])
    caller.unshow_busy()
    if res is not cmd_result.OK:
        if created:
            try:
                os.remove(lname)
            except OSError:
                pass
        dialogue.inform_user(os.linesep.join([sout, serr]), parent=gutils.get_gtk_window(caller))

def _copy_files_to_top_patch(caller, files, move=False):
    if len(files) == 0:
        return
    elif len(files) == 1:
        dest = dialogue.ask_file_name("Destination", existing=False, suggestion=files[0])
    else:
        dest = dialogue.ask_dir_name("Destination")
    if dest is None:
        return
    ldest = utils.path_relative_to_playground(dest)
    if ldest == None:
        dialogue.inform_user("Destination is outside playground", parent=gutils.get_gtk_window(caller))
        return
    for fname in files:
        caller.show_busy()
        if move:
            res, sout, serr = ifce.ifce.do_move_file(fname, ldest)
        else:
            res, sout, serr = ifce.ifce.do_copy_file(fname, ldest)
        caller.unshow_busy()
        if res & cmd_result.SUGGEST_FORCE:
            if dialogue.ask_force_or_cancel((res, sout, serr), parent=gutils.get_gtk_window(caller)) == dialogue.RESPONSE_FORCE:
                caller.show_busy()
                res, sout, serr = ifce.ifce.do_copy_file(fname, ldest, True)
                caller.unshow_busy()
        elif res != cmd_result.OK:
            if not dialogue.ask_ok_cancel(os.linesep.join([sout, serr]), parent=gutils.get_gtk_window(caller)):
                break

PATCH_FILES_UI_DESCR = \
'''
<ui>
  <menubar name='files_left_menubar'>
    <menu name='files_menu' action='menu_files'>
        <placeholder name='tail_end'/>
        <menuitem action='refresh_files'/>
        <menuitem action='auto_refresh_files'/>
    </menu>
  </menubar>
  <menubar name='files_right_menubar'/>
  <popup name='files_popup'>
    <placeholder name='selection_indifferent'/>
    <separator/>
    <placeholder name='selection'/>
        <menuitem action='diff_files'/>
        <menuitem action='ediff_files'/>
    <separator/>
    <placeholder name='selection_not_patched'/>
    <separator/>
    <placeholder name='unique_selection'/>
    <separator/>
    <placeholder name='no_selection'/>
    <separator/>
    <placeholder name='no_selection_not_patched'/>
    <separator/>
    <placeholder name='popup_tail_end'/>
        <menuitem action='refresh_files'/>
  </popup>
  <toolbar name='files_housekeeping_toolbar'>
    <toolitem action='refresh_files'/>
  </toolbar>
  <toolbar name='files_refresh_toolbar'>
    <toolitem action='refresh_files'/>
  </toolbar>
</ui>
'''

class PatchStore(Store):
    def __init__(self, patch=None):
        Store.__init__(self, show_hidden=True, populate_all=True, auto_expand=True)
        self.patch = patch
    def _get_file_db(self):
        return ifce.ifce.get_patch_file_db(patch=self.patch)

class PatchTree(Tree):
    def __init__(self, patch=None, busy_indicator=None):
        Tree.__init__(self, PatchStore(patch=patch), show_hidden=True, busy_indicator=busy_indicator)
        self.add_conditional_actions(actions.ON_IN_PGND_PMIC_SELN,
            [
                ('diff_files', icons.DIFF, '_Diff', None,
                 'Display diff for selected files', self._display_diff),
                ('ediff_files', icons.MELD, '_Meld', None,
                 'Display diff for selected files using "meld"',  self._display_diff_with_meld),
            ]
        )
        self.ui_manager.add_ui_from_string(PATCH_FILES_UI_DESCR)
        self.add_notification_cb(ws_event.CHANGE_WD|ws_event.PATCH_POP|ws_event.PATCH_PUSH, self._repopulate_tree_cb)
        self.add_notification_cb(ws_event.FILE_CHANGES|ws_event.PATCH_REFRESH, self._update_tree_cb)
    def _update_action_visibility(self):
        self.get_conditional_action('ediff_files').set_visible(utils.which("meld") is not None)
    def _display_diff(self, _arg):
        files = self.get_selected_files()
        _parent_window = gutils.get_gtk_window(self)
        dialog = diff.PmDiffTextDialog(parent=_parent_window, patch=self.model.patch, file_list=files)
        dialog.show()
    def _display_diff_with_meld(self, _arg):
        files = self.get_selected_files()
        ifce.ifce.display_files_diff_in_viewer("meld", files, self.model.patch)

class PatchFilesDialog(dialogue.AmodalDialog):
    def __init__(self, patch):
        dialogue.AmodalDialog.__init__(self, None, None,
                                       gtk.DIALOG_DESTROY_WITH_PARENT,
                                       (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
        self.set_title('patch: %s files: %s' % (patch, utils.cwd_rel_home()))
        # file tree view wrapped in scrolled window
        self.file_tree = PatchTree(busy_indicator=self, patch=patch)
        self.file_tree.seln.set_mode(gtk.SELECTION_MULTIPLE)
        self.file_tree.view.set_headers_visible(False)
        self.file_tree.view.set_size_request(240, 320)
        self.vbox.pack_start(self.file_tree)
        self.connect("response", self._close_cb)
        self.show_all()
    def _close_cb(self, dialog, response_id):
        self.destroy()

MUTABLE_PATCH_FILES_UI_DESCR = \
'''
<ui>
  <menubar name='files_left_menubar'>
    <menu name='files_menu' action='menu_files'>
        <placeholder name='tail_end'/>
        <menuitem action='refresh_files'/>
        <menuitem action='auto_refresh_files'/>
    </menu>
  </menubar>
  <menubar name='files_right_menubar'/>
  <popup name='files_popup'>
    <placeholder name='selection_indifferent'/>
    <separator/>
    <placeholder name='selection'/>
        <menuitem action='edit_files'/>
        <menuitem action='copy_files'/>
        <menuitem action='move_files'/>
        <menuitem action='remove_files'/>
        <menuitem action='delete_files'/>
        <menuitem action='resolve_files'/>
        <menuitem action='revert_files'/>
        <menuitem action='diff_files'/>
        <menuitem action='ediff_files'/>
    <separator/>
    <placeholder name='selection_not_patched'/>
    <separator/>
    <placeholder name='unique_selection'/>
    <separator/>
    <placeholder name='no_selection'/>
    <separator/>
    <placeholder name='no_selection_not_patched'/>
    <separator/>
    <placeholder name='popup_tail_end'/>
        <menuitem action='refresh_files'/>
  </popup>
  <toolbar name='files_housekeeping_toolbar'>
    <toolitem action='refresh_files'/>
  </toolbar>
  <toolbar name='files_refresh_toolbar'>
    <toolitem action='refresh_files'/>
  </toolbar>
</ui>
'''

class MutablePatchTree(PatchTree):
    def __init__(self, patch=None):
        PatchTree.__init__(self, patch=patch)
        self.patchname = patch
        self.add_conditional_actions(actions.ON_IN_PGND_PMIC_SELN_INDEP,
            [
                ('new_file', gtk.STOCK_NEW, '_New', None,
                 'Add a new file to the patch', self._add_new_file_to_patch),
            ]
        )
        self.add_conditional_actions(actions.ON_IN_PGND_PMIC_SELN,
            [
                ('remove_files', gtk.STOCK_REMOVE, '_Remove', None,
                 'Remove the selected files from the top patch', self._remove_selection_from_patch),
                ('copy_files', gtk.STOCK_COPY, '_Copy', None,
                 'Make copies of the selected files in the patch',  self._copy_selection),
                ('move_files', gtk.STOCK_DND, '_Move/Rename', None,
                 'Move/rename the selected files in the patch', self._move_selection),
                ('edit_files', gtk.STOCK_EDIT, '_Edit', None,
                 'Edit the selected files', self._edit_selected_files),
                ('resolve_files', gtk.STOCK_EDIT, 'Re_solve', None,
                 'Resolve problems with selected files in the top patch', self._resolve_selected_files),
                ('revert_files', gtk.STOCK_UNDO, 'Re_vert', None,
                 'Revert the selected files to their state before the last refresh', self._revert_selection_in_patch),
                ('delete_files', gtk.STOCK_DELETE, '_Delete', None,
                 'Delete the selected files', self._delete_selection),
            ]
        )
        self.ui_manager.add_ui_from_string(MUTABLE_PATCH_FILES_UI_DESCR)
    def _update_action_visibility(self):
        self.get_conditional_action('ediff_files').set_visible(utils.which("meld") is not None)
    def _display_diff(self, _arg):
        files = self.get_selected_files()
        _parent_window = gutils.get_gtk_window(self)
        dialog = diff.PmDiffTextDialog(parent=_parent_window, patch=self.model.patch, file_list=files)
        dialog.show()
    def _display_diff_with_meld(self, _arg):
        files = self.get_selected_files()
        ifce.ifce.display_files_diff_in_viewer("meld", files, self.model.patch)
    def _add_new_file_to_patch(self, _action=None):
        _add_new_file_to_patch(self)
    def _remove_selection_from_patch(self, _arg):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        elif len(files) == 1:
            emsg = os.linesep.join([files[0], "",
                                   "Confirm remove selected file from patch?"])
        else:
            emsg = os.linesep.join(files + ["", "Confirm remove selected files from patch?"])
        if not dialogue.ask_ok_cancel(emsg, parent=gutils.get_gtk_window(self)):
            return
        self.show_busy()
        res, sout, serr = ifce.ifce.do_remove_files_from_patch(files, self.patchname)
        self.unshow_busy()
        if res is not cmd_result.OK:
            dialogue.inform_user(os.linesep.join([sout, serr]))
    def _copy_selection(self, _arg):
        _copy_files_to_top_patch(self, self.get_selected_files(), move=False)
    def _move_selection(self, _arg):
        _copy_files_to_top_patch(self, self.get_selected_files(), move=True)
    def _edit_selected_files(self, _arg=None):
        files = self.get_selected_files()
        text_edit.edit_files_extern(files)
    def _resolve_selected_files(self, _arg=None):
        files = self.get_selected_files()
        if len(files) == 0:
            res, files, serr = ifce.ifce.get_patch_files(None, False)
            if res != cmd_result.OK:
                dialogue.inform_user(serr)
                return
        files_plus_rej = []
        for f in files:
            files_plus_rej.append(f)
            rejf = f + os.extsep + "rej"
            if os.path.exists(rejf):
                files_plus_rej.append(rejf)
        text_edit.edit_files_extern(files_plus_rej)
    def _revert_selection_in_patch(self, _arg):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        elif len(files) == 1:
            emsg = os.linesep.join([files[0], "",
                                   "Confirm revert changes to selected file?"])
        else:
            emsg = os.linesep.join(files + ["", "Confirm revert changes to selected files?"])
        if not dialogue.ask_ok_cancel(emsg, parent=gutils.get_gtk_window(self)):
            return
        self.show_busy()
        res, sout, serr = ifce.ifce.do_revert_files_in_patch(files, self.patchname)
        self.unshow_busy()
        if res is not cmd_result.OK:
            dialogue.inform_user(os.linesep.join([sout, serr]))
    def _delete_selection(self, _arg):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        elif len(files) == 1:
            emsg = os.linesep.join([files[0], "",
                                   "Confirm delete selected file?"])
        else:
            emsg = os.linesep.join(files + ["", "Confirm delete selected files?"])
        if not dialogue.ask_ok_cancel(emsg, parent=gutils.get_gtk_window(self)):
            return
        for fname in files:
            try:
                os.remove(fname)
            except OSError as edata:
                _errno, error_message = edata.args
                emsg = os.linesep.join([fname + ": " + error_message, "Continue?"])
                if dialogue.ask_ok_cancel(emsg, parent=gutils.get_gtk_window(self)):
                    continue
        ws_event.notify_events(ws_event.FILE_DEL)
