/*
 * GQview
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"

#include "collect.h"
#include "dnd.h"
#include "editors.h"
#include "filelist.h"
#include "layout.h"
#include "layout_image.h"
#include "menu.h"
#include "preferences.h"
#include "rcfile.h"
#include "similar.h"
#include "slideshow.h"
#include "utilops.h"
#include "ui_bookmark.h"
#include "ui_help.h"
#include "ui_fileops.h"
#include "ui_tabcomp.h"
#include "ui_utildlg.h"

#include <gdk/gdkkeysyms.h> /* for keyboard values */

#include "icons/icon.xpm"


/*
 *-----------------------------------------------------------------------------
 * misc (public)
 *-----------------------------------------------------------------------------
 */ 

typedef struct _WindowIconData WindowIconData;
struct _WindowIconData
{
	const char **icon;
	gchar *path;
};

static void window_set_icon_cb(GtkWidget *widget, gpointer data)
{
	WindowIconData *wid = data;
	GdkPixbuf *pb;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	if (wid->icon)
		{
		pb = gdk_pixbuf_new_from_xpm_data(wid->icon);
		}
	else
		{
		pb = gdk_pixbuf_new_from_file(wid->path, NULL);
		}

	g_free(wid->path);
	g_free(wid);

	if (!pb) return;

	gdk_pixbuf_render_pixmap_and_mask(pb, &pixmap, &mask, 128);
	gdk_pixbuf_unref(pb);

	gdk_window_set_icon(widget->window, NULL, pixmap, mask);
	/* apparently, gdk_window_set_icon does not ref the pixmap and mask, so don't unref it (leak?) */
}

void window_set_icon(GtkWidget *window, const char **icon, const gchar *file)
{
	WindowIconData *wid;

	if (!icon && !file) icon = (const char **)icon_xpm;

	wid = g_new0(WindowIconData, 1);
	wid->icon = icon;
	wid->path = g_strdup(file);

	g_signal_connect(G_OBJECT(window), "realize",
			 G_CALLBACK(window_set_icon_cb), wid);
}

gfloat get_zoom_increment(void)
{
	return (zoom_increment ? zoom_increment / 10.0 : 1.0);
}

static gint file_extension_match(const gchar *path, const gchar *ext)
{
	gint p;
	gint e;

	if (!path) return FALSE;
	if (!ext) return TRUE;

	p = strlen(path);
	e = strlen(ext);

	return (p > e && strncasecmp(path + p - e, ext, e) == 0);
}
/*
 *-----------------------------------------------------------------------------
 * help window
 *-----------------------------------------------------------------------------
 */ 

static GtkWidget *help_window = NULL;

static void help_window_destroy_cb(GtkWidget *window, gpointer data)
{
	help_window = NULL;
}

void help_window_show(const gchar *key)
{
	if (help_window)
		{
		gdk_window_raise(help_window->window);
		if (key) help_window_set_key(help_window, key);
		return;
		}

	help_window = help_window_new(_("Help - GQview"), "GQview", "help",
                                      GQVIEW_HELPDIR "/README", key);
	g_signal_connect(G_OBJECT(help_window), "destroy",
			 G_CALLBACK(help_window_destroy_cb), NULL);
}


/*
 *-----------------------------------------------------------------------------
 * keyboard functions
 *-----------------------------------------------------------------------------
 */

void keyboard_scroll_calc(gint *x, gint *y, GdkEventKey *event)
{
	static gint delta = 0;
	static guint32 time_old = 0;
	static guint keyval_old = 0;

	if (progressive_key_scrolling)
		{
		guint32 time_diff;

		time_diff = event->time - time_old;

		/* key pressed within 125ms ? (1/8 second) */
		if (time_diff > 125 || event->keyval != keyval_old) delta = 0;

		time_old = event->time;
		keyval_old = event->keyval;

		delta += 2;
		}
	else
		{
		delta = 8;
		}

	*x = *x * delta;
	*y = *y * delta;
}

/*
 *-----------------------------------------------------------------------------
 * command line parser (private) hehe, who needs popt anyway?
 *-----------------------------------------------------------------------------
 */ 

static gint startup_full_screen = FALSE;
static gint startup_in_slideshow = FALSE;
static gint startup_command_line_collection = FALSE;


static void parse_command_line_add_file(const gchar *new_path, gchar **path, gchar **file,
				        GList **list, GList **collection_list)
{
	if (file_extension_match(new_path, ".gqv"))
		{
		*collection_list = g_list_append(*collection_list, g_strdup(new_path));
		}
	else
		{
		if (!*path) *path = remove_level_from_path(new_path);
		if (!*file) *file = g_strdup(new_path);
		*list = g_list_append(*list, g_strdup(new_path));
		}
}

static void parse_command_line(int argc, char *argv[], gchar **path, gchar **file,
			       GList **cmd_list, GList **collection_list)
{
	GList *list = NULL;

	if (argc > 1)
		{
		gint i;
		gchar *base_dir = get_current_dir();
		i = 1;
		while (i < argc)
			{
			const gchar *cmd_line = argv[i];
			gchar *cmd_all = concat_dir_and_file(base_dir, cmd_line);

			if (!*path && cmd_line[0] == '/' && isdir(cmd_line))
				{
				*path = g_strdup(cmd_line);
				}
			else if (!*path && isdir(cmd_all))
				{
				*path = g_strdup(cmd_all);
				}
			else if (cmd_line[0] == '/' && isfile(cmd_line))
				{
				parse_command_line_add_file(cmd_line, path, file, &list, collection_list);
				}
			else if (isfile(cmd_all))
				{
				parse_command_line_add_file(cmd_all, path, file, &list, collection_list);
				}
			else if (strcmp(cmd_line, "--debug") == 0)
				{
				/* we now increment the debug state for verbosity */
				debug++;
				printf("debugging output enabled (level %d)\n", debug);
				}
			else if (strcmp(cmd_line, "+t") == 0 ||
				 strcmp(cmd_line, "--with-tools") == 0)
				{
				tools_float = FALSE;
				tools_hidden = FALSE;
				}
			else if (strcmp(cmd_line, "-t") == 0 ||
				 strcmp(cmd_line, "--without-tools") == 0)
				{
				tools_hidden = TRUE;
				}
			else if (strcmp(cmd_line, "-f") == 0 ||
				 strcmp(cmd_line, "--fullscreen") == 0)
				{
				startup_full_screen = TRUE;
				}
			else if (strcmp(cmd_line, "-s") == 0 ||
				 strcmp(cmd_line, "--slideshow") == 0)
				{
				startup_in_slideshow = TRUE;
				}
			else if (strcmp(cmd_line, "-l") == 0 ||
				 strcmp(cmd_line, "--list") == 0)
				{
				startup_command_line_collection = TRUE;
				}
			else if (strcmp(cmd_line, "-v") == 0 ||
				 strcmp(cmd_line, "--version") == 0)
				{
				printf("GQview %s\n", VERSION);
				exit (0);
				}
			else if (strcmp(cmd_line, "--alternate") == 0)
				{
				/* enable faster experimental algorithm */
				printf("Alternate similarity algorithm enabled\n");
				image_sim_alternate_set(TRUE);
				}
			else if (strcmp(cmd_line, "-h") == 0 ||
				 strcmp(cmd_line, "--help") == 0)
				{
				printf("GQview %s\n", VERSION);
				print_term(_("Usage: gqview [options] [path]\n\n"));
				print_term(_("valid options are:\n"));
				print_term(_("  +t, --with-tools           force show of tools\n"));
				print_term(_("  -t, --without-tools        force hide of tools\n"));
				print_term(_("  -f, --fullscreen           start in full screen mode\n"));
				print_term(_("  -s, --slideshow            start in slideshow mode\n"));
				print_term(_("  -l, --list                 open collection window for command line\n"));
				print_term(_("  --debug                    turn on debug output\n"));
				print_term(_("  -v, --version              print version info\n"));
				print_term(_("  -h, --help                 show this message\n\n"));
				
#if 0
				/* these options are not officially supported!
				 * only for testing new features, no need to translate them */
				print_term(  "  --alternate                use alternate similarity algorithm\n");
#endif
				
				exit (0);
				}
			else 
				{
				gchar *buf;

				buf = g_strdup_printf(_("invalid or ignored: %s\nUse --help for options\n"), cmd_line);
				print_term(buf);
				g_free(buf);
				}
			g_free(cmd_all);
			i++;
			}
		g_free(base_dir);
		parse_out_relatives(*path);
		parse_out_relatives(*file);
		}

	if (list && list->next)
		{
		GList *work;

		work = list;
		while (work)
			{
			parse_out_relatives((gchar *)work->data);
			work = work->next;
			}

		*cmd_list = list;
		}
	else
		{
		path_list_free(list);
		*cmd_list = NULL;
		}
}

/*
 *-----------------------------------------------------------------------------
 * startup, init, and exit
 *-----------------------------------------------------------------------------
 */ 

#define RC_HISTORY_NAME "history"

static void keys_load(void)
{
	gchar *path;

	path = g_strconcat(homedir(), "/", GQVIEW_RC_DIR, "/", RC_HISTORY_NAME, NULL);
	history_list_load(path);
	g_free(path);
}

static void keys_save(void)
{
	gchar *path;

	path = g_strconcat(homedir(), "/", GQVIEW_RC_DIR, "/", RC_HISTORY_NAME, NULL);
	history_list_save(path);
	g_free(path);
}

static void check_for_home_path(gchar *path)
{
	gchar *buf;

	buf = g_strconcat(homedir(), "/", path, NULL);
	if (!isdir(buf))
		{
		gchar *tmp;

		tmp = g_strdup_printf(_("Creating GQview dir:%s\n"), buf);
		print_term(tmp);
		g_free(tmp);

		if (!mkdir_utf8(buf, 0755))
			{
			tmp = g_strdup_printf(_("Could not create dir:%s\n"), buf);
			print_term(tmp);
			g_free(tmp);
			}
		}
	g_free(buf);
}

static void setup_default_options(void)
{
	gchar *path;
	gint i;

	for (i = 0; i < GQVIEW_EDITOR_SLOTS; i++)
		{
		editor_name[i] = NULL;
		editor_command[i] = NULL;
		}

	editor_reset_defaults();

	bookmark_add_default(_("Home"), homedir());
	path = concat_dir_and_file(homedir(), "Desktop");
	bookmark_add_default(_("Desktop"), path);
	g_free(path);
	path = concat_dir_and_file(homedir(), GQVIEW_RC_DIR_COLLECTIONS);
	bookmark_add_default(_("Collections"), path);
	g_free(path);

	g_free(safe_delete_path);
	safe_delete_path = concat_dir_and_file(homedir(), GQVIEW_RC_DIR_TRASH);
}

static void exit_gqview_final(void)
{
	gchar *path;
	gchar *pathl;

	layout_geometry_get(NULL, &main_window_x, &main_window_y,
			    &main_window_w, &main_window_h);

	layout_geometry_get_dividers(NULL, &window_hdivider_pos, &window_vdivider_pos);

	layout_views_get(NULL, &layout_view_tree, &layout_view_icons);

	thumbnails_enabled = layout_thumb_get(NULL);
	layout_sort_get(NULL, &file_sort_method, &file_sort_ascending);

	layout_geometry_get_tools(NULL, &float_window_x, &float_window_y,
				  &float_window_w, &float_window_h, &float_window_divider);
	layout_tools_float_get(NULL, &tools_float, &tools_hidden);
	toolbar_hidden = layout_toolbar_hidden(NULL);

	save_options();
	keys_save();

	path = g_strconcat(homedir(), "/", GQVIEW_RC_DIR, "/accels", NULL);
	pathl = path_from_utf8(path);
	gtk_accel_map_save(pathl);
	g_free(pathl);
	g_free(path);

	gtk_main_quit();
}

static GenericDialog *exit_dialog = NULL;

static void exit_confirm_cancel_cb(GenericDialog *gd, gpointer data)
{
	exit_dialog = NULL;
	generic_dialog_close(gd);
}

static void exit_confirm_exit_cb(GenericDialog *gd, gpointer data)
{
	exit_dialog = NULL;
	generic_dialog_close(gd);
	exit_gqview_final();
}

static gint exit_confirm_dlg(void)
{
	if (exit_dialog)
		{
		gdk_window_raise(exit_dialog->dialog->window);
		return TRUE;
		}

	if (!collection_window_modified_exists()) return FALSE;

	exit_dialog = generic_dialog_new(_("GQview - exit"), _("Collections have been modified.\nExit anyway?"),
				"GQview", "exit", FALSE,
				exit_confirm_cancel_cb, NULL);
	generic_dialog_add_stock(exit_dialog, _("Exit"), GTK_STOCK_QUIT, exit_confirm_exit_cb, TRUE);

	gtk_widget_show(exit_dialog->dialog);

	return TRUE;
}

void exit_gqview(void)
{
	layout_image_full_screen_stop(NULL);

	if (exit_confirm_dlg()) return;

	exit_gqview_final();
}

int main (int argc, char *argv[])
{
	LayoutWindow *lw;
	gchar *path = NULL;
	gchar *cmd_path = NULL;
	gchar *cmd_file = NULL;
	GList *cmd_list = NULL;
	GList *collection_list = NULL;
	gchar *buf;
	gchar *bufl;

	/* setup locale, i18n */
	gtk_set_locale();
	bindtextdomain (PACKAGE, LOCALEDIR);
	bind_textdomain_codeset (PACKAGE, "UTF-8");
	textdomain (PACKAGE);

	/* setup random seed for random slideshow */
        srand(time(NULL));

	gtk_init (&argc, &argv);

#if 0
	printf("GQview %s, This is a beta release!\n", VERSION);
#endif

	check_for_home_path(GQVIEW_RC_DIR);
	check_for_home_path(GQVIEW_RC_DIR_THUMBS);
	check_for_home_path(GQVIEW_RC_DIR_COLLECTIONS);

	layout_order = g_strdup("123");

	setup_default_options();
	load_options();
	keys_load();

	buf = g_strconcat(homedir(), "/", GQVIEW_RC_DIR, "/accels", NULL);
	bufl = path_from_utf8(buf);
	gtk_accel_map_load(bufl);
	g_free(bufl);
	g_free(buf);

	parse_command_line(argc, argv, &cmd_path, &cmd_file, &cmd_list, &collection_list);

	if (cmd_path)
		{
		path = g_strdup(cmd_path);
		}
	else if (startup_path_enable && startup_path && isdir(startup_path))
		{
		path = g_strdup(startup_path);
		}
	else
		{
		path = get_current_dir();
		}

	filter_add_defaults();
	filter_rebuild();

	lw = layout_new(NULL, tools_float, tools_hidden);
	layout_sort_set(lw, file_sort_method, file_sort_ascending);

	if (cmd_list)
		{
		CollectionData *cd;
		GList *work;

		if (startup_command_line_collection)
			{
			CollectWindow *cw;

			cw = collection_window_new("");
			cd = cw->cd;
			}
		else
			{
			cd = collection_new("");	/* if we pass NULL, untitled counter is falsely increm. */
			}

		g_free(cd->path);
		cd->path = NULL;
		g_free(cd->name);
		cd->name = g_strdup(_("Command line"));

		collection_path_changed(cd);

		work = cmd_list;
		while (work)
			{
			collection_add(cd, (gchar *)work->data, FALSE);
			work = work->next;
			}

		layout_set_path(lw, path);
		if (cd->list) layout_image_set_collection(lw, cd, cd->list->data);

		/* mem leak, we never unref this collection when !startup_command_line_collection
		 * (the image view of the main window does not hold a ref to the collection)
		 * this is sort of unavoidable, for if it did hold a ref, next/back
		 * may not work as expected when closing collection windows.
		 *
		 * collection_unref(cd);
		 */

		}
	else if (cmd_file)
		{
		layout_set_path(lw, cmd_file);
		}
	else
		{
		layout_set_path(lw, path);
		}

	if (collection_list)
		{
		GList *work;

		work = collection_list;
		while (work)
			{
			const gchar *path = work->data;
			work = work->next;

			collection_window_new(path);
			}
		}

	g_free(cmd_path);
	g_free(cmd_file);
	path_list_free(cmd_list);
	path_list_free(collection_list);
	g_free(path);

	if (startup_full_screen) layout_image_full_screen_start(lw);
	if (startup_in_slideshow) layout_image_slideshow_start(lw);

	gtk_main ();
	return 0;
}

