/*
 * GQview
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "fullscreen.h"

#include "image.h"
#include "ui_fileops.h"
#include "ui_menu.h"


/* see fullscreen.h for possible values and their meaning */
#define FULLSCREEN_SCREEN 0


/*
 *----------------------------------------------------------------------------
 * full screen functions
 *----------------------------------------------------------------------------
 */

static void clear_mouse_cursor(GtkWidget *widget, gint clear)
{
	if (clear)
		{
		GdkCursor *cursor;
		GdkPixmap *p;

		p = gdk_bitmap_create_from_data(widget->window, "\0\0\0", 1, 1);

		cursor = gdk_cursor_new_from_pixmap(p, p,
						    &widget->style->fg[GTK_STATE_ACTIVE],
						    &widget->style->bg[GTK_STATE_ACTIVE],
						    0, 0);

		gdk_window_set_cursor (widget->window, cursor);

		gdk_cursor_unref(cursor);
		g_object_unref(p);
		}
	else
		{
		gdk_window_set_cursor (widget->window, NULL);
		}
}

static gint fullscreen_hide_mouse_cb(gpointer data)
{
	FullScreenData *fs = data;

	if (fs->hide_mouse_id == -1) return FALSE;

	clear_mouse_cursor(fs->window, TRUE);

	fs->hide_mouse_id = -1;
	return FALSE;
}

static void fullscreen_hide_mouse_disable(FullScreenData *fs)
{
	if (fs->hide_mouse_id != -1)
		{
		gtk_timeout_remove(fs->hide_mouse_id);
		fs->hide_mouse_id = -1;
		}
}

static void fullscreen_hide_mouse_reset(FullScreenData *fs)
{
	fullscreen_hide_mouse_disable(fs);
	fs->hide_mouse_id = gtk_timeout_add(FULL_SCREEN_HIDE_MOUSE_DELAY, fullscreen_hide_mouse_cb, fs);
}

static gint fullscreen_mouse_moved(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	FullScreenData *fs = data;

	clear_mouse_cursor(fs->window, FALSE);
	fullscreen_hide_mouse_reset(fs);

	return FALSE;
}

static gint fullscreen_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	FullScreenData *fs = data;

	fullscreen_stop(fs);
	return TRUE;
}

FullScreenData *fullscreen_start(GtkWidget *window, ImageWindow *imd,
				 void (*stop_func)(FullScreenData *, gpointer), gpointer stop_data)
{
	FullScreenData *fs;
	GdkScreen *screen;
	gint same;
	gint x, y;
	gint w, h;

	if (!window || !imd) return NULL;

	fs = g_new0(FullScreenData, 1);

	fs->hide_mouse_id = -1;
	fs->normal_window = window;
	fs->normal_imd = imd;

	fs->stop_func = stop_func;
	fs->stop_data = stop_data;

	if (debug) printf("full screen requests screen %d\n", FULLSCREEN_SCREEN);
	fullscreen_prefs_get_geometry(FULLSCREEN_SCREEN, window, &x, &y, &w, &h,
				      &screen, &same);

	fs->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_fullscreen(GTK_WINDOW(fs->window));
	gtk_window_set_decorated(GTK_WINDOW(fs->window), FALSE);

	gtk_window_set_screen(GTK_WINDOW(fs->window), screen);
	gtk_container_set_border_width(GTK_CONTAINER(fs->window), 0);
	g_signal_connect(G_OBJECT(fs->window), "delete_event",
			 G_CALLBACK(fullscreen_delete_cb), fs);

	gtk_window_set_title(GTK_WINDOW(fs->window), _("GQview full screen"));
	gtk_window_set_default_size(GTK_WINDOW(fs->window), w, h);
	gtk_window_move(GTK_WINDOW(fs->window), x, y);

	fs->imd = image_new(FALSE);

	gtk_container_add(GTK_CONTAINER(fs->window), fs->imd->widget);

	/* set background to black */
	if (BLACK_BACKGROUND)
		{
		image_background_set_black(fs->imd, TRUE);
		}

	image_auto_refresh(fs->imd, fs->normal_imd->auto_refresh_interval);

	gtk_widget_show(fs->imd->widget);

	image_change_from_image(fs->imd, fs->normal_imd);

	gtk_widget_show(fs->window);

	/* for hiding the mouse */
	g_signal_connect(G_OBJECT(fs->imd->image), "motion_notify_event",
			   G_CALLBACK(fullscreen_mouse_moved), fs);
	clear_mouse_cursor(fs->window, TRUE);
#if 0
	fullscreen_hide_mouse_reset(fs);
#endif

	/* hide normal window
	 * FIXME: properly restore this window on show
	 */
#ifdef HIDE_WINDOW_IN_FULLSCREEN
	gtk_widget_hide(fs->normal_window);
#endif
	image_change_path(fs->normal_imd, NULL, image_zoom_get(fs->normal_imd));

	return fs;
}

void fullscreen_stop(FullScreenData *fs)
{
	if (!fs) return;

	fullscreen_hide_mouse_disable(fs);
	gdk_keyboard_ungrab(GDK_CURRENT_TIME);

	image_change_from_image(fs->normal_imd, fs->imd);
#ifdef HIDE_WINDOW_IN_FULLSCREEN
	gtk_widget_show(fs->normal_window);
#endif
	if (fs->stop_func) fs->stop_func(fs, fs->stop_data);

	gtk_widget_destroy(fs->window);

	g_free(fs);
}


/*
 *----------------------------------------------------------------------------
 * full screen preferences and utils
 *----------------------------------------------------------------------------
 */

GList *fullscreen_prefs_list(void)
{
	GList *list = NULL;
	GdkDisplay *display;
	gint number;
	gint i;

	display = gdk_display_get_default();
	number = gdk_display_get_n_screens(display);

	for (i = 0; i < number ; i++)
		{
		GdkScreen *screen;
		gint monitors;
		gint j;

		screen = gdk_display_get_screen(display, i);
		monitors = gdk_screen_get_n_monitors(screen);

		for (j = -1; j < monitors; j++)
			{
			ScreenData *sd;
			GdkRectangle rect;
			gchar *name;
			gchar *subname;

			name = gdk_screen_make_display_name(screen);

			if (j < 0)
				{
				rect.x = 0;
				rect.y = 0;
				rect.width = gdk_screen_get_width(screen);
				rect.height = gdk_screen_get_height(screen);
				subname = g_strdup(_("Full size"));
				}
			else
				{
				gdk_screen_get_monitor_geometry(screen, j, &rect);
				subname = g_strdup_printf("%s %d", _("Monitor"), j + 1);
				}

			sd = g_new0(ScreenData, 1);
			sd->number = (i+1) * 100 + j + 1;
			sd->description = g_strdup_printf("%s %s, %s", _("Screen"), name, subname);
			sd->x = rect.x;
			sd->y = rect.y;
			sd->width = rect.width;
			sd->height = rect.height;

			if (debug) printf("Screen %d %30s %4d,%4d (%4dx%4d)\n",
					  sd->number, sd->description, sd->x, sd->y, sd->width, sd->height);

			list = g_list_append(list, sd);

			g_free(name);
			g_free(subname);
			}
		}

	return list;
}

void fullscreen_prefs_list_free(GList *list)
{
	GList *work;

	work = list;
	while (work)
		{
		ScreenData *sd = work->data;
		work = work->next;

		g_free(sd->description);	
		g_free(sd);
		}

	g_list_free(list);
}

ScreenData *fullscreen_prefs_list_find(GList *list, gint screen)
{
	GList *work;

	work = list;
	while (work)
		{
		ScreenData *sd = work->data;
		work = work->next;

		if (sd->number == screen) return sd;
		}

	return NULL;
}

/* screen is interpreted as such:
 *   0  full size of screen containing widget
 *   1  size of monitor containing widget
 * 100  full size of screen 1 (screen, monitor counts start at 1)
 * 101  size of monitor 1 on screen 1
 * 203  size of monitor 3 on screen 2
 * returns:
 * dest_screen: screen to place widget [use gtk_window_set_screen()]
 * same_region: the returned region will overlap the current location of widget.
 */
void fullscreen_prefs_get_geometry(gint screen, GtkWidget *widget, gint *x, gint *y, gint *width, gint *height,
				   GdkScreen **dest_screen, gint *same_region)
{
	GList *list;
	ScreenData *sd;

	list = fullscreen_prefs_list();
	if (screen >= 100)
		{
		sd = fullscreen_prefs_list_find(list, screen);
		}
	else
		{
		sd = NULL;
		}

	if (sd)
		{
		GdkDisplay *display;
		GdkScreen *screen;
		gint n;

		display = gdk_display_get_default();
		n = sd->number / 100 - 1;
		if (n >= 0 && n < gdk_display_get_n_screens(display))
			{
			screen = gdk_display_get_screen(display, n);
			}
		else
			{
			screen = gdk_display_get_default_screen(display);
			}

		if (x) *x = sd->x;
		if (y) *y = sd->y;
		if (width) *width = sd->width;
		if (height) *height = sd->height;

		if (dest_screen) *dest_screen = screen;
		if (same_region) *same_region = (!widget || !widget->window ||
					(screen == gtk_widget_get_screen(widget) &&
					(sd->number%100 == 0 ||
					 sd->number%100 == gdk_screen_get_monitor_at_window(screen, widget->window)+1)));

		}
	else if (screen != 1 || !widget || !widget->window)
		{
		GdkScreen *screen;

		if (widget)
			{
			screen = gtk_widget_get_screen(widget);
			}
		else
			{
			screen = gdk_screen_get_default();
			}

		if (x) *x = 0;
		if (y) *y = 0;
		if (width) *width = gdk_screen_get_width(screen);
		if (height) *height = gdk_screen_get_height(screen);

		if (dest_screen) *dest_screen = screen;
		if (same_region) *same_region = TRUE;
		}
	else
		{
		GdkScreen *screen;
		gint monitor;
		GdkRectangle rect;

		screen = gtk_widget_get_screen(widget);
		monitor = gdk_screen_get_monitor_at_window(screen, widget->window);

		gdk_screen_get_monitor_geometry(screen, monitor, &rect);

		if (x) *x = rect.x;
		if (y) *y = rect.y;
		if (width) *width = rect.width;
		if (height) *height = rect.height;

		if (dest_screen) *dest_screen = screen;
		if (same_region) *same_region = TRUE;
		}

	fullscreen_prefs_list_free(list);
}

gint fullscreen_prefs_find_screen_for_widget(GtkWidget *widget)
{
	GdkScreen *screen;
	gint monitor;
	gint n;

	if (!widget || !widget->window) return 0;

	screen = gtk_widget_get_screen(widget);
	monitor = gdk_screen_get_monitor_at_window(screen, widget->window);

	n = (gdk_screen_get_number(screen)+1) * 100 + monitor + 1;

	if (debug || TRUE) printf("Screen appears to be %d\n", n);

	return n;
}

#define SCREEN_NUMBER_KEY "screen_number"

static void fullscreen_prefs_selection_cb(GtkWidget *widget, gpointer data)
{
	gint *value = data;

	if (!value) return;

	*value = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(widget), SCREEN_NUMBER_KEY));
}

GtkWidget *fullscreen_prefs_selection_new(const gchar *text, gint *value)
{
	GtkWidget *hbox;
	GtkWidget *omenu;
	GtkWidget *menu;
	GtkWidget *item;
	GList *list;
	GList *work;
	gint current = 0;
	gint n;

	if (!value) return NULL;

	hbox = gtk_hbox_new(FALSE, 0);

	if (text)
		{
		GtkWidget *label;

		label = gtk_label_new(text);
		gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
		gtk_widget_show(label);
		}

	omenu = gtk_option_menu_new();
	menu = gtk_menu_new();

	item = menu_item_add_simple(menu, _("Active screen"), G_CALLBACK(fullscreen_prefs_selection_cb), value);
	g_object_set_data(G_OBJECT(item), SCREEN_NUMBER_KEY, GINT_TO_POINTER(0));

	item = menu_item_add_simple(menu, _("Active monitor"), G_CALLBACK(fullscreen_prefs_selection_cb), value);
	g_object_set_data(G_OBJECT(item), SCREEN_NUMBER_KEY, GINT_TO_POINTER(1));
	if (*value == 1) current = 1;

	n = 2;
	list = fullscreen_prefs_list();
	work = list;
	while (work)
		{
		ScreenData *sd = work->data;

		item = menu_item_add_simple(menu, sd->description, G_CALLBACK(fullscreen_prefs_selection_cb), value);
		g_object_set_data(G_OBJECT(item), SCREEN_NUMBER_KEY, GINT_TO_POINTER(sd->number));
		if (*value == sd->number) current = n;

		work = work->next;
		n++;
		}
	fullscreen_prefs_list_free(list);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(omenu), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(omenu), current);

	gtk_box_pack_start(GTK_BOX(hbox), omenu, FALSE, FALSE, 0);
	gtk_widget_show(omenu);

	return hbox;
}

