#
# Gramps - a GTK+/GNOME based genealogy program
#
# Copyright (C) 2003-2006, 2008  Donald N. Allingham
# Copyright (C) 2008       Brian G. Matherly
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

# $Id: ExportFtree.py 12560 2009-05-21 17:58:31Z gbritton $

"Export to Web Family Tree"

#-------------------------------------------------------------------------
#
# standard python modules
#
#-------------------------------------------------------------------------
import os
from gettext import gettext as _

#------------------------------------------------------------------------
#
# Set up logging
#
#------------------------------------------------------------------------
import logging
log = logging.getLogger(".WriteFtree")

#-------------------------------------------------------------------------
#
# GNOME/GTK modules
#
#-------------------------------------------------------------------------
from gtk import glade

#-------------------------------------------------------------------------
#
# GRAMPS modules
#
#-------------------------------------------------------------------------
import Utils
from Filters import GenericFilter, Rules, build_filter_menu
import Errors
from QuestionDialog import ErrorDialog
from gen.plug import PluginManager, ExportPlugin

#-------------------------------------------------------------------------
#
# writeData
#
#-------------------------------------------------------------------------
def writeData(database, filename, option_box=None, callback=None):
    writer = FtreeWriter(database, filename, option_box, callback)
    return writer.export_data()
    
class FtreeWriterOptionBox(object):
    """
    Create a VBox with the option widgets and define methods to retrieve
    the options. 
    """
    def __init__(self, person):
        self.person = person
        self.restrict = True

    def get_option_box(self):
        glade_file = os.path.join(os.path.dirname(__file__),
                                  "ExportFtree.glade")
        
        self.top = glade.XML(glade_file, "top", "gramps")

        filter_obj = self.top.get_widget("filter")

        all = GenericFilter()
        all.set_name(_("Entire Database"))
        all.add_rule(Rules.Person.Everyone([]))

        the_filters = [all]
        
        if self.person:
            des = GenericFilter()
            des.set_name(_("Descendants of %s") %
                         self.person.get_primary_name().get_name())
            des.add_rule(Rules.Person.IsDescendantOf(
                [self.person.get_gramps_id(),1]))
        
            ans = GenericFilter()
            ans.set_name(_("Ancestors of %s")
                         % self.person.get_primary_name().get_name())
            ans.add_rule(Rules.Person.IsAncestorOf(
                [self.person.get_gramps_id(),1]))
        
            com = GenericFilter()
            com.set_name(_("People with common ancestor with %s") %
                         self.person.get_primary_name().get_name())
            com.add_rule(Rules.Person.HasCommonAncestorWith(
                [self.person.get_gramps_id()]))
        
            the_filters += [des, ans, com]

        from Filters import CustomFilters
        the_filters.extend(CustomFilters.get_filters('Person'))
        self.filter_menu = build_filter_menu(the_filters)
        filter_obj.set_menu(self.filter_menu)

        the_box = self.top.get_widget("vbox1")
        the_parent = self.top.get_widget('dialog-vbox1')
        the_parent.remove(the_box)
        self.top.get_widget("top").destroy()
        return the_box

    def parse_options(self):
        self.restrict = self.top.get_widget("restrict").get_active()
        self.cfilter = self.filter_menu.get_active().get_data("filter")

#-------------------------------------------------------------------------
#
# FtreeWriter
#
#-------------------------------------------------------------------------
class FtreeWriter(object):

    def __init__(self, database, filename="", option_box=None, 
                 callback = None):
        self.db = database
        self.option_box = option_box
        self.filename = filename
        self.callback = callback
        if callable(self.callback): # callback is really callable
            self.update = self.update_real
        else:
            self.update = self.update_empty

        self.plist = {}

        if not option_box:
            self.cl_setup()
        else:
            self.option_box.parse_options()

            self.restrict = self.option_box.restrict
            if self.option_box.cfilter is None:
                for p in self.db.get_person_handles(sort_handles=False):
                    self.plist[p] = 1
            else:
                try:
                    for p in self.option_box.cfilter.apply(self.db, self.db.get_person_handles(sort_handles=False)):
                        self.plist[p] = 1
                except Errors.FilterError, msg:
                    (m1, m2) = msg.messages()
                    ErrorDialog(m1, m2)
                    return

    def update_empty(self):
        pass

    def update_real(self):
        self.count += 1
        newval = int(100*self.count/self.total)
        if newval != self.oldval:
            self.callback(newval)
            self.oldval = newval

    def cl_setup(self):
        self.restrict = True
        for p in self.db.get_person_handles(sort_handles=False):
            self.plist[p] = 1

    def export_data(self):
        name_map = {}
        id_map = {}
        id_name = {}
        self.count = 0
        self.oldval = 0
        self.total = 2*len(self.plist)

        for key in self.plist:
            self.update()
            pn = self.db.get_person_from_handle(key).get_primary_name()
            sn = pn.get_surname()
            items = pn.get_first_name().split()
            if len(items) > 0:
                n = "%s %s" % (items[0], sn)
            else:
                n = sn

            count = -1
            if n in name_map:
                count = 0
                while 1:
                    nn = "%s%d" % (n, count)
                    if nn not in name_map:
                        break;
                    count += 1
                name_map[nn] = key
                id_map[key] = nn
            else:
                name_map[n] = key
                id_map[key] = n
            id_name[key] = get_name(pn, count)

        f = open(self.filename,"w")

        for key in self.plist:
            self.update()
            p = self.db.get_person_from_handle(key)
            name = id_name[key]
            father = ""
            mother = ""
            email = ""
            web = ""

            family_handle = p.get_main_parents_family_handle()
            if family_handle:
                family = self.db.get_family_from_handle(family_handle)
                if family.get_father_handle() and family.get_father_handle() in id_map:
                    father = id_map[family.get_father_handle()]
                if family.get_mother_handle() and family.get_mother_handle() in id_map:
                    mother = id_map[family.get_mother_handle()]

            #
            # Calculate Date
            #
            birth_ref = p.get_birth_ref()
            death_ref = p.get_death_ref()
            if birth_ref:
                birth_event = self.db.get_event_from_handle(birth_ref.ref)
                birth = birth_event.get_date_object()
            else:
                birth = None
            if death_ref:
                death_event = self.db.get_event_from_handle(death_ref.ref)
                death = death_event.get_date_object()
            else:
                death = None

            if self.restrict:
                alive = Utils.probably_alive(p, self.db)
            else:
                alive = 0
                
            if birth and not alive:
                if death and not alive :
                    dates = "%s-%s" % (fdate(birth), fdate(death))
                else:
                    dates = fdate(birth)
            else:
                if death and not alive:
                    dates = fdate(death)
                else:
                    dates = ""
                        
            f.write('%s;%s;%s;%s;%s;%s\n' % (name, father, mother, email, web, 
                                             dates))
            
        f.close()
        return True

def fdate(val):
    if val.get_year_valid():
        if val.get_month_valid():
            if val.get_day_valid():
                return "%d/%d/%d" % (val.get_day(), val.get_month(), 
                                     val.get_year())
            else:
                return "%d/%d" % (val.get_month(), val.get_year())
        else:
            return "%d" % val.get_year()
    else:
        return ""

def get_name(name, count):
    """returns a name string built from the components of the Name
    instance, in the form of Firstname Surname"""
    if count == -1:
        val = ""
    else:
        val = str(count)
        
    if (name.suffix == ""):
        if name.prefix:
            return "%s %s %s%s" % (name.first_name, name.prefix, name.surname, val)
        else:
            return "%s %s%s" % (name.first_name, name.surname, val)
    else:
        if name.prefix:
            return "%s %s %s%s, %s" % (name.first_name, name.prefix, name.surname, val, name.suffix)
        else:
            return "%s %s%s, %s" % (name.first_name, name.surname, val, name.suffix)

#------------------------------------------------------------------------
#
# Register with the plugin system
#
#------------------------------------------------------------------------
_config = (_('Web Family Tree export options'), FtreeWriterOptionBox)

pmgr = PluginManager.get_instance()
plugin = ExportPlugin(name            = _('_Web Family Tree'), 
                      description     = _('Web Family Tree format.'),
                      export_function = writeData,
                      extension       = "wft",
                      config          = _config )
pmgr.register_plugin(plugin)
