// -*- c++ -*-
//------------------------------------------------------------------------------
//                              Config.cpp
//------------------------------------------------------------------------------
// $Id: Config.cpp,v 1.37 2005/11/22 01:39:53 vlg Exp $
//------------------------------------------------------------------------------
//  Copyright (c) 2004-2005 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or
//  modify it under the terms of the GNU General Public License
//  as published by the Free Software Foundation; either version
//  2 of the License, or (at your option) any later version.
//------------------------------------------------------------------------------
// 01/03/2004 VLG  Created
//------------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#   include "config.h"
#endif

#include <unistd.h>				// getcwd(3)
#include <limits.h>				// PATH_MAX
#include <libgen.h>				// basename(3), dirname(3)
#include <string.h>				// strcpy(3)
#include <stdio.h>				// sprintf(3)
#include <sys/types.h>			// stat(3)
#include <sys/stat.h>			// stat(3)
#include <pwd.h>				// getpwnam(2), getlogin(2)

#include <sstream>
using namespace std;

#include <gtkmm/dialog.h>
#include <assa/IniFile.h>

#include "Config.h"
#include "Granule.h"

using namespace ASSA;
using ASSA::Utils::split_pair;

ASSA_DECL_SINGLETON(Config);

/******************************************************************************/
void 
SchedReview::
set_delay (const ASSA::TimeVal& load_time_,
		   int secs_in_day_, 
		   int secs_in_week_)
{
	ASSA::TimeVal hhmm (::atoi (m_hours.c_str ()) * 360 +
						::atoi (m_mins.c_str ()) * 60);
	m_tv = load_time_ + 
		ASSA::TimeVal (::atoi (m_days.c_str  ()) * secs_in_day_) +
		ASSA::TimeVal (::atoi (m_weeks.c_str ()) * secs_in_week_) +
		hhmm;
}

/******************************************************************************/
void 
SchedReview::
dump (int idx_)
{
	DL((GRAPP,"Deck %d sched = %s/%s/%s/%s (%ld secs : %s)\n", idx_,
		m_weeks.c_str (), m_days.c_str (), m_hours.c_str (), m_mins.c_str (), 
		m_tv.sec (), m_tv.fmtString ("%c").c_str ()));
}

/******************************************************************************/
Config::
Config () :
    m_proj_name         ("untitled.cdf"),
    m_dont_save_project (false),
	m_mwin_geometry     (0, 0, 430, 350),
	m_dplyr_geometry    (0, 0, 583, 350),
	m_crdview_geometry  (0, 0, 643, 386),
	m_dplyr_keep_aspect (true),
	m_dplyr_aspect      (0, 0, 5, 3),
	m_snd_player_cmd    ("sox"),
	m_snd_player_args   ("%s -t ossdsp -v 1.7  -r 48000 /dev/dsp"),
	m_config            (NULL),
	m_remove_dups       (true),
	m_with_relative_paths (false),
	m_load_time         (ASSA::TimeVal::gettimeofday ()),
	m_root_x (0),
    m_root_y (0)
{
    trace_with_mask("Config::Config",GUITRACE);

    /** Always start with the current directory as pathname
	 */
    char* ptr = NULL;
    ptr = new char [PATH_MAX+1];
    Assure_exit (getcwd (ptr, PATH_MAX) != NULL);
    m_path_name = ptr;
    delete [] ptr;

/*  If fonts description is missing, you might find out 
	theme's default font.

	Glib::RefPtr<Gtk::Style> style = m_example.get_style ();
	Pango::FontDescription default_font = style->get_font ();
	Glib::ustring usfont = default_font.to_string ();
	DL((APP, "Default font is \"%s\"\n", usfont.c_str ()));
*/
	set_question_font ("Sans 14");
	set_answer_font   ("Sans 14");
	set_example_font  ("Sans 14");

	/** Get Real User name. Retrieved from /etc/passwd, it comes in a format
		'John Doe,,301-JOH-NDOE'.
	 */
#ifdef GRAPP_DEPRICATED
#ifdef HAVE_CUSERID
	char* login_name = ::cuserid (NULL);
	if (login_name != NULL) {
		struct passwd* pw_info = ::getpwnam (login_name);
		if (pw_info != NULL) {
			m_user_name = pw_info->pw_gecos; // Not required by POSIX 1.01
			string::size_type idx = m_user_name.find (',');
			if (idx != string::npos) {
				m_user_name.replace (idx, m_user_name.size (), "");
			}
			DL((GRAPP,"Real User Name: \"%s\"\n", m_user_name.c_str ()));
		}
	}
#endif
#endif

	m_user_name = Glib::get_real_name ();
	if (m_user_name.length () == 0) {
		m_user_name = "John Doe";
	}
}

/******************************************************************************/
Config::
~Config ()
{
    trace_with_mask("Config::~Config",GUITRACE);
    dump ();

	if (m_config) {
		delete m_config;
	}
}

/******************************************************************************/
void
Config::
dump () const
{
    trace_with_mask("Config::dump",GUITRACE);

    DL((TRACE,"proj_name : \"%s\"\n", m_proj_name.c_str ()));
    DL((TRACE,"path_name : \"%s\"\n", m_path_name.c_str ()));

    dump_document_history ();
}

/******************************************************************************/
void
Config::
dump_document_history () const
{
    if (m_history.size () == 0) {
		DL((APP,"Document history is empty\n"));
		return;
    }
    DL((APP,"=== Document history ===\n"));
    DHList_CIter cit = m_history.begin ();
    uint idx = 0;
    while (cit != m_history.end ()) {
		DL((APP,"[%d] \"%s\"\n", idx++, (*cit).c_str ()));
		cit++;
    }
    DL((APP,"====== End history =====\n"));
}

/*******************************************************************************
 * Add the most recently visited project to the history list.
 * Keep up to the last five files only. 
 * 
 * RETURN: the file path if file was successfully added to the list;
 *         an empty string otherwise.
 */
string
Config::
add_document_history ()
{
    trace_with_mask("Config::add_document_history",GUITRACE);

    string fullpath;
	int hsz = 0;

    if (m_path_name.size () == 0) {
		fullpath = m_proj_name;
    }
    else {
		fullpath = m_path_name + G_DIR_SEPARATOR_S + m_proj_name;
    }

	if (fullpath != UNKNOWN_FNAME) {
		if ((hsz = m_history.size ()) > 0) {
			for (int idx = 0; idx < hsz; idx++) {
				if (m_history [idx] == fullpath) {
					m_history.erase (m_history.begin () + idx);
					break;
				}
			}
		}

		m_history.push_front (fullpath);
		if (m_history.size () > m_history_size) {
			m_history.pop_back ();
		} 
	}
	dump_document_history ();
    return fullpath;
}

/*******************************************************************************
 * Remove project from document history list.
 * This happens when user tries to open project that doesn't exist
 * any longer.
 */
void
Config::
remove_document_history (const string& fullpath_)
{
    trace_with_mask("Config::remove_document_history",GUITRACE);

	size_t hsz;
    if ((hsz = m_history.size ()) == 0) {
		return;
	}
	for (int idx = 0; idx < hsz; idx++) {
		if (m_history [idx] == fullpath_) {
			m_history.erase (m_history.begin () + idx);
			break;
		}
	}
}

/*******************************************************************************
 * Load configuration from ~/.granule
 *
 * History list can hold up to 5 elements.
 * Keys (history?) are not important.
 *
 * [History]
 *    0=/path/to/carddeck_0.gnl
 *    1=/path/to/carddeck_1.gnl
 *      ...
 *    4=/path/to/carddeck_4.gnl
 *
 * [FlipSide]
 *    front0=/path/to/Deck_One.xml
 *    back1=/path/to/Deck_Two.xml
 *    front2=/path/to/Deck_Three.xml
 *      ...
 *    {front|back}<index>=/path/to/file
 *
 * [Scheduling]  
 *    1=7/0/0/0
 *    2=0/1/0/0
 *    3=0/2/0/0
 *    4=0/3/0/0
 *    5=0/4/0/0
 *
 * [Default]
 *    filesel_path=/path/to/
 *    question_font_desc="Sans 14"
 *    answer_font_desc="Sans 14"
 *    example_font_desc="Sans 14"
 *    mainwin_geometry=430x350
 *    cardview_geometry_height=643x386
 *    deckplayer_geometry=583x350
 *    deckplayer_keep_aspect=true
 *    deckplayer_aspect_height=3x5
 *    snd_player_cmd="sox"
 *    snd_player_agrs="%s -t ossdsp -v 1.7  -r 48000 /dev/dsp"
 *    snd_archive_path=/usr/share/WyabdcRealPeopleTTS/
 *    remove_duplicates=true
 *    with_relative_paths=false
 *    history_size=5
 *    root_x=120
 *    root_y=240
 */
void
Config::
load_config (int secs_in_day_, int secs_in_week_)
{
    trace_with_mask("Config::load_config",GUITRACE);

    Glib::ustring usfont;
    std::string s;
    int idx;
    IniFile::const_config_iterator section;

    if (m_config == NULL) {
		s = GRANULE->get_config_file ();
		if (s.size () == 0) {
			s = GRANULE->get_default_config_file ();
		}
		m_config = new IniFile (s);
		if (m_config->load () < 0) {
			DL((APP, "Failed to load configuration file!\n"));
			return;
		}
		s = "";
    }
    /** Load [History] section
     */
    section = m_config->find_section ("History");
    if (section != m_config->sect_end ()) {
		DL((APP,"Scanning [History] section ...\n"));
		IniFile::const_tuple_iterator k = (*section).second.begin ();
		while (k != (*section).second.end ()) {
			m_history.push_back ((*k).second);
			k++;
		}
		DL((APP,"... loaded total %d history items.\n", m_history.size ()));
		dump_document_history ();
    }
    else {
		DL((APP,"Section [History] is not in ~/.granule\n"));
    }

    /** Load [FlipSide] section: front(first) = path(second).
     *  We store this info, however, path first to ease the lookup.
     */
    SideSelection ss;
    section = m_config->find_section ("FlipSide");
    DL((APP,"Scanning [FlipSide] section ...\n"));

    if (section != m_config->sect_end ()) {
		IniFile::const_tuple_iterator k = (*section).second.begin ();
		while (k != (*section).second.end ()) {
			if (Glib::file_test ((*k).second, Glib::FILE_TEST_IS_REGULAR)) {
				ss = ((*k).first.find ("front") == 0 ? FRONT : BACK);
				m_flipside_history.push_back (sidesel_t((*k).second, ss));
				DL((APP,"Found %s = %s\n", (*k).first.c_str (), 
					(*k).second.c_str ()));
			}
			k++;
		}
		DL((APP,"... loaded total %d flipside items.\n", 
			m_flipside_history.size ()));
    } 
    else {
		DL((APP,"Section [FlipSide] was not found in \"~/.granule\".\n"));
    }

    /** Load [Scheduling] section. 
		We have to detect and convert old format:

		   old:  days/weeks
		   new:  weeks/days/hours/minutes
     */
    section = m_config->find_section ("Scheduling");

    if (section != m_config->sect_end ()) {
		DL((APP,"Scanning [History] section ...\n"));
		IniFile::const_tuple_iterator k = (*section).second.begin ();
		std::string lhs;
		std::string rhs;

		while (k != (*section).second.end ()) { 
			idx = ::atoi ((*k).first.c_str ()) - 1;
			split_pair ((*k).second, '/',
						m_sched [idx].m_days, m_sched [idx].m_weeks);

			if (split_pair (m_sched [idx].m_weeks, '/', lhs, rhs) == 0) {
				m_sched [idx].m_weeks = m_sched [idx].m_days;
				m_sched [idx].m_days  = lhs;
				m_sched [idx].m_hours = rhs;
				split_pair (m_sched [idx].m_hours, '/', lhs, rhs);
				m_sched [idx].m_hours = lhs;
				m_sched [idx].m_mins  = rhs;
			}
			m_sched [idx].set_delay (m_load_time, secs_in_day_, secs_in_week_);
			m_sched [idx].dump (idx);
			k++;
		}
    }

    /** Load [Default] section
     */

	/** Default configuration file shipped with Granule comes
		with an empty path. This caused critical assertion with FileChooser:

           "Gtk-CRITICAL **: gtk_file_system_unix_get_folder: 
                             assertion `g_path_is_absolute (filename)' failed"

        If empty or non-existent, we initialize it with 
		the current working directory.
	*/

    m_filesel_path    = m_config->get_value ("Default", "filesel_path");
	if (m_filesel_path.length () == 0 ||
		Glib::path_is_absolute (m_filesel_path) == FALSE)
	{
		m_filesel_path = Glib::get_current_dir ();
	}

    m_snd_player_cmd  = m_config->get_value ("Default", "snd_player_cmd");
    m_snd_player_args = m_config->get_value ("Default", "snd_player_args");

    if (m_config->get_value ("Default", "remove_duplicates") == "false") {
		m_remove_dups = false;
    }
    m_snd_archive_path = m_config->get_value ("Default", "snd_archive_path");
	m_history_size = ::atoi (m_config->get_value ("Default", 
												  "history_size").c_str ());
	if (m_history_size == 0) {	// conversion failed
		m_history_size = 5;
	}

	if (m_config->get_value ("Default", "with_relative_paths") == "true") {
		m_with_relative_paths = true;
	}

    DL((APP,"m_filesel_path     = \"%s\"\n", m_filesel_path.c_str ()));
    DL((APP,"m_font_desc        = \"%s\"\n", usfont.c_str ()));
    DL((APP,"m_snd_player_cmd   = \"%s\"\n", m_snd_player_cmd.c_str ()));
    DL((APP,"m_snd_player_args  = \"%s\"\n", m_snd_player_args.c_str ()));
    DL((APP,"m_snd_archive_path = \"%s\"\n", m_snd_archive_path.c_str ()));

    usfont = m_config->get_value ("Default", "question_font_desc");
    if (! usfont.empty ()) {
		set_question_font (usfont);
    }

    usfont = m_config->get_value ("Default", "answer_font_desc");
    if (! usfont.empty ()) {
		set_answer_font (usfont);
    }

    usfont = m_config->get_value ("Default", "example_font_desc");
    if (! usfont.empty ()) {
		set_example_font (usfont);
    }

    s = m_config->get_value ("Default", "mainwin_geometry");
    if (! s.empty ()) {
		str_to_geometry (s, m_mwin_geometry);
    }

    s = m_config->get_value ("Default", "cardview_geometry");
    if (! s.empty ()) {
		str_to_geometry (s, m_crdview_geometry);
    }

    s = m_config->get_value ("Default", "deckplayer_geometry");
    if (! s.empty ()) {
		str_to_geometry (s, m_dplyr_geometry);
    }

    if (m_config->get_value ("Default", "deckplayer_keep_aspect") == "false") {
		m_dplyr_keep_aspect = false;
    }
	else {
		m_dplyr_keep_aspect = true;
	}

    s = m_config->get_value ("Default", "deckplayer_aspect");
    if (! s.empty ()) {
		str_to_geometry (s, m_dplyr_aspect);
		if (m_dplyr_keep_aspect) { // enforce aspect based on height.
			m_dplyr_geometry.set_width (
				m_dplyr_geometry.get_height () * m_dplyr_aspect.get_width () /
				m_dplyr_aspect.get_height ());
		}
    }

	m_root_x = ::atoi (m_config->get_value ("Default", "root_x").c_str ());
    m_root_y = ::atoi (m_config->get_value ("Default", "root_y").c_str ());

    if (m_config->get_value ("Default", "disable_key_shortcuts") == "true") {
		m_disable_key_shortcuts = true;
    }
	else {
		m_disable_key_shortcuts = false;
	}
}

/******************************************************************************
 * Save configuration to the user's configuration file, ~/.granule.
 *
 */
void
Config::
save_config ()
{
    trace_with_mask("Config::save_config",GUITRACE);
	std::string s;

    if (m_config == NULL) {
		s = GRANULE->get_config_file ();
		if (s.size () == 0) {
			s = GRANULE->get_default_config_file ();
		}
		m_config = new IniFile (s);
		s = "";
		if (m_config->load () < 0) {
			DL((APP,"Failed to load config file!\n"));
			return;
		}
    }
    m_config->drop_section ("History");
    m_config->add_section  ("History");

    m_config->drop_section ("Default");
    m_config->add_section  ("Default");

    m_config->drop_section ("FlipSide");
    m_config->add_section  ("FlipSide");

    m_config->drop_section ("Scheduling");
    m_config->add_section  ("Scheduling");

    int position = 0;
    ostringstream ckey;

    /** Save document history
     */
    if (m_history.size () != 0) {
		DHList_CIter cit = m_history.begin ();
		while (cit != m_history.end ()) {
			ckey.str ("");
			ckey  << position;
			DL((GRAPP,"Saving\nhistory[%d] %s=%s>\n", position,
				ckey.str ().c_str (), (*cit).c_str ()));
			m_config->set_pair ("History", 
								IniFile::tuple_type (ckey.str (), (*cit)));
			cit++, position++;
		}
    }
    /** Save Scheduling configuration
     */
    for (int j = 0; j < CARD_BOX_SIZE; j++) {
		ckey.str ("");
		ckey << (j + 1);
		m_config->set_pair ("Scheduling", 
							IniFile::tuple_type (ckey.str (),
     						 m_sched [j].m_weeks + '/' + 
							 m_sched [j].m_days  + '/' +
 							 m_sched [j].m_hours + '/' +
 							 m_sched [j].m_mins));
    }

    /** Flip the side so that next time it would be the back of the Deck.
     */
    DL((GRAPP,"Saving [FlipSide] section\n"));
    flipside_iter_t iter = m_flipside_history.begin ();
    position = 0;

    while (iter != m_flipside_history.end ()) {
		ckey.str ("");
		ckey  << ((*iter).second == FRONT ? "front_" : "back_") << position;
		DL((GRAPP,"%s = %s\n", ckey.str ().c_str (), (*iter).first.c_str ()));

		m_config->set_pair ("FlipSide", IniFile::tuple_type (ckey.str (),
															 (*iter).first));
		iter++, position++;
    }

    Glib::ustring usfont;
    m_config->set_pair ("Default", 
						IniFile::tuple_type("filesel_path", m_filesel_path));

    /** Fonts
     */
    usfont = m_question_font_desc.to_string ();
    m_config->set_pair ("Default", 
						IniFile::tuple_type("question_font_desc",usfont));
    usfont = m_answer_font_desc.to_string ();
    m_config->set_pair ("Default", 
						IniFile::tuple_type("answer_font_desc",usfont));
    usfont = m_example_font_desc.to_string ();
    m_config->set_pair ("Default", 
						IniFile::tuple_type("example_font_desc",usfont));

    /** Geometry
     */
    m_config->set_pair ("Default", 
						IniFile::tuple_type("mainwin_geometry",	
											geometry_to_str(m_mwin_geometry)));
    m_config->set_pair ("Default", 
						IniFile::tuple_type("cardview_geometry",
										geometry_to_str(m_crdview_geometry)));
    m_config->set_pair ("Default", 
						IniFile::tuple_type("deckplayer_geometry",
											geometry_to_str(m_dplyr_geometry)));

    m_config->set_pair ("Default", IniFile::tuple_type("deckplayer_keep_aspect",
									 (m_dplyr_keep_aspect ? "true" : "false")));
    m_config->set_pair ("Default", 
						IniFile::tuple_type("deckplayer_aspect",
											geometry_to_str (m_dplyr_aspect)));
    /** Sound playback
     */
    m_config->set_pair ("Default", IniFile::tuple_type("snd_player_cmd", 
													   m_snd_player_cmd));
    m_config->set_pair ("Default", IniFile::tuple_type("snd_player_args", 
													   m_snd_player_args));
    m_config->set_pair ("Default", IniFile::tuple_type("remove_duplicates",
										   (m_remove_dups ? "true" : "false")));
    m_config->set_pair ("Default", IniFile::tuple_type("snd_archive_path", 
													   m_snd_archive_path));
	m_config->set_pair ("Default", IniFile::tuple_type ("with_relative_paths",
								   (m_with_relative_paths ? "true" : "false")));
	char buf[32];
	sprintf (buf, "%d", m_history_size);
	m_config->set_pair ("Default", IniFile::tuple_type ("history_size", buf));

    sprintf (buf, "%d", m_root_x);
    m_config->set_pair ("Default", IniFile::tuple_type ("root_x", buf));
    sprintf (buf, "%d", m_root_y);
    m_config->set_pair ("Default", IniFile::tuple_type ("root_y", buf));

	m_config->set_pair ("Default", IniFile::tuple_type ("disable_key_shortcuts",
								(m_disable_key_shortcuts ? "true":"false")));
    /** Flush to disk
     */
    m_config->sync ();
}

/******************************************************************************/
void 
Config::
set_filesel_path (const string& fspath_) 
{ 
	string::size_type idx;
	idx = fspath_.rfind (G_DIR_SEPARATOR);

	if (idx != string::npos) {
		m_filesel_path = fspath_.substr (0, idx+1);
	}
	else {
		m_filesel_path = fspath_;
	}
}

/******************************************************************************/
SideSelection 
Config::
get_flipside_history (const string& fname_)
{
    trace_with_mask("Config::get_flipside_history",GUITRACE);

	if (fname_.find ("Untitled") != std::string::npos) {
		return (FRONT);
	}
	flipside_iter_t iter = m_flipside_history.begin ();
	while (iter != m_flipside_history.end ()) {
		if ((*iter).first == fname_) {
			return ((*iter).second);
		}
		iter++;
	}
	m_flipside_history.push_back (sidesel_t (fname_, FRONT));
	DL((APP,"Added new deck info \"%s\"\n", fname_.c_str ()));
	DL((APP,"%d elements in m_flipside_history\n", m_flipside_history.size ()));
	return (FRONT);
}

/******************************************************************************/
void
Config::
set_flipside_history (const string& fname_, SideSelection side_)
{
    trace_with_mask("Config::set_flipside_history",GUITRACE);

	flipside_iter_t iter = m_flipside_history.begin ();
	while (iter != m_flipside_history.end ()) {
		if ((*iter).first == fname_) {
			(*iter).second = side_;
			break;
		}
		iter++;
	}
	if (iter == m_flipside_history.end ()) {
		m_flipside_history.push_back (sidesel_t (fname_, side_));
		DL((APP,"Added new deck info \"%s\"\n", fname_.c_str ()));
		DL((APP,"%d elements in m_flipside_history\n", 
			m_flipside_history.size ()));	
	}
}

/** 
 * Given the full file path, split it into dirname and basename.
 *
 * From man dirname(3):
 *
 *   "Both dirname and basename may modify the contents of path,
 *    so if you need to  preserve  the  pathname string,
 *    copies should be passed to these functions.
 *
 *    Furthermore, dirname and basename may return  pointers  to
 *    statically  allocated  memory which may overwritten by subsequent
 *     calls."
 */
void
Config::
set_proj_name (const string& fname_)
{
    char s [PATH_MAX];
    strcpy (s, fname_.c_str ());
    m_proj_name = basename (s); // 's' might have been modified by basename() !
	DL ((GRAPP, "New project name: \"%s\"\n", m_proj_name.c_str ()));
                                                                                
    strcpy (s, fname_.c_str ());
    m_path_name = dirname  (s);
	DL ((GRAPP, "New project path: \"%s\"\n", m_path_name.c_str ()));
}

/*******************************************************************************
	Convert "HEIGHTxWIDTH" from string to Gtk::Allocation.
*/
void
Config::
str_to_geometry (const std::string& src_, Gdk::Rectangle& aspect_)
{
    trace_with_mask("Config::str_to_geometry",GUITRACE);

	int height;
	int width;

	sscanf (src_.c_str (), "%dx%d", &height, &width);
	aspect_.set_height (height);
	aspect_.set_width  (width);

	DL((GEOM,"aspect : h=%d, w=%d\n",
		aspect_.get_height (),aspect_.get_width ()));
}

/*******************************************************************************
	Convert Gtk::Allocation to string "HEIGHTxWIDTH".
*/
const char* 
Config::
geometry_to_str (const Gdk::Rectangle& aspect_)
{
    static char buf [26];
    sprintf (buf, "%dx%d", aspect_.get_height (), aspect_.get_width  ());
    return buf;
}
