# This file is part of the graph-includes package
#
# (c) 2005 Yann Dirson <ydirson@altern.org>
# Distributed under version 2 of the GNU GPL.

package graphincludes::project;
use strict;
use warnings;

use graphincludes::params;
our @ISA;

# set_language: class method that sets the language to be used when extracting deps.
# This is a hack, which does not allow to mix several languages in a single project.
# It is only a temporary measure that allows support for languages other than C/C++.
our $_language;
sub set_language {
  my $class = shift;
  $_language = shift;
  my $langmodule = "graphincludes::extractor::" . $_language;
  eval "require $langmodule" or die "cannot load $langmodule from " . join ':', @INC;
  push @ISA, $langmodule;
}

sub new {
  my $class = shift;
  my %args = @_;
  my $prefixstrip = $args{prefixstrip};
  my @files = @{$args{files}};	# take a copy of @ARGV
  my $self = {};

#   if (defined $_language) {
#     $self = ("graphincludes::extractor::" . $_language)->new;
#   }

  $self->{FILES}   = \@files;
  $self->{PFXSTRIP}= $prefixstrip;
  $self->{TREE}    = undef;
  $self->{NODEIDS} = undef;
  $self->{NODES}   = undef;	# nodes -> list of files
  $self->{DEPS}    = undef;
  $self->{SPECIALEDGES} = {};
  $self->{IGNOREDEDGES} = {};	# to be computed in getdeps
  $self->{REPORT} = { HDR => {},
		      SYS => {},
		    };
  $self->{_DROPCOUNT} = 0;

  bless ($self, $class);

  return $self;
}

sub init {
  my $self = shift;

  # store as a tree while keeping recording node id's
  ($self->{NODEIDS}, $self->{NODES}) = $self->_nodeids($self->{TREE}, @{$self->{FILES}});
  $self->{DEPS} = $self->getdeps();
}

sub get_dep_origins {
  my $self = shift;
  keys %{$self->{DEPS}};
}
sub get_dependencies {
  my $self = shift;
  my ($origin) = @_;
  keys %{$self->{DEPS}->{$origin}};
}
sub get_dependency_weight {
  my $self = shift;
  my ($src,$dst) = @_;
  $self->{DEPS}->{$src}->{$dst};
}

sub filelabel {
  my $self = shift;
  my ($file,$level) = @_;

  return $file;
}

sub _nodeids {
  my $self = shift;
  my ($tree, @files) = @_;
  my $i=0;
  my (%nodeid, %nodes);
  foreach my $file (@files) {
    my $label = $self->filelabel($file);
    if (!defined $nodeid{$label}) {
      $nodeid{$label} = "f" . ++$i;
      $nodeid{$label} .= '-' . $label if $graphincludes::params::debug;
    }
    push @{$nodes{$label}}, $file;
  }
  return (\%nodeid, \%nodes);
}

sub locatefile {
  my $self = shift;
  my ($dst, @path) = @_;

  print STDERR "Trying to locate \`$dst'\n" if $graphincludes::params::debug;

  sub fullpath {
    my ($dstpath, $strip, $srcpath) = @_;
    join ('/', @$srcpath[0..($#$srcpath-$strip)], @$dstpath);
  }

  my @dstpath = split (m|/|, $dst);
  # count number of leading "../" in the #include reference
  my $strip = 0;
  while ($dstpath[0] eq '..') {
    $strip++; shift @dstpath;
  }
  # find the file in @path
  my $dstfile;
  foreach my $dir (@path) {
    my @srcpath = split (m|/|, $dir);
    if (defined($dstfile = fullpath(\@dstpath,$strip,\@srcpath)) and
	grep { $_ eq $dstfile } @{$self->{FILES}}) {
      print STDERR " Found from $dir ($dstfile)\n" if $graphincludes::params::debug;
      last;
    } else {
      print STDERR " Not from $dir ($dstfile)\n" if $graphincludes::params::debug;
      $dstfile = undef;
    }
  }

  return $dstfile;		# can be undef !
}

sub _fileexists {
  my ($file, @path) = @_;
  foreach my $dir (@path) {
    my $f = "$dir/$file";
    return $f if -r $f;
  }
  return undef;
}

sub record_dep {
  my $self = shift;
  my ($deps, $src, $dst) = @_;

  my $orignode = $self->{NODEIDS}->{$self->filelabel($src)};
  my $destnode = $self->{NODEIDS}->{$self->filelabel($dst)};
  if (defined $self->{IGNOREDDEPS}->{$src}->{$dst}) {
    print STDERR "ignoring $src -> $dst\n" if $graphincludes::params::debug;
    $self->{IGNOREDEDGES}->{$orignode}->{$destnode} =
      $self->{IGNOREDDEPS}->{$src}->{$dst};
  }
  if (defined $deps->{$orignode}->{$destnode}) {
    $deps->{$orignode}->{$destnode} ++;
  } else {
    $deps->{$orignode}->{$destnode} = 1;
  }
}

sub record_missed_dep {
  my $self = shift;
  my ($src, $dst) = @_;

  if (defined _fileexists ($dst, @graphincludes::params::sysinclpath)) {
    # list as system include
    $self->{REPORT}->{SYS}->{$dst} = 1;
  } else {
    # list as unknown header
    push @{$self->{REPORT}->{HDR}->{$dst}}, $src;
  }
}

sub reduce {
  my $self = shift;
  my %reduceddeps = %{$self->{DEPS}};

  print STDERR "Doing transitive reduction " if $graphincludes::params::verbose;
  foreach my $node (keys %reduceddeps) {
    print STDERR '.' if $graphincludes::params::verbose;
    print STDERR "node $node\n" if $graphincludes::params::debug;
    if (defined $reduceddeps{$node}) {
      my %newdeps = %{$reduceddeps{$node}};
      delete $newdeps{$node};	# remove intra-node dependencies
      my @considered = ($node);
      foreach my $child (keys %{$reduceddeps{$node}}) {
	# do not explore children already removed, or some circles cause lost edges
	next unless defined $newdeps{$child};
	print STDERR " child $child\n" if $graphincludes::params::debug;
	  if (defined $reduceddeps{$child}) {
	    foreach my $gchild (keys %{$reduceddeps{$child}}) {
	      if ($gchild ne $node and $gchild ne $child) { # XXX
		print STDERR "  gchild $gchild\n" if $graphincludes::params::debug;
		$self->_suppress (\%newdeps, \%reduceddeps, $gchild, \@considered, ($node, $child, $gchild));
	      }
	    }
	  }
      }
      $reduceddeps{$node} = \%newdeps;
    }
  }
  print STDERR " $self->{_DROPCOUNT} cleared.\n" if $graphincludes::params::verbose;

  print STDERR "Verifying validity of transitive reduction " if $graphincludes::params::verbose;
  sub has_path {
    my ($graph, $from, $to, @seen) = @_;
    return 1 if $from eq $to;
    return 0 if grep { $_ eq $from } @seen;
    return 1 if defined $graph->{$from}->{$to}; # superfluous ?
    foreach my $child (keys %{$graph->{$from}}) {
      return 1 if has_path($graph, $child, $to, (@seen, $from));
    }
    return 0;
  }
  foreach my $node ($self->get_dep_origins) {
    print STDERR '.' if $graphincludes::params::verbose;
    foreach my $child ($self->get_dependencies($node)) {
      if (!has_path(\%reduceddeps, $node, $child)) {
	print STDERR "ERROR: missing edge from $node to $child\n";
      }
    }
  }
  print STDERR " done.\n" if $graphincludes::params::verbose;
  $self->{DEPS} = \%reduceddeps;
}

sub _suppress {
  my $self = shift;
  my ($list,			# childlist hash to cleanup
      $reduceddeps,		# ref to hash to be reduced
      $node,			# node to consider this time
      $considered,		# graph nodes already seen, not to reconsider
      @context)			# current path
    = @_;

  # Do not consider $node twice, prevent looping on circular deps.
  # We must take care of the special case of the child that led us to
  # the current node, or we would have to do special things to $gchild
  return if $node eq $context[1] or grep { $node eq $_ } (@$considered);
  push @$considered, $node;

  # remove $node from $list
  if (defined $list->{$node}) {
    if ($graphincludes::params::showdropped) {
      $self->{SPECIALEDGES}->{$considered->[0]}->{$node} = {color      => "#FFCCCC",
							    constraint => 'false'};
    } elsif (grep { $self->{NODEIDS}->{$_} eq $considered->[0] } @graphincludes::params::focus) {
      $self->{SPECIALEDGES}->{$considered->[0]}->{$node} = {color => "#FFCCCC"};
    } else {
      $self->{_DROPCOUNT}++;
      # increment "use count" on each step of the alternate path in @context
      for (my $i = 0; $i < $#context; $i++) {
	$reduceddeps->{$context[$i]}->{$context[$i+1]} += $list->{$node};
      }
      # remove it
      delete $list->{$node};
    }
    print STDERR "    --$node (", join (',', @context), ")\n" if $graphincludes::params::debug;
  }

  # remove $node's children
  if (defined $reduceddeps->{$node}) {
    foreach my $child (keys %{$reduceddeps->{$node}}) {
      if ($graphincludes::params::debug) {
	foreach (@context) {
	  print STDERR " ";
	}
	print STDERR "$child\n";
      }
      $self->_suppress ($list, $reduceddeps, $child, $considered, (@context, $child));
    }
  }
}

sub special_edge {
  my $self = shift;
  my ($src, $dst) = @_;

  my $attrs = $self->{SPECIALEDGES}->{$src}->{$dst};

  if (defined $attrs) {
    return $attrs;
  } else {
    return undef;
  }
}

1;
