"""!
@package gmodeler.py

@brief wxGUI Graphical Modeler for creating, editing, and managing models

Classes:
 - Model
 - ModelFrame
 - ModelCanvas
 - ModelObject
 - ModelAction
 - ModelSearchDialog
 - ModelData
 - ModelDataDialog
 - ModelRelation
 - ProcessModelFile
 - WriteModelFile
 - PreferencesDialog
 - PropertiesDialog
 - ModelParamDialog
 - ModelListCtrl
 - VariablePanel
 - ValiableListCtrl
 - ModelItem
 - ModelItemDialog
 - ModelLoop
 - ModelLoopDialog
 - ItemPanel
 - ItemListCtrl
 - ItemCheckListCtrl
 - ModelCondition
 - ModelConditionDialog
 - WritePythonFile

(C) 2010 by the GRASS Development Team
This program is free software under the GNU General Public License
(>=v2). Read the file COPYING that comes with GRASS for details.

@author Martin Landa <landa.martin gmail.com>
"""

import os
import sys
import shlex
import time
import traceback
import getpass
import stat
import textwrap
import tempfile
import copy
import re

try:
    import xml.etree.ElementTree as etree
except ImportError:
    import elementtree.ElementTree as etree # Python <= 2.4

import globalvar
if not os.getenv("GRASS_WXBUNDLED"):
    globalvar.CheckForWx()
import wx
import wx.lib.ogl             as ogl
import wx.lib.flatnotebook    as FN
import wx.lib.colourselect    as csel
import wx.lib.mixins.listctrl as listmix

import menu
import menudata
import toolbars
import menuform
import prompt
import utils
import goutput
import gselect
from debug        import Debug
from gcmd         import GMessage, GException, GWarning, GError
from gdialogs     import ElementDialog, GetImageHandlers
from preferences  import PreferencesBaseDialog, globalSettings as UserSettings
from ghelp        import SearchModuleWindow

from grass.script import core as grass

class Model(object):
    """!Class representing the model"""
    def __init__(self, canvas = None):
        self.items      = list() # list of actions/loops/...
        
        # model properties
        self.properties = { 'name'        : _("model"),
                            'description' : _("Script generated by wxGUI Graphical Modeler."),
                            'author'      : getpass.getuser() }
        # model variables
        self.variables = dict()
        self.variablesParams = dict()
        
        self.canvas  = canvas
        
    def GetCanvas(self):
        """!Get canvas or None"""
        return self.canvas
    
    def GetItems(self, objType = None):
        """!Get list of model items

        @param objType Object type to filter model objects
        """
        if not objType:
            return self.items
        
        result = list()
        for item in self.items:
            if isinstance(item, objType):
                result.append(item)
        
        return result

    def GetItem(self, aId):
        """!Get item of given id

        @param aId item id
        
        @return Model* instance
        @return None if no item found
        """
        ilist = self.GetItems()
        for item in ilist:
            if item.GetId() == aId:
                return item
        
        return None

    def GetNumItems(self, actionOnly = False):
        """!Get number of items"""
        if actionOnly:
            return len(self.GetItems(objType = ModelAction))
        
        return len(self.GetItems())
    
    def GetProperties(self):
        """!Get model properties"""
        return self.properties

    def GetVariables(self, params = False):
        """!Get model variables"""
        if params:
            return self.variablesParams
        
        return self.variables
    
    def SetVariables(self, data):
        """!Set model variables"""
        self.variables = data
    
    def Reset(self):
        """!Reset model"""
        self.items = list()
        
    def RemoveItem(self, item):
        """!Remove item from model
        
        @return list of related items to remove/update
        """
        relList = list()
        upList = list()
        
        if not isinstance(item, ModelData):
            self.items.remove(item)
        
        if isinstance(item, ModelAction):
            for rel in item.GetRelations():
                relList.append(rel)
                data = rel.GetData()
                if len(data.GetRelations()) < 2:
                    relList.append(data)
                else:
                    upList.append(data)
            
        elif isinstance(item, ModelData):
            for rel in item.GetRelations():
                relList.append(rel)
                if rel.GetFrom() == self:
                    relList.append(rel.GetTo())
                else:
                    relList.append(rel.GetFrom())
        
        return relList, upList
    
    def FindAction(self, aId):
        """!Find action by id"""
        alist = self.GetItems(objType = ModelAction)
        for action in alist:
            if action.GetId() == aId:
                return action
        
        return None

    def GetData(self):
        """!Get list of data items"""
        result = list()
        for action in self.GetItems(objType = ModelAction):
            for rel in action.GetRelations():
                dataItem = rel.GetData()
                if dataItem not in result:
                    result.append(dataItem)
        
        return result

    def FindData(self, value, prompt):
        """!Find data item in the model

        @param value value
        @param prompt prompt

        @return ModelData instance
        @return None if not found
        """
        for data in self.GetData():
            if data.GetValue() == value and \
                    data.GetPrompt() == prompt:
                return data
        
        return None
                
    def LoadModel(self, filename):
        """!Load model definition stored in GRASS Model XML file (gxm)
        
        @todo Validate against DTD
        
        Raise exception on error.
        """
        dtdFilename = os.path.join(globalvar.ETCWXDIR, "xml", "grass-gxm.dtd")
        
        # parse workspace file
        try:
            gxmXml = ProcessModelFile(etree.parse(filename))
        except StandardError, e:
            raise GException(e)
        
        if self.canvas:
            win = self.canvas.parent
            if gxmXml.pos:
                win.SetPosition(gxmXml.pos)
            if gxmXml.size:
                win.SetSize(gxmXml.size)
        
        # load properties
        self.properties = gxmXml.properties
        self.variables  = gxmXml.variables
        
        # load model.GetActions()
        for action in gxmXml.actions:
            actionItem = ModelAction(parent = self, 
                                     x = action['pos'][0],
                                     y = action['pos'][1],
                                     width = action['size'][0],
                                     height = action['size'][1],
                                     task = action['task'],
                                     id = action['id'])
            
            if action['disabled']:
                actionItem.Enable(False)
            
            self.AddItem(actionItem)
            
            task = actionItem.GetTask()
            parameterized = False
            valid = True
            for f in task.get_options()['flags']:
                if f.get('parameterized', False):
                    parameterized = True
                    break
            for p in task.get_options()['params']:
                if p.get('required', 'no') != 'no' and \
                        p.get('value', '') == '' and \
                        p.get('default', '') == '':
                    valid = False
                if p.get('parameterized', False):
                    parameterized = True
            
            actionItem.SetValid(valid)
            actionItem.SetParameterized(parameterized)
            actionItem.GetLog() # substitute variables (-> valid/invalid)
        
        # load data & relations
        for data in gxmXml.data:
            dataItem = ModelData(parent = self, 
                                 x = data['pos'][0],
                                 y = data['pos'][1],
                                 width = data['size'][0],
                                 height = data['size'][1],
                                 prompt = data['prompt'],
                                 value = data['value'])
            dataItem.SetIntermediate(data['intermediate'])
            
            for rel in data['rels']:
                actionItem = self.FindAction(rel['id'])
                if rel['dir'] == 'from':
                    relation = ModelRelation(dataItem, actionItem, rel['name'])
                else:
                    relation = ModelRelation(actionItem, dataItem, rel['name'])
                relation.SetControlPoints(rel['points'])
                actionItem.AddRelation(relation)
                dataItem.AddRelation(relation)
            
            dataItem.Update()
                   
        # load loops
        for loop in gxmXml.loops:
            loopItem = ModelLoop(parent = self, 
                                 x = loop['pos'][0],
                                 y = loop['pos'][1],
                                 width = loop['size'][0],
                                 height = loop['size'][1],
                                 text = loop['text'],
                                 id = loop['id'])
            self.AddItem(loopItem)

        # load conditions
        for condition in gxmXml.conditions:
            conditionItem = ModelCondition(parent = self, 
                                           x = condition['pos'][0],
                                           y = condition['pos'][1],
                                           width = condition['size'][0],
                                           height = condition['size'][1],
                                           text = condition['text'],
                                           id = condition['id'])
            self.AddItem(conditionItem)

        # define loops & if/else items
        for loop in gxmXml.loops:
            alist = list()
            for aId in loop['items']:
                action = self.GetItem(aId)
                alist.append(action)
            
            loopItem = self.GetItem(loop['id'])
            loopItem.SetItems(alist)
            
            for action in loopItem.GetItems():
                action.SetBlock(loopItem)
        
        for condition in gxmXml.conditions:
            conditionItem = self.GetItem(condition['id'])
            for b in condition['items'].keys():
                alist = list()
                for aId in condition['items'][b]:
                    action = self.GetItem(aId)
                    alist.append(action)
                conditionItem.SetItems(alist, branch = b)
            
            items = conditionItem.GetItems()
            for b in items.keys():
                for action in items[b]:
                    action.SetBlock(conditionItem)
        
    def AddItem(self, newItem):
        """!Add item to the list"""
        iId = newItem.GetId()
        
        i   = 0
        for item in self.items:
            if item.GetId() > iId:
                self.items.insert(i, newItem)
                return
            i += 1
        
        self.items.append(newItem)
        
    def IsValid(self):
        """Return True if model is valid"""
        if self.Validate():
            return False
        
        return True
    
    def Validate(self):
        """!Validate model, return None if model is valid otherwise
        error string"""
        errList = list()
        for action in self.GetItems(objType = ModelAction):
            task = menuform.GUI().ParseCommand(cmd = action.GetLog(string = False),
                                               show = None)
            errList += task.getCmdError()

        return errList

    def Run(self, log, onDone):
        """!Run model"""
        for action in self.actions:
            if not action.IsEnabled():
                continue
            log.RunCmd(command = action.GetLog(string = False),
                       onDone = onDone)
        
    def DeleteIntermediateData(self, log):
        """!Detele intermediate data"""
        rast, vect, rast3d, msg = self.GetIntermediateData()
        
        if rast:
            log.RunCmd(['g.remove', 'rast=%s' %','.join(rast)])
        if rast3d:
            log.RunCmd(['g.remove', 'rast3d=%s' %','.join(rast3d)])
        if vect:
            log.RunCmd(['g.remove', 'vect=%s' %','.join(vect)])
        
    def GetIntermediateData(self):
        """!Get info about intermediate data"""
        rast = list()
        rast3d = list()
        vect = list()
        for data in self.GetData():
            if not data.IsIntermediate():
                continue
            name = data.GetValue()
            prompt = data.GetPrompt()
            if prompt == 'raster':
                rast.append(name)
            elif prompt == 'vector':
                vect.append(name)
            elif prompt == 'rast3d':
                rast3d.append(name)
        
        msg = ''
        if rast:
            msg += '\n\n%s: ' % _('Raster maps')
            msg += ', '.join(rast)
        if rast3d:
            msg += '\n\n%s: ' % _('3D raster maps')
            msg += ', '.join(rast3d)
        if vect:
            msg += '\n\n%s: ' % _('Vector maps')
            msg += ', '.join(vect)
        
        return rast, vect, rast3d, msg

    def Update(self):
        """!Update model"""
        for item in self.items:
            item.Update()
        
    def IsParameterized(self):
        """!Return True if model is parameterized"""
        if self.Parameterize():
            return True
        
        return False
    
    def Parameterize(self):
        """!Return parameterized options"""
        result = dict()
        idx = 0
        if self.variables:
            params = list()
            result[_("Variables")] = { 'flags'  : list(),
                                       'params' : params,
                                       'idx' : idx }
            for name, values in self.variables.iteritems():
                gtype = values.get('type', 'string')
                if gtype in ('raster', 'vector'):
                    gisprompt = True
                    prompt = gtype
                    if gtype == 'raster':
                        element = 'cell'
                    else:
                        element = 'vector'
                    ptype = 'string'
                else:
                    gisprompt = False
                    prompt = None
                    element = None
                    ptype = gtype
                params.append({ 'gisprompt' : gisprompt,
                                'multiple'  : 'no',
                                'description' : values.get('description', ''),
                                'guidependency' : '',
                                'default' : '',
                                'age' : None,
                                'required' : 'yes',
                                'value' : values.get('value', ''),
                                'label' : '',
                                'guisection' : '',
                                'key_desc' : '',
                                'values' : list(),
                                'parameterized' : False,
                                'values_desc' : list(),
                                'prompt' : prompt,
                                'element' : element,
                                'type' : ptype,
                                'name' : name })
            
            idx += 1
        
        for action in self.GetItems(objType = ModelAction):
            if not action.IsEnabled():
                continue
            name   = action.GetName()
            params = action.GetParams()
            for f in params['flags']:
                if f.get('parameterized', False):
                    if not result.has_key(name):
                        result[name] = { 'flags' : list(),
                                         'params': list(),
                                         'idx'   : idx }
                    result[name]['flags'].append(f)
            for p in params['params']:
                if p.get('parameterized', False):
                    if not result.has_key(name):
                        result[name] = { 'flags' : list(),
                                         'params': list(),
                                         'idx'   : idx }
                    result[name]['params'].append(p)
            idx += 1

        self.variablesParams = result # record parameters
        
        return result
    
class ModelFrame(wx.Frame):
    def __init__(self, parent, id = wx.ID_ANY,
                 title = _("GRASS GIS Graphical Modeler"), **kwargs):
        """!Graphical modeler main window
        
        @param parent parent window
        @param id window id
        @param title window title

        @param kwargs wx.Frames' arguments
        """
        self.parent = parent
        self.searchDialog = None # module search dialog
        self.baseTitle = title
        self.modelFile = None    # loaded model
        self.modelChanged = False
        
        self.cursors = {
            "default" : wx.StockCursor(wx.CURSOR_ARROW),
            "cross"   : wx.StockCursor(wx.CURSOR_CROSS),
            }
        
        wx.Frame.__init__(self, parent = parent, id = id, title = title, **kwargs)
        self.SetName("Modeler")
        self.SetIcon(wx.Icon(os.path.join(globalvar.ETCICONDIR, 'grass.ico'), wx.BITMAP_TYPE_ICO))
        
        self.menubar = menu.Menu(parent = self, data = menudata.ModelerData())
        
        self.SetMenuBar(self.menubar)
        
        self.toolbar = toolbars.ModelToolbar(parent = self)
        self.SetToolBar(self.toolbar)
        
        self.statusbar = self.CreateStatusBar(number = 1)
        
        if globalvar.hasAgw:
            self.notebook = FN.FlatNotebook(parent = self, id = wx.ID_ANY,
                                            agwStyle = FN.FNB_FANCY_TABS | FN.FNB_BOTTOM |
                                            FN.FNB_NO_NAV_BUTTONS | FN.FNB_NO_X_BUTTON)
        else:
            self.notebook = FN.FlatNotebook(parent = self, id = wx.ID_ANY,
                                            style = FN.FNB_FANCY_TABS | FN.FNB_BOTTOM |
                                            FN.FNB_NO_NAV_BUTTONS | FN.FNB_NO_X_BUTTON)
        
        self.canvas = ModelCanvas(self)
        self.canvas.SetBackgroundColour(wx.WHITE)
        self.canvas.SetCursor(self.cursors["default"])
        
        self.model = Model(self.canvas)
        
        self.variablePanel = VariablePanel(parent = self)
        
        self.itemPanel = ItemPanel(parent = self)
        
        self.goutput = goutput.GMConsole(parent = self, pageid = 3,
                                         notebook = self.notebook)
        
        self.notebook.AddPage(self.canvas, text=_('Model'))
        self.notebook.AddPage(self.itemPanel, text=_('Items'))
        self.notebook.AddPage(self.variablePanel, text=_('Variables'))
        self.notebook.AddPage(self.goutput, text=_('Command output'))
        wx.CallAfter(self.notebook.SetSelection, 0)
        wx.CallAfter(self.ModelChanged, False)

        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow)
        self.Bind(wx.EVT_SIZE, self.OnSize)
        
        self._layout()
        self.SetMinSize((475, 300))
        self.SetSize((640, 480))
        
        # fix goutput's pane size
        if self.goutput:
            self.goutput.SetSashPosition(int(self.GetSize()[1] * .75))
        
    def _layout(self):
        """!Do layout"""
        sizer = wx.BoxSizer(wx.VERTICAL)

        sizer.Add(item = self.notebook, proportion = 1,
                  flag = wx.EXPAND)
        
        self.SetAutoLayout(True)
        self.SetSizer(sizer)
        sizer.Fit(self)
        
        self.Layout()

    def _addEvent(self, item):
        """!Add event to item"""
        evthandler = ModelEvtHandler(self.statusbar,
                                     self)
        evthandler.SetShape(item)
        evthandler.SetPreviousHandler(item.GetEventHandler())
        item.SetEventHandler(evthandler)

    def GetCanvas(self):
        """!Get canvas"""
        return self.canvas
    
    def GetModel(self):
        """!Get model"""
        return self.model
    
    def ModelChanged(self, changed = True):
        """!Update window title"""
        self.modelChanged = changed
        
        if self.modelFile:
            if self.modelChanged:
                self.SetTitle(self.baseTitle + " - " +  os.path.basename(self.modelFile) + '*')
            else:
                self.SetTitle(self.baseTitle + " - " +  os.path.basename(self.modelFile))
        else:
            self.SetTitle(self.baseTitle)
        
    def OnVariables(self, event):
        """!Switch to variables page"""
        self.notebook.SetSelection(2)
        
    def OnRemoveItem(self, event):
        """!Remove shape
        """
        self.GetCanvas().RemoveSelected()
        
    def OnCanvasRefresh(self, event):
        """!Refresh canvas"""
        self.SetStatusText(_("Redrawing model..."), 0)
        self.GetCanvas().Refresh()
        self.SetStatusText("", 0)

    def OnCmdRun(self, event):
        """!Run command"""
        try:
            action = self.GetModel().GetItems()[event.pid]
            if hasattr(action, "task"):
                action.Update(running = True)
        except IndexError:
            pass
        
    def OnCmdDone(self, event):
        """!Command done (or aborted)"""
        try:
            action = self.GetModel().GetItems()[event.pid]
            if hasattr(action, "task"):
                action.Update(running = True)
        except IndexError:
            pass
        
    def OnCloseWindow(self, event):
        """!Close window"""
        if self.modelChanged and \
                UserSettings.Get(group='manager', key='askOnQuit', subkey='enabled'):
            if self.modelFile:
                message = _("Do you want to save changes in the model?")
            else:
                message = _("Do you want to store current model settings "
                            "to model file?")
            
            # ask user to save current settings
            dlg = wx.MessageDialog(self,
                                   message = message,
                                   caption=_("Quit Graphical Modeler"),
                                   style = wx.YES_NO | wx.YES_DEFAULT |
                                   wx.CANCEL | wx.ICON_QUESTION | wx.CENTRE)
            ret = dlg.ShowModal()
            if ret == wx.ID_YES:
                if not self.modelFile:
                        self.OnWorkspaceSaveAs()
                else:
                    self.WriteModelFile(self.modelFile)
            elif ret == wx.ID_CANCEL:
                dlg.Destroy()
                return
            dlg.Destroy()
        
        self.Destroy()

    def OnSize(self, event):
        """Window resized, save to the model"""
        self.ModelChanged()
        event.Skip()
        
    def OnPreferences(self, event):
        """!Open preferences dialog"""
        dlg = PreferencesDialog(parent = self)
        dlg.CenterOnParent()
        
        dlg.ShowModal()
        self.canvas.Refresh()
        
    def OnHelp(self, event):
        """!Show help"""
        if self.parent and self.parent.GetName() == 'LayerManager':
            log = self.parent.GetLogWindow()
            log.RunCmd(['g.manual',
                        'entry=wxGUI.Modeler'])
        else:
            gcmd.RunCommand('g.manual',
                            quiet = True,
                            entry = 'wxGUI.Modeler')
        
    def OnModelProperties(self, event):
        """!Model properties dialog"""
        dlg = PropertiesDialog(parent = self)
        dlg.CentreOnParent()
        properties = self.model.GetProperties()
        dlg.Init(properties)
        if dlg.ShowModal() == wx.ID_OK:
            self.ModelChanged()
            for key, value in dlg.GetValues().iteritems():
                properties[key] = value
            for action in self.model.GetItems(objType = ModelAction):
                action.GetTask().set_flag('overwrite', properties['overwrite'])
        
        dlg.Destroy()
        
    def OnDeleteData(self, event):
        """!Delete intermediate data"""
        rast, vect, rast3d, msg = self.model.GetIntermediateData()
        
        if not rast and not vect and not rast3d:
            GMessage(parent = self,
                     message = _('Nothing to delete.'))
            return
        
        dlg = wx.MessageDialog(parent = self,
                               message= _("Do you want to permanently delete data?%s" % msg),
                               caption=_("Delete intermediate data?"),
                               style=wx.YES_NO | wx.YES_DEFAULT | wx.ICON_QUESTION)
        
        ret = dlg.ShowModal()
        if ret == wx.ID_YES:
            dlg.Destroy()
            
            if rast:
                self.goutput.RunCmd(['g.remove', 'rast=%s' %','.join(rast)])
            if rast3d:
                self.goutput.RunCmd(['g.remove', 'rast3d=%s' %','.join(rast3d)])
            if vect:
                self.goutput.RunCmd(['g.remove', 'vect=%s' %','.join(vect)])
            
            self.SetStatusText(_("%d maps deleted from current mapset") % \
                                 int(len(rast) + len(rast3d) + len(vect)))
            return
        
        dlg.Destroy()
                
    def OnModelNew(self, event):
        """!Create new model"""
        Debug.msg(4, "ModelFrame.OnModelNew():")
        
        # ask user to save current model
        if self.modelFile and self.modelChanged:
            self.OnModelSave()
        elif self.modelFile is None and \
                (self.model.GetNumItems() > 0 or len(self.model.GetData()) > 0):
            dlg = wx.MessageDialog(self, message=_("Current model is not empty. "
                                                   "Do you want to store current settings "
                                                   "to model file?"),
                                   caption=_("Create new model?"),
                                   style=wx.YES_NO | wx.YES_DEFAULT |
                                   wx.CANCEL | wx.ICON_QUESTION)
            ret = dlg.ShowModal()
            if ret == wx.ID_YES:
                self.OnModelSaveAs()
            elif ret == wx.ID_CANCEL:
                dlg.Destroy()
                return
            
            dlg.Destroy()
        
        # delete all items
        self.canvas.GetDiagram().DeleteAllShapes()
        self.model.Reset()
        self.canvas.Refresh()
        
        # no model file loaded
        self.modelFile = None
        self.modelChanged = False
        self.SetTitle(self.baseTitle)
        
    def OnModelOpen(self, event):
        """!Load model from file"""
        filename = ''
        dlg = wx.FileDialog(parent = self, message=_("Choose model file"),
                            defaultDir = os.getcwd(),
                            wildcard=_("GRASS Model File (*.gxm)|*.gxm"))
        if dlg.ShowModal() == wx.ID_OK:
            filename = dlg.GetPath()
                    
        if not filename:
            return
        
        Debug.msg(4, "ModelFrame.OnModelOpen(): filename=%s" % filename)
        
        # close current model
        self.OnModelClose()
        
        self.LoadModelFile(filename)
        
        self.modelFile = filename
        self.SetTitle(self.baseTitle + " - " +  os.path.basename(self.modelFile))
        self.SetStatusText(_('%(items)d items (%(actions)d actions) loaded into model') % \
                               { 'items' : self.model.GetNumItems(),
                                 'actions' : self.model.GetNumItems(actionOnly = True) }, 0)
        
    def OnModelSave(self, event = None):
        """!Save model to file"""
        if self.modelFile and self.modelChanged:
            dlg = wx.MessageDialog(self, message=_("Model file <%s> already exists. "
                                                   "Do you want to overwrite this file?") % \
                                       self.modelFile,
                                   caption=_("Save model"),
                                   style=wx.YES_NO | wx.YES_DEFAULT | wx.ICON_QUESTION)
            if dlg.ShowModal() == wx.ID_NO:
                dlg.Destroy()
            else:
                Debug.msg(4, "ModelFrame.OnModelSave(): filename=%s" % self.modelFile)
                self.WriteModelFile(self.modelFile)
                self.SetStatusText(_('File <%s> saved') % self.modelFile, 0)
                self.SetTitle(self.baseTitle + " - " +  os.path.basename(self.modelFile))
        elif not self.modelFile:
            self.OnModelSaveAs(None)
        
    def OnModelSaveAs(self, event):
        """!Create model to file as"""
        filename = ''
        dlg = wx.FileDialog(parent = self,
                            message = _("Choose file to save current model"),
                            defaultDir = os.getcwd(),
                            wildcard=_("GRASS Model File (*.gxm)|*.gxm"),
                            style=wx.FD_SAVE)
        
        
        if dlg.ShowModal() == wx.ID_OK:
            filename = dlg.GetPath()
        
        if not filename:
            return
        
        # check for extension
        if filename[-4:] != ".gxm":
            filename += ".gxm"
        
        if os.path.exists(filename):
            dlg = wx.MessageDialog(parent = self,
                                   message=_("Model file <%s> already exists. "
                                             "Do you want to overwrite this file?") % filename,
                                   caption=_("File already exists"),
                                   style=wx.YES_NO | wx.YES_DEFAULT | wx.ICON_QUESTION)
            if dlg.ShowModal() != wx.ID_YES:
                dlg.Destroy()
                return
        
        Debug.msg(4, "GMFrame.OnModelSaveAs(): filename=%s" % filename)
        
        self.WriteModelFile(filename)
        self.modelFile = filename
        self.SetTitle(self.baseTitle + " - " + os.path.basename(self.modelFile))
        self.SetStatusText(_('File <%s> saved') % self.modelFile, 0)

    def OnModelClose(self, event = None):
        """!Close model file"""
        Debug.msg(4, "ModelFrame.OnModelClose(): file=%s" % self.modelFile)
        # ask user to save current model
        if self.modelFile and self.modelChanged:
            self.OnModelSave()
        elif self.modelFile is None and \
                (self.model.GetNumItems() > 0 or len(self.model.GetData()) > 0):
            dlg = wx.MessageDialog(self, message=_("Current model is not empty. "
                                                   "Do you want to store current settings "
                                                   "to model file?"),
                                   caption=_("Create new model?"),
                                   style=wx.YES_NO | wx.YES_DEFAULT |
                                   wx.CANCEL | wx.ICON_QUESTION)
            ret = dlg.ShowModal()
            if ret == wx.ID_YES:
                self.OnModelSaveAs()
            elif ret == wx.ID_CANCEL:
                dlg.Destroy()
                return
            
            dlg.Destroy()
        
        self.modelFile = None
        self.SetTitle(self.baseTitle)
        
        self.canvas.GetDiagram().DeleteAllShapes()
        self.model.Reset()
        
        self.canvas.Refresh()
        
    def OnRunModel(self, event):
        """!Run entire model"""
        if self.model.GetNumItems() < 1:
            GMessage(parent = self, 
                     message = _('Model is empty. Nothing to run.'))
            return
        
        # validation
        errList = self._validateModel()
        if errList:
            dlg = wx.MessageDialog(parent = self,
                                   message = _('Model is not valid. Do you want to '
                                               'run the model anyway?\n\n%s') % '\n'.join(errList),
                                   caption=_("Run model?"),
                                   style = wx.YES_NO | wx.NO_DEFAULT |
                                   wx.ICON_QUESTION | wx.CENTRE)
            ret = dlg.ShowModal()
            if ret != wx.ID_YES:
                return
        
        # parametrization
        params = self.model.Parameterize()
        if params:
            dlg = ModelParamDialog(parent = self,
                                   params = params)
            dlg.CenterOnParent()
            
            ret = dlg.ShowModal()
            if ret != wx.ID_OK:
                dlg.Destroy()
                return
        
            err = dlg.GetErrors()
            if err:
                GError(parent = self,
                       message = unicode('\n'.join(err)))
                return
        
        self.goutput.cmdThread.SetId(-1)
        for item in self.model.GetItems():
            if not item.IsEnabled():
                continue
            if isinstance(item, ModelAction):
                if item.GetBlockId():
                    continue
                self._runAction(item, params)
            elif isinstance(item, ModelLoop):
                cond = item.GetText()
                # substitute variables in condition
                variables = self.model.GetVariables()
                for variable in variables:
                    pattern = re.compile('%' + variable)
                    if pattern.search(cond):
                        value = variables[variable].get('value', '')
                        vtype = variables[variable].get('type', 'string')
                        if vtype == 'string':
                            value = '"' + value + '"'
                        cond = pattern.sub(value, cond)
                # split condition
                condVar, condText = re.split('\s*in\s*', cond)
                
                for action in item.GetItems():
                    for vars()[condVar] in eval(condText):
                        if isinstance(action, ModelAction):
                            self._runAction(action, params)
        
        if params:
            dlg.Destroy()
        
    def _runAction(self, item, params):
        """!Run given action"""
        name = item.GetName()
        if params.has_key(name):
            paramsOrig = item.GetParams(dcopy = True)
            item.MergeParams(params[name])
            
        self.SetStatusText(_('Running model...'), 0)
        self.goutput.RunCmd(command = item.GetLog(string = False),
                            onDone = self.OnDone)
            
        if params.has_key(name):
            item.SetParams(paramsOrig)
        
    def OnDone(self, cmd, returncode):
        """!Computation finished"""
        self.SetStatusText('', 0)
        
    def OnValidateModel(self, event, showMsg = True):
        """!Validate entire model"""
        if self.model.GetNumItems() < 1:
            GMessage(parent = self, 
                     message = _('Model is empty. Nothing to validate.'))
            return
        
        errList = self._validateModel()
        
        if errList:
            GWarning(parent = self,
                     message = _('Model is not valid.\n\n%s') % '\n'.join(errList))
        else:
            GMessage(parent = self,
                     message = _('Model is valid.'))
    
    def OnExportImage(self, event):
        """!Export model to image (default image)
        """
        xminImg = 0
        xmaxImg = 0
        yminImg = 0
        ymaxImg = 0
        # get current size of canvas
        for shape in self.canvas.GetDiagram().GetShapeList():
            w, h = shape.GetBoundingBoxMax()
            x    = shape.GetX()
            y    = shape.GetY()
            xmin = x - w / 2
            xmax = x + w / 2
            ymin = y - h / 2
            ymax = y + h / 2
            if xmin < xminImg:
                xminImg = xmin
            if xmax > xmaxImg:
                xmaxImg = xmax
            if ymin < yminImg:
                yminImg = ymin
            if ymax > ymaxImg:
                ymaxImg = ymax
        size = wx.Size(int(xmaxImg - xminImg) + 50,
                       int(ymaxImg - yminImg) + 50)
        bitmap = wx.EmptyBitmap(width = size.width, height = size.height)
        
        filetype, ltype = GetImageHandlers(wx.ImageFromBitmap(bitmap))
        
        dlg = wx.FileDialog(parent = self,
                            message = _("Choose a file name to save the image (no need to add extension)"),
                            defaultDir = "",
                            defaultFile = "",
                            wildcard = filetype,
                            style=wx.SAVE | wx.FD_OVERWRITE_PROMPT)
        
        if dlg.ShowModal() == wx.ID_OK:
            path = dlg.GetPath()
            if not path:
                dlg.Destroy()
                return
            
            base, ext = os.path.splitext(path)
            fileType = ltype[dlg.GetFilterIndex()]['type']
            extType  = ltype[dlg.GetFilterIndex()]['ext']
            if ext != extType:
                path = base + '.' + extType
            
            dc = wx.MemoryDC(bitmap)
            dc.SetBackground(wx.WHITE_BRUSH)
            dc.SetBackgroundMode(wx.SOLID)
            
            dc.BeginDrawing()
            self.canvas.GetDiagram().Clear(dc)
            self.canvas.GetDiagram().Redraw(dc)
            dc.EndDrawing()
            
            bitmap.SaveFile(path, fileType)
            self.SetStatusText(_("Model exported to <%s>") % path)
        
        dlg.Destroy()
        
    def OnExportPython(self, event):
        """!Export model to Python script"""
        filename = ''
        dlg = wx.FileDialog(parent = self,
                            message = _("Choose file to save"),
                            defaultDir = os.getcwd(),
                            wildcard=_("Python script (*.py)|*.py"),
                            style=wx.FD_SAVE)
        
        if dlg.ShowModal() == wx.ID_OK:
            filename = dlg.GetPath()
        
        if not filename:
            return
        
        # check for extension
        if filename[-3:] != ".py":
            filename += ".py"
        
        if os.path.exists(filename):
            dlg = wx.MessageDialog(self, message=_("File <%s> already exists. "
                                                   "Do you want to overwrite this file?") % filename,
                                   caption=_("Save file"),
                                   style=wx.YES_NO | wx.YES_DEFAULT | wx.ICON_QUESTION)
            if dlg.ShowModal() == wx.ID_NO:
                dlg.Destroy()
                return
            
            dlg.Destroy()
        
        fd = open(filename, "w")
        try:
            WritePythonFile(fd, self.model)
        finally:
            fd.close()
        
        # executable file
        os.chmod(filename, stat.S_IRWXU | stat.S_IWUSR)
        
        self.SetStatusText(_("Model exported to <%s>") % filename)

    def _validateModel(self):
        """!Validate model"""
        self.SetStatusText(_('Validating model...'), 0)
        
        errList = self.model.Validate()
        
        self.SetStatusText('', 0)
        
        return errList
    
    def OnDefineRelation(self, event):
        """!Define relation between data and action items"""
        self.canvas.SetCursor(self.cursors["cross"])
        self.defineRelation = { 'from' : None,
                                'to'   : None }
        
    def OnDefineLoop(self, event):
        """!Define new loop in the model"""
        self.ModelChanged()
        
        width, height = self.canvas.GetSize()
        loop = ModelLoop(self, x = width/2, y = height/2,
                         id = self.model.GetNumItems() + 1)
        self.canvas.diagram.AddShape(loop)
        loop.Show(True)
        
        self._addEvent(loop)
        self.model.AddItem(loop)
        
        self.canvas.Refresh()
        
    def OnDefineCondition(self, event):
        """!Define new condition in the model"""
        self.ModelChanged()
        
        width, height = self.canvas.GetSize()
        cond = ModelCondition(self, x = width/2, y = height/2,
                              id = self.model.GetNumItems() + 1)
        self.canvas.diagram.AddShape(cond)
        cond.Show(True)
        
        self._addEvent(cond)
        self.model.AddItem(cond)
        
        self.canvas.Refresh()
    
    def OnAddAction(self, event):
        """!Add action to model"""
        if self.searchDialog is None:
            self.searchDialog = ModelSearchDialog(self)
            self.searchDialog.CentreOnParent()
        else:
            self.searchDialog.Reset()
            
        if self.searchDialog.ShowModal() == wx.ID_CANCEL:
            self.searchDialog.Hide()
            return
            
        cmd = self.searchDialog.GetCmd()
        self.searchDialog.Hide()
        
        self.ModelChanged()
        
        # add action to canvas
        width, height = self.canvas.GetSize()
        if cmd[0] == 'r.mapcalc':
            GMessage(parent = self,
                     message = _("Module r.mapcalc cannot be used in the model. "
                                 "Use r.mapcalculator instead."))
            return
        
        action = ModelAction(self.model, cmd = cmd, x = width/2, y = height/2,
                             id = self.model.GetNumItems() + 1)
        overwrite = self.model.GetProperties().get('overwrite', None)
        if overwrite is not None:
            action.GetTask().set_flag('overwrite', overwrite)
        
        self.canvas.diagram.AddShape(action)
        action.Show(True)

        self._addEvent(action)
        self.model.AddItem(action)
        
        self.itemPanel.Update()
        self.canvas.Refresh()
        time.sleep(.1)
        
        # show properties dialog
        win = action.GetPropDialog()
        if not win and action.GetLog(string = False):
            module = menuform.GUI().ParseCommand(action.GetLog(string = False),
                                                 completed = (self.GetOptData, action, action.GetParams()),
                                                 parentframe = self, show = True)
        elif win and not win.IsShown():
            win.Show()
        
        if win:
            win.Raise()
        
    def OnAddData(self, event):
        """!Add data item to model

        @todo
        """
        # add action to canvas
        width, height = self.canvas.GetSize()
        data = ModelData(self, x = width/2, y = height/2)
        self.canvas.diagram.AddShape(data)
        data.Show(True)
        
        self._addEvent(data)
        # self.model.AddData(data)
        
        self.canvas.Refresh()
        
    def OnHelp(self, event):
        """!Display manual page"""
        grass.run_command('g.manual',
                          entry = 'wxGUI.Modeler')

    def OnAbout(self, event):
        """!Display About window"""
        info = wx.AboutDialogInfo()

        info.SetIcon(wx.Icon(os.path.join(globalvar.ETCICONDIR, 'grass.ico'), wx.BITMAP_TYPE_ICO))
        info.SetName(_('wxGUI Graphical Modeler'))
        info.SetWebSite('http://grass.osgeo.org')
        info.SetDescription(_('(C) 2010 by the GRASS Development Team\n\n') + 
                            '\n'.join(textwrap.wrap(_('This program is free software under the GNU General Public License'
                                                      '(>=v2). Read the file COPYING that comes with GRASS for details.'), 75)))
        
        wx.AboutBox(info)
        
    def GetOptData(self, dcmd, layer, params, propwin):
        """!Process action data"""
        if params: # add data items
            for p in params['params']:
                if p.get('prompt', '') in ('raster', 'vector', 'raster3d'):
                    try:
                        name, mapset = p.get('value', '').split('@', 1)
                    except (ValueError, IndexError):
                        continue
                    
                    if mapset != grass.gisenv()['MAPSET']:
                        continue
                    
                    # don't use fully qualified names
                    p['value'] = p.get('value', '').split('@')[0]
                    for idx in range(1, len(dcmd)):
                        if p.get('name', '') in dcmd[idx]:
                            dcmd[idx] = p.get('name', '') + '=' + p.get('value', '')
                            break
            
            width, height = self.canvas.GetSize()
            x = [width/2 + 200, width/2 - 200]
            for p in params['params']:
                if p.get('prompt', '') in ('raster', 'vector', 'raster3d') and \
                        (p.get('value', None) or \
                             (p.get('age', 'old') != 'old' and p.get('required', 'no') == 'yes')):
                    data = layer.FindData(p.get('name', ''))
                    if data:
                        data.SetValue(p.get('value', ''))
                        data.Update()
                        continue
                    
                    data = self.model.FindData(p.get('value', ''),
                                               p.get('prompt', ''))
                    if data:
                        if p.get('age', 'old') == 'old':
                            rel = ModelRelation(data, layer, p.get('name', ''))
                        else:
                            rel = ModelRelation(layer, data, p.get('name', ''))
                        layer.AddRelation(rel)
                        data.AddRelation(rel)
                        self.AddLine(rel)
                        data.Update()
                        continue
                    
                    data = ModelData(self, value = p.get('value', ''),
                                     prompt = p.get('prompt', ''),
                                     x = x.pop(), y = height/2)
                    self._addEvent(data)
                    self.canvas.diagram.AddShape(data)
                    data.Show(True)
                                                            
                    if p.get('age', 'old') == 'old':
                        rel = ModelRelation(data, layer, p.get('name', ''))
                    else:
                        rel = ModelRelation(layer, data, p.get('name', ''))
                    layer.AddRelation(rel)
                    data.AddRelation(rel)
                    self.AddLine(rel)
                    data.Update()
            
            # valid ?
            valid = True
            for p in params['params']:
                if p.get('required', 'no') != 'no' and \
                        p.get('value', '') == '' and \
                        p.get('default', '') == '':
                    valid = False
                    break
            layer.SetValid(valid)

            # parameterized ?
            parameterized = False
            for f in params['flags']:
                if f.get('parameterized', False):
                    parameterized = True
                    break
            if not parameterized:
                for p in params['params']:
                    if p.get('parameterized', False):
                        parameterized = True
                        break
            layer.SetParameterized(parameterized)
            
            self.canvas.Refresh()
        
        if dcmd:
            layer.SetProperties(params, propwin)
            
        self.SetStatusText(layer.GetLog(), 0)
        
    def AddLine(self, rel):
        """!Add connection between model objects
        
        @param rel relation
        """
        fromShape = rel.GetFrom()
        toShape   = rel.GetTo()
        
        rel.SetCanvas(self)
        rel.SetPen(wx.BLACK_PEN)
        rel.SetBrush(wx.BLACK_BRUSH)
        rel.AddArrow(ogl.ARROW_ARROW)
        points = rel.GetControlPoints()
        rel.MakeLineControlPoints(2)
        if points:
            for x, y in points:
                rel.InsertLineControlPoint(point = wx.RealPoint(x, y))
        
        self._addEvent(rel)
        try:
            fromShape.AddLine(rel, toShape)
        except TypeError:
            pass # bug when connecting ModelCondition and ModelLoop - to be fixed
        
        self.canvas.diagram.AddShape(rel)
        rel.Show(True)
        
    def LoadModelFile(self, filename):
        """!Load model definition stored in GRASS Model XML file (gxm)
        """
        try:
            self.model.LoadModel(filename)
        except GException, e:
            GError(parent = self,
                   message = _("Reading model file <%s> failed.\n"
                               "Invalid file, unable to parse XML document.") % filename)
        
        self.modelFile = filename
        self.SetTitle(self.baseTitle + " - " +  os.path.basename(self.modelFile))
        
        self.SetStatusText(_("Please wait, loading model..."), 0)
        
        # load actions
        for item in self.model.GetItems(objType = ModelAction):
            self._addEvent(item)
            self.canvas.diagram.AddShape(item)
            item.Show(True)
            # relations/data
            for rel in item.GetRelations():
                if rel.GetFrom() == item:
                    dataItem = rel.GetTo()
                else:
                    dataItem = rel.GetFrom()
                self._addEvent(dataItem)
                self.canvas.diagram.AddShape(dataItem)
                self.AddLine(rel)
                dataItem.Show(True)
        
        # load loops
        for item in self.model.GetItems(objType = ModelLoop):
            self._addEvent(item)
            self.canvas.diagram.AddShape(item)
            item.Show(True)
            
            # connect items in the loop
            self.DefineLoop(item)

        # load conditions
        for item in self.model.GetItems(objType = ModelCondition):
            self._addEvent(item)
            self.canvas.diagram.AddShape(item)
            item.Show(True)
            
            # connect items in the condition
            self.DefineCondition(item)
        
        # load variables
        self.variablePanel.Update()
        self.itemPanel.Update()
        self.SetStatusText('', 0)
        
        self.canvas.Refresh(True)
        
    def WriteModelFile(self, filename):
        """!Save model to model file, recover original file on error.
        
        @return True on success
        @return False on failure
        """
        self.ModelChanged(False)
        tmpfile = tempfile.TemporaryFile(mode='w+b')
        try:
            WriteModelFile(fd = tmpfile, model = self.model)
        except StandardError:
            GError(parent = self,
                   message = _("Writing current settings to model file failed."))
            return False
        
        try:
            mfile = open(filename, "w")
            tmpfile.seek(0)
            for line in tmpfile.readlines():
                mfile.write(line)
        except IOError:
            wx.MessageBox(parent = self,
                          message = _("Unable to open file <%s> for writing.") % filename,
                          caption = _("Error"),
                          style = wx.OK | wx.ICON_ERROR | wx.CENTRE)
            return False
        
        mfile.close()
        
        return True
    
    def DefineLoop(self, loop):
        """!Define loop with given list of items"""
        parent = loop
        items = loop.GetItems()
        if not items:
            return
        
        # remove defined relations first
        for rel in loop.GetRelations():
            self.canvas.GetDiagram().RemoveShape(rel)
        loop.Clear()
        
        for item in items:
            rel = ModelRelation(parent, item)
            dx = item.GetX() - parent.GetX()
            dy = item.GetY() - parent.GetY()
            loop.AddRelation(rel)
            if dx != 0:
                rel.SetControlPoints(((parent.GetX(), parent.GetY() + dy / 2),
                                      (parent.GetX() + dx, parent.GetY() + dy / 2)))
            self.AddLine(rel)
            parent = item
        
        # close loop
        item = loop.GetItems()[-1]
        rel = ModelRelation(item, loop)
        loop.AddRelation(rel)
        self.AddLine(rel)
        dx = (item.GetX() - loop.GetX()) + loop.GetWidth() / 2 + 50
        dy = item.GetHeight() / 2 + 50
        rel.MakeLineControlPoints(0)
        rel.InsertLineControlPoint(point = wx.RealPoint(loop.GetX() - loop.GetWidth() / 2 ,
                                                        loop.GetY()))
        rel.InsertLineControlPoint(point = wx.RealPoint(item.GetX(),
                                                        item.GetY() + item.GetHeight() / 2))
        rel.InsertLineControlPoint(point = wx.RealPoint(item.GetX(),
                                                        item.GetY() + dy))
        rel.InsertLineControlPoint(point = wx.RealPoint(item.GetX() - dx,
                                                        item.GetY() + dy))
        rel.InsertLineControlPoint(point = wx.RealPoint(item.GetX() - dx,
                                                        loop.GetY()))
        
        self.canvas.Refresh()

    def DefineCondition(self, condition):
        """!Define if-else statement with given list of items"""
        parent = condition
        items = condition.GetItems()
        if not items['if'] and not items['else']:
            return
        
        # remove defined relations first
        for rel in condition.GetRelations():
            self.canvas.GetDiagram().RemoveShape(rel)
        condition.Clear()
        dxIf   = condition.GetX() + condition.GetWidth() / 2
        dxElse = condition.GetX() - condition.GetWidth() / 2
        dy     = condition.GetY()
        for branch in items.keys():
            for item in items[branch]:
                rel = ModelRelation(parent, item)
                condition.AddRelation(rel)
                self.AddLine(rel)
                rel.MakeLineControlPoints(0)
                if branch == 'if':
                    rel.InsertLineControlPoint(point = wx.RealPoint(item.GetX() - item.GetWidth() / 2, item.GetY()))
                    rel.InsertLineControlPoint(point = wx.RealPoint(dxIf, dy))
                else:
                    rel.InsertLineControlPoint(point = wx.RealPoint(dxElse, dy))
                    rel.InsertLineControlPoint(point = wx.RealPoint(item.GetX() - item.GetWidth() / 2, item.GetY()))
                parent = item
        
        self.canvas.Refresh()
        
class ModelCanvas(ogl.ShapeCanvas):
    """!Canvas where model is drawn"""
    def __init__(self, parent):
        self.parent = parent
        ogl.OGLInitialize()
        ogl.ShapeCanvas.__init__(self, parent)
        
        self.diagram = ogl.Diagram()
        self.SetDiagram(self.diagram)
        self.diagram.SetCanvas(self)
        
        self.SetScrollbars(20, 20, 1000/20, 1000/20)
        
        self.Bind(wx.EVT_CHAR,  self.OnChar)
        
    def OnChar(self, event):
        """!Key pressed"""
        kc = event.GetKeyCode()
        diagram = self.GetDiagram()
        if kc == wx.WXK_DELETE:
            self.RemoveSelected()
        
    def RemoveSelected(self):
        """!Remove selected shapes"""
        self.parent.ModelChanged()
        
        diagram = self.GetDiagram()
        for shape in diagram.GetShapeList():
            if not shape.Selected():
                continue
            remList, upList = self.parent.GetModel().RemoveItem(shape)
            shape.Select(False)
            diagram.RemoveShape(shape)
            del shape
            for item in remList:
                diagram.RemoveShape(item)
                item.__del__()
            
            for item in upList:
                item.Update()
        
        self.Refresh()
        
class ModelObject:
    def __init__(self, id = -1):
        self.id   = id
        self.rels = list() # list of ModelRelations
        
        self.isEnabled = True
        self.inBlock   = list() # list of related loops/conditions
        
    def __del__(self):
        pass
    
    def GetId(self):
        """!Get id"""
        return self.id
    
    def AddRelation(self, rel):
        """!Record new relation
        """
        self.rels.append(rel)

    def GetRelations(self, fdir = None):
        """!Get list of relations
        
        @param fdir True for 'from'
        """
        if fdir is None:
            return self.rels
        
        result = list()
        for rel in self.rels:
            if fdir == 'from':
                if rel.GetFrom() == self:
                    result.append(rel)
            else:
                if rel.GetTo() == self:
                    result.append(rel)
        
        return result
    
    def IsEnabled(self):
        """!Get True if action is enabled, otherwise False"""
        return self.isEnabled
    
    def Enable(self, enabled = True):
        """!Enable/disable action"""
        self.isEnabled = enabled
        self.Update()

    def Update(self):
        pass

    def SetBlock(self, item):
        """!Add object to the block (loop/condition)

        @param item reference to ModelLoop or ModelCondition which
        defines loops/condition
        """
        if item not in self.inBlock:
            self.inBlock.append(item)
        
    def UnSetBlock(self, item):
        """!Remove object from the block (loop/consition)

        @param item reference to ModelLoop or ModelCondition which
        defines loops/codition
        """
        if item in self.inBlock:
            self.inBlock.remove(item)
        
    def GetBlock(self):
        """!Get list of related ModelObject(s) which defines block
        (loop/condition)

        @return list of ModelObjects
        """
        return self.inBlock
    
    def GetBlockId(self):
        """!Get list of related ids which defines block

        @return list of ids
        """
        ret = list()
        for mo in self.inBlock:
            ret.append(mo.GetId())
        
        return ret
    
class ModelAction(ModelObject, ogl.RectangleShape):
    """!Action class (GRASS module)"""
    def __init__(self, parent, x, y, id = -1, cmd = None, task = None, width = None, height = None):
        ModelObject.__init__(self, id)
        
        self.parent  = parent
        self.task    = task
        
        if not width:
            width = UserSettings.Get(group='modeler', key='action', subkey=('size', 'width'))
        if not height:
            height = UserSettings.Get(group='modeler', key='action', subkey=('size', 'height'))
        
        if cmd:
            self.task = menuform.GUI().ParseCommand(cmd = cmd,
                                                    show = None)
        else:
            if task:
                self.task = task
            else:
                self.task = None
        
        self.propWin = None
        
        self.data = list()   # list of connected data items
        
        self.isValid = False
        self.isParameterized = False
        
        if self.parent.GetCanvas():
            ogl.RectangleShape.__init__(self, width, height)
            
            self.SetCanvas(self.parent)
            self.SetX(x)
            self.SetY(y)
            self.SetPen(wx.BLACK_PEN)
            self._setPen()
            self._setBrush()
            self.SetId(id)
            
    def _setBrush(self, running = False):
        """!Set brush"""
        if running:
            color = UserSettings.Get(group='modeler', key='action',
                                     subkey=('color', 'running'))
        elif not self.isEnabled:
            color = UserSettings.Get(group='modeler', key='action',
                                     subkey=('color', 'disabled'))
        elif self.isValid:
            color = UserSettings.Get(group='modeler', key='action',
                                     subkey=('color', 'valid'))
        else:
            color = UserSettings.Get(group='modeler', key='action',
                                     subkey=('color', 'invalid'))
        
        wxColor = wx.Color(color[0], color[1], color[2])
        self.SetBrush(wx.Brush(wxColor))
        
    def _setPen(self):
        """!Set pen"""
        if self.isParameterized:
            width = int(UserSettings.Get(group='modeler', key='action',
                                         subkey=('width', 'parameterized')))
        else:
            width = int(UserSettings.Get(group='modeler', key='action',
                                         subkey=('width', 'default')))
        pen = self.GetPen()
        pen.SetWidth(width)
        self.SetPen(pen)

    def SetId(self, id):
        """!Set id"""
        self.id = id
        cmd = self.task.getCmd(ignoreErrors = True)
        if cmd and len(cmd) > 0:
            self.ClearText()
            self.AddText('(%d) %s' % (self.id, cmd[0]))
        else:
            self.AddText('(%d) <<%s>>' % (self.id, _("unknown")))
        
    def SetProperties(self, params, propwin):
        """!Record properties dialog"""
        self.task.params = params['params']
        self.task.flags  = params['flags']
        self.propWin = propwin

    def GetPropDialog(self):
        """!Get properties dialog"""
        return self.propWin

    def GetLog(self, string = True):
        """!Get logging info"""
        cmd = self.task.getCmd(ignoreErrors = True)
        # substitute variables
        variables = self.parent.GetVariables()
        fparams = self.parent.GetVariables(params = True)
        params = None
        for values in fparams.itervalues():
            params = values['params']
            break
        
        for variable in variables:
            pattern= re.compile('%' + variable)
            value = None
            if params:
                for p in params:
                    if variable == p.get('name', ''):
                        value = p.get('value', '')
                        break
            if not value:
                value = variables[variable].get('value', '')
            
            for idx in range(len(cmd)):
                if pattern.search(cmd[idx]):
                    if value:
                        cmd[idx] = pattern.sub(value, cmd[idx])
                    else:
                        self.isValid = False
                    break
                idx += 1
        
        if string:
            if cmd is None:
                return ''
            else:
                return ' '.join(cmd)
        
        return cmd
    
    def GetName(self):
        """!Get name"""
        cmd = self.task.getCmd(ignoreErrors = True)
        if cmd and len(cmd) > 0:
            return cmd[0]
        
        return _('unknown')

    def GetParams(self, dcopy = False):
        """!Get dictionary of parameters"""
        if dcopy:
            return copy.deepcopy(self.task.get_options())
        
        return self.task.get_options()

    def GetTask(self):
        """!Get grassTask instance"""
        return self.task
    
    def SetParams(self, params):
        """!Set dictionary of parameters"""
        self.task.params = params['params']
        self.task.flags  = params['flags']
        
    def MergeParams(self, params):
        """!Merge dictionary of parameters"""
        if params.has_key('flags'):
            for f in params['flags']:
                self.task.set_flag(f['name'],
                                   f.get('value', False))
        if params.has_key('params'):
            for p in params['params']:
                self.task.set_param(p['name'],
                                    p.get('value', ''))
        
    def SetValid(self, isvalid):
        """!Set instance to be valid/invalid"""
        self.isValid = isvalid
        self._setBrush()
        
    def SetParameterized(self, isparameterized):
        """!Set action parameterized"""
        self.isParameterized = isparameterized
        self._setPen()

    def IsParameterized(self):
        """!Check if action is parameterized"""
        return self.isParameterized
    
    def FindData(self, name):
        """!Find data item by name"""
        for rel in self.GetRelations():
            data = rel.GetData()
            if name == rel.GetName() and name in data.GetName():
                return data
        
        return None

    def Update(self, running = False):
        """!Update action"""
        if running:
            self._setBrush(running = True)
        else:
            self._setBrush()
        self._setPen()

    def OnDraw(self, dc):
        """!Draw action in canvas"""
        self._setBrush()
        self._setPen()
        ogl.RectangleShape.OnDraw(self, dc)
    
class ModelData(ModelObject, ogl.EllipseShape):
    def __init__(self, parent, x, y, value = '', prompt = '', width = None, height = None):
        """Data item class
        
        @param parent window parent
        @param x, y   position of the shape
        @param fname, tname list of parameter names from / to
        @param value  value
        @param prompt type of GIS element
        @param width, height dimension of the shape
        """
        ModelObject.__init__(self)
        
        self.parent  = parent
        self.value   = value
        self.prompt  = prompt
        self.intermediate = False
        self.propWin = None
        if not width:
            width = UserSettings.Get(group='modeler', key='data', subkey=('size', 'width'))
        if not height:
            height = UserSettings.Get(group='modeler', key='data', subkey=('size', 'height'))
        
        if self.parent.GetCanvas():
            ogl.EllipseShape.__init__(self, width, height)
            
            self.SetCanvas(self.parent)
            self.SetX(x)
            self.SetY(y)
            self.SetPen(wx.BLACK_PEN)
            self._setBrush()
            
            self._setText()
            
    def IsIntermediate(self):
        """!Checks if data item is intermediate"""
        return self.intermediate
    
    def SetIntermediate(self, im):
        """!Set intermediate flag"""
        self.intermediate = im
  
    def OnDraw(self, dc):
        pen = self.GetPen()
        pen.SetWidth(1)
        if self.intermediate:
            pen.SetStyle(wx.SHORT_DASH)
        else:
            pen.SetStyle(wx.SOLID)
        self.SetPen(pen)
        
        ogl.EllipseShape.OnDraw(self, dc)
        
    def GetLog(self, string = True):
        """!Get logging info"""
        name = list()
        for rel in self.GetRelations():
            name.append(rel.GetName())
        if name:
            return '/'.join(name) + '=' + self.value + ' (' + self.prompt + ')'
        else:
            return _('unknown')

    def GetName(self):
        """!Get list of names"""
        name = list()
        for rel in self.GetRelations():
            name.append(rel.GetName())
        
        return name
    
    def GetPrompt(self):
        """!Get prompt"""
        return self.prompt

    def GetValue(self):
        """!Get value"""
        return self.value

    def SetValue(self, value):
        """!Set value"""
        self.value = value
        self._setText()
        for direction in ('from', 'to'):
            for rel in self.GetRelations(direction):
                if direction == 'from':
                    action = rel.GetTo()
                else:
                    action = rel.GetFrom()
                
                task = menuform.GUI().ParseCommand(cmd = action.GetLog(string = False),
                                                   show = None)
                task.set_param(rel.GetName(), self.value)
                action.SetParams(params = task.get_options())
        
    def AddName(self, name, direction):
        """!Record new name (parameter)

        @param direction direction - 'from' or 'to'
        """
        self.name[direction].append(name)
        
    def GetPropDialog(self):
        """!Get properties dialog"""
        return self.propWin

    def SetPropDialog(self, win):
        """!Get properties dialog"""
        self.propWin = win

    def _setBrush(self):
        """!Set brush"""
        if self.prompt == 'raster':
            color = UserSettings.Get(group='modeler', key='data',
                                     subkey=('color', 'raster'))
        elif self.prompt == 'raster3d':
            color = UserSettings.Get(group='modeler', key='data',
                                     subkey=('color', 'raster3d'))
        elif self.prompt == 'vector':
            color = UserSettings.Get(group='modeler', key='data',
                                     subkey=('color', 'vector'))
        else:
            color = UserSettings.Get(group='modeler', key='action',
                                     subkey=('color', 'invalid'))
        wxColor = wx.Color(color[0], color[1], color[2])
        self.SetBrush(wx.Brush(wxColor))
        
    def _setPen(self):
        """!Set pen"""
        isParameterized = False
        for rel in self.GetRelations('from'):
            if rel.GetTo().IsParameterized():
                isParameterized = True
                break
        if not isParameterized:
            for rel in self.GetRelations('to'):
                if rel.GetFrom().IsParameterized():
                    isParameterized = True
                    break

        if isParameterized:
            width = int(UserSettings.Get(group='modeler', key='action',
                                         subkey=('width', 'parameterized')))
        else:
            width = int(UserSettings.Get(group='modeler', key='action',
                                         subkey=('width', 'default')))
        pen = self.GetPen()
        pen.SetWidth(width)
        self.SetPen(pen)
        
    def _setText(self):
        """!Update text"""
        self.ClearText()
        name = []
        for rel in self.GetRelations():
            name.append(rel.GetName())
        self.AddText('/'.join(name))
        if self.value:
            self.AddText(self.value)
        else:
            self.AddText(_('<not defined>'))
        
    def Update(self):
        """!Update action"""
        self._setBrush()
        self._setPen()
        self._setText()
        
class ModelDataDialog(ElementDialog):
    """!Data item properties dialog"""
    def __init__(self, parent, shape, id = wx.ID_ANY, title = _("Data properties"),
                 style = wx.DEFAULT_DIALOG_STYLE | wx.RESIZE_BORDER):
        self.parent = parent
        self.shape = shape
        prompt = shape.GetPrompt()
        
        if prompt == 'raster':
            label = _('Name of raster map:')
        elif prompt == 'vector':
            label = _('Name of vector map:')
        else:
            label = _('Name of element:')
        
        ElementDialog.__init__(self, parent, title, label = label)
        
        self.element = gselect.Select(parent = self.panel, id = wx.ID_ANY,
                                      size = globalvar.DIALOG_GSELECT_SIZE,
                                      type = prompt)
        self.element.SetValue(shape.GetValue())
        
        self.Bind(wx.EVT_BUTTON, self.OnOK,     self.btnOK)
        self.Bind(wx.EVT_BUTTON, self.OnCancel, self.btnCancel)
        
        self.PostInit()
        
        if shape.GetValue():
            self.btnOK.Enable()
        
        self._layout()
        self.SetMinSize(self.GetSize())
        
    def _layout(self):
        """!Do layout"""
        self.dataSizer.Add(self.element, proportion=0,
                      flag=wx.EXPAND | wx.ALL, border=1)
        
        self.panel.SetSizer(self.sizer)
        self.sizer.Fit(self)

    def OnOK(self, event):
        """!Ok pressed"""
        self.shape.SetValue(self.GetElement())
        self.parent.canvas.Refresh()
        self.parent.SetStatusText('', 0)
        self.OnCancel(event)
        
    def OnCancel(self, event):
        """!Cancel pressed"""
        self.shape.SetPropDialog(None)
        self.Destroy()
        
class ModelEvtHandler(ogl.ShapeEvtHandler):
    """!Model event handler class"""
    def __init__(self, log, frame):
        ogl.ShapeEvtHandler.__init__(self)
        self.log = log
        self.frame = frame
        self.x = self.y = None
        
    def OnLeftClick(self, x, y, keys = 0, attachment = 0):
        """!Left mouse button pressed -> select item & update statusbar"""
        shape = self.GetShape()
        canvas = shape.GetCanvas()
        dc = wx.ClientDC(canvas)
        canvas.PrepareDC(dc)
        
        if hasattr(self.frame, 'defineRelation'):
            drel = self.frame.defineRelation
            if drel['from'] is None:
                drel['from'] = shape
            elif drel['to'] is None:
                drel['to'] = shape
                rel = ModelRelation(drel['from'], drel['to'])
                drel['from'].AddRelation(rel)
                self.frame.AddLine(rel)
                del self.frame.defineRelation
        
        if shape.Selected():
            shape.Select(False, dc)
        else:
            redraw = False
            shapeList = canvas.GetDiagram().GetShapeList()
            toUnselect = list()
            
            for s in shapeList:
                if s.Selected():
                    toUnselect.append(s)
            
            shape.Select(True, dc)
            
            for s in toUnselect:
                s.Select(False, dc)
                
        canvas.Refresh(False)

        if hasattr(shape, "GetLog"):
            self.log.SetStatusText(shape.GetLog(), 0)
        else:
            self.log.SetStatusText('', 0)
        
    def OnLeftDoubleClick(self, x, y, keys = 0, attachment = 0):
        """!Left mouse button pressed (double-click) -> show properties"""
        self.OnProperties()
        
    def OnProperties(self, event = None):
        """!Show properties dialog"""
        self.frame.ModelChanged()
        shape = self.GetShape()
        if isinstance(shape, ModelAction):
            module = menuform.GUI().ParseCommand(shape.GetLog(string = False),
                                                 completed = (self.frame.GetOptData, shape, shape.GetParams()),
                                                 parentframe = self.frame, show = True)
        
        elif isinstance(shape, ModelData):
            dlg = ModelDataDialog(parent = self.frame, shape = shape)
            shape.SetPropDialog(dlg)
            dlg.CentreOnParent()
            dlg.Show()
        
        elif isinstance(shape, ModelLoop):
            dlg = ModelLoopDialog(parent = self.frame, shape = shape)
            dlg.CentreOnParent()
            if dlg.ShowModal() == wx.ID_OK:
                shape.SetText(dlg.GetCondition())
                alist = list()
                ids = dlg.GetItems()
                for aId in ids['unchecked']:
                    action = self.frame.GetModel().GetItem(aId)
                    action.UnSetBlock(shape)
                for aId in ids['checked']:
                    action = self.frame.GetModel().GetItem(aId)
                    action.SetBlock(shape)
                    if action:
                        alist.append(action)
                shape.SetItems(alist)
                self.frame.DefineLoop(shape)
            self.frame.GetCanvas().Refresh()
            
            dlg.Destroy()
        
        elif isinstance(shape, ModelCondition):
            dlg = ModelConditionDialog(parent = self.frame, shape = shape)
            dlg.CentreOnParent()
            if dlg.ShowModal() == wx.ID_OK:
                shape.SetText(dlg.GetCondition())
                ids = dlg.GetItems()
                for b in ids.keys():
                    alist = list()
                    for aId in ids[b]['unchecked']:
                        action = self.frame.GetModel().GetItem(aId)
                        action.UnSetBlock(shape)
                    for aId in ids[b]['checked']:
                        action = self.frame.GetModel().GetItem(aId)
                        action.SetBlock(shape)
                        if action:
                            alist.append(action)
                    shape.SetItems(alist, branch = b)
                self.frame.DefineCondition(shape)
            self.frame.GetCanvas().Refresh()
            
            dlg.Destroy()
                   
    def OnBeginDragLeft(self, x, y, keys = 0, attachment = 0):
        """!Drag shape (begining)"""
        self.frame.ModelChanged()
        if self._previousHandler:
            self._previousHandler.OnBeginDragLeft(x, y, keys, attachment)
        
    def OnEndDragLeft(self, x, y, keys = 0, attachment = 0):
        """!Drag shape (end)"""
        if self._previousHandler:
            self._previousHandler.OnEndDragLeft(x, y, keys, attachment)
        
        shape = self.GetShape()
        if isinstance(shape, ModelLoop):
            self.frame.DefineLoop(shape)
        elif isinstance(shape, ModelCondition):
            self.frame.DefineCondition(shape)
        
        for mo in shape.GetBlock():
            if isinstance(mo, ModelLoop):
                self.frame.DefineLoop(mo)
            elif isinstance(mo, ModelCondition):
                self.frame.DefineCondition(mo)
        
    def OnEndSize(self, x, y):
        """!Resize shape"""
        self.frame.ModelChanged()
        if self._previousHandler:
            self._previousHandler.OnEndSize(x, y)
        
    def OnRightClick(self, x, y, keys = 0, attachment = 0):
        """!Right click -> pop-up menu"""
        if not hasattr (self, "popupID1"):
            self.popupID1 = wx.NewId()
            self.popupID2 = wx.NewId()
            self.popupID3 = wx.NewId()
            self.popupID4 = wx.NewId()

        # record coordinates
        self.x = x
        self.y = y
        
        shape = self.GetShape()
        popupMenu = wx.Menu()
        popupMenu.Append(self.popupID1, text=_('Remove'))
        self.frame.Bind(wx.EVT_MENU, self.OnRemove, id = self.popupID1)
        if isinstance(shape, ModelAction):
            if shape.IsEnabled():
                popupMenu.Append(self.popupID3, text=_('Disable'))
                self.frame.Bind(wx.EVT_MENU, self.OnDisable, id = self.popupID3)
            else:
                popupMenu.Append(self.popupID3, text=_('Enable'))
                self.frame.Bind(wx.EVT_MENU, self.OnEnable, id = self.popupID3)
        
        if isinstance(shape, ModelRelation):
            popupMenu.AppendSeparator()
            popupMenu.Append(self.popupID2, text=_('Add control point'))
            self.frame.Bind(wx.EVT_MENU, self.OnAddPoint, id = self.popupID2)
            popupMenu.Append(self.popupID3, text=_('Remove control point'))
            self.frame.Bind(wx.EVT_MENU, self.OnRemovePoint, id = self.popupID3)
            if len(shape.GetLineControlPoints()) == 2:
                popupMenu.Enable(self.popupID3, False)
        
        if isinstance(shape, ModelData) and '@' not in shape.GetValue():
            popupMenu.AppendSeparator()
            popupMenu.Append(self.popupID3, text=_('Intermediate'),
                             kind = wx.ITEM_CHECK)
            if self.GetShape().IsIntermediate():
                popupMenu.Check(self.popupID3, True)
            
            self.frame.Bind(wx.EVT_MENU, self.OnIntermediate, id = self.popupID3)
            
        if isinstance(shape, ModelData) or \
                isinstance(shape, ModelAction) or \
                isinstance(shape, ModelLoop):
            popupMenu.AppendSeparator()
            popupMenu.Append(self.popupID2, text=_('Properties'))
            self.frame.Bind(wx.EVT_MENU, self.OnProperties, id = self.popupID2)
        
        if isinstance(shape, ModelAction):
            popupMenu.Append(self.popupID4, text=_('Change ID'))
            self.frame.Bind(wx.EVT_MENU, self.OnChangeId, id = self.popupID3)
        
        self.frame.PopupMenu(popupMenu)
        popupMenu.Destroy()

    def OnChangeId(self, event):
        """!Change action id"""
        pass
    
    def OnDisable(self, event):
        """!Disable action"""
        action = self.GetShape()
        action.Enable(False)
        self.frame.ModelChanged()
        self.frame.canvas.Refresh()

    def OnEnable(self, event):
        """!Disable action"""
        action = self.GetShape()
        action.Enable(True)
        self.frame.ModelChanged()
        self.frame.canvas.Refresh()
        
    def OnAddPoint(self, event):
        """!Add control point"""
        shape = self.GetShape()
        shape.InsertLineControlPoint(point = wx.RealPoint(self.x, self.y))
        shape.ResetShapes()
        shape.Select(True)
        self.frame.ModelChanged()
        self.frame.canvas.Refresh()
        
    def OnRemovePoint(self, event):
        """!Remove control point"""
        shape = self.GetShape()
        shape.DeleteLineControlPoint()
        shape.Select(False)
        shape.Select(True)
        self.frame.ModelChanged()
        self.frame.canvas.Refresh()
        
    def OnIntermediate(self, event):
        """!Mark data as intermediate"""
        self.frame.ModelChanged()
        shape = self.GetShape()
        shape.SetIntermediate(event.IsChecked())
        self.frame.canvas.Refresh()

    def OnRemove(self, event):
        """!Remove shape
        """
        self.frame.GetCanvas().RemoveSelected()
        self.frame.itemPanel.Update()
        
class ModelSearchDialog(wx.Dialog):
    def __init__(self, parent, id = wx.ID_ANY, title = _("Add new GRASS module to the model"),
                 style = wx.DEFAULT_DIALOG_STYLE | wx.RESIZE_BORDER, **kwargs):
        """!Graphical modeler module search window
        
        @param parent parent window
        @param id window id
        @param title window title
        @param kwargs wx.Dialogs' arguments
        """
        self.parent = parent
        
        wx.Dialog.__init__(self, parent = parent, id = id, title = title, **kwargs)
        self.SetName("ModelerDialog")
        self.SetIcon(wx.Icon(os.path.join(globalvar.ETCICONDIR, 'grass.ico'), wx.BITMAP_TYPE_ICO))
        
        self.panel = wx.Panel(parent = self, id = wx.ID_ANY)
        
        self.cmdBox = wx.StaticBox(parent = self.panel, id = wx.ID_ANY,
                                   label=" %s " % _("Command"))
        
        self.cmd_prompt = prompt.GPromptSTC(parent = self)
        self.search = SearchModuleWindow(parent = self.panel, cmdPrompt = self.cmd_prompt, showTip = True)
        wx.CallAfter(self.cmd_prompt.SetFocus)
        
        # get commands
        items = self.cmd_prompt.GetCommandItems()
        
        self.btnCancel = wx.Button(self.panel, wx.ID_CANCEL)
        self.btnOk     = wx.Button(self.panel, wx.ID_OK)
        self.btnOk.SetDefault()
        self.btnOk.Enable(False)

        self.cmd_prompt.Bind(wx.EVT_CHAR, self.OnText)
        self.search.searchChoice.Bind(wx.EVT_CHOICE, self.OnText)
        self.Bind(wx.EVT_BUTTON, self.OnOk, self.btnOk)
        
        self._layout()
        
        self.SetSize((500, 275))
        
    def _layout(self):
        cmdSizer = wx.StaticBoxSizer(self.cmdBox, wx.VERTICAL)
        cmdSizer.Add(item = self.cmd_prompt, proportion = 1,
                     flag = wx.EXPAND)
        
        btnSizer = wx.StdDialogButtonSizer()
        btnSizer.AddButton(self.btnCancel)
        btnSizer.AddButton(self.btnOk)
        btnSizer.Realize()
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(item = self.search, proportion = 0,
                      flag = wx.EXPAND | wx.ALL, border = 3)
        mainSizer.Add(item = cmdSizer, proportion = 1,
                      flag = wx.EXPAND | wx.LEFT | wx.RIGHT | wx.TOP, border = 3)
        mainSizer.Add(item = btnSizer, proportion = 0,
                      flag = wx.EXPAND | wx.ALL | wx.ALIGN_CENTER, border = 5)
        
        self.panel.SetSizer(mainSizer)
        mainSizer.Fit(self.panel)
        
        self.Layout()

    def GetPanel(self):
        """!Get dialog panel"""
        return self.panel

    def GetCmd(self):
        """!Get command"""
        line = self.cmd_prompt.GetCurLine()[0].strip()
        if len(line) == 0:
            list()
        
        try:
            cmd = shlex.split(str(line))
        except UnicodeError:
            cmd = shlex.split(utils.EncodeString((line)))
            
        return cmd
    
    def OnOk(self, event):
        self.btnOk.SetFocus()
        cmd = self.GetCmd()
        
        if len(cmd) < 1:
            GError(parent = self,
                   message = _("Command not defined.\n\n"
                               "Unable to add new action to the model."))
            return
        
        if cmd[0] not in globalvar.grassCmd['all']:
            GError(parent = self,
                   message = _("'%s' is not a GRASS module.\n\n"
                               "Unable to add new action to the model.") % cmd[0])
            return
        
        self.EndModal(wx.ID_OK)
        
    def OnText(self, event):
        """!Text entered"""
        if self.cmd_prompt.AutoCompActive():
            event.Skip()
            return
        
        if isinstance(event, wx.KeyEvent):
            entry = self.cmd_prompt.GetTextLeft() # FIXME
        else:
            entry = event.GetString()
        
        if entry:
            self.btnOk.Enable()
        else:
            self.btnOk.Enable(False)
            
        event.Skip()
        
    def Reset(self):
        """!Reset dialog"""
        self.search.Reset()
        self.cmd_prompt.OnCmdErase(None)

class ModelRelation(ogl.LineShape):
    """!Data - action relation"""
    def __init__(self, fromShape, toShape, param = ''):
        self.fromShape = fromShape
        self.toShape   = toShape
        self.param     = param
        
        self._points    = None
        
        ogl.LineShape.__init__(self)
    
    def __del__(self):
        self.fromShape.rels.remove(self)
        self.toShape.rels.remove(self)

    def GetFrom(self):
        """!Get id of 'from' shape"""
        return self.fromShape
    
    def GetTo(self):
        """!Get id of 'to' shape"""
        return self.toShape
    
    def GetData(self):
        """!Get related ModelData instance

        @return ModelData instance
        @return None if not found
        """
        if isinstance(self.fromShape, ModelData):
            return self.fromShape
        elif isinstance(self.toShape, ModelData):
            return self.toShape
        
        return None
    
    def GetName(self):
        """!Get parameter name"""
        return self.param
    
    def ResetShapes(self):
        """!Reset related objects"""
        self.fromShape.ResetControlPoints()
        self.toShape.ResetControlPoints()
        self.ResetControlPoints()
        
    def SetControlPoints(self, points):
        """!Set control points"""
        self._points = points
        
    def GetControlPoints(self):
        """!Get list of control points"""
        return self._points
    
    def _setPen(self):
        """!Set pen"""
        pen = self.GetPen()
        pen.SetWidth(1)
        pen.SetStyle(wx.SOLID)
        self.SetPen(pen)
        
    def OnDraw(self, dc):
        """!Draw relation"""
        self._setPen()
        ogl.LineShape.OnDraw(self, dc)
        
class ProcessModelFile:
    """!Process GRASS model file (gxm)"""
    def __init__(self, tree):
        """!A ElementTree handler for the GXM XML file, as defined in
        grass-gxm.dtd.
        """
        self.tree = tree
        self.root = self.tree.getroot()
        
        # list of actions, data
        self.properties = dict()
        self.variables  = dict() 
        self.actions = list()
        self.data    = list()
        self.loops   = list()
        self.conditions = list()
        
        self._processWindow()
        self._processProperties()
        self._processVariables()
        self._processItems()
        self._processData()
        
    def _filterValue(self, value):
        """!Filter value
        
        @param value
        """
        value = value.replace('&lt;', '<')
        value = value.replace('&gt;', '>')
        
        return value
        
    def _getNodeText(self, node, tag, default = ''):
        """!Get node text"""
        p = node.find(tag)
        if p is not None:
            if p.text:
                return utils.normalize_whitespace(p.text)
            else:
                return ''
        
        return default
    
    def _processWindow(self):
        """!Process window properties"""
        node = self.root.find('window')
        if node is None:
            self.pos = self.size = None
            return
        
        self.pos, self.size = self._getDim(node)
        
    def _processProperties(self):
        """!Process model properties"""
        node = self.root.find('properties')
        if node is None:
            return
        for key in ('name', 'description', 'author'):
            self._processProperty(node, key)
        
        for f in node.findall('flag'):
            name = f.get('name', '')
            if name == 'overwrite':
                self.properties['overwrite'] = True
        
    def _processProperty(self, pnode, name):
        """!Process given property"""
        node = pnode.find(name)
        if node is not None:
            self.properties[name] = node.text
        else:
            self.properties[name] = ''

    def _processVariables(self):
        """!Process model variables"""
        vnode = self.root.find('variables')
        if vnode is None:
            return
        for node in vnode.findall('variable'):
            name = node.get('name', '')
            if not name:
                continue # should not happen
            self.variables[name] = { 'type' : node.get('type', 'string') }
            for key in ('description', 'value'):
                self._processVariable(node, name, key)
        
    def _processVariable(self, pnode, name, key):
        """!Process given variable"""
        node = pnode.find(key)
        if node is not None:
            if node.text:
                self.variables[name][key] = node.text

    def _processItems(self):
        """!Process model items (actions, loops, conditions)"""
        self._processActions()
        self._processLoops()
        self._processConditions()
        
    def _processActions(self):
        """!Process model file"""
        for action in self.root.findall('action'):
            pos, size = self._getDim(action)
            disabled  = False
            
            task = action.find('task')
            if task is not None:
                if task.find('disabled') is not None:
                    disabled = True
                task = self._processTask(task)
            else:
                task = None
            
            aId = int(action.get('id', -1))
            
            self.actions.append({ 'pos'      : pos,
                                  'size'     : size,
                                  'task'     : task,
                                  'id'       : aId,
                                  'disabled' : disabled })
            
    def _getDim(self, node):
        """!Get position and size of shape"""
        pos = size = None
        posAttr = node.get('pos', None)
        if posAttr:
            posVal = map(int, posAttr.split(','))
            try:
                pos = (posVal[0], posVal[1])
            except:
                pos = None
        
        sizeAttr = node.get('size', None)
        if sizeAttr:
            sizeVal = map(int, sizeAttr.split(','))
            try:
                size = (sizeVal[0], sizeVal[1])
            except:
                size = None
        
        return pos, size        
    
    def _processData(self):
        """!Process model file"""
        for data in self.root.findall('data'):
            pos, size = self._getDim(data)
            param = data.find('data-parameter')
            prompt = value = None
            if param is not None:
                prompt = param.get('prompt', None)
                value = self._filterValue(self._getNodeText(param, 'value'))
            
            if data.find('intermediate') is None:
                intermediate = False
            else:
                intermediate = True
            
            rels = list()
            for rel in data.findall('relation'):
                defrel = { 'id'  : int(rel.get('id', -1)),
                           'dir' : rel.get('dir', 'to'),
                           'name' : rel.get('name', '') }
                points = list()
                for point in rel.findall('point'):
                    x = self._filterValue(self._getNodeText(point, 'x'))
                    y = self._filterValue(self._getNodeText(point, 'y'))
                    points.append((float(x), float(y)))
                defrel['points'] = points
                rels.append(defrel)
            
            self.data.append({ 'pos' : pos,
                               'size': size,
                               'prompt' : prompt,
                               'value' : value,
                               'intermediate' : intermediate,
                               'rels' : rels })
        
    def _processTask(self, node):
        """!Process task

        @return grassTask instance
        @return None on error
        """
        cmd = list()
        parameterized = list()
        
        name = node.get('name', None)
        if not name:
            return None
        
        cmd.append(name)
        
        # flags
        for f in node.findall('flag'):
            flag = f.get('name', '')
            if f.get('parameterized', '0') == '1':
                parameterized.append(('flag', flag))
                if f.get('value', '1') == '0':
                    continue
            if len(flag) > 1:
                cmd.append('--' + flag)
            else:
                cmd.append('-' + flag)
        # parameters
        for p in node.findall('parameter'):
            name = p.get('name', '')
            if p.find('parameterized') is not None:
                parameterized.append(('param', name))
            cmd.append('%s=%s' % (name,
                                  self._filterValue(self._getNodeText(p, 'value'))))
        
        task, err = menuform.GUI().ParseCommand(cmd = cmd,
                                                show = None, checkError = True)
        if err:
            GWarning(os.linesep.join(err))
        
        for opt, name in parameterized:
            if opt == 'flag':
                task.set_flag(name, True, element = 'parameterized')
            else:
                task.set_param(name, True, element = 'parameterized')
        
        return task

    def _processLoops(self):
        """!Process model loops"""
        for node in self.root.findall('loop'):
            pos, size = self._getDim(node)
            text = self._filterValue(self._getNodeText(node, 'condition')).strip()
            aid = list()
            for anode in node.findall('item'):
                try:
                    aid.append(int(anode.text))
                except ValueError:
                    pass
            
            self.loops.append({ 'pos'     : pos,
                                'size'    : size,
                                'text'    : text,
                                'id'      : int(node.get('id', -1)),
                                'items'   : aid })
        
    def _processConditions(self):
        """!Process model conditions"""
        for node in self.root.findall('if-else'):
            pos, size = self._getDim(node)
            text = self._filterValue(self._getNodeText(node, 'condition')).strip()
            aid = { 'if'   : list(),
                    'else' : list() }
            for b in aid.keys():
                bnode = node.find(b)
                if bnode is None:
                    continue
                for anode in bnode.findall('item'):
                    try:
                        aid[b].append(int(anode.text))
                    except ValueError:
                        pass
            
            self.conditions.append({ 'pos'     : pos,
                                     'size'    : size,
                                     'text'    : text,
                                     'id'      : int(node.get('id', -1)),
                                     'items'   : aid })
        
class WriteModelFile:
    """!Generic class for writing model file"""
    def __init__(self, fd, model):
        self.fd         = fd
        self.model      = model
        self.properties = model.GetProperties()
        self.variables  = model.GetVariables()
        self.items      = model.GetItems()
        
        self.indent = 0
        
        self._header()
        
        self._window()
        self._properties()
        self._variables()
        self._items()
        
        dataList = list()
        for action in model.GetItems(objType = ModelAction):
            for rel in action.GetRelations():
                dataItem = rel.GetData()
                if dataItem not in dataList:
                    dataList.append(dataItem)
        self._data(dataList)
        
        self._footer()

    def _filterValue(self, value):
        """!Make value XML-valid"""
        value = value.replace('<', '&lt;')
        value = value.replace('>', '&gt;')
        
        return value
        
    def _header(self):
        """!Write header"""
        self.fd.write('<?xml version="1.0" encoding="UTF-8"?>\n')
        self.fd.write('<!DOCTYPE gxm SYSTEM "grass-gxm.dtd">\n')
        self.fd.write('%s<gxm>\n' % (' ' * self.indent))
        self.indent += 4
                
    def _footer(self):
        """!Write footer"""
        self.indent -= 4
        self.fd.write('%s</gxm>\n' % (' ' * self.indent))

    def _window(self):
        """!Write window properties"""
        canvas = self.model.GetCanvas()
        if canvas is None:
            return
        win  = canvas.parent
        pos  = win.GetPosition()
        size = win.GetSize()
        self.fd.write('%s<window pos="%d,%d" size="%d,%d" />\n' % \
                          (' ' * self.indent, pos[0], pos[1], size[0], size[1]))
        
    def _properties(self):
        """!Write model properties"""
        self.fd.write('%s<properties>\n' % (' ' * self.indent))
        self.indent += 4
        if self.properties['name']:
            self.fd.write('%s<name>%s</name>\n' % (' ' * self.indent, self.properties['name']))
        if self.properties['description']:
            self.fd.write('%s<description>%s</description>\n' % (' ' * self.indent, self.properties['description']))
        if self.properties['author']:
            self.fd.write('%s<author>%s</author>\n' % (' ' * self.indent, self.properties['author']))
        
        if self.properties.has_key('overwrite') and \
                self.properties['overwrite']:
            self.fd.write('%s<flag name="overwrite" />\n' % (' ' * self.indent))
        self.indent -= 4
        self.fd.write('%s</properties>\n' % (' ' * self.indent))

    def _variables(self):
        """!Write model variables"""
        if not self.variables:
            return
        self.fd.write('%s<variables>\n' % (' ' * self.indent))
        self.indent += 4
        for name, values in self.variables.iteritems():
            self.fd.write('%s<variable name="%s" type="%s">\n' % \
                              (' ' * self.indent, name, values['type']))
            self.indent += 4
            if values.has_key('value'):
                self.fd.write('%s<value>%s</value>\n' % \
                                  (' ' * self.indent, values['value']))
            if values.has_key('description'):
                self.fd.write('%s<description>%s</description>\n' % \
                                  (' ' * self.indent, values['description']))
            self.indent -= 4
            self.fd.write('%s</variable>\n' % (' ' * self.indent))
        self.indent -= 4
        self.fd.write('%s</variables>\n' % (' ' * self.indent))
        
    def _items(self):
        """!Write actions/loops/conditions"""
        for item in self.items:
            if isinstance(item, ModelAction):
                self._action(item)
            elif isinstance(item, ModelLoop):
                self._loop(item)
            elif isinstance(item, ModelCondition):
                self._condition(item)
        
    def _action(self, action):
        """!Write actions"""
        self.fd.write('%s<action id="%d" name="%s" pos="%d,%d" size="%d,%d">\n' % \
                          (' ' * self.indent, action.GetId(), action.GetName(), action.GetX(), action.GetY(),
                           action.GetWidth(), action.GetHeight()))
        self.indent += 4
        self.fd.write('%s<task name="%s">\n' % (' ' * self.indent, action.GetLog(string = False)[0]))
        self.indent += 4
        if not action.IsEnabled():
            self.fd.write('%s<disabled />\n' % (' ' * self.indent))
        for key, val in action.GetParams().iteritems():
            if key == 'flags':
                for f in val:
                    if f.get('value', False) or f.get('parameterized', False):
                        if f.get('parameterized', False):
                            if f.get('value', False) == False:
                                self.fd.write('%s<flag name="%s" value="0" parameterized="1" />\n' %
                                              (' ' * self.indent, f.get('name', '')))
                            else:
                                self.fd.write('%s<flag name="%s" parameterized="1" />\n' %
                                              (' ' * self.indent, f.get('name', '')))
                        else:
                            self.fd.write('%s<flag name="%s" />\n' %
                                          (' ' * self.indent, f.get('name', '')))
            else: # parameter
                for p in val:
                    if not p.get('value', ''):
                        continue
                    self.fd.write('%s<parameter name="%s">\n' %
                                  (' ' * self.indent, p.get('name', '')))
                    self.indent += 4
                    if p.get('parameterized', False):
                        self.fd.write('%s<parameterized />\n' % (' ' * self.indent))
                    self.fd.write('%s<value>%s</value>\n' %
                                  (' ' * self.indent, self._filterValue(p.get('value', ''))))
                    self.indent -= 4
                    self.fd.write('%s</parameter>\n' % (' ' * self.indent))
        self.indent -= 4
        self.fd.write('%s</task>\n' % (' ' * self.indent))
        self.indent -= 4
        self.fd.write('%s</action>\n' % (' ' * self.indent))
                
    def _data(self, dataList):
        """!Write data"""
        for data in dataList:
            self.fd.write('%s<data pos="%d,%d" size="%d,%d">\n' % \
                              (' ' * self.indent, data.GetX(), data.GetY(),
                               data.GetWidth(), data.GetHeight()))
            self.indent += 4
            self.fd.write('%s<data-parameter prompt="%s">\n' % \
                              (' ' * self.indent, data.GetPrompt()))
            self.indent += 4
            self.fd.write('%s<value>%s</value>\n' %
                          (' ' * self.indent, self._filterValue(data.GetValue())))
            self.indent -= 4
            self.fd.write('%s</data-parameter>\n' % (' ' * self.indent))
            
            if data.IsIntermediate():
                self.fd.write('%s<intermediate />\n' % (' ' * self.indent))

            # relations
            for ft in ('from', 'to'):
                for rel in data.GetRelations(ft):
                    if ft == 'from':
                        aid = rel.GetTo().GetId()
                    else:
                        aid  = rel.GetFrom().GetId()
                    self.fd.write('%s<relation dir="%s" id="%d" name="%s">\n' % \
                                      (' ' * self.indent, ft, aid, rel.GetName()))
                    self.indent += 4
                    for point in rel.GetLineControlPoints()[1:-1]:
                        self.fd.write('%s<point>\n' % (' ' * self.indent))
                        self.indent += 4
                        x, y = point.Get()
                        self.fd.write('%s<x>%d</x>\n' % (' ' * self.indent, int(x)))
                        self.fd.write('%s<y>%d</y>\n' % (' ' * self.indent, int(y)))
                        self.indent -= 4
                        self.fd.write('%s</point>\n' % (' ' * self.indent))
                    self.indent -= 4
                    self.fd.write('%s</relation>\n' % (' ' * self.indent))
                
            self.indent -= 4
            self.fd.write('%s</data>\n' % (' ' * self.indent))

    def _loop(self, loop):
        """!Write loops"""
        self.fd.write('%s<loop id="%d" pos="%d,%d" size="%d,%d">\n' % \
                          (' ' * self.indent, loop.GetId(), loop.GetX(), loop.GetY(),
                           loop.GetWidth(), loop.GetHeight()))
        text = loop.GetText()
        self.indent += 4
        if text:
            self.fd.write('%s<condition>%s</condition>\n' %
                          (' ' * self.indent, self._filterValue(text)))
        for item in loop.GetItems():
            self.fd.write('%s<item>%d</item>\n' %
                          (' ' * self.indent, item.GetId()))
        self.indent -= 4
        self.fd.write('%s</loop>\n' % (' ' * self.indent))

    def _condition(self, condition):
        """!Write conditions"""
        bbox = condition.GetBoundingBoxMin()
        self.fd.write('%s<if-else id="%d" pos="%d,%d" size="%d,%d">\n' % \
                          (' ' * self.indent, condition.GetId(), condition.GetX(), condition.GetY(),
                           bbox[0], bbox[1]))
        text = condition.GetText()
        self.indent += 4
        if text:
            self.fd.write('%s<condition>%s</condition>\n' %
                          (' ' * self.indent, self._filterValue(text)))
        items = condition.GetItems()
        for b in items.keys():
            if len(items[b]) < 1:
                continue
            self.fd.write('%s<%s>\n' % (' ' * self.indent, b))
            self.indent += 4
            for item in items[b]:
                self.fd.write('%s<item>%d</item>\n' %
                              (' ' * self.indent, item.GetId()))
            self.indent -= 4
            self.fd.write('%s</%s>\n' % (' ' * self.indent, b))
        
        self.indent -= 4
        self.fd.write('%s</if-else>\n' % (' ' * self.indent))
        
class PreferencesDialog(PreferencesBaseDialog):
    """!User preferences dialog"""
    def __init__(self, parent, settings = UserSettings,
                 title = _("Modeler settings")):
        
        PreferencesBaseDialog.__init__(self, parent = parent, title = title,
                                       settings = settings)
        
        # create notebook pages
        self._createActionPage(self.notebook)
        self._createDataPage(self.notebook)
                
        self.SetMinSize(self.GetBestSize())
        self.SetSize(self.size)

    def _createActionPage(self, notebook):
        """!Create notebook page for action settings"""
        panel = wx.Panel(parent = notebook, id = wx.ID_ANY)
        notebook.AddPage(page = panel, text = _("Action"))
        
        # colors
        border = wx.BoxSizer(wx.VERTICAL)
        box   = wx.StaticBox (parent = panel, id = wx.ID_ANY,
                              label = " %s " % _("Validity"))
        sizer = wx.StaticBoxSizer(box, wx.VERTICAL)
        
        gridSizer = wx.GridBagSizer (hgap = 3, vgap = 3)
        gridSizer.AddGrowableCol(0)
        
        row = 0
        gridSizer.Add(item = wx.StaticText(parent = panel, id = wx.ID_ANY,
                                         label = _("Valid:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 0))
        vColor = csel.ColourSelect(parent = panel, id = wx.ID_ANY,
                                   colour = self.settings.Get(group='modeler', key='action', subkey=('color', 'valid')),
                                   size = globalvar.DIALOG_COLOR_SIZE)
        vColor.SetName('GetColour')
        self.winId['modeler:action:color:valid'] = vColor.GetId()
        
        gridSizer.Add(item = vColor,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))

        row += 1
        gridSizer.Add(item = wx.StaticText(parent = panel, id = wx.ID_ANY,
                                         label = _("Invalid:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 0))
        iColor = csel.ColourSelect(parent = panel, id = wx.ID_ANY,
                                   colour = self.settings.Get(group='modeler', key='action', subkey=('color', 'invalid')),
                                   size = globalvar.DIALOG_COLOR_SIZE)
        iColor.SetName('GetColour')
        self.winId['modeler:action:color:invalid'] = iColor.GetId()
        
        gridSizer.Add(item = iColor,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))

        row += 1
        gridSizer.Add(item = wx.StaticText(parent = panel, id = wx.ID_ANY,
                                         label = _("Running:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 0))
        rColor = csel.ColourSelect(parent = panel, id = wx.ID_ANY,
                                   colour = self.settings.Get(group='modeler', key='action', subkey=('color', 'running')),
                                   size = globalvar.DIALOG_COLOR_SIZE)
        rColor.SetName('GetColour')
        self.winId['modeler:action:color:running'] = rColor.GetId()
        
        gridSizer.Add(item = rColor,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))

        row += 1
        gridSizer.Add(item = wx.StaticText(parent = panel, id = wx.ID_ANY,
                                         label = _("Disabled:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 0))
        rColor = csel.ColourSelect(parent = panel, id = wx.ID_ANY,
                                   colour = self.settings.Get(group='modeler', key='action', subkey=('color', 'disabled')),
                                   size = globalvar.DIALOG_COLOR_SIZE)
        rColor.SetName('GetColour')
        self.winId['modeler:action:color:disabled'] = rColor.GetId()
        
        gridSizer.Add(item = rColor,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))

        sizer.Add(item = gridSizer, proportion = 1, flag = wx.ALL | wx.EXPAND, border = 5)
        border.Add(item = sizer, proportion = 0, flag = wx.LEFT | wx.RIGHT | wx.BOTTOM | wx.EXPAND, border = 3)
        
        # size
        box   = wx.StaticBox (parent = panel, id = wx.ID_ANY,
                              label = " %s " % _("Shape size"))
        sizer = wx.StaticBoxSizer(box, wx.VERTICAL)
        
        gridSizer = wx.GridBagSizer (hgap=3, vgap=3)
        gridSizer.AddGrowableCol(0)

        row = 0
        gridSizer.Add(item = wx.StaticText(parent = panel, id = wx.ID_ANY,
                                         label = _("Width:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 0))
        
        width = wx.SpinCtrl(parent = panel, id = wx.ID_ANY,
                            min = 0, max = 500,
                            initial = self.settings.Get(group='modeler', key='action', subkey=('size', 'width')))
        width.SetName('GetValue')
        self.winId['modeler:action:size:width'] = width.GetId()
        
        gridSizer.Add(item = width,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))

        row += 1
        gridSizer.Add(item = wx.StaticText(parent=panel, id=wx.ID_ANY,
                                         label=_("Height:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos=(row, 0))
        
        height = wx.SpinCtrl(parent = panel, id = wx.ID_ANY,
                             min = 0, max = 500,
                             initial = self.settings.Get(group='modeler', key='action', subkey=('size', 'height')))
        height.SetName('GetValue')
        self.winId['modeler:action:size:height'] = height.GetId()
        
        gridSizer.Add(item = height,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))
        
        sizer.Add(item=gridSizer, proportion=1, flag=wx.ALL | wx.EXPAND, border=5)
        border.Add(item=sizer, proportion=0, flag=wx.LEFT | wx.RIGHT | wx.BOTTOM | wx.EXPAND, border=3)
                
        panel.SetSizer(border)
        
        return panel

    def _createDataPage(self, notebook):
        """!Create notebook page for data settings"""
        panel = wx.Panel(parent = notebook, id = wx.ID_ANY)
        notebook.AddPage(page = panel, text = _("Data"))
        
        # colors
        border = wx.BoxSizer(wx.VERTICAL)
        box   = wx.StaticBox (parent = panel, id = wx.ID_ANY,
                              label = " %s " % _("Type"))
        sizer = wx.StaticBoxSizer(box, wx.VERTICAL)
        
        gridSizer = wx.GridBagSizer (hgap = 3, vgap = 3)
        gridSizer.AddGrowableCol(0)
        
        row = 0
        gridSizer.Add(item = wx.StaticText(parent = panel, id = wx.ID_ANY,
                                         label = _("Raster:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 0))
        rColor = csel.ColourSelect(parent = panel, id = wx.ID_ANY,
                                   colour = self.settings.Get(group='modeler', key='data', subkey=('color', 'raster')),
                                   size = globalvar.DIALOG_COLOR_SIZE)
        rColor.SetName('GetColour')
        self.winId['modeler:data:color:raster'] = rColor.GetId()
        
        gridSizer.Add(item = rColor,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))

        row += 1
        gridSizer.Add(item = wx.StaticText(parent = panel, id = wx.ID_ANY,
                                         label = _("3D raster:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 0))
        r3Color = csel.ColourSelect(parent = panel, id = wx.ID_ANY,
                                    colour = self.settings.Get(group='modeler', key='data', subkey=('color', 'raster3d')),
                                    size = globalvar.DIALOG_COLOR_SIZE)
        r3Color.SetName('GetColour')
        self.winId['modeler:data:color:raster3d'] = r3Color.GetId()
        
        gridSizer.Add(item = r3Color,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))
        
        row += 1
        gridSizer.Add(item = wx.StaticText(parent = panel, id = wx.ID_ANY,
                                         label = _("Vector:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 0))
        vColor = csel.ColourSelect(parent = panel, id = wx.ID_ANY,
                                   colour = self.settings.Get(group='modeler', key='data', subkey=('color', 'vector')),
                                   size = globalvar.DIALOG_COLOR_SIZE)
        vColor.SetName('GetColour')
        self.winId['modeler:data:color:vector'] = vColor.GetId()
        
        gridSizer.Add(item = vColor,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))
        
        sizer.Add(item = gridSizer, proportion = 1, flag = wx.ALL | wx.EXPAND, border = 5)
        border.Add(item = sizer, proportion = 0, flag = wx.LEFT | wx.RIGHT | wx.BOTTOM | wx.EXPAND, border = 3)

        # size
        box   = wx.StaticBox (parent = panel, id = wx.ID_ANY,
                              label = " %s " % _("Shape size"))
        sizer = wx.StaticBoxSizer(box, wx.VERTICAL)
        
        gridSizer = wx.GridBagSizer (hgap=3, vgap=3)
        gridSizer.AddGrowableCol(0)
        
        row = 0
        gridSizer.Add(item = wx.StaticText(parent = panel, id = wx.ID_ANY,
                                         label = _("Width:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 0))
        
        width = wx.SpinCtrl(parent = panel, id = wx.ID_ANY,
                            min = 0, max = 500,
                            initial = self.settings.Get(group='modeler', key='data', subkey=('size', 'width')))
        width.SetName('GetValue')
        self.winId['modeler:data:size:width'] = width.GetId()
        
        gridSizer.Add(item = width,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))

        row += 1
        gridSizer.Add(item = wx.StaticText(parent=panel, id=wx.ID_ANY,
                                         label=_("Height:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos=(row, 0))
        
        height = wx.SpinCtrl(parent = panel, id = wx.ID_ANY,
                             min = 0, max = 500,
                             initial = self.settings.Get(group='modeler', key='data', subkey=('size', 'height')))
        height.SetName('GetValue')
        self.winId['modeler:data:size:height'] = height.GetId()
        
        gridSizer.Add(item = height,
                      flag = wx.ALIGN_RIGHT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (row, 1))
        
        sizer.Add(item=gridSizer, proportion=1, flag=wx.ALL | wx.EXPAND, border=5)
        border.Add(item=sizer, proportion=0, flag=wx.LEFT | wx.RIGHT | wx.BOTTOM | wx.EXPAND, border=3)
        
        panel.SetSizer(border)
        
        return panel

    def OnApply(self, event):
        """!Button 'Apply' pressed"""
        PreferencesBaseDialog.OnApply(self, event)
        
        self.parent.GetModel().Update()
        self.parent.GetCanvas().Refresh()

    def OnSave(self, event):
        """!Button 'Save' pressed"""
        PreferencesBaseDialog.OnSave(self, event)
        
        self.parent.GetModel().Update()
        self.parent.GetCanvas().Refresh()

class PropertiesDialog(wx.Dialog):
    """!Model properties dialog
    """
    def __init__(self, parent, id = wx.ID_ANY,
                 title = _('Model properties'),
                 size = (350, 400),
                 style = wx.DEFAULT_DIALOG_STYLE | wx.RESIZE_BORDER):
        wx.Dialog.__init__(self, parent, id, title, size = size,
                           style = style)
        
        self.metaBox = wx.StaticBox(parent = self, id = wx.ID_ANY,
                                    label=" %s " % _("Metadata"))
        self.cmdBox = wx.StaticBox(parent = self, id = wx.ID_ANY,
                                   label=" %s " % _("Commands"))
        
        self.name = wx.TextCtrl(parent = self, id = wx.ID_ANY,
                                size = (300, 25))
        self.desc = wx.TextCtrl(parent = self, id = wx.ID_ANY,
                                style = wx.TE_MULTILINE,
                                size = (300, 50))
        self.author = wx.TextCtrl(parent = self, id = wx.ID_ANY,
                                size = (300, 25))
        
        # commands
        self.overwrite = wx.CheckBox(parent = self, id=wx.ID_ANY,
                                     label=_("Allow output files to overwrite existing files"))
        self.overwrite.SetValue(UserSettings.Get(group='cmd', key='overwrite', subkey='enabled'))
        
        # buttons
        self.btnOk     = wx.Button(self, wx.ID_OK)
        self.btnCancel = wx.Button(self, wx.ID_CANCEL)
        self.btnOk.SetDefault()
        
        self.btnOk.SetToolTipString(_("Apply properties"))
        self.btnOk.SetDefault()
        self.btnCancel.SetToolTipString(_("Close dialog and ignore changes"))
        
        self.Bind(wx.EVT_CLOSE, self.OnCloseWindow)
        
        self._layout()

    def _layout(self):
        metaSizer = wx.StaticBoxSizer(self.metaBox, wx.VERTICAL)
        gridSizer = wx.GridBagSizer(hgap = 3, vgap = 3)
        gridSizer.AddGrowableCol(0)
        gridSizer.AddGrowableRow(1)
        gridSizer.Add(item = wx.StaticText(parent = self, id = wx.ID_ANY,
                                         label = _("Name:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (0, 0))
        gridSizer.Add(item = self.name,
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL | wx.EXPAND,
                      pos = (0, 1))
        gridSizer.Add(item = wx.StaticText(parent = self, id = wx.ID_ANY,
                                         label = _("Description:")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (1, 0))
        gridSizer.Add(item = self.desc,
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL | wx.EXPAND,
                      pos = (1, 1))
        gridSizer.Add(item = wx.StaticText(parent = self, id = wx.ID_ANY,
                                         label = _("Author(s):")),
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL,
                      pos = (2, 0))
        gridSizer.Add(item = self.author,
                      flag = wx.ALIGN_LEFT |
                      wx.ALIGN_CENTER_VERTICAL | wx.EXPAND,
                      pos = (2, 1))
        metaSizer.Add(item = gridSizer)
        
        cmdSizer = wx.StaticBoxSizer(self.cmdBox, wx.VERTICAL)
        cmdSizer.Add(item = self.overwrite,
                     flag = wx.EXPAND | wx.ALL, border = 3)
        
        btnStdSizer = wx.StdDialogButtonSizer()
        btnStdSizer.AddButton(self.btnCancel)
        btnStdSizer.AddButton(self.btnOk)
        btnStdSizer.Realize()
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(item=metaSizer, proportion=1,
                      flag=wx.EXPAND | wx.ALL, border=5)
        mainSizer.Add(item=cmdSizer, proportion=0,
                      flag=wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, border=5)
        mainSizer.Add(item=btnStdSizer, proportion=0,
                      flag=wx.EXPAND | wx.ALL | wx.ALIGN_RIGHT, border=5)
        
        self.SetSizer(mainSizer)
        mainSizer.Fit(self)

    def OnCloseWindow(self, event):
        self.Hide()
        
    def GetValues(self):
        """!Get values"""
        return { 'name'        : self.name.GetValue(),
                 'description' : self.desc.GetValue(),
                 'author'      : self.author.GetValue(),
                 'overwrite'   : self.overwrite.IsChecked() }
    
    def Init(self, prop):
        """!Initialize dialog"""
        self.name.SetValue(prop['name'])
        self.desc.SetValue(prop['description'])
        self.author.SetValue(prop['author'])
        if prop.has_key('overwrite'):
            self.overwrite.SetValue(prop['overwrite'])

class ModelParamDialog(wx.Dialog):
    def __init__(self, parent, params, id = wx.ID_ANY, title = _("Model parameters"),
                 style = wx.DEFAULT_DIALOG_STYLE | wx.RESIZE_BORDER, **kwargs):
        """!Model parameters dialog
        """
        self.parent = parent
        self.params = params
        self.tasks  = list() # list of tasks/pages
        
        wx.Dialog.__init__(self, parent = parent, id = id, title = title, style = style, **kwargs)
        
        if globalvar.hasAgw:
            self.notebook = FN.FlatNotebook(self, id = wx.ID_ANY,
                                            agwStyle = FN.FNB_FANCY_TABS |
                                            FN.FNB_BOTTOM |
                                            FN.FNB_NO_NAV_BUTTONS |
                                            FN.FNB_NO_X_BUTTON)
        else:
            self.notebook = FN.FlatNotebook(self, id = wx.ID_ANY,
                                            style = FN.FNB_FANCY_TABS |
                                            FN.FNB_BOTTOM |
                                            FN.FNB_NO_NAV_BUTTONS |
                                            FN.FNB_NO_X_BUTTON)
        
        panel = self._createPages()
        wx.CallAfter(self.notebook.SetSelection, 0)
        
        self.btnCancel = wx.Button(parent = self, id = wx.ID_CANCEL)
        self.btnRun    = wx.Button(parent = self, id = wx.ID_OK,
                                   label = _("&Run"))
        self.btnRun.SetDefault()
        
        self._layout()
        
        size = self.GetBestSize()
        self.SetMinSize(size)
        self.SetSize((size.width, size.height +
                      panel.constrained_size[1] -
                      panel.panelMinHeight))
                
    def _layout(self):
        btnSizer = wx.StdDialogButtonSizer()
        btnSizer.AddButton(self.btnCancel)
        btnSizer.AddButton(self.btnRun)
        btnSizer.Realize()
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(item = self.notebook, proportion = 1,
                      flag = wx.EXPAND)
        mainSizer.Add(item=btnSizer, proportion=0,
                      flag=wx.EXPAND | wx.ALL | wx.ALIGN_CENTER, border=5)
        
        self.SetSizer(mainSizer)
        mainSizer.Fit(self)
        
    def _createPages(self):
        """!Create for each parameterized module its own page"""
        nameOrdered = [''] * len(self.params.keys())
        for name, params in self.params.iteritems():
            nameOrdered[params['idx']] =  name
        for name in nameOrdered:
            params = self.params[name]
            panel = self._createPage(name, params)
            self.notebook.AddPage(panel, text = name)
        
        return panel
    
    def _createPage(self, name, params):
        """!Define notebook page"""
        if name in globalvar.grassCmd['all']:
            task = menuform.grassTask(name)
        else:
            task = menuform.grassTask()
        task.flags  = params['flags']
        task.params = params['params']
        
        panel = menuform.cmdPanel(parent = self, id = wx.ID_ANY, task = task)
        self.tasks.append(task)
        
        return panel

    def GetErrors(self):
        """!Check for errors, get list of messages"""
        errList = list()
        for task in self.tasks:
            errList += task.getCmdError()
        
        return errList
    
class ModelListCtrl(wx.ListCtrl,
                    listmix.ListCtrlAutoWidthMixin,
                    listmix.TextEditMixin,
                    listmix.ColumnSorterMixin):
    def __init__(self, parent, columns, id = wx.ID_ANY,
                 style = wx.LC_REPORT | wx.BORDER_NONE |
                 wx.LC_SORT_ASCENDING |wx.LC_HRULES |
                 wx.LC_VRULES, **kwargs):
        """!List of model variables"""
        self.parent = parent
        self.columns = columns
        self.shape = None
        try:
            self.frame  = parent.parent
        except AttributeError:
            self.frame = None
        
        wx.ListCtrl.__init__(self, parent, id = id, style = style, **kwargs)
        listmix.ListCtrlAutoWidthMixin.__init__(self)
        listmix.TextEditMixin.__init__(self)
        listmix.ColumnSorterMixin.__init__(self, 4)
        
        i = 0
        for col in columns:
            self.InsertColumn(i, col)
            self.SetColumnWidth(i, wx.LIST_AUTOSIZE_USEHEADER)
            i += 1
        
        self.itemDataMap = {} # requested by sorter
        self.itemCount   = 0
        
        self.Bind(wx.EVT_LIST_BEGIN_LABEL_EDIT, self.OnBeginEdit)
        self.Bind(wx.EVT_LIST_END_LABEL_EDIT, self.OnEndEdit)
        self.Bind(wx.EVT_LIST_COL_CLICK, self.OnColClick)
        self.Bind(wx.EVT_COMMAND_RIGHT_CLICK, self.OnRightUp) #wxMSW
        self.Bind(wx.EVT_RIGHT_UP, self.OnRightUp)            #wxGTK
                
    def OnBeginEdit(self, event):
        """!Editing of item started"""
        event.Allow()

    def OnEndEdit(self, event):
        """!Finish editing of item"""
        pass
    
    def OnColClick(self, event):
        """!Click on column header (order by)"""
        event.Skip()

class VariablePanel(wx.Panel):
    def __init__(self, parent, id = wx.ID_ANY,
                 **kwargs):
        """!Manage model variables panel
        """
        self.parent = parent
        
        wx.Panel.__init__(self, parent = parent, id = id, **kwargs)
        
        self.listBox = wx.StaticBox(parent = self, id = wx.ID_ANY,
                                    label=" %s " % _("List of variables - right-click to delete"))
        
        self.list = VariableListCtrl(parent = self,
                                     columns = [_("Name"), _("Data type"),
                                                _("Default value"), _("Description")])
        
        # add new category
        self.addBox = wx.StaticBox(parent = self, id = wx.ID_ANY,
                                   label = " %s " % _("Add new variable"))
        self.name = wx.TextCtrl(parent = self, id = wx.ID_ANY)
        wx.CallAfter(self.name.SetFocus)
        self.type = wx.Choice(parent = self, id = wx.ID_ANY,
                              choices = [_("integer"),
                                         _("float"),
                                         _("string"),
                                         _("raster"),
                                         _("vector")])
        self.value = wx.TextCtrl(parent = self, id = wx.ID_ANY)
        self.desc = wx.TextCtrl(parent = self, id = wx.ID_ANY)
        
        # buttons
        self.btnAdd = wx.Button(parent = self, id = wx.ID_ADD)
        self.btnAdd.SetToolTipString(_("Add new variable to the model"))
        self.btnAdd.Enable(False)
        
        # bindings
        self.name.Bind(wx.EVT_TEXT, self.OnText)
        self.value.Bind(wx.EVT_TEXT, self.OnText)
        self.desc.Bind(wx.EVT_TEXT, self.OnText)
        self.btnAdd.Bind(wx.EVT_BUTTON, self.OnAdd)
        
        self._layout()

    def _layout(self):
        """!Layout dialog"""
        listSizer = wx.StaticBoxSizer(self.listBox, wx.VERTICAL)
        listSizer.Add(item = self.list, proportion = 1,
                      flag = wx.EXPAND)
        
        addSizer = wx.StaticBoxSizer(self.addBox, wx.VERTICAL)
        gridSizer = wx.GridBagSizer(hgap = 5, vgap = 5)
        gridSizer.AddGrowableCol(1)
        gridSizer.Add(item = wx.StaticText(parent = self, id = wx.ID_ANY,
                                           label = "%s:" % _("Name")),
                      flag = wx.ALIGN_CENTER_VERTICAL,
                      pos = (0, 0))
        gridSizer.Add(item = self.name,
                      pos = (0, 1),
                      flag = wx.EXPAND)
        gridSizer.Add(item = wx.StaticText(parent = self, id = wx.ID_ANY,
                                           label = "%s:" % _("Data type")),
                      flag = wx.ALIGN_CENTER_VERTICAL,
                      pos = (0, 2))
        gridSizer.Add(item = self.type,
                      pos = (0, 3))
        gridSizer.Add(item = wx.StaticText(parent = self, id = wx.ID_ANY,
                                           label = "%s:" % _("Default value")),
                      flag = wx.ALIGN_CENTER_VERTICAL,
                      pos = (1, 0))
        gridSizer.Add(item = self.value,
                      pos = (1, 1), span = (1, 3),
                      flag = wx.EXPAND)
        gridSizer.Add(item = wx.StaticText(parent = self, id = wx.ID_ANY,
                                           label = "%s:" % _("Description")),
                      flag = wx.ALIGN_CENTER_VERTICAL,
                      pos = (2, 0))
        gridSizer.Add(item = self.desc,
                      pos = (2, 1), span = (1, 3),
                      flag = wx.EXPAND)
        addSizer.Add(item = gridSizer,
                     flag = wx.EXPAND)
        addSizer.Add(item = self.btnAdd, proportion = 0,
                     flag = wx.TOP | wx.ALIGN_RIGHT, border = 5)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(item = listSizer, proportion = 1,
                      flag = wx.EXPAND | wx.ALL | wx.ALIGN_CENTER, border = 5)
        mainSizer.Add(item = addSizer, proportion = 0,
                      flag = wx.EXPAND | wx.ALIGN_CENTER |
                      wx.LEFT | wx.RIGHT | wx.BOTTOM, border = 5)
        
        self.SetSizer(mainSizer)
        mainSizer.Fit(self)
        
    def OnText(self, event):
        """!Text entered"""
        if self.name.GetValue():
            self.btnAdd.Enable()
        else:
            self.btnAdd.Enable(False)
    
    def OnAdd(self, event):
        """!Add new variable to the list"""
        msg = self.list.Append(self.name.GetValue(),
                               self.type.GetStringSelection(),
                               self.value.GetValue(),
                               self.desc.GetValue())
        self.name.SetValue('')
        self.name.SetFocus()
        
        if msg:
            GError(parent = self,
                   message = msg)
        else:
            self.type.SetSelection(0)
            self.value.SetValue('')
            self.desc.SetValue('')
            self.UpdateModelVariables()
        
    def UpdateModelVariables(self):
        """!Update model variables"""
        variables = dict()
        for values in self.list.GetData().itervalues():
            name = values[0]
            variables[name] = { 'type' : str(values[1]) }
            if values[2]:
                variables[name]['value'] = values[2]
            if values[3]:
                variables[name]['description'] = values[3]
        
        self.parent.GetModel().SetVariables(variables)
        self.parent.ModelChanged()

    def Update(self):
        """!Reload list of variables"""
        self.list.OnReload(None)
        
class VariableListCtrl(ModelListCtrl):
    def __init__(self, parent, columns, **kwargs):
        """!List of model variables"""
        ModelListCtrl.__init__(self, parent, columns, **kwargs)
        
    def GetListCtrl(self):
        """!Used by ColumnSorterMixin"""
        return self
    
    def GetData(self):
        """!Get list data"""
        return self.itemDataMap
    
    def Populate(self, data):
        """!Populate the list"""
        self.itemDataMap = dict()
        i = 0
        for name, values in data.iteritems():
            self.itemDataMap[i] = [name, values['type'],
                                   values.get('value', ''),
                                   values.get('description', '')]
            i += 1
        
        self.itemCount = len(self.itemDataMap.keys())
        self.DeleteAllItems()
        i = 0
        for name, vtype, value, desc in self.itemDataMap.itervalues():
            index = self.InsertStringItem(sys.maxint, name)
            self.SetStringItem(index, 0, name)
            self.SetStringItem(index, 1, vtype)
            self.SetStringItem(index, 2, value)
            self.SetStringItem(index, 3, desc)
            self.SetItemData(index, i)
            i += 1
        
    def Append(self, name, vtype, value, desc):
        """!Append new item to the list

        @return None on success
        @return error string
        """
        for iname, ivtype, ivalue, idesc in self.itemDataMap.itervalues():
            if iname == name:
                return _("Variable <%s> already exists in the model. "
                         "Adding variable failed.") % name
        
        index = self.InsertStringItem(sys.maxint, name)
        self.SetStringItem(index, 0, name)
        self.SetStringItem(index, 1, vtype)
        self.SetStringItem(index, 2, value)
        self.SetStringItem(index, 3, desc)
        self.SetItemData(index, self.itemCount)
        
        self.itemDataMap[self.itemCount] = [name, vtype, value, desc]
        self.itemCount += 1
        
        return None

    def OnRemove(self, event):
        """!Remove selected variable(s) from the model"""
        item = self.GetFirstSelected()
        while item != -1:
            self.DeleteItem(item)
            del self.itemDataMap[item]
            item = self.GetFirstSelected()
        self.parent.UpdateModelVariables()
        
        event.Skip()
        
    def OnRemoveAll(self, event):
        """!Remove all variable(s) from the model"""
        dlg = wx.MessageBox(parent=self,
                            message=_("Do you want to delete all variables from "
                                      "the model?"),
                            caption=_("Delete variables"),
                            style=wx.YES_NO | wx.CENTRE)
        if dlg != wx.YES:
            return
        
        self.DeleteAllItems()
        self.itemDataMap = dict()
        
        self.parent.UpdateModelVariables()
        
    def OnEndEdit(self, event):
        """!Finish editing of item"""
        itemIndex = event.GetIndex()
        columnIndex = event.GetColumn()
        nameOld = self.GetItem(itemIndex, 0).GetText()

        if columnIndex == 0: # TODO
            event.Veto()
        
        self.itemDataMap[itemIndex][columnIndex] = event.GetText()
        
        self.parent.UpdateModelVariables()

    def OnReload(self, event):
        """!Reload list of variables"""
        self.Populate(self.parent.parent.GetModel().GetVariables())

    def OnRightUp(self, event):
        """!Mouse right button up"""
        if not hasattr(self, "popupID1"):
            self.popupID1 = wx.NewId()
            self.popupID2 = wx.NewId()
            self.popupID3 = wx.NewId()
            self.Bind(wx.EVT_MENU, self.OnRemove,    id = self.popupID1)
            self.Bind(wx.EVT_MENU, self.OnRemoveAll, id = self.popupID2)
            self.Bind(wx.EVT_MENU, self.OnReload,    id = self.popupID3)
        
        # generate popup-menu
        menu = wx.Menu()
        menu.Append(self.popupID1, _("Delete selected"))
        menu.Append(self.popupID2, _("Delete all"))
        if self.GetFirstSelected() == -1:
            menu.Enable(self.popupID1, False)
            menu.Enable(self.popupID2, False)
        
        menu.AppendSeparator()
        menu.Append(self.popupID3, _("Reload"))
        
        self.PopupMenu(menu)
        menu.Destroy()
        
class ModelItem(ModelObject):
    def __init__(self, parent, x, y, id = -1, width = None, height = None, text = '', items = []):
        """!Abstract class for loops and conditions"""
        ModelObject.__init__(self, id)
        self.parent  = parent
        self.text    = text
        self.items   = items  # list of items in the loop
        
    def GetText(self):
        """!Get loop text"""
        return self.text

    def GetItems(self):
        """!Get items (id)"""
        return self.items

    def SetId(self, id):
        """!Set loop id"""
        self.id = id

    def SetText(self, cond):
        """!Set loop text (condition)"""
        self.text = cond
        self.ClearText()
        self.AddText('(' + str(self.id) + ') ' + self.text)

    def GetLog(self):
        """!Get log info"""
        if self.text:
            return _("Condition: ") + self.text
        else:
            return _("Condition: not defined")

    def AddRelation(self, rel):
        """!Record relation"""
        self.rels.append(rel)
        
    def Clear(self):
        """!Clear object, remove rels"""
        self.rels = list()
        
class ModelItemDialog(wx.Dialog):
    """!Abstract item properties dialog"""
    def __init__(self, parent, shape, title, id = wx.ID_ANY,
                 style = wx.DEFAULT_DIALOG_STYLE | wx.RESIZE_BORDER, **kwargs):
        self.parent = parent
        self.shape = shape
        
        wx.Dialog.__init__(self, parent, id, title = title, style = style, **kwargs)
        
        self.panel = wx.Panel(parent = self, id = wx.ID_ANY)
        
        self.condBox = wx.StaticBox(parent = self.panel, id = wx.ID_ANY,
                                    label=" %s " % _("Condition"))
        self.condText = wx.TextCtrl(parent = self.panel, id = wx.ID_ANY,
                                    value = shape.GetText())
        
        self.itemList = ItemCheckListCtrl(parent = self.panel,
                                          window = self,
                                          columns = [_("ID"), _("Name"),
                                                     _("Command")],
                                          shape = shape)
        self.itemList.Populate(self.parent.GetModel().GetItems())
        
        self.btnCancel = wx.Button(parent = self.panel, id = wx.ID_CANCEL)
        self.btnOk     = wx.Button(parent = self.panel, id = wx.ID_OK)
        self.btnOk.SetDefault()
        
    def _layout(self):
        """!Do layout (virtual method)"""
        pass
    
    def GetCondition(self):
        """!Get loop condition"""
        return self.condText.GetValue()
    
class ModelLoop(ModelItem, ogl.RectangleShape):
    def __init__(self, parent, x, y, id = -1, width = None, height = None, text = '', items = []):
        """!Defines a loop"""
        ModelItem.__init__(self, parent, x, y, id, width, height, text, items)
        
        if not width:
            width = UserSettings.Get(group='modeler', key='loop', subkey=('size', 'width'))
        if not height:
            height = UserSettings.Get(group='modeler', key='loop', subkey=('size', 'height'))
        
        if self.parent.GetCanvas():
            ogl.RectangleShape.__init__(self, width, height)
            
            self.SetCanvas(self.parent)
            self.SetX(x)
            self.SetY(y)
            self.SetPen(wx.BLACK_PEN)
            self.SetCornerRadius(100)
            if text:
                self.AddText('(' + str(self.id) + ') ' + text)
            else:
                self.AddText('(' + str(self.id) + ')')
        
    def GetName(self):
        """!Get name"""
        return _("loop")
    
    def SetItems(self, items):
        """!Set items (id)"""
        self.items = items

class ModelLoopDialog(ModelItemDialog):
    """!Loop properties dialog"""
    def __init__(self, parent, shape, id = wx.ID_ANY, title = _("Loop properties"),
                 style = wx.DEFAULT_DIALOG_STYLE | wx.RESIZE_BORDER, **kwargs):
        ModelItemDialog.__init__(self, parent, shape, title,
                                 style = style, **kwargs)
        
        self.listBox = wx.StaticBox(parent = self.panel, id = wx.ID_ANY,
                                    label=" %s " % _("List of items in loop"))
        
        self._layout()
        self.SetMinSize(self.GetSize())
        self.SetSize((500, 400))
        
    def _layout(self):
        """!Do layout"""
        sizer = wx.BoxSizer(wx.VERTICAL)
        
        condSizer = wx.StaticBoxSizer(self.condBox, wx.VERTICAL)
        condSizer.Add(item = self.condText, proportion = 1,
                      flag = wx.EXPAND)
        
        listSizer = wx.StaticBoxSizer(self.listBox, wx.VERTICAL)
        listSizer.Add(item = self.itemList, proportion = 1,
                      flag = wx.EXPAND)
        
        btnSizer = wx.StdDialogButtonSizer()
        btnSizer.AddButton(self.btnCancel)
        btnSizer.AddButton(self.btnOk)
        btnSizer.Realize()

        sizer.Add(item = condSizer, proportion = 0,
                  flag = wx.EXPAND | wx.ALL, border = 3)
        sizer.Add(item = listSizer, proportion = 1,
                  flag = wx.EXPAND | wx.LEFT | wx.RIGHT, border = 3)
        sizer.Add(item = btnSizer, proportion=0,
                  flag = wx.EXPAND | wx.ALL | wx.ALIGN_CENTER, border=5)
        
        self.panel.SetSizer(sizer)
        sizer.Fit(self.panel)
        
        self.Layout()
        
    def GetItems(self):
        """!Get list of selected actions"""
        return self.itemList.GetItems()

class ItemPanel(wx.Panel):
    def __init__(self, parent, id = wx.ID_ANY,
                 **kwargs):
        """!Manage model items
        """
        self.parent = parent
        
        wx.Panel.__init__(self, parent = parent, id = id, **kwargs)
        
        self.listBox = wx.StaticBox(parent = self, id = wx.ID_ANY,
                                    label=" %s " % _("List of items - right-click to delete"))
        
        self.list = ItemListCtrl(parent = self,
                                 columns = [_("ID"), _("Name"), _("In block"),
                                            _("Command / Condition")])
        
        self._layout()

    def _layout(self):
        """!Layout dialog"""
        listSizer = wx.StaticBoxSizer(self.listBox, wx.VERTICAL)
        listSizer.Add(item = self.list, proportion = 1,
                      flag = wx.EXPAND)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(item = listSizer, proportion = 1,
                      flag = wx.EXPAND | wx.ALL | wx.ALIGN_CENTER, border = 5)
        
        self.SetSizer(mainSizer)
        mainSizer.Fit(self)
        
    def Update(self):
        """!Reload list of variables"""
        self.list.OnReload(None)
        
class ItemListCtrl(ModelListCtrl):
    def __init__(self, parent, columns, disablePopup = False, **kwargs):
        """!List of model actions"""
        self.disablePopup = disablePopup
                
        ModelListCtrl.__init__(self, parent, columns, **kwargs)
        self.SetColumnWidth(1, 100)
        self.SetColumnWidth(2, 65)
        
    def GetListCtrl(self):
        """!Used by ColumnSorterMixin"""
        return self
    
    def GetData(self):
        """!Get list data"""
        return self.itemDataMap
    
    def Populate(self, data):
        """!Populate the list"""
        self.itemDataMap = dict()
        if self.shape:
            if isinstance(self.shape, ModelCondition):
                if self.GetName() == 'ElseBlockList':
                    shapeItems = map(lambda x: x.GetId(), self.shape.GetItems()['else'])
                else:
                    shapeItems = map(lambda x: x.GetId(), self.shape.GetItems()['if'])
            else:
                shapeItems = map(lambda x: x.GetId(), self.shape.GetItems())
        else:
            shapeItems = list()
        i = 0
        if len(self.columns) == 3: # ItemCheckList
            checked = list()
        for action in data:
            if len(self.columns) == 3:
                self.itemDataMap[i] = [str(action.GetId()),
                                       action.GetName(),
                                       action.GetLog()]
                aId = action.GetBlockId()
                if action.GetId() in shapeItems:
                    checked.append(aId)
                else:
                    checked.append(None)
            else:
                bId = action.GetBlockId()
                if not bId:
                    bId = ''
                self.itemDataMap[i] = [str(action.GetId()),
                                       action.GetName(),
                                       ','.join(map(str, bId)),
                                       action.GetLog()]
            
            i += 1
        
        self.itemCount = len(self.itemDataMap.keys())
        self.DeleteAllItems()
        i = 0
        if len(self.columns) == 3:
            for aid, name, desc in self.itemDataMap.itervalues():
                index = self.InsertStringItem(sys.maxint, aid)
                self.SetStringItem(index, 0, aid)
                self.SetStringItem(index, 1, name)
                self.SetStringItem(index, 2, desc)
                self.SetItemData(index, i)
                if checked[i]:
                    self.CheckItem(index, True)
                i += 1
        else:
            for aid, name, inloop, desc in self.itemDataMap.itervalues():
                index = self.InsertStringItem(sys.maxint, aid)
                self.SetStringItem(index, 0, aid)
                self.SetStringItem(index, 1, name)
                self.SetStringItem(index, 2, inloop)
                self.SetStringItem(index, 3, desc)
                self.SetItemData(index, i)
                i += 1
                
    def OnRemove(self, event):
        """!Remove selected action(s) from the model"""
        model = self.frame.GetModel()
        canvas = self.frame.GetCanvas()
        
        item = self.GetFirstSelected()
        while item != -1:
            self.DeleteItem(item)
            del self.itemDataMap[item]
            
            aId = self.GetItem(item, 0).GetText()
            action = model.GetItem(int(aId))
            if not action:
                item = self.GetFirstSelected()
                continue
            
            model.RemoveItem(action)
            canvas.GetDiagram().RemoveShape(action)
            self.frame.ModelChanged()
            
            item = self.GetFirstSelected()
        
        canvas.Refresh()
        
        event.Skip()
    
    def OnRemoveAll(self, event):
        """!Remove all variable(s) from the model"""
        deleteDialog = wx.MessageBox(parent=self,
                                     message=_("Selected data records (%d) will permanently deleted "
                                               "from table. Do you want to delete them?") % \
                                         (len(self.listOfSQLStatements)),
                                     caption=_("Delete records"),
                                     style=wx.YES_NO | wx.CENTRE)
        if deleteDialog != wx.YES:
            return False
        
        self.DeleteAllItems()
        self.itemDataMap = dict()

        self.parent.UpdateModelVariables()

    def OnEndEdit(self, event):
        """!Finish editing of item"""
        itemIndex = event.GetIndex()
        columnIndex = event.GetColumn()
        
        self.itemDataMap[itemIndex][columnIndex] = event.GetText()
        
        aId = int(self.GetItem(itemIndex, 0).GetText())
        action = self.frame.GetModel().GetItem(aId)
        if not action:
            event.Veto()
        if columnIndex == 0:
            action.SetId(int(event.GetText()))
        
        self.frame.ModelChanged()

    def OnReload(self, event = None):
        """!Reload list of actions"""
        self.Populate(self.frame.GetModel().GetItems())

    def OnRightUp(self, event):
        """!Mouse right button up"""
        if self.disablePopup:
            return
        
        if not hasattr(self, "popupID1"):
            self.popupID1 = wx.NewId()
            self.popupID2 = wx.NewId()
            self.popupID3 = wx.NewId()
            self.popupID4 = wx.NewId()
            self.Bind(wx.EVT_MENU, self.OnRemove,    id = self.popupID1)
            self.Bind(wx.EVT_MENU, self.OnRemoveAll, id = self.popupID2)
            self.Bind(wx.EVT_MENU, self.OnReload,    id = self.popupID3)
            self.Bind(wx.EVT_MENU, self.OnNormalize, id = self.popupID4)

        # generate popup-menu
        menu = wx.Menu()
        menu.Append(self.popupID1, _("Delete selected"))
        menu.Append(self.popupID2, _("Delete all"))
        if self.GetFirstSelected() == -1:
            menu.Enable(self.popupID1, False)
            menu.Enable(self.popupID2, False)
        
        menu.AppendSeparator()
        menu.Append(self.popupID4, _("Normalize"))
        menu.Append(self.popupID3, _("Reload"))
        
        self.PopupMenu(menu)
        menu.Destroy()
    
    def OnNormalize(self, event):
        """!Update id of actions"""
        model = self.frame.GetModel()
        
        aId = 1
        for item in model.GetItems():
            item.SetId(aId)
            aId += 1
        
        self.OnReload(None)
        self.frame.GetCanvas().Refresh()
        self.frame.ModelChanged()

class ItemCheckListCtrl(ItemListCtrl, listmix.CheckListCtrlMixin):
    def __init__(self, parent, shape, columns, window = None, **kwargs):
        self.parent = parent
        self.window = window
        
        ItemListCtrl.__init__(self, parent, columns, disablePopup = True, **kwargs)
        listmix.CheckListCtrlMixin.__init__(self)
        self.SetColumnWidth(0, 50)
        
        self.shape  = shape
        
    def OnBeginEdit(self, event):
        """!Disable editing"""
        event.Veto()
        
    def OnCheckItem(self, index, flag):
        """!Item checked/unchecked"""
        name = self.GetName()
        if name == 'IfBlockList' and self.window:
            self.window.OnCheckItemIf(index, flag)
        elif name == 'ElseBlockList' and self.window:
            self.window.OnCheckItemElse(index, flag)
        
    def GetItems(self):
        """!Get list of selected actions"""
        ids = { 'checked'   : list(),
                'unchecked' : list() }
        for i in range(self.GetItemCount()):
            iId = int(self.GetItem(i, 0).GetText())
            if self.IsChecked(i):
                ids['checked'].append(iId)
            else:
                ids['unchecked'].append(iId)
            
        return ids

    def CheckItemById(self, aId, flag):
        """!Check/uncheck given item by id"""
        for i in range(self.GetItemCount()):
            iId = int(self.GetItem(i, 0).GetText())
            if iId == aId:
                self.CheckItem(i, flag)
                break
        
class ModelCondition(ModelItem, ogl.PolygonShape):
    def __init__(self, parent, x, y, id = -1, width = None, height = None, text = '',
                 items = { 'if' : [], 'else' : [] }):
        """!Defines a if-else condition"""
        ModelItem.__init__(self, parent, x, y, id, width, height, text, items)
        
        if not width:
            self.width = UserSettings.Get(group='modeler', key='if-else', subkey=('size', 'width'))
        else:
            self.width = width
        if not height:
            self.height = UserSettings.Get(group='modeler', key='if-else', subkey=('size', 'height'))
        else:
            self.height = height
        
        if self.parent.GetCanvas():
            ogl.PolygonShape.__init__(self)
            
            points = [(0, - self.height / 2),
                      (self.width / 2, 0),
                      (0, self.height / 2),
                      (- self.width / 2, 0)]
            self.Create(points)
            
            self.SetCanvas(self.parent)
            self.SetX(x)
            self.SetY(y)
            self.SetPen(wx.BLACK_PEN)
            if text:
                self.AddText('(' + str(self.id) + ') ' + text)
            else:
                self.AddText('(' + str(self.id) + ')')

    def GetName(self):
        """!Get name"""
        return _("if-else")

    def GetWidth(self):
        """!Get object width"""
        return self.width

    def GetHeight(self):
        """!Get object height"""
        return self.height

    def SetItems(self, items, branch = 'if'):
        """!Set items (id)

        @param items list of items
        @param branch 'if' / 'else'
        """
        if branch in ['if', 'else']:
            self.items[branch] = items
        
class ModelConditionDialog(ModelItemDialog):
    """!Condition properties dialog"""
    def __init__(self, parent, shape, id = wx.ID_ANY, title = _("If-else properties"),
                 style = wx.DEFAULT_DIALOG_STYLE | wx.RESIZE_BORDER, **kwargs):
        ModelItemDialog.__init__(self, parent, shape, title,
                                 style = style, **kwargs)
        
        self.listBoxIf = wx.StaticBox(parent = self.panel, id = wx.ID_ANY,
                                      label=" %s " % _("List of items in 'if' block"))
        self.itemListIf = self.itemList
        self.itemListIf.SetName('IfBlockList')
        
        self.listBoxElse = wx.StaticBox(parent = self.panel, id = wx.ID_ANY,
                                        label=" %s " % _("List of items in 'else' block"))
        self.itemListElse = ItemCheckListCtrl(parent = self.panel,
                                              window = self,
                                              columns = [_("ID"), _("Name"),
                                                         _("Command")],
                                              shape = shape)
        self.itemListElse.SetName('ElseBlockList')
        self.itemListElse.Populate(self.parent.GetModel().GetItems())
        
        self._layout()
        self.SetMinSize(self.GetSize())
        self.SetSize((500, 400))
        
    def _layout(self):
        """!Do layout"""
        sizer = wx.BoxSizer(wx.VERTICAL)
        
        condSizer = wx.StaticBoxSizer(self.condBox, wx.VERTICAL)
        condSizer.Add(item = self.condText, proportion = 1,
                      flag = wx.EXPAND)
        
        listIfSizer = wx.StaticBoxSizer(self.listBoxIf, wx.VERTICAL)
        listIfSizer.Add(item = self.itemListIf, proportion = 1,
                        flag = wx.EXPAND)
        listElseSizer = wx.StaticBoxSizer(self.listBoxElse, wx.VERTICAL)
        listElseSizer.Add(item = self.itemListElse, proportion = 1,
                          flag = wx.EXPAND)
        
        btnSizer = wx.StdDialogButtonSizer()
        btnSizer.AddButton(self.btnCancel)
        btnSizer.AddButton(self.btnOk)
        btnSizer.Realize()

        sizer.Add(item = condSizer, proportion = 0,
                  flag = wx.EXPAND | wx.ALL, border = 3)
        sizer.Add(item = listIfSizer, proportion = 1,
                  flag = wx.EXPAND | wx.LEFT | wx.RIGHT, border = 3)
        sizer.Add(item = listElseSizer, proportion = 1,
                  flag = wx.EXPAND | wx.LEFT | wx.RIGHT, border = 3)
        sizer.Add(item = btnSizer, proportion=0,
                  flag = wx.EXPAND | wx.ALL | wx.ALIGN_CENTER, border=5)
        
        self.panel.SetSizer(sizer)
        sizer.Fit(self.panel)
        
        self.Layout()

    def OnCheckItemIf(self, index, flag):
        """!Item in if-block checked/unchecked"""
        if flag is False:
            return
        
        aId = int(self.itemListIf.GetItem(index, 0).GetText())
        if aId in self.itemListElse.GetItems()['checked']:
            self.itemListElse.CheckItemById(aId, False)
            
    def OnCheckItemElse(self, index, flag):
        """!Item in else-block checked/unchecked"""
        if flag is False:
            return
        
        aId = int(self.itemListElse.GetItem(index, 0).GetText())
        if aId in self.itemListIf.GetItems()['checked']:
            self.itemListIf.CheckItemById(aId, False)
        
    def GetItems(self):
        """!Get items"""
        return { 'if'   : self.itemListIf.GetItems(),
                 'else' : self.itemListElse.GetItems() }

class WritePythonFile:
    def __init__(self, fd, model):
        """!Class for exporting model to Python script

        @param fd file desciptor
        """
        self.fd     = fd
        self.model  = model
        self.indent = 4

        self._writePython()
        
    def _writePython(self):
        """!Write model to file"""
        properties = self.model.GetProperties()
        
        self.fd.write(
r"""#!/usr/bin/env python
#
############################################################################
#
# MODULE:       %s
#
# AUTHOR(S):	%s
#               
# PURPOSE:      %s
#
# DATE:         %s
#
#############################################################################
""" % (properties['name'],
       properties['author'],
       properties['description'],
       time.asctime()))
        
        self.fd.write(
r"""
import sys
import os
import atexit

import grass.script as grass
""")
        
        # cleanup()
        rast, vect, rast3d, msg = self.model.GetIntermediateData()
        self.fd.write(
r"""
def cleanup():
""")
        if rast:
            self.fd.write(
r"""    grass.run_command('g.remove',
                      rast=%s)
""" % ','.join(map(lambda x: "'" + x + "'", rast)))
        if vect:
            self.fd.write(
r"""    grass.run_command('g.remove',
                      vect = %s)
""" % ','.join(map(lambda x: "'" + x + "'", vect)))
        if rast3d:
            self.fd.write(
r"""    grass.run_command('g.remove',
                      rast3d = %s)
""" % ','.join(map(lambda x: "'" + x + "'", rast3d)))
        if not rast and not vect and not rast3d:
            self.fd.write('    pass\n')
        
        self.fd.write("\ndef main():\n")
        for item in self.model.GetItems():
            self._writePythonItem(item)
        
        self.fd.write("\n    return 0\n")
        
        self.fd.write(
r"""
if __name__ == "__main__":
    options, flags = grass.parser()
    atexit.register(cleanup)
    sys.exit(main())
""")
        
    def _writePythonItem(self, item, ignoreBlock = True):
        """!Write model object to Python file"""
        if isinstance(item, ModelAction):
            if ignoreBlock and item.GetBlockId(): # ignore items in loops of conditions
                return
            self._writePythonAction(item)
        elif isinstance(item, ModelLoop) or isinstance(item, ModelCondition):
            # substitute condition
            variables = self.model.GetVariables()
            cond = item.GetText()
            for variable in variables:
                pattern= re.compile('%' + variable)
                if pattern.search(cond):
                    value = variables[variable].get('value', '')
                    if variables[variable].get('type', 'string') == 'string':
                        value = '"' + value + '"'
                    cond = pattern.sub(value, cond)
            if isinstance(item, ModelLoop):
                self.fd.write('%sfor %s:\n' % (' ' * self.indent, cond))
                self.indent += 4
                for action in item.GetItems():
                    self._writePythonItem(action, ignoreBlock = False)
                self.indent -= 4
            else: # ModelCondition
                self.fd.write('%sif %s:\n' % (' ' * self.indent, cond))
                self.indent += 4
                condItems = item.GetItems()
                for action in condItems['if']:
                    self._writePythonItem(action, ignoreBlock = False)
                if condItems['else']:
                    self.indent -= 4
                    self.fd.write('%selse:\n' % (' ' * self.indent))
                    self.indent += 4
                    for action in condItems['else']:
                        self._writePythonItem(action, ignoreBlock = False)
                self.indent += 4
        
    def _writePythonAction(self, item):
        """!Write model action to Python file"""
        task = menuform.GUI().ParseCommand(cmd = item.GetLog(string = False),
                                           show = None)
        opts = task.get_options()
        flags = ''
        params = list()
        strcmd = "%sgrass.run_command(" % (' ' * self.indent)
        cmdIndent = len(strcmd)
        for f in opts['flags']:
            if f.get('value', False) == True:
                name = f.get('name', '')
                if len(name) > 1:
                    params.append('%s = True' % name)
                else:
                    flags += name
            
        for p in opts['params']:
            name = p.get('name', None)
            value = p.get('value', None)
            if name and value:
                ptype = p.get('type', 'string')
                if ptype == 'string':
                    params.append('%s = "%s"' % (name, value))
                else:
                    params.append("%s = %s" % (name, value))
        
        self.fd.write(strcmd + '"%s"' % task.get_name())
        if flags:
            self.fd.write(",\n%sflags = '%s'" % (' ' * cmdIndent, flags))
        if len(params) > 0:
            self.fd.write(",\n")
            for opt in params[:-1]:
                self.fd.write("%s%s,\n" % (' ' * cmdIndent, opt))
            self.fd.write("%s%s)\n" % (' ' * cmdIndent, params[-1]))
        else:
            self.fd.write(")\n")

        
def main():
    app = wx.PySimpleApp()
    wx.InitAllImageHandlers()
    frame = ModelFrame(parent = None)
    if len(sys.argv) > 1:
        frame.LoadModelFile(sys.argv[1])
    frame.Show()
    
    app.MainLoop()
    
if __name__ == "__main__":
    main()
