/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include "config.h"
#include "grdcpublic.h"
#include "grdcplug.h"

G_DEFINE_TYPE (GrdcPlug, grdc_plug, GTK_TYPE_FIXED)

enum {
    CONNECT_SIGNAL,
    DISCONNECT_SIGNAL,
    LAST_SIGNAL
};

typedef struct _GrdcPlugSignalData
{
    GrdcPlug *gp;
    const gchar *signal_name;
} GrdcPlugSignalData;

static guint grdc_plug_signals[LAST_SIGNAL] = { 0 };

static void
grdc_plug_class_init (GrdcPlugClass *klass)
{
    grdc_plug_signals[CONNECT_SIGNAL] =
        g_signal_new ("connect",
            G_TYPE_FROM_CLASS (klass),
            G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
            G_STRUCT_OFFSET (GrdcPlugClass, connect),
            NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    grdc_plug_signals[DISCONNECT_SIGNAL] =
        g_signal_new ("disconnect",
            G_TYPE_FROM_CLASS (klass),
            G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
            G_STRUCT_OFFSET (GrdcPlugClass, disconnect),
            NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}

static void
grdc_plug_init_cancel (GrdcInitDialog *dialog, gint response_id, GrdcPlug *gp)
{
    if ((response_id == GTK_RESPONSE_CANCEL || response_id == GTK_RESPONSE_DELETE_EVENT)
        && dialog->mode == GRDC_INIT_MODE_CONNECTING)
    {
        grdc_plug_close_connection (gp);
    }
}

static void
grdc_plug_show_init_dialog (GrdcPlug *gp, const gchar *name)
{
    if (gp->init_dialog)
    {
        gtk_widget_destroy (gp->init_dialog);
    }
    gp->init_dialog = grdc_init_dialog_new (NULL, _("Connecting to '%s'..."), name);
    g_signal_connect (G_OBJECT (gp->init_dialog), "response", G_CALLBACK (grdc_plug_init_cancel), gp);
    gtk_widget_show (gp->init_dialog);
}

static void
grdc_plug_hide_init_dialog (GrdcPlug *gp)
{
    if (gp->init_dialog)
    {
        gtk_widget_destroy (gp->init_dialog);
        gp->init_dialog = NULL;
    }
}

static void
grdc_plug_destroy (GrdcPlug *gp, gpointer data)
{
    grdc_plug_hide_init_dialog (gp);
}

static void
grdc_plug_connect (GrdcPlug *gp, gpointer data)
{
    grdc_plug_hide_init_dialog (gp);
}

static void
grdc_plug_disconnect (GrdcPlug *gp, gpointer data)
{
    grdc_plug_hide_init_dialog (gp);
}

static void
grdc_plug_init (GrdcPlug *gp)
{
    g_signal_connect (G_OBJECT (gp), "destroy", G_CALLBACK (grdc_plug_destroy), NULL);
    g_signal_connect (G_OBJECT (gp), "connect", G_CALLBACK (grdc_plug_connect), NULL);
    g_signal_connect (G_OBJECT (gp), "disconnect", G_CALLBACK (grdc_plug_disconnect), NULL);

    gp->init_dialog = NULL;
    gp->has_error = FALSE;
    gp->error_message[0] = '\0';
}

gboolean
grdc_plug_open_connection (GrdcPlug *gp, GrdcFile *grdcfile)
{
    /* Show "server" instead of "name" for quick connect */
    grdc_plug_show_init_dialog (gp, (grdcfile->filename ? grdcfile->name : grdcfile->server));
    return (* (GRDC_PLUG_GET_CLASS (gp)->open_connection)) (gp, grdcfile);
}

gboolean
grdc_plug_close_connection (GrdcPlug *gp)
{
    return (* (GRDC_PLUG_GET_CLASS (gp)->close_connection)) (gp);
}

gboolean
grdc_plug_query_feature (GrdcPlug *gp, GrdcPlugFeature feature)
{
    return (* (GRDC_PLUG_GET_CLASS (gp)->query_feature)) (gp, feature);
}

void
grdc_plug_call_feature (GrdcPlug *gp, GrdcPlugFeature feature, gpointer data)
{
    (* (GRDC_PLUG_GET_CLASS (gp)->call_feature)) (gp, feature, data);
}

static gboolean
grdc_plug_emit_signal_timeout (gpointer user_data)
{
    GrdcPlugSignalData *data = (GrdcPlugSignalData*) user_data;
    g_signal_emit_by_name (G_OBJECT (data->gp), data->signal_name);
    g_free (data);
    return FALSE;
}

void
grdc_plug_emit_signal (GrdcPlug *gp, const gchar *signal_name)
{
    GrdcPlugSignalData *data;

    data = g_new (GrdcPlugSignalData, 1);
    data->gp = gp;
    data->signal_name = signal_name;
    TIMEOUT_ADD (0, grdc_plug_emit_signal_timeout, data);
}

