/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"

#ifdef HAVE_LIBSSH

/* Define this before stdlib.h to have posix_openpt */
#define _XOPEN_SOURCE 600

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <libssh/libssh.h>
#include <pthread.h>
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_TERMIOS_H
#include <termios.h>
#endif
#ifdef HAVE_LIBVTE
#include <vte/vte.h>
#endif
#include "grdcpixmaps.h"
#include "grdcssh.h"

/*************************** SSH Base *********************************/

static const gchar *common_identities[] = {
    ".ssh/id_rsa",
    ".ssh/id_dsa",
    ".ssh/identity",
    NULL
};

gchar*
grdc_ssh_identity_path (const gchar *id)
{
    if (id == NULL) return NULL;
    if (id[0] == '/') return g_strdup (id);
    return g_strdup_printf ("%s/%s", g_get_home_dir (), id);
}

gchar*
grdc_ssh_find_identity (void)
{
    gchar *path;
    gint i;

    for (i = 0; common_identities[i]; i++)
    {
        path = grdc_ssh_identity_path (common_identities[i]);
        if (g_file_test (path, G_FILE_TEST_IS_REGULAR | G_FILE_TEST_EXISTS))
        {
            g_free (path);
            return g_strdup (common_identities[i]);
        }
        g_free (path);
    }
    return NULL;
}

void
grdc_ssh_set_error (GrdcSSH *ssh, const gchar *fmt)
{
    const gchar *err;

    err = ssh_get_error (ssh->session);
    ssh->error = g_strdup_printf (fmt, err);
}

static gint
grdc_ssh_auth_password (GrdcSSH *ssh)
{
    gint ret;

    if (ssh->authenticated) return 1;
    if (ssh->password == NULL) return -1;

    ret = ssh_userauth_password (ssh->session, NULL, ssh->password);
    if (ret != SSH_AUTH_SUCCESS)
    {
        grdc_ssh_set_error (ssh, _("SSH password authentication failed: %s"));
        return 0;
    }

    ssh->authenticated = TRUE;
    return 1;
}

static gint
grdc_ssh_auth_pubkey (GrdcSSH *ssh)
{
    gint ret;
    STRING *pubkey;
    PRIVATE_KEY *privkey;
    gint keytype;

    if (ssh->authenticated) return 1;

    if (ssh->pubkeyfile == NULL || ssh->privkeyfile == NULL)
    {
        ssh->error = g_strdup_printf (_("SSH public key authentication failed: %s"),
            _("SSH Key file not yet set."));
        return 0;
    }

    pubkey = publickey_from_file (ssh->session, ssh->pubkeyfile, &keytype);
    if (pubkey == NULL)
    {
        grdc_ssh_set_error (ssh, _("SSH public key authentication failed: %s"));
        return 0;
    }

    privkey = privatekey_from_file (ssh->session, ssh->privkeyfile, keytype, _
        (ssh->password ? ssh->password : ""));
    if (privkey == NULL)
    {
#if LIBSSH_VERSION_MAJOR >= 0 && LIBSSH_VERSION_MINOR >= 3
        string_free (pubkey);
#endif
        if (ssh->password == NULL || ssh->password[0] == '\0') return -1;

        grdc_ssh_set_error (ssh, _("SSH public key authentication failed: %s"));
        return 0;
    }

    ret = ssh_userauth_pubkey (ssh->session, NULL, pubkey, privkey);
#if LIBSSH_VERSION_MAJOR >= 0 && LIBSSH_VERSION_MINOR >= 3
    string_free (pubkey);
#endif
    privatekey_free (privkey);

    if (ret != SSH_AUTH_SUCCESS)
    {
        grdc_ssh_set_error (ssh, _("SSH public key authentication failed: %s"));
        return 0;
    }

    ssh->authenticated = TRUE;
    return 1;
}

gint
grdc_ssh_auth (GrdcSSH *ssh, const gchar *password)
{
    if (password)
    {
        g_free (ssh->password);
        ssh->password = g_strdup (password);
    }

    switch (ssh->auth)
    {

    case SSH_AUTH_PASSWORD:
        return grdc_ssh_auth_password (ssh);

    case SSH_AUTH_PUBLICKEY:
        return grdc_ssh_auth_pubkey (ssh);

    default:
        return 0;
    }
}

gint
grdc_ssh_auth_gui (GrdcSSH *ssh, GrdcInitDialog *dialog, gboolean threaded)
{
    gchar *tips;
    gchar *keyname;
    gint ret;

    /* Try empty password or existing password first */
    ret = grdc_ssh_auth (ssh, NULL);
    if (ret > 0) return 1;

    /* Requested for a non-empty password */
    if (ret < 0)
    {
        if (!dialog) return -1;

        switch (ssh->auth)
        {
        case SSH_AUTH_PASSWORD:
            tips = _("Authenticating %s's password to SSH server %s...");
            keyname = _("SSH Password");
            break;
        case SSH_AUTH_PUBLICKEY:
            tips = _("Authenticating %s's identity to SSH server %s...");
            keyname = _("SSH Private Key Passphrase");
            break;
        default:
            return FALSE;
        }

        if (threaded) gdk_threads_enter();
        grdc_init_dialog_set_status (dialog, tips, ssh->user, ssh->server);

        ret = grdc_init_dialog_authpwd (dialog, keyname, FALSE);
        if (threaded) {gdk_flush();gdk_threads_leave();}

        if (ret != GTK_RESPONSE_OK) return -1;

        ret = grdc_ssh_auth (ssh, dialog->password);
    }

    if (ret <= 0)
    {
        return 0;
    }

    return 1;
}

gboolean
grdc_ssh_init_session (GrdcSSH *ssh)
{
    SSH_OPTIONS *options;

    options = ssh_options_new ();
    ssh_options_set_username (options, ssh->user);
    ssh_options_set_host (options, ssh->server);
    ssh_options_set_port (options, ssh->port);

    /* Now init & startup the SSH session */    
    ssh->session = ssh_new ();
    ssh_set_options (ssh->session, options);
    if (ssh_connect (ssh->session))
    {
        grdc_ssh_set_error (ssh, _("Failed to startup SSH session: %s"));
        return FALSE;
    }

    /* Try the "none" authentication */
    if (ssh_userauth_none (ssh->session, NULL) == SSH_AUTH_SUCCESS)
    {
        ssh->authenticated = TRUE;
    }
    return TRUE;
}

gboolean
grdc_ssh_init_from_file (GrdcSSH *ssh, GrdcFile *grdcfile)
{
    gchar *ptr, *s;

    ssh->session = NULL;
    ssh->authenticated = FALSE;
    ssh->error = NULL;
    pthread_mutex_init (&ssh->ssh_mutex, NULL);

    /* Parse the address and port */
    if (grdcfile->ssh_server && grdcfile->ssh_server[0] != '\0')
    {
        ssh->server = g_strdup (grdcfile->ssh_server);
        ptr = g_strrstr (ssh->server, ":");
        if (ptr != NULL)
        {
            *ptr++ =  '\0';
            ssh->port = (guint) atoi (ptr);
        }
        else
        {
            ssh->port = 22;
        }
    }
    else if (grdcfile->server == NULL || grdcfile->server[0] == '\0')
    {
        return FALSE;
    }
    else
    {
        ssh->server = g_strdup (grdcfile->server);
        ptr = g_strrstr (ssh->server, ":");
        if (ptr) *ptr = '\0';
        ssh->port = 22;
    }

    ssh->user = g_strdup ((grdcfile->ssh_username && grdcfile->ssh_username[0] != '\0' ?
        grdcfile->ssh_username : g_get_user_name ()));
    ssh->password = NULL;
    ssh->auth = grdcfile->ssh_auth;
    ssh->charset = g_strdup (grdcfile->ssh_charset);

    /* Public/Private keys */
    s = (grdcfile->ssh_privatekey && grdcfile->ssh_privatekey[0] != '\0' ?
        g_strdup (grdcfile->ssh_privatekey) : grdc_ssh_find_identity ());
    if (s)
    {
        ssh->privkeyfile = grdc_ssh_identity_path (s);
        ssh->pubkeyfile = g_strdup_printf ("%s.pub", ssh->privkeyfile);
        g_free (s);
    }
    else
    {
        ssh->privkeyfile = NULL;
        ssh->pubkeyfile = NULL;
    }

    return TRUE;
}

static gboolean
grdc_ssh_init_from_ssh (GrdcSSH *ssh, const GrdcSSH *ssh_src)
{
    ssh->session = NULL;
    ssh->authenticated = FALSE;
    ssh->error = NULL;
    pthread_mutex_init (&ssh->ssh_mutex, NULL);

    ssh->server = g_strdup (ssh_src->server);
    ssh->port = ssh_src->port;
    ssh->user = g_strdup (ssh_src->user);
    ssh->auth = ssh_src->auth;
    ssh->password = g_strdup (ssh_src->password);
    ssh->pubkeyfile = g_strdup (ssh_src->pubkeyfile);
    ssh->privkeyfile = g_strdup (ssh_src->privkeyfile);
    ssh->charset = g_strdup (ssh_src->charset);

    return TRUE;
}

gchar*
grdc_ssh_convert (GrdcSSH *ssh, const gchar *from)
{
    gchar *to = NULL;

    if (ssh->charset && from)
    {
        to = g_convert (from, -1, "UTF-8", ssh->charset, NULL, NULL, NULL);
    }
    if (!to) to = g_strdup (from);
    return to;
}

gchar*
grdc_ssh_unconvert (GrdcSSH *ssh, const gchar *from)
{
    gchar *to = NULL;

    if (ssh->charset && from)
    {
        to = g_convert (from, -1, ssh->charset, "UTF-8", NULL, NULL, NULL);
    }
    if (!to) to = g_strdup (from);
    return to;
}

void
grdc_ssh_free (GrdcSSH *ssh)
{
    if (ssh->session)
    {
        ssh_disconnect (ssh->session);
        ssh->session = NULL;
    }
    g_free (ssh->server);
    g_free (ssh->user);
    g_free (ssh->password);
    g_free (ssh->pubkeyfile);
    g_free (ssh->privkeyfile);
    g_free (ssh->charset);
    g_free (ssh->error);
    pthread_mutex_destroy (&ssh->ssh_mutex);
    g_free (ssh);
}

/*************************** SSH Tunnel *********************************/

GrdcSSHTunnel*
grdc_ssh_tunnel_new_from_file (GrdcFile *grdcfile)
{
    GrdcSSHTunnel *tunnel;

    tunnel = g_new (GrdcSSHTunnel, 1);

    grdc_ssh_init_from_file (GRDC_SSH (tunnel), grdcfile);

    tunnel->channel = NULL;
    tunnel->tunnel_server_sock = -1;
    tunnel->tunnel_client_sock = -1;
    tunnel->thread = 0;
    tunnel->running = FALSE;
    tunnel->dest_host = NULL;
    tunnel->dest_port = 0;
    tunnel->buffer = NULL;
    tunnel->buffer_len = 0;

    return tunnel;
}

static gpointer
grdc_ssh_tunnel_main_thread (gpointer data)
{
    GrdcSSHTunnel *tunnel = (GrdcSSHTunnel*) data;
    gint flags;
    gchar *ptr;
    ssize_t len, lenw;
    fd_set set;
    struct timeval timeout;
    CHANNEL *ch[2], *chout[2];
    gint ret;
     
    pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL);
    pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL);

    /* Accept a local connection */
    tunnel->tunnel_client_sock = accept (tunnel->tunnel_server_sock, NULL, NULL);
    if (tunnel->tunnel_client_sock < 0)
    {
        GRDC_SSH (tunnel)->error = g_strdup ("Failed to accept local socket");
        tunnel->thread = 0;
        return NULL;
    }
    close (tunnel->tunnel_server_sock);
    tunnel->tunnel_server_sock = -1;

    if ((flags = fcntl (tunnel->tunnel_client_sock, F_GETFL, 0)) < 0) 
    { 
        GRDC_SSH (tunnel)->error = g_strdup ("fcntl(1) failed");
        tunnel->thread = 0;
        return NULL;
    } 

    if (fcntl (tunnel->tunnel_client_sock, F_SETFL, flags | O_NONBLOCK) < 0) 
    { 
        GRDC_SSH (tunnel)->error = g_strdup ("fcntl(2) failed");
        tunnel->thread = 0;
        return NULL;
    } 

    /* Request the SSH server to connect to the destination */
    tunnel->channel = channel_new (tunnel->ssh.session);
    if (channel_open_forward (tunnel->channel, tunnel->dest_host, tunnel->dest_port, "127.0.0.1", 0) != SSH_OK)
    {
        grdc_ssh_set_error (GRDC_SSH (tunnel), _("Failed to connect to the SSH tunnel destination: %s"));
        tunnel->thread = 0;
        return NULL;
    }

    tunnel->buffer_len = 10240;
    tunnel->buffer = g_malloc (tunnel->buffer_len);

    ch[0] = tunnel->channel;
    ch[1] = NULL;

    /* Start the tunnel data transmittion */
    while (tunnel->running)
    {
        timeout.tv_sec = 1;
        timeout.tv_usec = 0;

        FD_ZERO (&set);
        FD_SET (tunnel->tunnel_client_sock, &set);

        ret = ssh_select (ch, chout, FD_SETSIZE, &set, &timeout);
        if (!tunnel->running) break;
        if (ret == SSH_EINTR) continue;
        if (ret == -1) break;

        while (tunnel->running &&
            (len = read (tunnel->tunnel_client_sock, tunnel->buffer, tunnel->buffer_len)) > 0)
        {
            for (ptr = tunnel->buffer, lenw = 0; len > 0; len -= lenw, ptr += lenw)
            {
                lenw = channel_write (tunnel->channel, (char*) ptr, len);
                if (lenw <= 0)
                {
                    tunnel->running = FALSE;
                    break;
                }
            }
        }
        if (!tunnel->running || len == 0) break;

        len = channel_poll (tunnel->channel, 0);
        if (len == SSH_ERROR) break;
        if (len > 0)
        {
            if (len > tunnel->buffer_len)
            {
                tunnel->buffer_len = len;
                tunnel->buffer = (gchar*) g_realloc (tunnel->buffer, tunnel->buffer_len);
            }
            len = channel_read_nonblocking (tunnel->channel, tunnel->buffer, len, 0);
            if (len <= 0) break;

            for (ptr = tunnel->buffer, lenw = 0; len > 0; len -= lenw, ptr += lenw)
            {
                lenw = write (tunnel->tunnel_client_sock, ptr, len);
                if (lenw < 0 && tunnel->running)
                {
                    lenw = 0;
                    continue;
                }
                if (lenw <= 0)
                {
                    tunnel->running = FALSE;
                    break;
                }
            }
        }
        if (len == SSH_ERROR) break;
    }

    pthread_mutex_lock (&GRDC_SSH (tunnel)->ssh_mutex);

    tunnel->thread = 0;

    close (tunnel->tunnel_client_sock);
    tunnel->tunnel_client_sock = -1;

    channel_close (tunnel->channel);
    channel_free (tunnel->channel);
    tunnel->channel = NULL;

    pthread_mutex_unlock (&GRDC_SSH (tunnel)->ssh_mutex);

    return NULL;
}

gboolean
grdc_ssh_tunnel_start (GrdcSSHTunnel* tunnel, const gchar *dest, gint local_port)
{
    gchar *ptr;
    gint sock;
    gint sockopt = 1;
    struct sockaddr_in sin;

    tunnel->dest_host = g_strdup (dest);
    ptr = g_strrstr (tunnel->dest_host, ":");
    if (ptr != NULL)
    {
        *ptr++ =  '\0';
        tunnel->dest_port = atoi (ptr);
    }
    else
    {
        GRDC_SSH (tunnel)->error = g_strdup ("Destination port has not been assigned");
        return FALSE;
    }

    /* Create the server socket that listens on the local port */
    sock = socket (AF_INET, SOCK_STREAM, 0);
    if (sock < 0)
    {
        GRDC_SSH (tunnel)->error = g_strdup ("Failed to create socket.");
        return FALSE;
    }
    setsockopt (sock, SOL_SOCKET, SO_REUSEADDR, &sockopt, sizeof (sockopt));

    sin.sin_family = AF_INET;
    sin.sin_port = htons (local_port);
    sin.sin_addr.s_addr = inet_addr("127.0.0.1");

    if (bind (sock, (struct sockaddr *) &sin, sizeof(sin)))
    {
        GRDC_SSH (tunnel)->error = g_strdup ("Failed to bind on local port.");
        close (sock);
        return FALSE;
    }

    if (listen (sock, 1))
    {
        GRDC_SSH (tunnel)->error = g_strdup ("Failed to listen on local port.");
        close (sock);
        return FALSE;
    }

    tunnel->tunnel_server_sock = sock;
    tunnel->running = TRUE;

    if (pthread_create (&tunnel->thread, NULL, grdc_ssh_tunnel_main_thread, tunnel))
    {
        GRDC_SSH (tunnel)->error = g_strdup ("Failed to initialize pthread.");
        tunnel->thread = 0;
        return FALSE;
    }
    return TRUE;
}

gboolean
grdc_ssh_tunnel_terminated (GrdcSSHTunnel* tunnel)
{
    return (tunnel->thread == 0);
}

void
grdc_ssh_tunnel_free (GrdcSSHTunnel* tunnel)
{
    pthread_mutex_lock (&GRDC_SSH (tunnel)->ssh_mutex);

    if (tunnel->thread != 0)
    {
        tunnel->running = FALSE;
        pthread_cancel (tunnel->thread);
        pthread_join (tunnel->thread, NULL);
    }
    if (tunnel->tunnel_client_sock >= 0) close (tunnel->tunnel_client_sock);
    if (tunnel->tunnel_server_sock >= 0) close (tunnel->tunnel_server_sock);
    if (tunnel->channel)
    {
        channel_close (tunnel->channel);
        channel_free (tunnel->channel);
        tunnel->channel = NULL;
    }

    pthread_mutex_unlock (&GRDC_SSH (tunnel)->ssh_mutex);

    if (tunnel->buffer)
    {
        g_free (tunnel->buffer);
        tunnel->buffer = NULL;
    }
    g_free (tunnel->dest_host);

    grdc_ssh_free (GRDC_SSH (tunnel));
}

/*************************** SFTP *********************************/

GrdcSFTP*
grdc_sftp_new_from_file (GrdcFile *grdcfile)
{
    GrdcSFTP *sftp;

    sftp = g_new (GrdcSFTP, 1);

    grdc_ssh_init_from_file (GRDC_SSH (sftp), grdcfile);

    sftp->sftp_sess = NULL;

    return sftp;
}

GrdcSFTP*
grdc_sftp_new_from_ssh (GrdcSSH *ssh)
{
    GrdcSFTP *sftp;

    sftp = g_new (GrdcSFTP, 1);

    grdc_ssh_init_from_ssh (GRDC_SSH (sftp), ssh);

    sftp->sftp_sess = NULL;

    return sftp;
}

gboolean
grdc_sftp_open (GrdcSFTP *sftp)
{
    sftp->sftp_sess = sftp_new (sftp->ssh.session);
    if (!sftp->sftp_sess)
    {
        grdc_ssh_set_error (GRDC_SSH (sftp), _("Failed to create sftp session: %s"));
        return FALSE;
    }
    if (sftp_init (sftp->sftp_sess))
    {
        grdc_ssh_set_error (GRDC_SSH (sftp), _("Failed to initialize sftp session: %s"));
        return FALSE;
    }
    return TRUE;
}

void
grdc_sftp_free (GrdcSFTP *sftp)
{
    if (sftp->sftp_sess)
    {
        sftp_free (sftp->sftp_sess);
        sftp->sftp_sess = NULL;
    }
    grdc_ssh_free (GRDC_SSH (sftp));
}

/*************************** SSH Terminal *********************************/

GrdcSSHTerminal*
grdc_ssh_terminal_new_from_ssh (GrdcSSH *ssh)
{
    GrdcSSHTerminal *term;

    term = g_new (GrdcSSHTerminal, 1);

    grdc_ssh_init_from_ssh (GRDC_SSH (term), ssh);

    term->thread = 0;
    term->window = NULL;
    term->master = -1;
    term->slave = -1;
    term->closed = FALSE;

    return term;
}

#ifdef HAVE_LIBVTE

#define GRDC_SSH_TERMINAL_THREAD_EXIT \
    gdk_threads_enter(); \
    gtk_widget_destroy (term->window); \
    if (error) \
    { \
        dialog = gtk_message_dialog_new (NULL, \
            GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, \
            error, NULL); \
        g_signal_connect (G_OBJECT (dialog), "response", G_CALLBACK (gtk_widget_destroy), NULL); \
        gtk_widget_show (dialog); \
        g_free (error); \
    } \
    gdk_flush();gdk_threads_leave(); \
    close (term->master); \
    close (term->slave); \
    channel_close (channel); \
    channel_free (channel); \
    grdc_ssh_free (GRDC_SSH (term)); \
    g_free (buf); \
    return NULL;

static gpointer
grdc_ssh_terminal_thread (gpointer data)
{
    GrdcSSHTerminal *term = (GrdcSSHTerminal*) data;
    fd_set fds;
    struct timeval timeout;
    CHANNEL *channel = NULL;
    CHANNEL *ch[2], *chout[2];
    gchar *buf = NULL;
    gint buf_len;
    gint len;
    gint i, ret;
    gchar *error = NULL;
    GtkWidget *dialog;

    if (!grdc_ssh_init_session (GRDC_SSH (term)) ||
        grdc_ssh_auth (GRDC_SSH (term), NULL) <= 0 ||
        (channel = channel_new (GRDC_SSH (term)->session)) == NULL ||
        channel_open_session (channel))
    {
        error = g_strdup_printf ("Failed to open channel : %s", ssh_get_error (GRDC_SSH (term)->session));
        GRDC_SSH_TERMINAL_THREAD_EXIT
    }

    channel_request_pty (channel);
    if (channel_request_shell(channel))
    {
        error = g_strdup_printf ("Failed to request shell : %s", ssh_get_error (GRDC_SSH (term)->session));
        GRDC_SSH_TERMINAL_THREAD_EXIT
    }

    buf_len = 1000;
    buf = g_malloc (buf_len + 1);

    ch[0] = channel;
    ch[1] = NULL;

    while (!term->closed)
    {
        timeout.tv_sec = 1;
        timeout.tv_usec = 0;

        FD_ZERO (&fds);
        FD_SET (term->master, &fds);

        ret = ssh_select (ch, chout, FD_SETSIZE, &fds, &timeout);
        if (ret == SSH_EINTR) continue;
        if (ret == -1) break;

        if (FD_ISSET (term->master, &fds))
        {
            len = read (term->master, buf, buf_len);
            if (len <= 0) break;
            channel_write (channel, buf, len);
        }
        for (i = 0; i < 2; i++)
        {
            len = channel_poll (channel, i);
            if (len == SSH_ERROR || len == SSH_EOF)
            {
                term->closed = TRUE;
                break;
            }
            if (len <= 0) continue;
            if (len > buf_len)
            {
                buf_len = len;
                buf = (gchar*) g_realloc (buf, buf_len + 1);
            }
            len = channel_read_nonblocking (channel, buf, len, i);
            if (len <= 0)
            {
                term->closed = TRUE;
                break;
            }
            write (term->master, buf, len);
        }
    }

    GRDC_SSH_TERMINAL_THREAD_EXIT
}

static gboolean
grdc_ssh_terminal_on_delete_event (GtkWidget *window, GdkEvent *event, GrdcSSHTerminal *term)
{
    term->closed = TRUE;
    return TRUE;
}

static void
grdc_ssh_terminal_create_window (GrdcSSHTerminal *term)
{
    GtkWidget *window;
    GtkWidget *hbox;
    GtkWidget *vscrollbar;
    GtkWidget *vte;
    gchar *charset;

    charset = GRDC_SSH (term)->charset;

    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (window), GRDC_SSH (term)->server);
    gtk_window_set_resizable (GTK_WINDOW (window), FALSE);
    gtk_window_set_icon (GTK_WINDOW (window), grdc_pixmap (GRDC_PIXMAP_TYPE_SSH, 0));
    g_signal_connect (G_OBJECT (window), "delete-event",
        G_CALLBACK (grdc_ssh_terminal_on_delete_event), term);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox);
    gtk_container_add (GTK_CONTAINER (window), hbox);

    vte = vte_terminal_new ();
    gtk_widget_show (vte);
    vte_terminal_set_size (VTE_TERMINAL (vte), 80, 25);
    vte_terminal_set_scroll_on_keystroke (VTE_TERMINAL (vte), TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), vte, TRUE, TRUE, 0);
    if (charset && charset[0] != '\0')
    {
        vte_terminal_set_encoding (VTE_TERMINAL (vte), charset);
    }

    vscrollbar = gtk_vscrollbar_new (vte_terminal_get_adjustment (VTE_TERMINAL (vte)));
    gtk_widget_show (vscrollbar);
    gtk_box_pack_start (GTK_BOX (hbox), vscrollbar, FALSE, TRUE, 0);

    vte_terminal_set_pty (VTE_TERMINAL (vte), term->slave);

    term->window = window;
}

gboolean
grdc_ssh_terminal_open (GrdcSSHTerminal *term)
{
    gchar *slavedevice;
    struct termios stermios;

    term->master = posix_openpt (O_RDWR | O_NOCTTY);
    if (term->master == -1 ||
        grantpt (term->master) == -1 ||
        unlockpt (term->master) == -1 ||
        (slavedevice = ptsname (term->master)) == NULL ||
        (term->slave = open (slavedevice, O_RDWR | O_NOCTTY)) < 0)
    {
        GRDC_SSH (term)->error = g_strdup ("Failed to create pty device.");
        return FALSE;
    }

    /* These settings works fine with OpenSSH... */
    tcgetattr (term->slave, &stermios);
    stermios.c_lflag &= ~(ECHO | ECHOE | ECHOK | ECHONL | ICANON);
    stermios.c_iflag &= ~(ICRNL);
    tcsetattr (term->slave, TCSANOW, &stermios);

    grdc_ssh_terminal_create_window (term);

    gtk_widget_show (term->window);
    gtk_window_present (GTK_WINDOW (term->window));

    /* Once the process started, we should always TRUE and assume the pthread will be created always */
    pthread_create (&term->thread, NULL, grdc_ssh_terminal_thread, term);

    return TRUE;
}

#else /* HAVE_LIBVTE */

gboolean
grdc_ssh_terminal_open (GrdcSSHTerminal *term)
{
    GRDC_SSH (term)->error = g_strdup ("No terminal support.");
    return FALSE;
}

#endif /* HAVE_LIBVTE */

#endif /* HAVE_LIBSSH */

