/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <string.h>
#include "grdcpublic.h"
#include "grdcstringlist.h"

G_DEFINE_TYPE (GrdcStringList, grdc_string_list, GTK_TYPE_TABLE)

static gboolean
grdc_string_list_on_key_press (GtkWidget *widget, GdkEventKey *event, GrdcStringList *gsl)
{
    if (gsl->has_error)
    {
        gtk_label_set_text (GTK_LABEL (gsl->status_label), NULL);
        gsl->has_error = FALSE;
    }
    return FALSE;
}

static void
grdc_string_list_add (GtkWidget *entry, GrdcStringList *gsl)
{
    gchar *text, *ptr, *error;
    GtkTreeSelection *selection;
    GtkTreeIter iter;
    GtkTreePath *path;
    gboolean passed;

    text = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
    passed = TRUE;
    if (text && text[0] != '\0')
    {
        /* Eliminate delimitors... */
        while ((ptr = strchr (text, STRING_DELIMITOR)) != NULL) *ptr = ' ';

        if (gsl->validation_func)
        {
            if (!((*gsl->validation_func) (text, &error)))
            {
                gtk_label_set_text (GTK_LABEL (gsl->status_label), error);
                g_free (error);
                gsl->has_error = TRUE;
                passed = FALSE;
            }
        }

        if (passed)
        {
            selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (gsl->list));
            gtk_list_store_append (gsl->store, &iter);
            gtk_list_store_set (gsl->store, &iter, 0, text, -1);
            gtk_tree_selection_select_iter (selection, &iter);

            path = gtk_tree_model_get_path (GTK_TREE_MODEL (gsl->store), &iter);
            gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (gsl->list), path, NULL, 0, 0, 0);
            gtk_tree_path_free (path);
        }
    }
    g_free (text);
    if (passed) gtk_entry_set_text (GTK_ENTRY (entry), "");
    gtk_widget_grab_focus (entry);
}

static void
grdc_string_list_remove (GtkWidget *widget, GrdcStringList *gsl)
{
    GtkTreeSelection *selection;
    GtkTreeIter iter;

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (gsl->list));
    if (gtk_tree_selection_get_selected(selection, NULL, &iter))
    {
        gtk_list_store_remove (gsl->store, &iter);
    }
}

static void
grdc_string_list_move (GrdcStringList *gsl, GtkTreeIter *from, GtkTreeIter *to)
{
    GtkTreePath *path;

    gtk_list_store_swap (gsl->store, from, to);
    path = gtk_tree_model_get_path (GTK_TREE_MODEL (gsl->store), from);
    gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (gsl->list), path, NULL, 0, 0, 0);
    gtk_tree_path_free (path);
}

static void
grdc_string_list_down (GtkWidget *widget, GrdcStringList *gsl)
{
    GtkTreeSelection *selection;
    GtkTreeIter iter;
    GtkTreeIter target_iter;

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (gsl->list));
    if (gtk_tree_selection_get_selected(selection, NULL, &iter))
    {
        gtk_tree_selection_get_selected(selection, NULL, &target_iter);
        if (gtk_tree_model_iter_next (GTK_TREE_MODEL (gsl->store), &target_iter))
        {
            grdc_string_list_move (gsl, &iter, &target_iter);
        }
    }
}

static void
grdc_string_list_up (GtkWidget *widget, GrdcStringList *gsl)
{
    GtkTreeSelection *selection;
    GtkTreeIter iter;
    GtkTreeIter target_iter;
    GtkTreePath *path;

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (gsl->list));
    if (gtk_tree_selection_get_selected(selection, NULL, &iter))
    {
        gtk_tree_selection_get_selected(selection, NULL, &target_iter);
        path = gtk_tree_model_get_path (GTK_TREE_MODEL (gsl->store), &target_iter);
        if (gtk_tree_path_prev (path))
        {
            gtk_tree_model_get_iter (GTK_TREE_MODEL (gsl->store), &target_iter, path);
            gtk_tree_path_free (path);
            grdc_string_list_move (gsl, &iter, &target_iter);
        }
    }
}

static void
grdc_string_list_class_init (GrdcStringListClass *klass)
{
}

static void
grdc_string_list_init (GrdcStringList *gsl)
{
    GtkWidget *widget;
    GtkWidget *image;
    GtkWidget *scrolled_window;
    GtkWidget *vbox;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    GtkWidget *frame;
    GdkColor color;

    gtk_table_resize (GTK_TABLE (gsl), 3, 2);

    /* Create the frame and add a new scrolled window, followed by the group list */
    frame = gtk_frame_new (NULL);
    gtk_widget_show (frame);
    gtk_table_attach_defaults (GTK_TABLE (gsl), frame, 0, 1, 0, 1);

    scrolled_window = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolled_window);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
        GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_add (GTK_CONTAINER (frame), scrolled_window);

    gsl->store = gtk_list_store_new (1, G_TYPE_STRING);
    gsl->list = gtk_tree_view_new_with_model (GTK_TREE_MODEL (gsl->store));
    gtk_widget_show (gsl->list);
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (gsl->list), FALSE);
    gtk_container_add (GTK_CONTAINER (scrolled_window), gsl->list);

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (NULL, renderer, "text", 0, NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (gsl->list), column);

    /* entry */
    gsl->new_entry = gtk_entry_new_with_max_length (50);
    gtk_widget_show (gsl->new_entry);
    gtk_table_attach (GTK_TABLE (gsl), gsl->new_entry, 0, 1, 1, 2, GTK_EXPAND | GTK_FILL, 0, 0, 0);
    g_signal_connect (G_OBJECT (gsl->new_entry), "activate", G_CALLBACK (grdc_string_list_add), gsl);
    g_signal_connect (G_OBJECT (gsl->new_entry), "key-press-event", G_CALLBACK (grdc_string_list_on_key_press), gsl);

    /* buttons packed into a vbox */
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (vbox);
    gtk_table_attach (GTK_TABLE (gsl), vbox, 1, 2, 0, 3, 0, GTK_EXPAND | GTK_FILL, 0, 0);

    image = gtk_image_new_from_stock (GTK_STOCK_GO_UP, GTK_ICON_SIZE_MENU);
    gtk_widget_show (image);
    widget = gtk_button_new ();
    gtk_widget_show (widget);
    gtk_container_add (GTK_CONTAINER (widget), image);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 0);
    g_signal_connect (G_OBJECT (widget), "clicked", G_CALLBACK (grdc_string_list_up), gsl);

    image = gtk_image_new_from_stock (GTK_STOCK_GO_DOWN, GTK_ICON_SIZE_MENU);
    gtk_widget_show (image);
    widget = gtk_button_new ();
    gtk_widget_show (widget);
    gtk_container_add (GTK_CONTAINER (widget), image);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 0);
    g_signal_connect (G_OBJECT (widget), "clicked", G_CALLBACK (grdc_string_list_down), gsl);

    image = gtk_image_new_from_stock (GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU);
    gtk_widget_show (image);
    widget = gtk_button_new ();
    gtk_widget_show (widget);
    gtk_container_add (GTK_CONTAINER (widget), image);
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 0);
    g_signal_connect (G_OBJECT (widget), "clicked", G_CALLBACK (grdc_string_list_remove), gsl);

    /* The last status label */
    gsl->status_label = gtk_label_new (NULL);
    gtk_widget_show (gsl->status_label);
    gdk_color_parse ("red", &color);
    gtk_widget_modify_fg (gsl->status_label, GTK_STATE_NORMAL, &color);
    gtk_misc_set_alignment (GTK_MISC (gsl->status_label), 0.0, 0.5);
    gtk_table_attach (GTK_TABLE (gsl), gsl->status_label, 0, 2, 2, 3, GTK_EXPAND | GTK_FILL, 0, 0, 0);

    gsl->validation_func = NULL;
    gsl->has_error = FALSE;
}

GtkWidget*
grdc_string_list_new (void)
{
    return GTK_WIDGET (g_object_new (GRDC_TYPE_STRING_LIST, NULL));
}

void
grdc_string_list_set_text (GrdcStringList *gsl, const gchar *text)
{
    GtkTreeIter iter;
    gchar *buf, *ptr1, *ptr2;

    gtk_list_store_clear (gsl->store);

    buf = g_strdup (text);
    ptr1 = buf;
    while (ptr1 && *ptr1 != '\0')
    {
        ptr2 = strchr (ptr1, STRING_DELIMITOR);
        if (ptr2) *ptr2++ = '\0';
        
        gtk_list_store_append (gsl->store, &iter);
        gtk_list_store_set (gsl->store, &iter, 0, ptr1, -1);

        ptr1 = ptr2;
    }

    g_free (buf);
}

gchar*
grdc_string_list_get_text (GrdcStringList *gsl)
{
    GString *str;
    GtkTreeIter iter;
    gboolean first, ret;
    GValue value = {0};

    str = g_string_new (NULL);
    first = TRUE;

    ret = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (gsl->store), &iter);
    while (ret)
    {
        if (!first)
        {
            g_string_append_c (str, STRING_DELIMITOR);
        }
        else
        {
            first = FALSE;
        }
        gtk_tree_model_get_value (GTK_TREE_MODEL (gsl->store), &iter, 0, &value);
        g_string_append (str, g_value_get_string (&value));
        g_value_unset (&value);

        ret = gtk_tree_model_iter_next (GTK_TREE_MODEL (gsl->store), &iter);
    }

    return g_string_free (str, FALSE);
}

void
grdc_string_list_set_new_entry_tooltip_text (GrdcStringList *gsl, const gchar *text)
{
    gtk_widget_set_tooltip_text (gsl->new_entry, text);
}

void
grdc_string_list_set_validation_func (GrdcStringList *gsl, GrdcStringListValidationFunc func)
{
    gsl->validation_func = func;
}

