/*
 * Groach eyes2 theme module
 * Eyes keep watching mouse pointer.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <math.h>
#include <gnome.h>

#include "groach.h"


/* Constant numbers */
/* Every 3 seconds, get pointer coordinates */
#define GET_POINTER_INTERVAL	3000

enum {
	EYES2_STAT_R  = 0,
	EYES2_STAT_RU = 1,
	EYES2_STAT_U  = 2,
	EYES2_STAT_LU = 3,
	EYES2_STAT_L  = 4,
	EYES2_STAT_LD = 5,
	EYES2_STAT_D  = 6,
	EYES2_STAT_RD = 7
};


/* Data structure definitions */	
/* These are derived from theme-default.c */
typedef struct _DefaultMoveEnv DefaultMoveEnv;
typedef struct _AngleInfo AngleInfo;

struct _DefaultMoveEnv {
	gboolean turn_left;
	guint steps;	/* number of steps until recomputation */
	AngleInfo *angle_infos;/* array of AngleInfo for all directions */
};

struct _AngleInfo {
	gdouble sine;
	gdouble cosine;
};


/* Private variables */
/* Every %GET_POINTER_INTERVAL seconds, query the pointer's coordinate
   and keep it in static variable to use in callback functions. */
static gint get_pointer_timer_id;
static GdkPoint pointer_coordinate;


/* Private function declarations */
static gint get_pointer_timer_cb(gpointer data);
static void change_stat(GroMove *gmove);
static void move_push_cb(GroMove *gmove, const GroController *controller, GdkEventButton *event, gpointer data);
static void turn_gmove(GroMove *gmove);

gint
theme_init(const GroController *controller)
{
	pointer_coordinate.x = 0;
	pointer_coordinate.y = 0;
	get_pointer_timer_id = gtk_timeout_add(GET_POINTER_INTERVAL,
										   get_pointer_timer_cb,
										   (gpointer)controller);
	return 1;
}

gint
theme_finalize(const GroController *controller)
{
	gtk_timeout_remove(get_pointer_timer_id);
	return 1;
}

gint
move_init(const GroController *controller, GroMove *gmove)
{
	GroVector init_pos;
	DefaultMoveEnv *move_env;
	AngleInfo *angle_infos;
	int nd;

	move_env = g_new(DefaultMoveEnv, 1);
	gmove->move_env = move_env;/* Set it for later use */
	move_env->turn_left = RAND_INT(100) >= 50;
	move_env->steps = RAND_INT(200);
	angle_infos = g_new(AngleInfo, gmove->num_direct);
	move_env->angle_infos = angle_infos;
	for (nd = 0; nd < gmove->num_direct; nd++) {
		gdouble angle;

		angle = 2 * M_PI * nd / gmove->num_direct;
		angle_infos[nd].sine = sin(angle);
		angle_infos[nd].cosine = cos(angle);
	}

	/* Initialize */
	gro_move_change_gstat(gmove, GRO_STAT_WAKE);
	gro_move_change_direct(gmove, RAND_INT(gmove->num_direct));

	/* Initial position is decided with randum number */
	init_pos.ix = RAND_INT(gcontroller_window_width(controller) - gmove_width(gmove));
	init_pos.iy = RAND_INT(gcontroller_window_height(controller) - gmove_height(gmove));
	gro_move_move(gmove, controller, &init_pos);
	
	gtk_signal_connect(GTK_OBJECT(gmove), "push",
					   GTK_SIGNAL_FUNC(move_push_cb),
					   NULL);

	return 1;
}

GroMoveRet
move_compute(const GroController *controller, GroMove *gmove, const GdkRegion *vis_region, GroVector *ret_vec)
{
	DefaultMoveEnv *move_env = gmove->move_env;
	const AngleInfo *angle_infos = move_env->angle_infos;
	GdkRectangle tmp_gmove_rect = gmove->cur_rect;/* local copy */
	gint ix;
	gint iy;

	g_return_val_if_fail(gmove->cur_gstat != GRO_STAT_DEAD, GRO_RET_DONT_MOVE);
	
	/* Every theme is recommended to call this at first */
	DONTCARE_hidden_gmove(gmove, vis_region);
	
	/* Compute a new position temporarily to check whether it is
	   within the window and to check collisions with other gmoves. */
	ix = gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].cosine;
	iy = -(gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].sine);
	tmp_gmove_rect.x += ix;
	tmp_gmove_rect.y += iy;

	if (is_rect_in_gcontroller_window(controller, &tmp_gmove_rect) == TRUE) {
		const GList *other_gmoves = controller->gmove_list;
		const GList *node;

		if (move_env->steps-- <= 0) {
			turn_gmove(gmove);
			move_env->steps = RAND_INT(200);
		}

		/* Detect collision and avoid it.
		   This is intended to follow xroach's method. */
		for (node = other_gmoves; node; node = node->next) {
			GroMove *other = node->data;
			if (gmove == other)
				continue;
			if (is_rect_intersect(&tmp_gmove_rect, &other->cur_rect)) {
				turn_gmove(gmove);
				break;
			}
		}

		/* Recompute the move vector after turn */
		ix = gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].cosine;
		iy = -(gcontroller_step_pixel(controller) * angle_infos[gmove->cur_direct].sine);
	} else {
		/* If a new position is out of the window, turn it */
		turn_gmove(gmove);
		ix = iy = 0;
    }
	
	ret_vec->ix = ix;
	ret_vec->iy = iy;

	change_stat(gmove);
	
	return GRO_RET_MOVE;
}

gint
move_finalize(const GroController *controller, GroMove *gmove)
{
	DefaultMoveEnv *move_env;
	AngleInfo *angle_infos;

	gtk_signal_disconnect_by_func(GTK_OBJECT(gmove),
								  GTK_SIGNAL_FUNC(move_push_cb),
								  NULL);
	move_env = gmove->move_env;
	angle_infos = move_env->angle_infos;
	g_free(angle_infos);
	g_free(move_env);

	return 1;
}



/* ---The followings are private functions--- */
static gint
get_pointer_timer_cb(gpointer data)
{
	const GroController *controller = data;

	gdk_window_get_pointer(gcontroller_gdkwindow(controller),
						   &pointer_coordinate.x,
						   &pointer_coordinate.y,
						   NULL);

	return TRUE;
}

static void
change_stat(GroMove *gmove)
{
	int x = gmove_center_x(gmove);
	int y = gmove_center_y(gmove);
	
	if (ABS(x - pointer_coordinate.x) < 50) {
		if (y < pointer_coordinate.y) {
			gro_move_change_gstat(gmove, EYES2_STAT_D);
		} else {
			gro_move_change_gstat(gmove, EYES2_STAT_U);
		}
	} else if (x < pointer_coordinate.x) {
		if (ABS(y - pointer_coordinate.y) < 50) {
			gro_move_change_gstat(gmove, EYES2_STAT_R);
		} else if (y < pointer_coordinate.y) {
			gro_move_change_gstat(gmove, EYES2_STAT_RD);
		} else {
			gro_move_change_gstat(gmove, EYES2_STAT_RU);
		}
	} else {
		if (ABS(y - pointer_coordinate.y) < 50) {
			gro_move_change_gstat(gmove, EYES2_STAT_L);
		} else if (y < pointer_coordinate.y) {
			gro_move_change_gstat(gmove, EYES2_STAT_LD);
		} else {
			gro_move_change_gstat(gmove, EYES2_STAT_LU);
		}
	}
}

static void
move_push_cb(GroMove *gmove, const GroController *controller, GdkEventButton *event, gpointer data)
{
	gdk_beep();
	gdk_window_clear_area(gcontroller_gdkwindow(controller),
						  gmove->cur_rect.x, gmove->cur_rect.y,
						  gmove->cur_rect.width, gmove->cur_rect.height);
	gro_move_change_gstat(gmove, GRO_STAT_DEAD);
}

static void
turn_gmove(GroMove *gmove)
{
	const DefaultMoveEnv *move_env = gmove->move_env;
	gint cur_direct = gmove->cur_direct;/* not guint */
	int turn_base;

	turn_base = gmove->num_direct / 8;	/* heuristic */
    if (move_env->turn_left == TRUE) {
		cur_direct += RAND_INT(turn_base) + 1; /* between 1 and turn_base */
		if (cur_direct >= gmove->num_direct)
			cur_direct -= gmove->num_direct;
    } else {
		cur_direct -= RAND_INT(turn_base) + 1; /* between 1 and turn_base */
		if (cur_direct < 0)
			cur_direct += gmove->num_direct;
    }
	gro_move_change_direct(gmove, cur_direct);
}
