/* GStreamer Filter
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-fields
 *
 * <refsect2>
 * <para>
 * This filter provides some basic field operations.
 * Depending on how <link linkend="GstFields--operation">operation</link> is
 * set, this can range from simply dropping a field (crude de-interlacing),
 * flipping fields (if top field were not first),
 * or more involved de-interlacing of frames (by linear interpolation or
 * even linear blind, both output fields are then linear interpolations
 * of input fields).
 * </para>
 * <para>
 * If <link linkend="GstFields--dynamic-deinter">dynamic-deinter</link> is set,
 * it will only perform (linear or linear blend) de-interlacing if an incoming
 * frame is marked as interlaced by means of a preceding custom event
 * (containing an empty structure simply called 'detectinter',
 * e.g. <link linkend="GstDetectInter">detectinter</link>
 * can produce such marking).
 * </para>
 * <para>
 * Using the flip, shift (and combinations thereof) methods, the filter can shift,
 * reorder and generally rearrange independent fields of an interlaced video input.
 * As is (well-)known, input retrieved from  broadcast  (PAL,  NTSC,  etc)
 * video  sources generally comes in an interlaced form where each pass
 * from top to bottom of the screen displays every other scanline,
 * and then the next pass displays the lines between the lines from the first pass.
 * Each pass is known as a "field" (whence the name of the filter);
 * there are generally two fields per frame.
 * When  this form of video is captured and manipulated digitally, the two fields
 * of each frame are usually merged together into one flat (planar) image per frame.
 * This usually produces reasonable results, however there are conditions which
 * can cause this merging to  be  performed  incorrectly or less-than-optimally,
 * which is where this filter can help.
 * </para>
 * <para>
 * In shift mode, the video is shifted by one field (half a frame),
 * changing frame boundaries appropriately.  This is useful if a video capture
 * started grabbing video half a frame (one field) off from where frame
 * boundaries were actually intended to be.
 * </para>
 * <para>
 * In flip mode, the top field and bottom field of each frame is exchanged.
 * This can be useful if the video signal was sent "bottom field first"
 * (which can happen sometimes with PAL video sources) or other oddities
 * occurred which caused the frame boundaries to be at the right place,
 * but the scanlines to be swapped.
 * </para>
 * <para>
 * It is also possible to perform both operations, in which case shifting
 * before flipping is expected to be more commonly useful, though flipping
 * before shifting may be in order for some extremely odd material.
 * Note that the former basically comes down to keeping the top frame
 * while shifting (delaying) the bottom frame,
 * whereas the latter keeps the bottom frame and shifts the top frame.
 * </para>
 * <para>
 * As a technical note, the shift function (when only shifting)
 * might produce slight color discrepancies, as the subsampled nature of
 * YV12 (and alike) format does not contain enough information to do
 * field shifting cleanly.
 * </para>
 * <para>
 * The split mode separates a frame into its composing fields, yielding
 * a half-height stream at twice the framerate.  This can be useful
 * for subsequent per-field processing.  Conversely, merge mode merges
 * treats 2 consecutive frames as fields and yields a double-height stream
 * at half the framerate.  This would then be used to re-assemble a split
 * stream.
 * </para>
 * <para>
 * The mix mode tries to mimic actual display of an interlaced stream.
 * It produces a double framerate stream in which each frame is followed
 * by a frame composed of its own bottom field and the top field replaced
 * by the next frame's top field.
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * It is akin to (a combination of) transcode's 'core' de-interlacing operation,
 * fields filter, and doublefps filter [Thomas Oestreich, Tilmann Bitterberg, Alex Stewart].
 * Some of the documentation is also inspired by these components.
 * </listitem>
 * <listitem>
 * Similar filters also available in mplayer, e.g. phase filter, ..., avidemux, etc
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-entrans.h"

#include <string.h>


#define GST_TYPE_FIELDS \
  (gst_fields_get_type())
#define GST_FIELDS(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_FIELDS,GstFields))
#define GST_FIELDS_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_FIELDS,GstFieldsClass))
#define GST_IS_FIELDS(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_FIELDS))
#define GST_IS_FIELDS_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_FIELDS))


typedef struct _GstFields GstFields;
typedef struct _GstFieldsClass GstFieldsClass;

struct _GstFields
{
  GstVideoFilter videofilter;

  gint width, height;
  guint method;

  /* hold on to the previous buffer/frame */
  GstBuffer *buffer;

  /* tells splitting to take the top */
  gboolean do_top;

  /* dynamic deinterlacing driven by detection */
  gboolean dynamic_deinter;
  gboolean interlaced;

  /* the parent's chain function */
  GstPadChainFunction btrans_chain;
};


struct _GstFieldsClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (fields_debug);
#define GST_CAT_DEFAULT fields_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_METHOD,
  PROP_DYNAMIC_DEINTER
      /* FILL ME */
};

#define DEFAULT_METHOD           METHOD_DROP
#define DEFAULT_DYNAMIC_DEINTER  FALSE

static GstStaticPadTemplate gst_fields_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_fields_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static gboolean gst_fields_set_caps (GstBaseTransform * btrans,
    GstCaps * incaps, GstCaps * outcaps);
static GstCaps *gst_fields_transform_caps (GstBaseTransform * btrans,
    GstPadDirection direction, GstCaps * caps);
static gboolean gst_fields_get_unit_size (GstBaseTransform * btrans,
    GstCaps * caps, guint * size);
static GstFlowReturn gst_fields_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static GstFlowReturn gst_fields_chain (GstPad * pad, GstBuffer * buffer);
static gboolean gst_fields_event (GstBaseTransform * btrans, GstEvent * event);
static gboolean gst_fields_start (GstBaseTransform * btrans);
static gboolean gst_fields_stop (GstBaseTransform * btrans);

static void gst_fields_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_fields_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstFields, gst_fields, GstVideoFilter, GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE (GstFields, gst_fields);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_fields);

#define GST_TYPE_FIELDS_METHOD (gst_fields_method_get_type ())

typedef enum
{
  METHOD_FLIP,
  METHOD_SHIFT,
  METHOD_SHIFT_FLIP,
  METHOD_FLIP_SHIFT,
  METHOD_DROP,
  METHOD_LINEAR,
  METHOD_LINEAR_BLEND,
  METHOD_SPLIT,
  METHOD_MERGE,
  METHOD_MIX
} GstFieldsMethods;

static GType
gst_fields_method_get_type (void)
{
  static GType fields_method_type = 0;

  if (!fields_method_type) {
    static const GEnumValue fields_methods[] = {
      {METHOD_FLIP, "Flip fields", "flip"},
      {METHOD_SHIFT, "Shift fields", "shift"},
      {METHOD_SHIFT_FLIP, "Shift fields, then flip fields", "ship"},
      {METHOD_FLIP_SHIFT, "Flip fields, then shift fields", "flift"},
      {METHOD_DROP, "Drop (bottom) field", "drop"},
      {METHOD_LINEAR, "Keep top field, bottom field is average", "lin"},
      {METHOD_LINEAR_BLEND, "Both top and bottom are averages of input",
          "blend"},
      {METHOD_SPLIT, "Split fields, double framerate", "split"},
      {METHOD_MERGE, "Merge fields, halve framerate", "merge"},
      {METHOD_MIX, "Mix consecutive frames' fields, double framerate", "mix"},
      {0, NULL, NULL},
    };

    fields_method_type =
        g_enum_register_static ("GstFieldsMethods", fields_methods);
  }

  return fields_method_type;
}

static void
gst_fields_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Fields",
      "Filter/Effect/Video", "Field Manipulator",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_fields_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_fields_src_template));
}

static void
gst_fields_class_init (GstFieldsClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (fields_debug, "fields", 0, "fields");

  gobject_class->set_property = gst_fields_set_property;
  gobject_class->get_property = gst_fields_get_property;

  g_object_class_install_property (gobject_class, PROP_METHOD,
      g_param_spec_enum ("operation", "Operation", "Field operation",
          GST_TYPE_FIELDS_METHOD, DEFAULT_METHOD, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_DYNAMIC_DEINTER,
      g_param_spec_boolean ("dynamic-deinter", "Dynamic Deinterlace",
          "Perform deinterlacing only when requested.",
          DEFAULT_DYNAMIC_DEINTER, G_PARAM_READWRITE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_fields_set_caps);
  trans_class->transform_caps = GST_DEBUG_FUNCPTR (gst_fields_transform_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_fields_get_unit_size);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_fields_transform);
  trans_class->event = GST_DEBUG_FUNCPTR (gst_fields_event);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_fields_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_fields_stop);
}

static void
gst_fields_init (GstFields * filter, GstFieldsClass * g_class)
{
  GstBaseTransform *btrans = GST_BASE_TRANSFORM (filter);

  filter->method = DEFAULT_METHOD;
  filter->dynamic_deinter = DEFAULT_DYNAMIC_DEINTER;
  filter->buffer = NULL;

  /* as 'inheriting' element,
   * we override chain function on what is actually our own pad;
   * this allows dropping a buffer or holding on to it */
  if (btrans->sinkpad) {
    filter->btrans_chain = GST_PAD_CHAINFUNC (btrans->sinkpad);
    gst_pad_set_chain_function (btrans->sinkpad,
        GST_DEBUG_FUNCPTR (gst_fields_chain));
  } else
    GST_WARNING_OBJECT (filter, "GstBaseTransform provided no chain; "
        "no splitting or merging possible.");
}

static GstCaps *
gst_fields_transform_caps (GstBaseTransform * btrans,
    GstPadDirection direction, GstCaps * caps)
{
  GstFields *filter = GST_FIELDS (btrans);
  GstCaps *ret;
  gint i;

  ret = gst_caps_copy (caps);

  if (filter->method != METHOD_DROP && filter->method != METHOD_MERGE
      && filter->method != METHOD_SPLIT && filter->method != METHOD_MIX)
    return ret;

  for (i = 0; i < gst_caps_get_size (ret); ++i) {
    GstStructure *s = gst_caps_get_structure (ret, i);
    gint height;
    const GValue *fps;
    if (gst_structure_get_int (s, "height", &height)) {
      if (filter->method != METHOD_MIX) {
        if ((direction == GST_PAD_SINK && filter->method != METHOD_MERGE)
            || (direction == GST_PAD_SRC && filter->method == METHOD_MERGE))
          height /= 2;
        else
          height *= 2;
      }
      gst_structure_set (s, "height", G_TYPE_INT, height, NULL);
    }
    if (filter->method != METHOD_DROP
        && (fps = gst_structure_get_value (s, "framerate"))
        && GST_VALUE_HOLDS_FRACTION (fps)) {
      gint fps_n = gst_value_get_fraction_numerator (fps);
      gint fps_d = gst_value_get_fraction_denominator (fps);
      if ((direction == GST_PAD_SINK && filter->method == METHOD_MERGE)
          || (direction == GST_PAD_SRC
              && (filter->method == METHOD_SPLIT
                  || filter->method == METHOD_MIX)))
        fps_d *= 2;
      else
        fps_n *= 2;
      gst_structure_set (s, "framerate", GST_TYPE_FRACTION, fps_n, fps_d, NULL);
    }
  }

  return ret;
}

static inline void
gst_fields_copy (gpointer in, gpointer out, guint stride, guint height)
{
  guint increment;

  increment = stride << 1;
  height >>= 1;

  while (height--) {
    oil_memcpy (out, in, stride);
    out += increment;
    in += increment;
  }
}

static gboolean
gst_fields_event (GstBaseTransform * btrans, GstEvent * event)
{
  GstFields *filter;

  filter = GST_FIELDS (btrans);

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_CUSTOM_DOWNSTREAM:
      if (gst_structure_has_name (gst_event_get_structure (event),
              "detectinter"))
        filter->interlaced = TRUE;
      return TRUE;
    default:
      break;
  }

  return GST_BASE_TRANSFORM_CLASS (parent_class)->event (btrans, event);
}

static void
gst_fields_drop (gpointer in, gpointer out, guint stride, guint height)
{
  guint skip;

  skip = stride << 1;
  height >>= 1;

  /* interpolate where needed */
  while (height--) {
    oil_memcpy (out, in, stride);

    in += skip;
    out += stride;
  }
}


static void
gst_fields_undrop (gpointer in, gpointer out, guint stride, guint height)
{
  guint skip;

  skip = stride << 1;

  /* interpolate where needed */
  while (height--) {
    oil_memcpy (out, in, stride);

    in += stride;
    out += skip;
  }
}


static inline void
average (guint8 * in1, guint8 * in2, guint8 * out, guint size)
{
  while (size) {
    *out = ((*in1 + *in2) >> 1);
    ++in1;
    ++in2;
    ++out;
    --size;
  }
}

static void
gst_fields_linear (gpointer in, gpointer out, guint stride, guint height)
{
  guint y;
  guint skip;

  /* first copy all */
  oil_memcpy (out, in, stride * height);

  /* shift to next line for destination */
  out += stride;
  skip = stride << 1;

  /* interpolate where needed */
  for (y = 0; y < height / 2 - 1; ++y) {
    average (in, in + skip, out, stride);

    in += skip;
    out += skip;
  }

  /* last row is made clone of last-but-one */
  oil_memcpy (out, in, stride);
}

static void
gst_fields_linear_blend (gpointer in, gpointer out, guint stride, guint height)
{
  guint y, skip;
  gpointer src, dest;

  /* first copy all */
  oil_memcpy (out, in, stride * height);

  /* first field to full frame in the input buffer by interpolation */

  /* shift to next line for destination */
  src = in;
  dest = in + stride;
  skip = stride << 1;

  /* interpolate where needed */
  for (y = 0; y < height / 2 - 1; ++y) {
    average (src, src + skip, dest, stride);

    src += skip;
    dest += skip;
  }

  /* second field to full frame in the output buffer by interpolation */

  /* shift to next line for destination */
  src = out + stride;
  dest = out;
  skip = stride << 1;

  /* interpolate where needed */
  for (y = 0; y < height / 2 - 1; ++y) {
    average (src, src + skip, dest, stride);

    src += skip;
    dest += skip;
  }

  /* now merge both frames */
  average (in, out, out, stride * height);
}

static GstFlowReturn
gst_fields_transform (GstBaseTransform * btrans, GstBuffer * in,
    GstBuffer * out)
{
  GstFields *filter;
  guint8 *src, *dest, *buf;
  GstFlowReturn ret = GST_FLOW_OK;

  filter = GST_FIELDS (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);
  dest = (guint8 *) GST_BUFFER_DATA (out);

  guint width = filter->width;
  guint height = filter->height;

  guint stride = GST_VIDEO_I420_Y_ROWSTRIDE (width);

  switch (filter->method) {
    case METHOD_FLIP:
      /* copying all already takes care of chroma */
      oil_memcpy (dest, src, GST_BUFFER_SIZE (in));
      /* only fields in luma plane are flipped */
      /* top field to bottom field */
      gst_fields_copy (src, dest + stride, stride, height);
      /* bottom field to top field */
      gst_fields_copy (src + stride, dest, stride, height);
      break;
    case METHOD_SHIFT:
      /* initialize buffer with current at "bootstrap" */
      if (!filter->buffer) {
        filter->buffer = gst_buffer_ref (in);
      }
      buf = GST_BUFFER_DATA (filter->buffer);
      /* copying all already takes care of chroma */
      oil_memcpy (dest, src, GST_BUFFER_SIZE (in));
      /* the bottom field of the previous frame goes in top of current */
      gst_fields_copy (buf + stride, dest, stride, height);
      /* the top field of the current frame goes to bottom */
      gst_fields_copy (src, dest + stride, stride, height);
      /* release previous frame, put current in place */
      gst_buffer_unref (filter->buffer);
      filter->buffer = gst_buffer_ref (in);
      break;
    case METHOD_SHIFT_FLIP:
      /* this means the bottom field is delayed to the next frame */
      if (!filter->buffer) {
        filter->buffer = gst_buffer_ref (in);
      }
      buf = GST_BUFFER_DATA (filter->buffer);
      /* copying all already takes care of chroma */
      oil_memcpy (dest, src, GST_BUFFER_SIZE (in));
      /* the top field of the current frame just stays on top */
      gst_fields_copy (src, dest, stride, height);
      /* the bottom field of the previous frame goes in bottom of current */
      gst_fields_copy (buf + stride, dest + stride, stride, height);
      /* release previous frame, put current in place */
      gst_buffer_unref (filter->buffer);
      filter->buffer = gst_buffer_ref (in);
      break;
    case METHOD_FLIP_SHIFT:
      /* this means the top field is delayed to the next frame */
      if (!filter->buffer) {
        filter->buffer = gst_buffer_ref (in);
      }
      buf = GST_BUFFER_DATA (filter->buffer);
      /* copying all already takes care of chroma */
      oil_memcpy (dest, src, GST_BUFFER_SIZE (in));
      /* the top field of the previous frame goes in top of current */
      gst_fields_copy (buf, dest, stride, height);
      /* the bottom field of the current frame stays in bottom field */
      gst_fields_copy (src + stride, dest + stride, stride, height);
      /* release previous frame, put current in place */
      gst_buffer_unref (filter->buffer);
      filter->buffer = gst_buffer_ref (in);
      // TODO ? consider moving the chroma around, possibly sampled from top field */
      break;
    case METHOD_DROP:
      /* keep top field in luma */
      gst_fields_drop (src, dest, stride, height);
      /* keep top field in chroma */
      gst_fields_drop (src + GST_VIDEO_I420_U_OFFSET (width, height),
          dest + GST_VIDEO_I420_U_OFFSET (width, height / 2),
          GST_VIDEO_I420_U_ROWSTRIDE (width), height / 2);
      gst_fields_drop (src + GST_VIDEO_I420_V_OFFSET (width, height),
          dest + GST_VIDEO_I420_V_OFFSET (width, height / 2),
          GST_VIDEO_I420_V_ROWSTRIDE (width), height / 2);
      break;
    case METHOD_LINEAR:
      /* copying all already takes care of chroma */
      oil_memcpy (dest, src, GST_BUFFER_SIZE (in));
      if (filter->dynamic_deinter && !filter->interlaced)
        break;
      /* only luma comp need be averaged */
      gst_fields_linear (src, dest, width, height);
      break;
    case METHOD_LINEAR_BLEND:
      if (filter->dynamic_deinter && !filter->interlaced) {
        oil_memcpy (dest, src, GST_BUFFER_SIZE (in));
        break;
      }
      /* first blend luma */
      gst_fields_linear_blend (src, dest,
          GST_VIDEO_I420_Y_ROWSTRIDE (width), height);
      /* then on the chroma planes */
      gst_fields_linear_blend (src + GST_VIDEO_I420_U_OFFSET (width, height),
          dest + GST_VIDEO_I420_U_OFFSET (width, height),
          GST_VIDEO_I420_U_ROWSTRIDE (width), height / 2);
      gst_fields_linear_blend (src + GST_VIDEO_I420_V_OFFSET (width, height),
          dest + GST_VIDEO_I420_V_OFFSET (width, height),
          GST_VIDEO_I420_V_ROWSTRIDE (width), height / 2);
      break;
    case METHOD_SPLIT:
      if (filter->do_top) {
        /* output top field in luma */
        gst_fields_drop (src, dest, stride, height);
        /* output top field in chroma */
        gst_fields_drop (src + GST_VIDEO_I420_U_OFFSET (width, height),
            dest + GST_VIDEO_I420_U_OFFSET (width, height / 2),
            GST_VIDEO_I420_U_ROWSTRIDE (width), height / 2);
        gst_fields_drop (src + GST_VIDEO_I420_V_OFFSET (width, height),
            dest + GST_VIDEO_I420_V_OFFSET (width, height / 2),
            GST_VIDEO_I420_V_ROWSTRIDE (width), height / 2);

        GST_BUFFER_DURATION (out) = GST_BUFFER_DURATION (in) / 2;
      } else {
        /* output bottom field in luma */
        gst_fields_drop (src + stride, dest, stride, height);
        /* output bottom field in chroma */
        gst_fields_drop (src + GST_VIDEO_I420_U_OFFSET (width, height)
            + GST_VIDEO_I420_U_ROWSTRIDE (width),
            dest + GST_VIDEO_I420_U_OFFSET (width, height / 2),
            GST_VIDEO_I420_U_ROWSTRIDE (width), height / 2);
        gst_fields_drop (src + GST_VIDEO_I420_V_OFFSET (width, height)
            + GST_VIDEO_I420_V_ROWSTRIDE (width),
            dest + GST_VIDEO_I420_V_OFFSET (width, height / 2),
            GST_VIDEO_I420_V_ROWSTRIDE (width), height / 2);

        GST_BUFFER_TIMESTAMP (out)
            = GST_BUFFER_TIMESTAMP (in) + GST_BUFFER_DURATION (in) / 2;
        GST_BUFFER_DURATION (out) = GST_BUFFER_DURATION (in) / 2;
      }
      break;
    case METHOD_MERGE:
      /* buffer contains the top field put there by chain */
      buf = GST_BUFFER_DATA (filter->buffer);
      /* put in output for luma */
      gst_fields_undrop (buf, dest, stride, height);
      /* output for chroma */
      gst_fields_undrop (buf + GST_VIDEO_I420_U_OFFSET (width, height),
          dest + GST_VIDEO_I420_U_OFFSET (width, height * 2),
          GST_VIDEO_I420_U_ROWSTRIDE (width), height / 2);
      gst_fields_undrop (src + GST_VIDEO_I420_V_OFFSET (width, height),
          dest + GST_VIDEO_I420_V_OFFSET (width, height * 2),
          GST_VIDEO_I420_V_ROWSTRIDE (width), height / 2);
      /* now put the current buffer as bottom field in output */
      /* for luma */
      gst_fields_undrop (src, dest + stride, stride, height);
      /* output for chroma */
      gst_fields_undrop (src + GST_VIDEO_I420_U_OFFSET (width, height),
          dest + GST_VIDEO_I420_U_OFFSET (width, height * 2)
          + GST_VIDEO_I420_V_ROWSTRIDE (width),
          GST_VIDEO_I420_U_ROWSTRIDE (width), height / 2);
      gst_fields_undrop (src + GST_VIDEO_I420_V_OFFSET (width, height),
          dest + GST_VIDEO_I420_V_OFFSET (width, height * 2)
          + GST_VIDEO_I420_V_ROWSTRIDE (width),
          GST_VIDEO_I420_V_ROWSTRIDE (width), height / 2);

      GST_BUFFER_TIMESTAMP (out)
          = GST_BUFFER_TIMESTAMP (filter->buffer);
      GST_BUFFER_DURATION (out)
          = GST_BUFFER_DURATION (filter->buffer) + GST_BUFFER_DURATION (in);
      break;
    case METHOD_MIX:
      if (G_UNLIKELY (!filter->buffer)) {
        filter->buffer = gst_buffer_ref (in);
      }
      if (filter->do_top) {
        /* output top field in luma */
        gst_fields_copy (src, dest, stride, height);
        /* output top field in chroma */
        gst_fields_copy (src + GST_VIDEO_I420_U_OFFSET (width, height),
            dest + GST_VIDEO_I420_U_OFFSET (width, height),
            GST_VIDEO_I420_U_ROWSTRIDE (width), height / 2);
        gst_fields_copy (src + GST_VIDEO_I420_V_OFFSET (width, height),
            dest + GST_VIDEO_I420_V_OFFSET (width, height),
            GST_VIDEO_I420_V_ROWSTRIDE (width), height / 2);
        /* now put the saved buffer as bottom field in output */
        /* for luma */
        buf = GST_BUFFER_DATA (filter->buffer);
        gst_fields_copy (buf + stride, dest + stride, stride, height);
        /* output for chroma */
        gst_fields_copy (buf + GST_VIDEO_I420_U_OFFSET (width, height)
            + GST_VIDEO_I420_U_ROWSTRIDE (width),
            dest + GST_VIDEO_I420_U_OFFSET (width, height)
            + GST_VIDEO_I420_V_ROWSTRIDE (width),
            GST_VIDEO_I420_U_ROWSTRIDE (width), height / 2);
        gst_fields_copy (buf + GST_VIDEO_I420_V_OFFSET (width, height)
            + GST_VIDEO_I420_V_ROWSTRIDE (width),
            dest + GST_VIDEO_I420_V_OFFSET (width, height)
            + GST_VIDEO_I420_V_ROWSTRIDE (width),
            GST_VIDEO_I420_V_ROWSTRIDE (width), height / 2);

        GST_BUFFER_DURATION (out) = GST_BUFFER_DURATION (in) / 2;
        filter->do_top = FALSE;

        /* release previous frame, put current in place */
        gst_buffer_unref (filter->buffer);
        filter->buffer = gst_buffer_ref (in);
      } else {
        /* output the whole frame */
        oil_memcpy (dest, src, GST_BUFFER_SIZE (in));

        GST_BUFFER_TIMESTAMP (out)
            = GST_BUFFER_TIMESTAMP (in) + GST_BUFFER_DURATION (in) / 2;
        GST_BUFFER_DURATION (out) = GST_BUFFER_DURATION (in) / 2;
      }
      break;
  }

  /* whatever the method, we forget about this */
  filter->interlaced = FALSE;

  return ret;
}


static GstFlowReturn
gst_fields_chain (GstPad * pad, GstBuffer * buffer)
{
  GstFields *filter = GST_FIELDS (GST_PAD_PARENT (pad));
  GstFlowReturn ret;

  switch (filter->method) {
      /* note: do_top should only be changed here */
    case METHOD_SPLIT:
    case METHOD_MIX:
      /* let base transform go and treat first stage */
      /* take ref to prevent buffer going into oblivion */
      gst_buffer_ref (buffer);
      ret = filter->btrans_chain (pad, buffer);
      if (G_UNLIKELY (ret != GST_FLOW_OK)) {
        gst_buffer_unref (buffer);
        return ret;
      }
      /* and now the next stage */
      filter->do_top = FALSE;
      /* this will drop the extra ref because it is not in place transform */
      ret = filter->btrans_chain (pad, buffer);
      filter->do_top = TRUE;
      return ret;
      break;
    case METHOD_MERGE:
      if (!filter->buffer) {
        filter->buffer = buffer;
        return GST_FLOW_OK;
      } else {
        /* ref of current buffer will be dropped by base transform */
        ret = filter->btrans_chain (pad, buffer);
        /* now loose the stored buffer */
        gst_buffer_unref (filter->buffer);
        filter->buffer = NULL;
        return ret;
      }
      break;
    default:
      /* hand over to GstBaseTransform if nothing special */
      return filter->btrans_chain (pad, buffer);
      break;
  }
}

static gboolean
gst_fields_start (GstBaseTransform * btrans)
{
  GstFields *filter = GST_FIELDS (btrans);

  filter->buffer = NULL;
  filter->do_top = TRUE;
  filter->interlaced = FALSE;

  return TRUE;
}

static gboolean
gst_fields_stop (GstBaseTransform * btrans)
{
  GstFields *filter = GST_FIELDS (btrans);

  if (filter->buffer) {
    gst_buffer_unref (filter->buffer);
    filter->buffer = NULL;
  }

  return TRUE;
}

static void
gst_fields_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstFields *src;

  g_return_if_fail (GST_IS_FIELDS (object));
  src = GST_FIELDS (object);

  switch (prop_id) {
    case PROP_METHOD:
      src->method = g_value_get_enum (value);
      break;
    case PROP_DYNAMIC_DEINTER:
      src->dynamic_deinter = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_fields_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstFields *src;

  g_return_if_fail (GST_IS_FIELDS (object));
  src = GST_FIELDS (object);

  switch (prop_id) {
    case PROP_METHOD:
      g_value_set_enum (value, src->method);
      break;
    case PROP_DYNAMIC_DEINTER:
      g_value_set_boolean (value, src->dynamic_deinter);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
