/* GStreamer Element
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * mplayer filter:
 * Copyright (C) 2002 Rémi Guyomarch <rguyom@pobox.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-unsharp
 *
 * <refsect2>
 * <para>
 * This filter blurs or sharpens an image depending on the sign of
 * <link linkend="GstUnsharp--amount">amount</link>. You can either
 * set amount for both luma and chroma or you can set it individually
 * (recommended), which will be the case if
 * <link linkend="GstUnsharp--chroma-amount">chroma-amount</link>
 * provides a valid (amount) value as override.
 * A positive value for amount will sharpen the image, a negative
 * value will blur it. A sane range for amount is -1.5 to 1.5.
 * </para>
 * <para>
 * The (square) <link linkend="GstUnsharp--chroma-matrix">matrix</link>
 * (and possible override
 * <link linkend="GstUnsharp--chroma-matrix">chroma-matrix</link>) sizes
 * must be odd and define the range/strength of the effect.
 * Sensible ranges are 3x3 to 7x7.
 * </para>
 * <para>
 * It sometimes  makes  sense  to  sharpen  the sharpen the luma
 * and to blur the chroma, for example:
 * <programlisting>
 * unsharp luma=0.8 matrix=5 chroma_amount=-0.2 chroma_matrix=3
 * </programlisting>
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * mplayer unsharp filter [Rémi Guyomarch]
 * </listitem>
 * <listitem>
 * Also available in transcode (unsharp filter)
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-mencoder.h"

#include <string.h>


#define GST_TYPE_UNSHARP \
  (gst_unsharp_get_type())
#define GST_UNSHARP(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_UNSHARP,GstUnsharp))
#define GST_UNSHARP_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_UNSHARP,GstUnsharpClass))
#define GST_IS_UNSHARP(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_UNSHARP))
#define GST_IS_UNSHARP_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_UNSHARP))


#define MIN_AMOUNT      -2
#define MAX_AMOUNT      -MIN_AMOUNT
#define MIN_MATRIX      3
#define MAX_MATRIX      63
#define DEFAULT_AMOUNT  1.0
#define DEFAULT_MATRIX   3

/* legacy structure */
typedef struct FilterParam
{
  int msizeX, msizeY;
  double amount;
  guint32 *SC[MAX_MATRIX - 1];
} FilterParam;


typedef struct _GstUnsharp GstUnsharp;
typedef struct _GstUnsharpClass GstUnsharpClass;

struct _GstUnsharp
{
  GstVideoFilter videofilter;

  gint width, height;

  gdouble amount, chroma_amount;
  guint matrix, chroma_matrix;

  FilterParam fp, chroma_fp;
};

/* chroma values only provide override if they are valid */
#define UNSHARP_GET_CHROMA_AMOUNT(unsharp)   \
  unsharp->chroma_amount >= MIN_AMOUNT ? unsharp->chroma_amount : unsharp->amount
#define UNSHARP_GET_CHROMA_MATRIX(unsharp)   \
  unsharp->chroma_amount >= MIN_MATRIX ? unsharp->chroma_matrix : unsharp->matrix


struct _GstUnsharpClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (unsharp_debug);
#define GST_CAT_DEFAULT unsharp_debug

/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_AMOUNT,
  PROP_MATRIX,
  PROP_CHROMA_AMOUNT,
  PROP_CHROMA_MATRIX
      /* FILL ME */
};

static GstStaticPadTemplate gst_unsharp_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_unsharp_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static gboolean gst_unsharp_hook_caps (GstUnsharp * filter,
    GstCaps * incaps, GstCaps * outcaps);
static GstFlowReturn gst_unsharp_transform (GstBaseTransform * btrans,
    GstBuffer * in, GstBuffer * out);
static gboolean gst_unsharp_start (GstBaseTransform * btrans);
static gboolean gst_unsharp_stop (GstBaseTransform * btrans);

static void gst_unsharp_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_unsharp_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstUnsharp, gst_unsharp, GstVideoFilter,
    GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE_FULL (GstUnsharp, gst_unsharp,
    gst_unsharp_hook_caps);

GST_VIDEO_FILTER_GET_UNIT_SIZE_BOILERPLATE (gst_unsharp);

static void
gst_unsharp_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Unsharp",
      "Filter/Effect/Video", "(Un)Sharpen using Gaussian blur",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Rémi Guyomarch");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_unsharp_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_unsharp_src_template));
}

static void
gst_unsharp_class_init (GstUnsharpClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (unsharp_debug, "unsharp", 0, "unsharp");

  gobject_class->set_property = gst_unsharp_set_property;
  gobject_class->get_property = gst_unsharp_get_property;

  // TODO properties for non-square matrices
  g_object_class_install_property (gobject_class, PROP_AMOUNT,
      g_param_spec_double ("amount", "Amount",
          "[luma and chroma] (Un)sharpness amount",
          MIN_AMOUNT, MAX_AMOUNT, DEFAULT_AMOUNT, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_MATRIX,
      g_param_spec_uint ("matrix", "Matrix Size",
          "[luma and chroma] Search Matrix Size",
          MIN_MATRIX, MAX_MATRIX, DEFAULT_MATRIX, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_CHROMA_AMOUNT,
      g_param_spec_double ("chroma-amount", "Chroma Amount",
          "[chroma override] (Un)sharpness amount",
          2 * MIN_AMOUNT, MAX_AMOUNT, DEFAULT_AMOUNT, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_CHROMA_MATRIX,
      g_param_spec_uint ("chroma-matrix", "Chroma Matrix Size",
          "[chroma override] Search Matrix Size",
          0, MAX_MATRIX, DEFAULT_MATRIX, G_PARAM_READWRITE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_unsharp_set_caps);
  trans_class->get_unit_size = GST_DEBUG_FUNCPTR (gst_unsharp_get_unit_size);
  trans_class->transform = GST_DEBUG_FUNCPTR (gst_unsharp_transform);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_unsharp_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_unsharp_stop);
}

static void
gst_unsharp_init (GstUnsharp * filter, GstUnsharpClass * g_class)
{
  filter->amount = DEFAULT_AMOUNT;
  filter->matrix = DEFAULT_MATRIX;

  /* chroma are out-of-range by default to avoid override */
  filter->chroma_amount = 2 * MIN_AMOUNT;
  filter->chroma_matrix = 0;
}

static void
gst_unsharp_free (GstUnsharp * filter, FilterParam * fp)
{
  guint z;

  /* either NULL, or an allocated chunk */
  for (z = 0; z < MAX_MATRIX - 1; z++) {
    g_free (fp->SC[z]);
    fp->SC[z] = NULL;
  }
}

static void
gst_unsharp_configure_and_allocate (GstUnsharp * filter, FilterParam * fp,
    gdouble amount, guint mx, guint my)
{
  guint sx = mx / 2;
  guint sy = my / 2;
  guint z;

  fp->amount = amount;

  fp->msizeX = mx;
  fp->msizeY = my;

  gst_unsharp_free (filter, fp);
  memset (fp->SC, 0, sizeof (fp->SC));
  for (z = 0; z < 2 * sy; z++) {
    fp->SC[z] = g_malloc (sizeof (*(fp->SC[z])) * (filter->width + 2 * sx));
  }
}

static gboolean
gst_unsharp_hook_caps (GstUnsharp * filter, GstCaps * incaps, GstCaps * outcaps)
{
  GstStructure *structure;

  structure = gst_caps_get_structure (incaps, 0);

  /* allocate for luma */
  gst_unsharp_configure_and_allocate (filter, &filter->fp, filter->amount,
      filter->matrix, filter->matrix);
  /* allocate for chroma */
  gst_unsharp_configure_and_allocate (filter, &filter->chroma_fp,
      UNSHARP_GET_CHROMA_AMOUNT (filter), UNSHARP_GET_CHROMA_MATRIX (filter),
      UNSHARP_GET_CHROMA_MATRIX (filter));

  return TRUE;
}


/* ====

This code is based on :

An Efficient algorithm for Gaussian blur using finite-state machines
Frederick M. Waltz and John W. V. Miller

SPIE Conf. on Machine Vision Systems for Inspection and Metrology VII
Originally published Boston, Nov 98

==== */

static void
gst_unsharp (guint8 * dst, guint8 * src,
    int dstStride, int srcStride, int width, int height, FilterParam * fp)
{

  guint32 **SC = fp->SC;
  guint32 SR[MAX_MATRIX - 1], Tmp1, Tmp2;
  guint8 *src2 = src;

  gint32 res;
  int x, y, z;
  int amount = fp->amount * 65536.0;
  int stepsX = fp->msizeX / 2;
  int stepsY = fp->msizeY / 2;
  int scalebits = (stepsX + stepsY) * 2;
  int32_t halfscale = 1 << ((stepsX + stepsY) * 2 - 1);

  if (!fp->amount) {
    if (src == dst)
      return;
    if (dstStride == srcStride)
      memcpy (dst, src, srcStride * height);
    else
      for (y = 0; y < height; y++, dst += dstStride, src += srcStride)
        memcpy (dst, src, width);
    return;
  }

  for (y = 0; y < 2 * stepsY; y++)
    memset (SC[y], 0, sizeof (SC[y][0]) * (width + 2 * stepsX));

  for (y = -stepsY; y < height + stepsY; y++) {
    if (y < height)
      src2 = src;
    memset (SR, 0, sizeof (SR[0]) * (2 * stepsX - 1));
    for (x = -stepsX; x < width + stepsX; x++) {
      Tmp1 = x <= 0 ? src2[0] : (x >= width ? src2[width - 1] : src2[x]);
      for (z = 0; z < stepsX * 2; z += 2) {
        Tmp2 = SR[z + 0] + Tmp1;
        SR[z + 0] = Tmp1;
        Tmp1 = SR[z + 1] + Tmp2;
        SR[z + 1] = Tmp2;
      }
      for (z = 0; z < stepsY * 2; z += 2) {
        Tmp2 = SC[z + 0][x + stepsX] + Tmp1;
        SC[z + 0][x + stepsX] = Tmp1;
        Tmp1 = SC[z + 1][x + stepsX] + Tmp2;
        SC[z + 1][x + stepsX] = Tmp2;
      }
      if (x >= stepsX && y >= stepsY) {
        guint8 *srx = src - stepsY * srcStride + x - stepsX;
        guint8 *dsx = dst - stepsY * dstStride + x - stepsX;

        res = (gint32) (*srx)
            + ((((gint32) (*srx) - (gint32) ((Tmp1 +
                            halfscale) >> scalebits)) * amount) >> 16);
        *dsx = res > 255 ? 255 : res < 0 ? 0 : (guint8) res;
      }
    }
    if (y >= 0) {
      dst += dstStride;
      src += srcStride;
    }
  }
}

static GstFlowReturn
gst_unsharp_transform (GstBaseTransform * btrans, GstBuffer * in,
    GstBuffer * out)
{
  GstUnsharp *filter = GST_UNSHARP (btrans);

  guint8 *src = (guint8 *) GST_BUFFER_DATA (in);
  guint8 *dest = (guint8 *) GST_BUFFER_DATA (out);

  guint width = filter->width;
  guint height = filter->height;

  guint stride, offset;

  /* process luma */
  stride = GST_VIDEO_I420_Y_ROWSTRIDE (width);
  gst_unsharp (dest, src, stride, stride, width, height, &filter->fp);

  /* process chroma */
  stride = GST_VIDEO_I420_U_ROWSTRIDE (width);
  offset = GST_VIDEO_I420_U_OFFSET (width, height);
  gst_unsharp (dest + offset, src + offset, stride, stride,
      width / 2, height / 2, &filter->fp);

  stride = GST_VIDEO_I420_V_ROWSTRIDE (width);
  offset = GST_VIDEO_I420_V_OFFSET (width, height);
  gst_unsharp (dest + offset, src + offset, stride, stride,
      width / 2, height / 2, &filter->fp);

  return GST_FLOW_OK;
}


static gboolean
gst_unsharp_start (GstBaseTransform * btrans)
{
  return TRUE;
}

static gboolean
gst_unsharp_stop (GstBaseTransform * btrans)
{
  GstUnsharp *filter = GST_UNSHARP (btrans);

  gst_unsharp_free (filter, &filter->fp);
  gst_unsharp_free (filter, &filter->chroma_fp);

  return TRUE;
}

static void
gst_unsharp_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstUnsharp *src;

  g_return_if_fail (GST_IS_UNSHARP (object));
  src = GST_UNSHARP (object);

  switch (prop_id) {
    case PROP_AMOUNT:
      src->amount = g_value_get_double (value);
      break;
    case PROP_MATRIX:
      /* must be odd */
      src->matrix = g_value_get_uint (value) | 1U;
      break;
    case PROP_CHROMA_AMOUNT:
      src->chroma_amount = g_value_get_double (value);
      break;
    case PROP_CHROMA_MATRIX:
      /* must be odd */
      src->chroma_matrix = g_value_get_uint (value) | 1U;
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_unsharp_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstUnsharp *src;

  g_return_if_fail (GST_IS_UNSHARP (object));
  src = GST_UNSHARP (object);

  switch (prop_id) {
    case PROP_AMOUNT:
      g_value_set_double (value, src->amount);
      break;
    case PROP_MATRIX:
      g_value_set_uint (value, src->matrix);
      break;
    case PROP_CHROMA_AMOUNT:
      g_value_set_double (value, src->chroma_amount);
      break;
    case PROP_CHROMA_MATRIX:
      g_value_set_uint (value, src->chroma_matrix);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
