/* GStreamer Filter
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * transcode filter:
 * Copyright (C) Thomas Oestreich, Chad Page - February/March 2002
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-cshift
 *
 * <refsect2>
 * <para>
 * Shift the chroma (color) component in YUV/I420 image data
 * to the left by the amount specified by
 * <link linkend="GstCshift--shift">shift</link>.
 * The amount is in real-image size dimension (not chroma planes stride).
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * <para>
 * Transcode cshift filter [Thomas Oestreich, Chad Page]
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <string.h>


#define GST_TYPE_CSHIFT \
  (gst_cshift_get_type())
#define GST_CSHIFT(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_CSHIFT,GstCshift))
#define GST_CSHIFT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_CSHIFT,GstCshiftClass))
#define GST_IS_CSHIFT(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_CSHIFT))
#define GST_IS_CSHIFT_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_CSHIFT))


typedef struct _GstCshift GstCshift;
typedef struct _GstCshiftClass GstCshiftClass;

struct _GstCshift
{
  GstVideoFilter videofilter;

  gint width, height;

  guint shift;
};


struct _GstCshiftClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (cshift_debug);
#define GST_CAT_DEFAULT cshift_debug


/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_SHIFT
      /* FILL ME */
};

#define DEFAULT_SHIFT      0

static GstStaticPadTemplate gst_cshift_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_cshift_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstFlowReturn gst_cshift_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);
static gboolean gst_cshift_start (GstBaseTransform * btrans);
static gboolean gst_cshift_stop (GstBaseTransform * btrans);

static void gst_cshift_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_cshift_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstCshift, gst_cshift, GstVideoFilter, GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE (GstCshift, gst_cshift);

static void
gst_cshift_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Cshift",
      "Filter/Effect/Video", "Chroma-lag shifter",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n"
      "Thomas Oestreich, Chad Page");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_cshift_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_cshift_src_template));
}

static void
gst_cshift_class_init (GstCshiftClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (cshift_debug, "cshift", 0, "cshift");

  gobject_class->set_property = gst_cshift_set_property;
  gobject_class->get_property = gst_cshift_get_property;

  g_object_class_install_property (gobject_class, PROP_SHIFT,
      g_param_spec_uint ("shift", "Shift",
          "Shift chroma (color) to the left",
          0, G_MAXUINT, DEFAULT_SHIFT,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_cshift_set_caps);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_cshift_transform_ip);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_cshift_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_cshift_stop);
}

static void
gst_cshift_init (GstCshift * filter, GstCshiftClass * g_class)
{
  filter->shift = DEFAULT_SHIFT;
}

static GstFlowReturn
gst_cshift_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  GstCshift *filter;
  guint8 *src, *cubuf, *cvbuf, *cuaddr, *cvaddr;
  guint width, height, stride, shift;
  guint x, y;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_CSHIFT (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);

  width = filter->width;
  height = filter->height;
  /* shift property is with respect to real image dimension */
  shift = MIN (filter->shift, width) >> 1;

  if (G_UNLIKELY (!shift))
    return GST_FLOW_OK;

  stride = GST_VIDEO_I420_U_ROWSTRIDE (width);

  cubuf = src + GST_VIDEO_I420_U_OFFSET (width, height);
  cvbuf = src + GST_VIDEO_I420_V_OFFSET (width, height);

  for (y = 0; y < height / 2; y++) {
    cuaddr = cubuf + y * stride;
    cvaddr = cvbuf + y * stride;
    for (x = 0; x < (width / 2) - shift; x++) {
      *cuaddr = cuaddr[shift];
      *cvaddr = cvaddr[shift];
      cuaddr++;
      cvaddr++;
    }
  }

  return GST_FLOW_OK;
}


static gboolean
gst_cshift_start (GstBaseTransform * btrans)
{
  return TRUE;
}

static gboolean
gst_cshift_stop (GstBaseTransform * btrans)
{
  return TRUE;
}

static void
gst_cshift_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstCshift *src;

  g_return_if_fail (GST_IS_CSHIFT (object));
  src = GST_CSHIFT (object);

  switch (prop_id) {
    case PROP_SHIFT:
      src->shift = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_cshift_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstCshift *src;

  g_return_if_fail (GST_IS_CSHIFT (object));
  src = GST_CSHIFT (object);

  switch (prop_id) {
    case PROP_SHIFT:
      g_value_set_uint (value, src->shift);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
