/* GStreamer Filter
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * aviindex fragment
 * Copyright (C) Tilmann Bitterberg - June 2003
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1307  USA
 */

/**
 * SECTION:element-divxkey
 *
 * <refsect2>
 * <para>
 * Checks for divx/xvid/mpeg4 keyframes and marks such buffers accordingly
 * (non-delta-unit), and others as not keyframe (delta-unit).
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * <para>
 * keyframe recognition in aviindex tool [Tilmann Bitterberg]
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <string.h>


#define GST_TYPE_DIVX_KEY \
  (gst_divx_key_get_type())
#define GST_DIVX_KEY(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_DIVX_KEY,GstDivxKey))
#define GST_DIVX_KEY_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_DIVX_KEY,GstDivxKeyClass))
#define GST_IS_DIVX_KEY(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_DIVX_KEY))
#define GST_IS_DIVX_KEY_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_DIVX_KEY))


typedef struct _GstDivxKey GstDivxKey;
typedef struct _GstDivxKeyClass GstDivxKeyClass;

struct _GstDivxKey
{
  GstBaseTransform parent;

  /* caps info */
  gboolean old;
};


struct _GstDivxKeyClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (divx_key_debug);
#define GST_CAT_DEFAULT divx_key_debug


/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_DELAY
      /* FILL ME */
};

static GstStaticPadTemplate gst_divx_key_sink_template =
    GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-divx, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], "
        "framerate = (fraction) [ 0, MAX ], "
        "divxversion = (int) [ 3, 5 ]; "
        "video/x-xvid, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], "
        "framerate = (fraction) [ 0, MAX ]; "
        "video/x-msmpeg, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], "
        "framerate = (fraction) [ 0, MAX ], "
        "msmpegversion = (int) 41; "
        "video/mpeg, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], "
        "framerate = (fraction) [ 0, MAX ], "
        "mpegversion = (int) 4, " "systemstream = (boolean) FALSE ")
    );

static GstStaticPadTemplate gst_divx_key_src_template =
    GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("video/x-divx, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], "
        "framerate = (fraction) [ 0, MAX ], "
        "divxversion = (int) [ 3, 5 ]; "
        "video/x-xvid, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], "
        "framerate = (fraction) [ 0, MAX ]; "
        "video/x-msmpeg, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], "
        "framerate = (fraction) [ 0, MAX ], "
        "msmpegversion = (int) 41; "
        "video/mpeg, "
        "width = (int) [ 16, 4096 ], "
        "height = (int) [ 16, 4096 ], "
        "framerate = (fraction) [ 0, MAX ], "
        "mpegversion = (int) 4, " "systemstream = (boolean) FALSE ")
    );


static gboolean gst_divx_key_set_caps (GstBaseTransform * btrans,
    GstCaps * incaps, GstCaps * outcaps);
static GstFlowReturn gst_divx_key_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);
static gboolean gst_divx_key_start (GstBaseTransform * btrans);
static gboolean gst_divx_key_stop (GstBaseTransform * btrans);

static void gst_divx_key_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_divx_key_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstDivxKey, gst_divx_key, GstBaseTransform,
    GST_TYPE_BASE_TRANSFORM);

static void
gst_divx_key_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Divxkey",
      "Codec/Parser/Video", "Check for divx/xvid/mpeg4 keyframes",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>,\n" "Tilmann Bitterberg");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_divx_key_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_divx_key_src_template));
}

static void
gst_divx_key_class_init (GstDivxKeyClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (divx_key_debug, "divxkey", 0, "divxkey");

  gobject_class->set_property = gst_divx_key_set_property;
  gobject_class->get_property = gst_divx_key_get_property;

  /* not setting passthrough assures (metadata) writable buffers */
  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_divx_key_set_caps);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_divx_key_transform_ip);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_divx_key_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_divx_key_stop);
}

static void
gst_divx_key_init (GstDivxKey * filter, GstDivxKeyClass * g_class)
{

}


static gboolean
gst_divx_key_set_caps (GstBaseTransform * btrans, GstCaps * incaps,
    GstCaps * outcaps)
{
  GstDivxKey *filter;
  GstStructure *structure;

  filter = GST_DIVX_KEY (btrans);

  structure = gst_caps_get_structure (incaps, 0);

  filter->old = FALSE;

  if (gst_structure_has_name (structure, "video/x-divx")) {
    gint divxversion = 0;

    gst_structure_get_int (structure, "divxversion", &divxversion);

    switch (divxversion) {
      case 3:
        filter->old = TRUE;
        break;
      case 4:
      case 5:
        filter->old = FALSE;
        break;
      default:
        return FALSE;
    }
  }

  return TRUE;
}

static gboolean
gst_divx_key_is_key (guint8 * data, glong size, gboolean old)
{
  if (old) {
    gint32 c;

    c = ((data[0] << 24) | (data[1] << 16) | (data[2] << 8) | (data[3] & 0xff));
    if (c & 0x40000000)
      return FALSE;
    else
      return TRUE;
  } else {
    gint result = FALSE;
    gint i;

    for (i = 0; i < size - 5; i++) {
      if (data[i] == 0x00 && data[i + 1] == 0x00 &&
          data[i + 2] == 0x01 && data[i + 3] == 0xb6) {
        if ((data[i + 4] & 0xc0) == 0x0)
          return TRUE;
        else
          return FALSE;
      }
    }

    return result;
  }
}

static GstFlowReturn
gst_divx_key_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  GstDivxKey *filter;

  filter = GST_DIVX_KEY (btrans);

  if (gst_divx_key_is_key (GST_BUFFER_DATA (in), GST_BUFFER_SIZE (in),
          filter->old))
    GST_BUFFER_FLAG_UNSET (in, GST_BUFFER_FLAG_DELTA_UNIT);
  else
    GST_BUFFER_FLAG_SET (in, GST_BUFFER_FLAG_DELTA_UNIT);

  return GST_FLOW_OK;
}


static gboolean
gst_divx_key_start (GstBaseTransform * btrans)
{
  return TRUE;
}

static gboolean
gst_divx_key_stop (GstBaseTransform * btrans)
{
  return TRUE;
}

static void
gst_divx_key_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstDivxKey *src;

  g_return_if_fail (GST_IS_DIVX_KEY (object));
  src = GST_DIVX_KEY (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_divx_key_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstDivxKey *src;

  g_return_if_fail (GST_IS_DIVX_KEY (object));
  src = GST_DIVX_KEY (object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
