/* GStreamer Filter
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-mask
 *
 * <refsect2>
 * <para>
 * This filter applies a rectangular mask to an image.
 * The mask is specified by an upper left and bottom right corner;
 * everything outside of the maks is set to color indicated by
 * <link linkend="GstMask--fill">fill</link> (as borders are
 * in the <link linkend="GstVideoBox">videobox</link> filter).
 * </para>
 * <title>History</title>
 * <para>
 * <itemizedlist>
 * <listitem>
 * <para>
 * It is akin to and named after transcode mask filter
 * [Thomas Oestreich, Chad Page].
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "plugin-tc.h"

#include <string.h>


#define GST_TYPE_MASK \
  (gst_mask_get_type())
#define GST_MASK(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_MASK,GstMask))
#define GST_MASK_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_MASK,GstMaskClass))
#define GST_IS_MASK(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_MASK))
#define GST_IS_MASK_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_MASK))


typedef struct _GstMask GstMask;
typedef struct _GstMaskClass GstMaskClass;

struct _GstMask
{
  GstVideoFilter videofilter;

  gint width, height;

  /* properties */
  guint left, right, top, bottom;
  guint fill_type;
};


struct _GstMaskClass
{
  GstVideoFilterClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (mask_debug);
#define GST_CAT_DEFAULT mask_debug


/* signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  PROP_0,
  PROP_FILL,
  PROP_LEFT,
  PROP_TOP,
  PROP_RIGHT,
  PROP_BOTTOM
      /* FILL ME */
};

#define DEFAULT_LEFT      0
#define DEFAULT_RIGHT    32
#define DEFAULT_TOP       0
#define DEFAULT_BOTTOM   32
#define DEFAULT_FILL      0

static GstStaticPadTemplate gst_mask_src_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstStaticPadTemplate gst_mask_sink_template =
GST_STATIC_PAD_TEMPLATE (GST_BASE_TRANSFORM_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_YUV ("{ IYUV, I420, YV12 }"))
    );

static GstFlowReturn gst_mask_transform_ip (GstBaseTransform * btrans,
    GstBuffer * in);
static gboolean gst_mask_start (GstBaseTransform * btrans);
static gboolean gst_mask_stop (GstBaseTransform * btrans);

static void gst_mask_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec);
static void gst_mask_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec);

GST_BOILERPLATE (GstMask, gst_mask, GstVideoFilter, GST_TYPE_VIDEO_FILTER);

GST_VIDEO_FILTER_SET_CAPS_BOILERPLATE (GstMask, gst_mask);


enum
{
  MASK_FILL_BLACK,
  MASK_FILL_GREEN,
  MASK_FILL_BLUE,
};

/* fill type as in videobox */
#define GST_TYPE_MASK_FILL (gst_mask_fill_get_type())
static GType
gst_mask_fill_get_type (void)
{
  static GType mask_fill_type = 0;
  static const GEnumValue mask_fill[] = {
    {MASK_FILL_BLACK, "Black", "black"},
    {MASK_FILL_GREEN, "Colorkey green", "green"},
    {MASK_FILL_BLUE, "Colorkey blue", "blue"},
    {0, NULL, NULL},
  };

  if (!mask_fill_type) {
    mask_fill_type = g_enum_register_static ("GstMaskFill", mask_fill);
  }
  return mask_fill_type;
}

static void
gst_mask_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details_simple (element_class, "Mask",
      "Filter/Effect/Video", "Filter through rectangular mask",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_mask_sink_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_mask_src_template));
}

static void
gst_mask_class_init (GstMaskClass * g_class)
{
  GObjectClass *gobject_class;
  GstBaseTransformClass *trans_class;

  gobject_class = G_OBJECT_CLASS (g_class);
  trans_class = GST_BASE_TRANSFORM_CLASS (g_class);

  GST_DEBUG_CATEGORY_INIT (mask_debug, "mask", 0, "mask");

  gobject_class->set_property = gst_mask_set_property;
  gobject_class->get_property = gst_mask_get_property;

  g_object_class_install_property (gobject_class, PROP_FILL,
      g_param_spec_enum ("fill", "Fill", "How to fill masked parts",
          GST_TYPE_MASK_FILL, DEFAULT_FILL,
          (GParamFlags) G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_LEFT,
      g_param_spec_uint ("upper-left-x", "upper-left-x",
          "Upper left corner x-coordinate",
          0, G_MAXUINT, DEFAULT_LEFT,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_TOP,
      g_param_spec_uint ("upper-left-y", "upper-left-y",
          "Upper left corner y-coordinate",
          0, G_MAXUINT, DEFAULT_TOP,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_RIGHT,
      g_param_spec_uint ("bottom-right-x", "bottom-right-x",
          "Bottom right corner x-coordinate",
          0, G_MAXUINT, DEFAULT_RIGHT,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  g_object_class_install_property (gobject_class, PROP_BOTTOM,
      g_param_spec_uint ("bottom-right-y", "bottom-right-y",
          "Bottom right corner y-coordinate",
          0, G_MAXUINT, DEFAULT_BOTTOM,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE));

  trans_class->set_caps = GST_DEBUG_FUNCPTR (gst_mask_set_caps);
  trans_class->transform_ip = GST_DEBUG_FUNCPTR (gst_mask_transform_ip);
  trans_class->start = GST_DEBUG_FUNCPTR (gst_mask_start);
  trans_class->stop = GST_DEBUG_FUNCPTR (gst_mask_stop);
}

static void
gst_mask_init (GstMask * filter, GstMaskClass * g_class)
{
  filter->left = DEFAULT_LEFT;
  filter->right = DEFAULT_RIGHT;
  filter->top = DEFAULT_TOP;
  filter->bottom = DEFAULT_BOTTOM;
  filter->fill_type = DEFAULT_FILL;
}

static inline void
gst_mask_mask (guint8 * data, guint stride, guint height,
    guint left, guint right, guint top, guint bottom, guint8 value)
{
  guint j, extra;

  /* top border */
  for (j = 0; j < top; ++j) {
    oil_splat_u8_ns (data, &value, stride);
    data += stride;
  }

  extra = stride - right - 1;
  /* middle; left and right border */
  for (j = top; j <= bottom; ++j) {
    oil_splat_u8_ns (data, &value, left);
    oil_splat_u8_ns (data + right + 1, &value, extra);
    data += stride;
  }

  /* bottom border */
  for (j = bottom + 1; j < height; ++j) {
    oil_splat_u8_ns (data, &value, stride);
    data += stride;
  }
}

static int yuv_colors_Y[] = { 0, 150, 29 };
static int yuv_colors_U[] = { 128, 46, 255 };
static int yuv_colors_V[] = { 128, 21, 107 };

static GstFlowReturn
gst_mask_transform_ip (GstBaseTransform * btrans, GstBuffer * in)
{
  GstMask *filter;
  guint8 *src;
  guint width, height, stride, left, right, top, bottom;

  gst_object_sync_values (G_OBJECT (btrans), GST_BUFFER_TIMESTAMP (in));

  filter = GST_MASK (btrans);

  src = (guint8 *) GST_BUFFER_DATA (in);

  width = filter->width;
  height = filter->height;

  /* safeties */
  left = MIN (filter->left, width - 1);
  right = MAX (MIN (filter->right, width - 1), left);
  top = MIN (filter->top, height - 1);
  bottom = MAX (MIN (filter->bottom, height - 1), top);

  /* luma */
  stride = GST_VIDEO_I420_Y_ROWSTRIDE (width);
  gst_mask_mask (src, stride, height, left, right, top, bottom,
      yuv_colors_Y[filter->fill_type]);

  /* chroma */
  stride = GST_VIDEO_I420_U_ROWSTRIDE (width);
  gst_mask_mask (src + GST_VIDEO_I420_U_OFFSET (width, height),
      stride, height / 2, left / 2, right / 2, top / 2, bottom / 2,
      yuv_colors_U[filter->fill_type]);
  gst_mask_mask (src + GST_VIDEO_I420_V_OFFSET (width, height),
      stride, height / 2, left / 2, right / 2, top / 2, bottom / 2,
      yuv_colors_V[filter->fill_type]);

  return GST_FLOW_OK;
}


static gboolean
gst_mask_start (GstBaseTransform * btrans)
{
  return TRUE;
}

static gboolean
gst_mask_stop (GstBaseTransform * btrans)
{
  return TRUE;
}

static void
gst_mask_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstMask *src;

  g_return_if_fail (GST_IS_MASK (object));
  src = GST_MASK (object);

  switch (prop_id) {
    case PROP_LEFT:
      src->left = g_value_get_uint (value);
      break;
    case PROP_RIGHT:
      src->right = g_value_get_uint (value);
      break;
    case PROP_TOP:
      src->top = g_value_get_uint (value);
      break;
    case PROP_BOTTOM:
      src->bottom = g_value_get_uint (value);
      break;
    case PROP_FILL:
      src->fill_type = g_value_get_enum (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_mask_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstMask *src;

  g_return_if_fail (GST_IS_MASK (object));
  src = GST_MASK (object);

  switch (prop_id) {
    case PROP_LEFT:
      g_value_set_uint (value, src->left);
      break;
    case PROP_RIGHT:
      g_value_set_uint (value, src->right);
      break;
    case PROP_TOP:
      g_value_set_uint (value, src->bottom);
      break;
    case PROP_BOTTOM:
      g_value_set_uint (value, src->bottom);
      break;
    case PROP_FILL:
      g_value_set_enum (value, src->fill_type);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}
