/* GStreamer yuv (mjpegtools) runtime wrapper
 * Copyright (C) 2006 Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * gsty4mtrans.c: gstreamer yuv tool runtime wrapper
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1307, USA.
 */

/**
 * SECTION:element-y4mtrans
 *
 * <refsect2>
 * <para>
 * This filter is a special case of <link linkend="GstProcPipe">procpipe</link>,
 * in as much that both the input to and output from the child process are
 * in YUV4MPEG2 format, as e.g. used by various tools in the
 * <ulink url="http://mjpeg.sourceforge.net/">mjpegtools</ulink> suite.
 * </para>
 * </refsect2>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>

#include "gstproctrans.h"

#define GST_TYPE_Y4M_TRANS \
  gst_y4m_trans_get_type ()
#define GST_Y4M_TRANS(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), GST_TYPE_Y4M_TRANS, GstY4mTrans))
#define GST_Y4M_TRANS_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), GST_TYPE_Y4M_TRANS, GstY4mTransClass))
#define GST_IS_Y4M_TRANS(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), GST_TYPE_Y4M_TRANS))
#define GST_IS_Y4M_TRANS_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), GST_TYPE_Y4M_TRANS))

typedef struct _GstY4mTrans GstY4mTrans;
typedef struct _GstY4mTransClass GstY4mTransClass;

struct _GstY4mTrans
{
  GstProcTrans parent;

  /* properties */
  gchar *args;
};

struct _GstY4mTransClass
{
  GstProcTransClass parent_class;
};

GST_DEBUG_CATEGORY_STATIC (y4m_trans_debug);
#define GST_CAT_DEFAULT y4m_trans_debug

static GstStaticPadTemplate sink_template =
GST_STATIC_PAD_TEMPLATE (GST_PROC_TRANS_SINK_NAME,
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("application/x-yuv4mpeg, " "y4mversion = (int) 2 ")
    );

static GstStaticPadTemplate src_template =
GST_STATIC_PAD_TEMPLATE (GST_PROC_TRANS_SRC_NAME,
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("application/x-yuv4mpeg, " "y4mversion = (int) 2 ")
    );

/* properties */
enum
{
  PROP_0,
  PROP_ARGS
};

static gboolean gst_y4m_trans_set_caps (GstProcTrans * ptrans, GstCaps * incaps,
    GstCaps ** outcaps);
static GstStateChangeReturn gst_y4m_trans_change_state (GstElement * element,
    GstStateChange transition);

/* properties */
static void gst_y4m_trans_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_y4m_trans_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

static void gst_y4m_trans_finalize (GObject * object);

GST_BOILERPLATE (GstY4mTrans, gst_y4m_trans, GstProcTrans, GST_TYPE_PROC_TRANS);

static void
gst_y4m_trans_base_init (gpointer klass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_set_details_simple (element_class, "y4m runtime wrapper",
      "Filter/Effect/Video", "yuv/y4m mjpegtools runtime wrapper",
      "Mark Nauwelaerts <mnauw@users.sourceforge.net>");

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&src_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&sink_template));
}

static void
gst_y4m_trans_class_init (GstY4mTransClass * klass)
{
  GstElementClass *element_class;
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  element_class = GST_ELEMENT_CLASS (klass);

  GST_DEBUG_CATEGORY_INIT (y4m_trans_debug, "y4mtrans", 0, "y4m wrapper");

  gobject_class->finalize = GST_DEBUG_FUNCPTR (gst_y4m_trans_finalize);

  gobject_class->set_property = gst_y4m_trans_set_property;
  gobject_class->get_property = gst_y4m_trans_get_property;

  g_object_class_install_property (gobject_class, PROP_ARGS,
      g_param_spec_string ("args", "Args", "Command arguments",
          NULL, G_PARAM_READWRITE));

  element_class->change_state = GST_DEBUG_FUNCPTR (gst_y4m_trans_change_state);
}

static void
gst_y4m_trans_init (GstY4mTrans * trans, GstY4mTransClass * klass)
{
  GstProcTransClass *proctransclass = GST_PROC_TRANS_CLASS (klass);

  proctransclass->set_caps = GST_DEBUG_FUNCPTR (gst_y4m_trans_set_caps);

  /* most properties are initialized by base element */
  trans->args = NULL;
}

static void
gst_y4m_trans_finalize (GObject * object)
{
  GstY4mTrans *trans = GST_Y4M_TRANS (object);

  g_free (trans->args);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}


static gboolean
gst_y4m_trans_set_caps (GstProcTrans * ptrans, GstCaps * incaps,
    GstCaps ** outcaps)
{
  GstY4mTrans *trans = GST_Y4M_TRANS (ptrans);
  GstStructure *structure;
  gint version;

  structure = gst_caps_get_structure (incaps, 0);

  if (!(gst_structure_has_name (structure, "application/x-yuv4mpeg")
          && gst_structure_get_int (structure, "y4mversion", &version)
          && version == 2))
    return FALSE;


  if (trans->args) {
    gchar **args;
    gint nargs;
    GError *err = NULL;

    GST_INFO_OBJECT (trans, "parsing arguments %s", trans->args);

    if (!g_shell_parse_argv (trans->args, &nargs, &args, &err)) {
      g_return_val_if_fail (err != NULL, FALSE);
      GST_ELEMENT_ERROR (trans, RESOURCE, FAILED,
          ("glib error %s", err->message ? err->message : ""),
          ("parsing %s", trans->args));
      return FALSE;
    }
    g_array_append_vals (ptrans->args, args, nargs);
    /* only free the list of pointers, not their contents */
    g_free (args);
  }

  /* set the outgoing caps */
  *outcaps = gst_caps_new_simple ("application/x-yuv4mpeg",
      "y4mversion", G_TYPE_INT, 2, NULL);

  return TRUE;
}

static void
gst_y4m_trans_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstY4mTrans *trans;

  g_return_if_fail (GST_IS_Y4M_TRANS (object));

  trans = GST_Y4M_TRANS (object);

  switch (prop_id) {
    case PROP_ARGS:
      g_free (trans->args);
      trans->args = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_y4m_trans_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstY4mTrans *trans;

  g_return_if_fail (GST_IS_Y4M_TRANS (object));

  trans = GST_Y4M_TRANS (object);

  switch (prop_id) {
    case PROP_ARGS:
      g_value_take_string (value, g_strdup (trans->args));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstStateChangeReturn
gst_y4m_trans_change_state (GstElement * element, GstStateChange transition)
{
  GstStateChangeReturn ret;

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    goto done;

  switch (transition) {
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      break;
    default:
      break;
  }

done:
  return ret;
}
