/* GStreamer
 *
 * unit test for videofilter elements
 *
 * Copyright (C) <2006> Mark Nauwelaerts <mnauw@users.sourceforge.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <unistd.h>
#include <stdarg.h>

#include <gst/check/gstcheck.h>

#include "videofilter.h"

GList *buffers = NULL;

static GList *padtemplates = NULL;

/* For ease of programming we use globals to keep refs for our floating
 * src and sink pads we create; otherwise we always have to do get_pad,
 * get_peer, and then remove references in every test function */
static GstPad *mysrcpad, *mysinkpad;

static GstElement *
setup_filter (const gchar * name,
    GstStaticPadTemplate * sink, GstStaticPadTemplate * src,
    const gchar * prop, va_list var_args)
{
  GstElement *element;

  GST_DEBUG ("setup_element");

  element = gst_check_setup_element (name);
  g_object_set_valist (G_OBJECT (element), prop, var_args);
  mysrcpad = gst_check_setup_src_pad (element, src, NULL);
  mysinkpad = gst_check_setup_sink_pad (element, sink, NULL);
  gst_pad_set_active (mysrcpad, TRUE);
  gst_pad_set_active (mysinkpad, TRUE);

  return element;
}

static void
cleanup_filter (GstElement * filter)
{
  GST_DEBUG ("cleanup_element");

  gst_pad_set_active (mysrcpad, FALSE);
  gst_pad_set_active (mysinkpad, FALSE);
  gst_check_teardown_src_pad (filter);
  gst_check_teardown_sink_pad (filter);
  gst_check_teardown_element (filter);
}

void
check_filter (const gchar * name, RunInfo * info, const gchar * prop, ...)
{
  GstElement *filter;
  GstBuffer *inbuffer, *outbuffer;
  GstCaps *caps;
  int i, size;
  va_list varargs;
  GstPadTemplate *src, *sink;
  GstStaticPadTemplate *templ;

  /* setup pad templates */
  GstStaticPadTemplate srctemplate = GST_STATIC_PAD_TEMPLATE ("src",
      GST_PAD_SRC,
      GST_PAD_ALWAYS,
      GST_STATIC_CAPS (info->in_caps)
    );
  GstStaticPadTemplate sinktemplate = GST_STATIC_PAD_TEMPLATE ("sink",
      GST_PAD_SINK,
      GST_PAD_ALWAYS,
      GST_STATIC_CAPS (info->out_caps)
    );
  /* setup element */
  va_start (varargs, prop);
  filter = setup_filter (name, &sinktemplate, &srctemplate, prop, varargs);
  va_end (varargs);
  fail_unless (gst_element_set_state (filter,
          GST_STATE_PLAYING) == GST_STATE_CHANGE_SUCCESS,
      "could not set to playing");

  /* send out buffers */
  size = info->in_size;
  for (i = 0; i < info->in_count; ++i) {
    inbuffer = gst_buffer_new_and_alloc (size);
    /* makes valgrind's memcheck happier */
    memset (GST_BUFFER_DATA (inbuffer), 0, GST_BUFFER_SIZE (inbuffer));
    caps = gst_caps_from_string (info->in_caps);
    gst_buffer_set_caps (inbuffer, caps);
    gst_caps_unref (caps);
    GST_BUFFER_TIMESTAMP (inbuffer) = 0;
    ASSERT_BUFFER_REFCOUNT (inbuffer, "inbuffer", 1);
    fail_unless (gst_pad_push (mysrcpad, inbuffer) == GST_FLOW_OK);
  }

  /* eos to ask for flushing if needed */
  gst_pad_push_event (mysrcpad, gst_event_new_eos ());

  GST_DEBUG ("got %d output buffers", g_list_length (buffers));
  fail_unless (g_list_length (buffers) == info->out_count);

  /* clean up element; makes sure element cleans up any refs */
  cleanup_filter (filter);

  /* check and clean up buffers */
  for (i = 0; i < info->out_count; ++i) {
    outbuffer = GST_BUFFER (buffers->data);
    fail_if (outbuffer == NULL);

    switch (i) {
      case 0:
        fail_unless (GST_BUFFER_SIZE (outbuffer) == info->out_size);
        /* no check on filter operation itself */
        break;
      default:
        break;
    }
    buffers = g_list_remove (buffers, outbuffer);

    ASSERT_BUFFER_REFCOUNT (outbuffer, "outbuffer", 1);
    gst_buffer_unref (outbuffer);
    outbuffer = NULL;
  }

  g_list_free (buffers);
  buffers = NULL;

  /* make the not so really static caps look a bit more like static
   * by keeping some refs to them around to make valgrind happy;
   * this hack is forced due to check API requiring static templates
   * and the way gst_static_caps_get works */
  templ = g_new (GstStaticPadTemplate, 1);
  *templ = srctemplate;
  padtemplates = g_list_append (padtemplates, templ);
  templ = g_new (GstStaticPadTemplate, 1);
  *templ = sinktemplate;
  padtemplates = g_list_append (padtemplates, templ);
}
