/* Copyright (C) <2006> Nokia Corporation.
 *
 * Contact: Zeeshan Ali <zeenix@gstreamer.net>
 *
 * gstrtpclock.c: clock that synchronizes itself according to RTP timestamps
 * provided to it
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include "gstrtpclock.h"

GST_DEBUG_CATEGORY (rtpclock_debug);
#define GST_CAT_DEFAULT (rtpclock_debug)

#define _do_init(type) \
  GST_DEBUG_CATEGORY_INIT (rtpclock_debug, "netclock", 0, "Network client clock");

GST_BOILERPLATE_FULL (GstRTPClock, gst_rtp_clock,
    GstSystemClock, GST_TYPE_SYSTEM_CLOCK, _do_init);

static void gst_rtp_clock_finalize (GObject * object);

static void
gst_rtp_clock_base_init (gpointer g_class)
{
  /* nop */
}

static void
gst_rtp_clock_class_init (GstRTPClockClass * klass)
{
  GObjectClass *gobject_class;

  gobject_class = (GObjectClass *) klass;

  gobject_class->finalize = gst_rtp_clock_finalize;
}

static void
gst_rtp_clock_init (GstRTPClock * self, GstRTPClockClass * g_class)
{
  /* GstClock *clock = GST_CLOCK_CAST (self); */
}

static void
gst_rtp_clock_finalize (GObject * object)
{
  G_OBJECT_CLASS (parent_class)->finalize (object);
}

void
gst_rtp_clock_update (GstRTPClock * self, GstClockTime timestamp,
    gboolean first_packet)
{
  gdouble r_squared;
  GstClockTime internal;
  GstClockTime external;
  GstClockTime rate_num;
  GstClockTime rate_denom;

  gst_clock_get_calibration (GST_CLOCK (self),
      &internal, &external, &rate_num, &rate_denom);

  //g_print ("before: %llu %llu %llu %llu\n", internal, external, rate_num, rate_denom);

  internal = gst_clock_get_internal_time (GST_CLOCK (self));

  if (first_packet)
    self->base_time = internal - timestamp;

  gst_clock_add_observation (GST_CLOCK (self),
      internal, self->base_time + timestamp, &r_squared);

  gst_clock_get_calibration (GST_CLOCK (self),
      &internal, &external, &rate_num, &rate_denom);

  //g_print ("after: %llu %llu %llu %llu\n", internal, external, rate_num, rate_denom);
}

/**
 * gst_rtp_clock_new:
 * @base_time: initial time of the clock
 *
 * Create a new #GstRTPClock that will synchronize itself with the
 * timestamps on the RTP packets provided to it.
 *
 * Returns: a new #GstClock 
 * 
 */
GstClock *
gst_rtp_clock_new (GstClockTime base_time)
{
  GstRTPClock *ret;
  GstClockTime internal;

  g_return_val_if_fail (base_time != GST_CLOCK_TIME_NONE, NULL);

  ret = g_object_new (GST_TYPE_RTP_CLOCK, NULL);

  ret->base_time = base_time;
  /* update our internal time so get_time() give something around base_time.
     assume that the rate is 1 in the beginning. */
  internal = gst_clock_get_internal_time (GST_CLOCK (ret));
  gst_clock_set_calibration (GST_CLOCK (ret), internal, base_time, 1, 1);

  {
    GstClockTime now = gst_clock_get_time (GST_CLOCK (ret));

    if (now < base_time || now > base_time + GST_SECOND)
      g_warning ("unable to set the base time, expect sync problems!");
  }

  gst_element_set_name (ret, "rtpclock");

  /* all systems go, cap'n */
  return (GstClock *) ret;
}
