/* gtd_fusr.c - Find User functions
 *
 * Copyright (C) 1994, 95, 96, 97, 98, 99 Free Software Foundation
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org
 * 
 * Description:
 * 
 *   This file contains functions used to lookup users.  It should be
 * flexible enough to handle GNU finger style lookups, or standard
 * /etc/utmp (UTMP_FILE) lookups.
 * 
 * History:
 * zappo   9/26/94    Created
 * 
 * $Log: gtd_fusr.c,v $
 * Revision 1.20  1999/02/28 12:29:45  zappo
 * Added idle time into the tty finding equasion.
 *
 * Revision 1.19  1998/01/04 13:33:27  zappo
 * Fixed warnings
 *
 * Revision 1.18  1997/12/14 19:19:13  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.17  1997/12/02 22:54:51  zappo
 * Now accounts for filter data for each user.
 *
 * Revision 1.16  1997/10/08 00:50:59  zappo
 * Cast output of LIST_find to the correct type.
 *
 * Revision 1.15  1997/07/27 20:41:10  zappo
 * User object now uses the generic list
 *
 * Revision 1.14  1997/02/21 04:01:27  zappo
 * Added check for group write permission before writing on a tty.
 *
 * Revision 1.13  1997/02/21 03:40:46  zappo
 * Added ability to choose the tty with the least idle time for selection
 * when making an announcement.  Require new sys/stat.h to get the file status.
 * Added solo test.
 *
 * Revision 1.12  1996/06/25 03:03:59  zappo
 * changed how users are looked up because it sometime chose the wrong
 * TTY.  May need some work on other types of systems.
 *
 * Revision 1.11  1995/12/10  03:54:06  zappo
 * error messages now use DISP_message
 *
 * Revision 1.10  1995/07/13  01:17:22  zappo
 * Ifdef'd out "free(pws)" which is returned as static space on some
 * systems.
 *
 * Revision 1.9  1995/05/09  23:00:46  zappo
 * Last fix assigned temp variable in the wrong place.  Fixed
 *
 * Revision 1.8  1995/05/03  23:09:27  zappo
 * Utmp name field is NOT null terminated!  Make sure we move the data
 * somewhere useful where it is null terminated.
 *
 * Revision 1.7  1995/04/08  20:03:13  zappo
 * Replaced several malloc/strcpy combinations with strdup.
 *
 * Revision 1.6  1995/04/04  01:11:36  zappo
 * Added references to UTMP_FILE instead of the direct string /etc/utmp
 *
 * Revision 1.5  1995/04/01  18:56:03  zappo
 * Modified to use a configure variable to access a /etc/utmp field
 * whose name is different on different systems.
 *
 * Revision 1.4  1995/03/25  04:22:29  zappo
 * Updated copyright
 *
 * Revision 1.3  1995/03/04  14:46:08  zappo
 * Added use of syslog to report error conditions
 *
 * Revision 1.2  1995/02/11  17:18:52  zappo
 * changed formatting of contents print statement
 *
 * Revision 1.1  1995/02/01  03:49:35  zappo
 * Initial revision
 *
 * Tokens: ::Header:: gtalkd.h
 */
#include "gtalklib.h"
#include "gtalkd.h"

#include "sitecnfg.h"

#ifdef HAVE_UTMP_H
#include <utmp.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

/* This defines how to check for a user process type */
#ifdef USER_PROCESS
#define IS_USER_PROCESS (u.ut_type == USER_PROCESS)
#else
/* How to do this on other systems ?? */
#define IS_USER_PROCESS 1
#endif

static MakeList(list);

/* static definitions */
static char *FUSR_utmp_lookup();
static int FUSR_print_ou();


/*
 * Function: FUSR_alloc
 *
 *   Allocates and initializes a new user user object which will
 * define a new individual, and various important attributes.
 *
 * Returns:     struct UserObject  - the new user
 * Parameters:  name - Name of the user
 *              tty  - Tty requested for pestering
 * History:
 * zappo   11/20/94   Created
 */
static unsigned char MatchName(uo, name)
     struct UserObject *uo;
     char              *name;
{
  return !strcmp(uo->name, name);
}
struct UserObject *FUSR_alloc(name, tty)
     char *name;
     char *tty;
{
  struct passwd *pws;
  struct UserObject *new;

  new = (struct UserObject *)LIST_find(&list, MatchName, name);

  if(new) 
    {
      if(verbose)
	{
	  printf("User allocator recycling old user object.\n");
	}
      /* Free any tty already in there */
      if(new->tty) 
	{
	  free(new->tty);
	  new->tty = NULL;
	}
      if(tty && (strlen(tty) > 1))
	{
	  new->tty = strdup(tty);
	  if(!new->tty)
	    {
	      DISP_message(NULL, "FUSR_alloc: malloc failed.", LOG_ERR);
	      exit(1);
	    }
	}
      else
	new->tty = NULL;

      FUSR_print_ou(new);

      return new;
    }

  /* Hmm, this is new, try finding out more from the operating system.
   */
  pws = getpwnam(name);
  
  if(pws == NULL)
    {
      char buff[100];
      sprintf(buff, "FUSR_alloc: getpwname: Cannot find user %s", name);
      DISP_message(NULL, buff, LOG_ERR);
      return NULL;
    }
  
  /* Now that we know more about him, allocate something for im!
   *
   * NOTE: With the exception of forever mode, gtalkd will exit when
   * there are no usefull MESSAGES.  This means we can allocate stuff,
   * and it will all be freed up when the program exits.  Your typical
   * talk daemon is not active for more that one or two calls at most
   * anyways.
   */
  new = (struct UserObject *)LIST_alloc(&list, sizeof(struct UserObject));

  if(new == NULL)
    {
      DISP_message(NULL, "FUSR_alloc: Malloc failure!", LOG_ERR);
      exit(1);
    }

  new->name = strdup(name);
  if(!new->name)
    {
      DISP_message(NULL, "FUSR_alloc: Malloc failure!", LOG_ERR);
      exit(1);
    }
  new->uid = pws->pw_uid;

  if(tty && (strlen(tty) > 1))
    {
      new->tty = strdup(tty);
      if(!new->tty)
	{
	  DISP_message(NULL, "FUSR_alloc: Malloc failure!", LOG_ERR);
	  exit(1);
	}
    }
  else
    new->tty = NULL;

  /* lets build some file names */
  new->ringf = (char *)malloc(strlen(pws->pw_dir) + strlen(RINGER_RC) + 2);

  if(!new->ringf)
    {
      DISP_message(NULL, "FUSR_alloc: Malloc failure!", LOG_ERR);
      exit(1);
    }
  strcpy(new->ringf, pws->pw_dir);
  strcat(new->ringf, "/");
  strcat(new->ringf, RINGER_RC);

  new->filterf = (char*)malloc(strlen(pws->pw_dir)+strlen(BLACKLIST_FILE) + 2);

  if(!new->filterf)
    {
      DISP_message(NULL, "FUSR_alloc: Malloc failure!", LOG_ERR);
      exit(1);
    }
  strcpy(new->filterf, pws->pw_dir);
  strcat(new->filterf, "/");
  strcat(new->filterf, BLACKLIST_FILE);

  /* This is not true on all systems.  Since the talk daemon exits
   * after a minimal delay, if this is a memory leak on some systems,
   * then it certainly  won't be a bad one.  What we need is a configure
   * script to learn this information.
   */
#ifdef FREE_PWS_STRUCTURE
  /* The password structure is allocated, so we must free it. */
  free(pws);
#endif

  /* Load in the filters */
  FILT_ReadFile(new);

  FUSR_print_ou(new);

  return new;
}

/*
 * Function: FUSR_lookup
 *
 *   Lookup user defined in UO and return their TTY device.  If GNU
 * finger is available, use that, otherwise look up locally in
 * /etc/utmp, which has all the current logins listed.
 *
 * Returns:     char * - the tty
 * Parameters:  uo - Pointer to a user object
 *
 * History:
 * zappo   9/26/94    Created 
 */
char *FUSR_lookup(uo)
     struct UserObject *uo;
{
#ifdef GNUFINGER
  if(I have gnu finger)
    return FUSER_gnufinger_lookup(uo)
  else
#endif
    return FUSR_utmp_lookup(uo);
}


/*
 * Function: FUSR_utmp_lookup
 *
 *   Locally defined function which looks up a user in /etc/utmp based
 * on the user object UO.  Returns a string such as "/dev/ttyp1" to
 * indicate what to write to.
 *
 * Returns:     static char * - the device to write to.
 * Parameters:  uo - Pointer to user object
 *
 * History:
 * zappo   9/26/94    Created
 */
static char *FUSR_utmp_lookup(uo)
     struct UserObject *uo;
{
  static struct utmp u;
  static char        device[30] = "\0";
  time_t             device_age = -1;
  struct stat        status_buffer;
  char               mbuff[100];
  char               namebuff[20]; /* temp name buffer */
  FILE              *ustream;
  int                status;
  time_t             current_time;

  ustream = fopen(UTMP_FILE, "r");

  if(!ustream)
    {
      sprintf(mbuff, "Cannot open %s.  Cannot announce.", UTMP_FILE);
      DISP_message(NULL, mbuff, LOG_ERR);
      exit(1);
    }

  /* find a match! */
  if(verbose)
    printf("Scanning %s for pattern: [%s on %s] ...\n", UTMP_FILE,
	   uo->name, uo->tty?uo->tty:"(unspecified)");

  /* Get the current time for comparison to tty times in same format */
  time(&current_time);

  while((status = fread(&u, sizeof(u), 1, ustream)) > 0)
    {
      /*
       * Use configure macro to fill in the name of the field
       * that holds the username.  This can be ut_user on linux,
       * and ut_name on scumos.
       */
      strncpy(namebuff, u.UT_USER_OR_NAME, 8);
      namebuff[8] = 0;

      /* Check for loop exit conditions */
      if((strcmp(uo->name, namebuff) == 0) &&	     /* name match        */
	 IS_USER_PROCESS &&			     /* is a user process */
	 (!uo->tty || !strcmp(uo->tty, u.ut_line)))  /* matching tty name */
	{
	  char buff[sizeof(device)];
	  time_t age;

#ifdef _PATH_DEV
	  strcpy(buff, _PATH_DEV);
#else
	  strcpy(buff, "/dev/");
#endif
	  strcat(buff, u.ut_line);

	  /* Get the age of the line... */
	  if(stat(buff, &status_buffer) < 0)
	    {
	      if(verbose) {
		printf("Error Getting Idle Time!\n");
	      }
	    }
	  
	  /* Get the time differential */
	  age = current_time - status_buffer.st_atime;

	  /* Only use the youngest tty, and make sure others have
	   * permission to pester that tty as well.  This is denoted
	   * when GROUP write permission is available on Linux
	   */
#ifndef S_IWGRP
#define S_IWGRP 0020
#endif
	  if((status_buffer.st_mode & S_IWGRP) && 
	     ((device[0] == 0) || (device_age < 0) || (age < device_age)))
	    {
	      if(verbose) printf("\nTime %ld < %ld", age, device_age);
	      strcpy(device, buff);
	      device_age = age;
	    }

	  if(verbose) {
	    printf("\n(%d)[%s on %s] ", u.ut_type, namebuff, u.ut_line);
	    printf(" Found on %s with %ld idle seconds\n", buff, age);
	  }

	} else {
	  if(verbose)
	    printf("(%d)[%s on %s] ", u.ut_type, namebuff, u.ut_line);
	}
    }

  if(verbose) printf("\n");

  fclose(ustream);

  /* If we have a match, go for it! */
  if(device[0])
    return device;
  else
    return NULL;
}


/*
 * Function: FUSR_print_ou
 *
 *   Locally defined function which prints out user information found
 * in a user structure.
 *
 * Returns:     static int  - 
 * Parameters:  uo - Pointer to the user object.
 *
 * History:
 * zappo   1/14/95    Created
 */
static int FUSR_print_ou(uo)
     struct UserObject *uo;
{
  if(verbose)
    {
      printf("Contents of user object @ %p\n", uo);
      printf("name    : %s\n", uo->name);
      printf("uid     : %d\n", uo->uid);
      printf("tty     : %s\n", uo->tty?uo->tty:"None");
      printf("ringf   : %s\n", uo->ringf);
      printf("filterf : %s\n", uo->filterf);
    }
  return 0;
}

#ifdef TEST
int verbose = 1;

/* Mock disp message */
void DISP_message(struct DaemonContext *Ctxt, char *msg, int severity)
{
  printf("%s\n", msg);
}
/* Mock main */
int main(int argc, char *argv[])
{
  struct DaemonContext Ctxt;
  struct UserObject *uo;
  char *tty;
  int ans;

  if(argc < 2) {
    fprintf(stderr, "use: %s NAME [TTY]\n", argv[0]);
    return 1;
  }

  Ctxt.type = GTALKD;

  /* Set up a user struct */
  if(argc >= 3)
    uo = FUSR_alloc(argv[1], argv[2]);
  else
    uo = FUSR_alloc(argv[1], NULL);

  /* Make the call */
  tty = FUSR_utmp_lookup(uo);
  
  printf("TTY is [%s]\n", tty?tty:"NULL");

  ans = FILT_CheckFilterSimple(uo, HOST_gen_local_host(), argv[1], &Ctxt);

  printf("Filter Answer = %d\n", ans);

  return 0;
}
#endif /* TEST */
