/*
** automaton.c: An automaton executing the program, creating widgets. 
** copyright: (c) 2003 by Lszl Pere
** email: pipas@linux.pte.hu
**
** This program is free software; you can redistribute it and/or 
** modify  it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of
** the License, or (at your option) any later version.
**
** $Id: automaton.c,v 1.3 2004/11/25 18:26:48 pipas Exp pipas $
** $Log: automaton.c,v $
** Revision 1.3  2004/11/25 18:26:48  pipas
** Menu seems to be much more better, and the actions are just the same.
**
** Revision 1.2  2004/11/18 16:45:52  root
**   The button now can have a pixmap and a label simultaneously.
**
** Revision 1.1  2004/11/16 20:45:22  root
** Initial revision
**
*/
#include "widgets.h"
#include "automaton.h"
#include "gtkdialog.h"
#include "stringman.h"

#undef TOOLTIPS

#undef EXECUTION_DEBUG

extern char *includefile;
extern char *program_name;
char *wtitle;

//
// Command line -geometry options
//
extern gboolean have_geometry_xy;
extern gboolean have_geometry_dxdy;
extern gint geometry_dx;
extern gint geometry_dy;
extern gint geometry_x;
extern gint geometry_y;



instruction *program = NULL;
int instruction_counter = 0;	/* The first available memory cell. */
size_t memory_counter = 0;	/* The size of program memory.     */
GtkWidget *window = NULL;	/* The actual window.              */

//
// Locale function definitions
//
static
int instruction_execute_push(
		token Token, 
		AttributeSet *Attr,
		tag_attr *Tag_Attributes);

gint delete(GtkWidget * widget, GtkWidget * event, gpointer data)
{
	gtk_main_quit();
	printf("EXIT=abort\n");
	exit(EXIT_SUCCESS);
}

void table_selection_made(GtkWidget * clist, gint row, gint column,
			  GdkEventButton * event, gpointer data)
{
	if (data == NULL)
		return;
#ifdef DEBUG
	fprintf(stderr, "%s(): Selected %d, %d\n", __func__, column, row);
	fflush(stderr);
#endif
	variables_set_row_column(data, row, column);
}

void table_selected(GtkWidget * clist, gint row, gint column,
		    GdkEventButton * event, gpointer data)
{

#ifdef DEBUG
	fprintf(stderr, "%s(): %s\n", __func__, data);
	fflush(stderr);
#endif
	g_assert(data != NULL);

	button_pressed(clist, data);
#if 0
	if (strncasecmp(data, "refresh:", 8) == 0) {
		variables_export_all();
		variables_refresh(&((char *) data)[8]);
		return;
	}
	if (strncasecmp(data, "clear:", 6) == 0) {
		action_clearwidget(widget, command);
		return TRUE;
	}

	/*
	 ** If there is no known prefix at the beginning of the command
	 ** string, we start it as a shell command.
	 */
	action_shellcommand(clist, data);
#endif
}

void entry_inserted(GtkWidget * entry, gpointer str)
{
	button_pressed(entry, str);
}

void button_toggled(GtkToggleButton * button, gpointer str)
{
	/*
	 ** First we check for "if true" and "if false" prefixes.
	 */
	if (strncasecmp(str, "if true ", 8) == 0) {
		if (gtk_toggle_button_get_active
		    (GTK_TOGGLE_BUTTON(button)))
			button_pressed((GtkWidget *) button,
				       &((char *) str)[8]);
		return;
	}
	if (strncasecmp(str, "if false ", 9) == 0) {
		if (!gtk_toggle_button_get_active
		    (GTK_TOGGLE_BUTTON(button)))
			button_pressed((GtkWidget *) button,
				       &((char *) str)[9]);
		return;
	}

	/*
	 ** Otherwise we just call the function we used to call when a
	 ** button is pressed.
	 */
	button_pressed((GtkWidget *) button, str);
	return;
}

void list_selection_changed(GtkWidget * list, gpointer str)
{
	button_pressed(list, str);
	return;
}


int 
execute_action(GtkWidget *widget, const char *command, const char *type){
	if (strlen(type) == 0){
		action_shellcommand(widget, command);
		return TRUE;
	}
	
	if (strcasecmp(type, "exit") == 0){
		action_exitprogram(widget, command);
		return TRUE;
	}
	
	if (strcasecmp(type, "closewindow") == 0) {
		action_closewindow(widget, command);
		return TRUE;
	}

	if (strcasecmp(type, "launch") == 0) {
		action_launchwindow(widget, command);
		return TRUE;
	}

	if (strcasecmp(type, "enable") == 0) {
		action_enable(widget, command);
		return TRUE;
	}
	if (strcasecmp(type, "disable") == 0) {
		action_disable(widget, command);
		return TRUE;
	}

	if (strcasecmp(type, "refresh") == 0) {
		variables_export_all();
		action_refreshwidget(widget, command);
		return TRUE;
	}

	if (strcasecmp(type, "save") == 0) {
		/* FIXME: is it working? */
		action_savewidget(widget, command);
		return TRUE;
	}

	if (strcasecmp(type, "fileselect") == 0) {
		action_fileselect(widget, command);
		return TRUE;
	}

	if (strcasecmp(type, "clear") == 0) {
		action_clearwidget(widget, command);
		return TRUE;
	}

	if (strcasecmp(type, "removeselected") == 0) {
		action_removeselected(widget, command);
		return TRUE;
	}

	if (strcasecmp(type, "insert") == 0) {
		action_append(widget, command);
		return TRUE;
	}
	
	if (strcasecmp(type, "append") == 0) {
		action_append(widget, command);
		return TRUE;
	}
	
	return FALSE;
}

/*
** This function is called when a button is pressed. The first
** argument is a pointer to the window, the second is the command
** string.
*/
void button_pressed(GtkWidget * button, gpointer str)
{
#ifdef DEBUG
	fprintf(stderr, "%s(): action: '%s'\n", __func__, str);
	fflush(stderr);
#endif

	if (strncasecmp(str, "Closewindow:", 12) == 0) {
		action_closewindow(button, &((char *) str)[12]);
		return;
	}

	if (strncasecmp(str, "Launch:", 7) == 0) {
		action_launchwindow(button, &((char *) str)[7]);
		return;
	}

	if (strncasecmp(str, "Enable:", 7) == 0) {
		action_enable(button, &((char *) str)[7]);
		return;
	}
	if (strncasecmp(str, "Disable:", 8) == 0) {
		action_disable(button, &((char *) str)[8]);
		return;
	}

	if (strncasecmp(str, "exit", 4) == 0) {
		/*
		 ** This function will not return, it exits the
		 ** application.
		 */
		action_exitprogram(button, &((char *) str)[4]);
		return;
	}

	if (strncasecmp(str, "Refresh:", 8) == 0) {
		variables_export_all();
		action_refreshwidget(button, &((char *) str)[8]);
		return;
	}

	if (strncasecmp(str, "Save:", 5) == 0) {
		/* FIXME: is it working? */
		action_savewidget(button, &((char *) str)[5]);
		return;
	}

	if (strncasecmp(str, "fileselect:", 11) == 0) {
		action_fileselect(button, &((char *) str)[11]);
		return;
	}

	if (strncasecmp(str, "clear:", 6) == 0) {
		action_clearwidget(button, &((char *) str)[6]);
		return;
	}

	if (strncasecmp(str, "removeselected:", 15) == 0) {
		action_removeselected(button, &((char *) str)[15]);
		return;
	}

	if (strncasecmp(str, "append:", 6) == 0) {
		action_append(button, &((char *) str)[7]);
		return;
	}


	/*
	 ** If there is no known prefix at the beginning of the command
	 ** string, we start it as a shell command.
	 */
	action_shellcommand(button, str);
	return;
}

/* 
** We get the string between <wtitle> and </wtitle>, and give it for
** gtk_window_set_title() function in run_program. If the string is NULL
** we set the title to MAIN_DIALOG.
*/
void window_title(char *string)
{
	wtitle = string;
}

/*
** This function will run the program in memory.
*/
void run_program()
{
	int pc, q;
	stackelement s;
	AttributeSet *Attr;
	
	/*
	 ** First we create a new window which will be the parent of
	 ** all widgets.
	 */
#ifdef EXECUTION_DEBUG
	fprintf(stderr, "%s(): program starting.\n", __func__);
	fflush(stderr);
#endif

	Attr = attributeset_new();
	
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	//
	// If we have geometry given in the command line, we set that.
	//
	if (have_geometry_dxdy)
		gtk_widget_set_usize(window, geometry_dx, geometry_dy);
	if (have_geometry_xy)
		gtk_widget_set_uposition(window, geometry_x, geometry_y);
	
	//
	// Setting the title of the window
	//
	if (wtitle == NULL)
		wtitle = "gtkdialog";
	gtk_window_set_title(GTK_WINDOW(window), wtitle);
	gtk_container_set_border_width(GTK_CONTAINER(window), 5);
	gtk_signal_connect(GTK_OBJECT(window), "delete_event",
			   GTK_SIGNAL_FUNC(delete), NULL);

	variables_new_with_widget(Attr, window, WIDGET_WINDOW);
	variables_set_parent(program_name, window);
	
	for (pc = 0; pc < instruction_counter; ++pc)
		instruction_execute(program[pc]);
#ifdef EXECUTION_DEBUG
	fprintf(stderr, "%s(): program ended.\n", __func__);
	fflush(stderr);
#endif

	s = pop();

	gtk_container_add(GTK_CONTAINER(window), s.widgets[0]);
#ifdef EXECUTION_DEBUG
	fprintf(stderr, "%s(): window: %p.\n", __func__, window);
	fflush(stderr);
#endif
	gtk_widget_show_all(window);

	//
	// Now, all of the widgets are realized, so we can initialize
	// them as needed.
	//
	variables_initialize_all();	
	
	instruction_counter = 0;
	charsreaded = 0;
	gtk_main();
}

void print_token(token Token)
{
	int Instr_Code;
	int Widget_Type;
	int Attribute_Number;
	Instr_Code = Token & COMMAND;
	Widget_Type = Token & WIDGET_TYPE;
	Attribute_Number = Token & ATTRIBUTE;

	switch (Instr_Code) {
		case PUSH:
			printf("   %s(): instruction PUSH ", __func__);
			break;
		case SUM:
			printf("   %s(): instruction SUM ", __func__);
			break;
		case SET:
			printf("   %s(): instruction SET ", __func__);
			break;
		default:
			printf("   %s(): unknown instruction: '%d' ", __func__, 
					Instr_Code);
			break;
	}

	switch (Widget_Type) {
	case WIDGET_LABEL:
		printf("LABEL");
		break;
	case WIDGET_ENTRY:
		printf("ENTRY");
		break;
	case WIDGET_EDIT:
		printf("EDIT");
		break;
	case WIDGET_BUTTON:
		printf("BUTTON");
		break;
	case WIDGET_CHECKBOX:
		printf("CHECKBOX");
		break;
	case WIDGET_RADIO:
		printf("RADIO");
		break;
	case WIDGET_LIST:
		printf("LIST");
		break;
	case WIDGET_TABLE:
		printf("TABLE");
		break;
	case WIDGET_COMBO:
		printf("COMBO");
		break;
	case WIDGET_SCROLLEDW:
		printf("SCROLLEDW");
		break;
	case WIDGET_VBOX:
		printf("VBOX");
		break;
	case WIDGET_HBOX:
		printf("HBOX");
		break;
	case WIDGET_FRAME:
		printf("FRAME");
		break;
	case WIDGET_PIXMAP:
		printf("PIXMAP");
		break;
	default:
		printf("Unknown Widget (%d)", Widget_Type);
	}
	printf("\n");
	fflush(stdout);
}


int instruction_execute(instruction command)
{
	token Token;
	char *Argument;
	int Instr_Code;
	int Widget_Type;
	int Attribute_Number;
	static AttributeSet *Attr = NULL;
	tag_attr *Tag_Attributes = NULL;

	Token = command.command;
	Argument = command.argument;
	Tag_Attributes = command.tag_attributes;

	Instr_Code = Token & COMMAND;
	Widget_Type = Token & WIDGET_TYPE;
	Attribute_Number = Token & ATTRIBUTE;
#ifdef EXECUTION_DEBUG
	fprintf(stderr,
		"%s(): Executing token %08x with argument '%s'.\n",
		__func__, Token, Argument);
	fprintf(stderr,
		"   inst:%08x, widgt:%08x, argn:%08x\n",
		Instr_Code, Widget_Type, Attribute_Number);
	print_token(Token);
	fflush(stderr);
#endif

	if (Attr == NULL)
		Attr = attributeset_new();

	switch (Instr_Code) {
	case PUSH:
		/* These instructions will push widgets into the stack, some
		 ** of them will remove elements from it.
		 */
		instruction_execute_push(Token, Attr, Tag_Attributes);
		variables_refresh(attributeset_get_first(Attr, ATTR_VARIABLE));
		Attr = NULL;
		break;
	case SET:
		/* 
		 ** SET instructions stores strings in the attributes.
		 */
		attributeset_insert_with_tagattrs(Attr, 
				    Attribute_Number, 
				    Argument,
				    Tag_Attributes);
		break;
	case SUM:
		/*
		 ** This instruction will add the to uppermost stack element and
		 ** push the result to the stack, so the upper element will 
		 ** contain all the widgets.
		 */
		push(_sum(pop(), pop()));
		break;
	default:
		fprintf(stderr, "%s(): Unknown instruction.\n", __func__);
	}

#ifdef DEBUG
	fprintf(stderr, "%s(): end.\n", __func__);
	fflush(stderr);
#endif
	return 0;
}

static void
menu_callback (GtkWidget *menu_item, AttributeSet *Attr)
{
	char *command;
	char *type;
	
#ifdef DEBUG
	fprintf(stderr, "%s(): Start.\n", __func__);
	fflush(stderr);
#endif
	command = attributeset_get_first(Attr, ATTR_ACTION);
	if (command == NULL){
		yywarning("Menu item without action.");
		return;
	}
	while (command != NULL){
		type = attributeset_get_this_tagattr(Attr, ATTR_ACTION, "type");
		if (type == NULL){
			type = "";
		}
		execute_action(menu_item, command, type);
next_command:   command = attributeset_get_next(Attr, ATTR_ACTION);
	}
}


static
GtkWidget *create_menuitem(AttributeSet *Attr){
	GtkWidget *menu_item;
	// Example: GTK_STOCK_OPEN
	if (strncmp("gtk-",
			attributeset_get_first(Attr, ATTR_LABEL),
			4) == 0){
		menu_item = gtk_image_menu_item_new_from_stock(
				attributeset_get_first(Attr, ATTR_LABEL), NULL);
	}else{
		menu_item = gtk_menu_item_new_with_label (
				attributeset_get_first(Attr, ATTR_LABEL));
	}
	
	if (attributeset_is_avail(Attr, ATTR_ACTION))
		g_signal_connect(G_OBJECT(menu_item),
				"activate",
                                menu_callback,
				Attr
				);
	
	
	return menu_item;
}

static
GtkWidget *create_menu(AttributeSet *Attr, stackelement items){
	int n;
	GtkWidget *menu;
	GtkWidget *root_menu;
	GtkWidget *menu_items;
	
	menu = gtk_menu_new();
	for (n = 0; n < items.nwidgets; ++n){
		menu_items = items.widgets[n];
        	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menu_items);
	}
	
	root_menu = gtk_menu_item_new_with_label (
			attributeset_get_first(Attr, ATTR_LABEL));
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
	
	return root_menu;
}

static
GtkWidget *create_menubar(AttributeSet * Attr, stackelement menus)
{
	int n;
	GtkWidget *menu_bar;
	GtkWidget *root_menu;
	
	menu_bar = gtk_menu_bar_new ();
	for (n = 0; n < menus.nwidgets; ++n){
		root_menu = menus.widgets[n];
		gtk_menu_shell_append (GTK_MENU_SHELL (menu_bar), root_menu);
	}
	
	return menu_bar;
}

static
GtkWidget *create_label(AttributeSet * Attr)
{
	GtkWidget *Label;
	char *text = NULL;
#if 0
	if (attributeset_is_avail(Attr, ATTR_LABEL))
		text = attributeset_get_first(Attr, ATTR_LABEL);
	if (text == NULL &&
			attributeset_is_avail(Attr
#endif
	/* 
	 ** Setting up the default values for the new label.
	 */
	attributeset_set_if_unset(Attr, ATTR_LABEL,
				  "(unnamed label)");
	/*
	 ** Creating the widget, and pushing it into the stack.
	 */
	Label = gtk_label_new(attributeset_get_first
			  (Attr, ATTR_LABEL));
	gtk_label_set_line_wrap(GTK_LABEL(Label), TRUE);

	if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
		gtk_widget_set_sensitive(Label, FALSE);

	return Label;
}

static
gboolean widget_moved(GtkWidget *widget,
                      GdkEvent *event,
                      gpointer user_data){
	GtkRequisition size;
	GtkWindow *window = user_data;
	GdkEventType type;
	GdkEventConfigure  *configure;
	
	type = event->type;
	
	if (type == 2)
		return FALSE;
#if 0
	fprintf(stderr, "%s(): event->type: %d mask %x.\n", __func__, 
			type,
			gtk_widget_get_events(widget)
			);
#endif	
  	if (type == GDK_EXPOSE ){
		return FALSE;
	}
	
	if (type == GDK_CONFIGURE){
		configure = (GdkEventConfigure *)event;
#if 0
		printf("   ->x = %d y = %d dx = %d dy = %d\n", 
				configure->x,
				configure->y,
				configure->width,
				configure->height);
#endif
		//gtk_widget_set_usize(window, 
		//		configure->width -20,
		//		configure->height);
		gtk_widget_set_uposition(GTK_WIDGET(window), 
				configure->x,
				configure->y);
	}
	
	fflush(stderr);

	return FALSE;
}

static
GtkWidget *create_gvim(AttributeSet * Attr)
{
	GtkWidget *Window = gtk_window_new(GTK_WINDOW_POPUP);
	GtkWidget *socket = gtk_socket_new();
	GtkWidget *fake = gtk_drawing_area_new();
	gtk_widget_set_events(fake, GDK_ALL_EVENTS_MASK);
	gtk_widget_set_extension_events (fake,
			GDK_EXTENSION_EVENTS_ALL);
	char socket_id[32];

	//gtk_window_set_transient_for(Window, window);
	gtk_widget_set_usize(Window, 350, 150);
	gtk_widget_set_usize(fake, 350, 150);
	
	gtk_container_add (GTK_CONTAINER (Window), socket);
	gtk_widget_show (socket);
	gtk_widget_show (Window);
	gtk_widget_realize (socket);
	sprintf(socket_id, "%d", gtk_socket_get_id (GTK_SOCKET(socket)));
	attributeset_set_if_unset(Attr, ATTR_SOCKET, socket_id); 
	//
	// 
	//
	//gtk_widget_add_events(fake, GDK_ALL_EVENTS_MASK);
	//g_signal_connect(G_OBJECT(fake), "configure-event",
	//		 widget_moved, Window);
	g_signal_connect(G_OBJECT(window), "configure-event",
			 widget_moved, Window);
	//g_signal_connect(G_OBJECT(window), "event",
	//		 widget_moved, Window);
	return fake;
}


static 
gboolean is_item_with_this_tagattr(AttributeSet *Attr, const char *name){
	if (attributeset_get_first(Attr, ATTR_ITEM) == NULL)
		return FALSE;
	do {
		if (attributeset_get_this_tagattr(Attr, ATTR_ITEM, name) != NULL)
			return TRUE;
	} while (attributeset_get_next(Attr, ATTR_ITEM));

	return FALSE;
}

static 
gboolean is_input_with_this_tagattr(AttributeSet *Attr, const char *name){
	if (attributeset_get_first(Attr, ATTR_INPUT) == NULL)
		return FALSE;
	do {
		if (attributeset_get_this_tagattr(Attr, ATTR_INPUT, name) != NULL)
			return TRUE;
	} while (attributeset_get_next(Attr, ATTR_INPUT));

	return FALSE;
}

static 
GtkTreeStore *create_store_from_input(AttributeSet *Attr){
	GtkTreeStore *store;
	GType types[32];
	GtkTreeIter iter[32];
	list_t *lines = NULL;
	list_t *lines_before = NULL;
	int column_number;
	char *line = NULL;
	int size = 0;
	int level;
	gboolean need_stock;
	list_t *columns;
	int n, q;
	char *stock_name;
	
#ifdef DEBUG
	fprintf(stderr, "%s(): Start.\n", __func__);
	fflush(stderr);
#endif

	if (!attributeset_is_avail(Attr, ATTR_LABEL) ||
	    !attributeset_is_avail(Attr, ATTR_INPUT)) 
		return NULL;
	
	columns = linecutter(strdup(attributeset_get_first(Attr, ATTR_LABEL)), '|');
	column_number = columns->n_lines;
	/*
	 * Creating a store
	 */
	for (n = 0; n < 32; ++n){
		types[n] = G_TYPE_STRING;
	}
	types[n] = G_TYPE_INT;

	need_stock = is_input_with_this_tagattr(Attr, "stock");

	if (need_stock) {
		store = gtk_tree_store_newv(column_number + 1, types);
	}else{
		store = gtk_tree_store_newv(column_number, types);
	}
#ifdef DEBUG	
	fprintf(stderr, "%s(): columns: %d, need stock: %d\n", 
			__func__, column_number, need_stock);
	fflush(stderr);
#endif

	/*
	 * Opening file
	 */
	FILE *infile = widget_opencommand(attributeset_get_first(Attr, ATTR_INPUT) );
	if (infile == NULL) {
		fprintf(stderr, "%s(): %m\n", __func__);
		fflush(stderr);
		return NULL;
	}
	
	/*
	 * Inserting lines
	 */
	level = 0;
	while (getline(&line, &size, infile) != -1) {
		//
		// In the tree the lines must not end with a new line character,
		// because the tree actually shows this
		//
		if (line[strlen(line) - 1] == '\n')
			line[strlen(line) - 1] = '\0';

		stock_name = attributeset_get_this_tagattr(Attr, ATTR_INPUT, "stock");
		lines = linecutter(strdup(line), '|'); 
		
		gtk_tree_store_append (store, &iter[0], NULL);
	
		//
		// This is what I have to rewrite
		// 
		if (lines->n_lines == 2){
			gtk_tree_store_set (store, &iter[0],
                    			0, lines->line[0],
                    			1, lines->line[1],
                    			-1);
		}
		if (lines->n_lines == 3){
			gtk_tree_store_set (store, &iter[0],
                    			0, lines->line[0],
                    			1, lines->line[1],
                    			2, lines->line[2],
                    			-1);
		}
		if (lines->n_lines == 4){
			gtk_tree_store_set (store, &iter[0],
                    			0, lines->line[0],
                    			1, lines->line[1],
                    			2, lines->line[2],
                    			3, lines->line[3],
                    			-1);
		}
		if (lines->n_lines == 5){
			gtk_tree_store_set (store, &iter[0],
                    			0, lines->line[0],
                    			1, lines->line[1],
                    			2, lines->line[2],
                    			3, lines->line[3],
                    			4, lines->line[4],
                    			-1);
		}
		if (lines->n_lines == 6){
			gtk_tree_store_set (store, &iter[0],
                    			0, lines->line[0],
                    			1, lines->line[1],
                    			2, lines->line[2],
                    			3, lines->line[3],
                    			4, lines->line[4],
                    			5, lines->line[5],
                    			-1);
		}
	}

	pclose(infile);
	return store;
}

static 
GtkTreeStore *create_store_from_attr(AttributeSet *Attr){
	GtkTreeStore *store;
	GType types[32];
	GtkTreeIter iter[32];
	list_t *lines = NULL;
	list_t *lines_before = NULL;
	int column_number;
	char *line;
	int level;
	gboolean need_stock;
	list_t *columns;
	int n, q;
	char *stock_name;
	
	if (!attributeset_is_avail(Attr, ATTR_LABEL) ||
	    !attributeset_is_avail(Attr, ATTR_ITEM)) 
		return NULL;
	
	columns = linecutter(strdup(attributeset_get_first(Attr, ATTR_LABEL)), '|');
	column_number = columns->n_lines;
	/*
	 * Creating a store
	 */
	for (n = 0; n < 32; ++n){
		types[n] = G_TYPE_STRING;
	}
	types[n] = G_TYPE_INT;

	need_stock = is_item_with_this_tagattr(Attr, "stock");
	if (need_stock) {
		store = gtk_tree_store_newv(column_number + 1, types);
	}else{
		store = gtk_tree_store_newv(column_number, types);
	}
#ifdef DEBUG	
	fprintf(stderr, "%s(): columns: %d, need stock: %d\n", 
			__func__, column_number, need_stock);
	fflush(stderr);
#endif
	
	/*
	 * Inserting lines
	 */
	line = attributeset_get_first(Attr, ATTR_ITEM);
	level = 0;
	while (line != NULL){
		stock_name = attributeset_get_this_tagattr(Attr, ATTR_ITEM, "stock");
		lines = linecutter(strdup(line), '|'); 
		
		if (lines_before == NULL || strcmp(lines->line[0], lines_before->line[0]) != 0){
			gtk_tree_store_append (store, &iter[0], NULL);
			level = 0;
			goto go;
		}
		
		if (level == 0 && strcmp(lines->line[0], lines_before->line[0]) == 0){
			gtk_tree_store_append (store, &iter[1], &iter[0]);
			level = 1;
			goto go;
		}
		
		if (level < column_number){
			for(q = 1; q <= level; ++q)
				if (strcmp(lines->line[q], lines_before->line[q]) != 0){
					gtk_tree_store_append (store, &iter[q], &iter[q-1]);
					level = q;
					goto go;
			}
			gtk_tree_store_append (store, &iter[level+1], &iter[level]);
			++level;
		}


go:		//
		// This is what I have to rewrite
		// 
		if (lines->n_lines == 1 && need_stock){
			gtk_tree_store_set (store, &iter[level],
                    			0, stock_name,
                    			1, lines->line[0],
                    			-1);
		}
		
		if (lines->n_lines == 1 && !need_stock){
			gtk_tree_store_set (store, &iter[level],
                    			0, lines->line[0],
                    			-1);
		}
		
		if (lines->n_lines == 2 && need_stock){
			gtk_tree_store_set (store, &iter[level],
                    			0, stock_name,
                    			1, lines->line[0],
                    			2, lines->line[1],
                    			-1);
		}
		
		if (lines->n_lines == 2 && !need_stock){
			gtk_tree_store_set (store, &iter[level],
                    			0, lines->line[0],
                    			1, lines->line[1],
                    			-1);
		}
		
		if (lines->n_lines == 3 && need_stock){
			gtk_tree_store_set (store, &iter[level],
                    			0, stock_name,
                    			1, lines->line[0],
                    			2, lines->line[1],
                    			3, lines->line[2],
                    			-1);
		}
		
		if (lines->n_lines == 3 && !need_stock){
			gtk_tree_store_set (store, &iter[level],
                    			0, lines->line[0],
                    			1, lines->line[1],
                    			2, lines->line[2],
                    			-1);
		}
		
		line = attributeset_get_next(Attr, ATTR_ITEM);
		lines_before = lines;
	}
	return store;
}

#if GTK_CHECK_VERSION(2,4,0)
static
GtkWidget *create_tree_view_from_attr( AttributeSet *Attr, 
					GtkTreeStore *store){
	GtkWidget *tree;
	GtkCellRenderer *renderer;
	GtkCellRenderer *pixmap_renderer;
	GtkTreeViewColumn *column;
	list_t *columns;
	char *headline;
	int n, q;
	gboolean need_pixmap = FALSE;
	gboolean need_stock = FALSE;

	if (attributeset_is_avail(Attr, ATTR_LABEL)) {
		headline = strdup(attributeset_get_first(Attr, ATTR_LABEL));
		columns = linecutter(headline, '|');
	}
	
	/*
	 * We create the TreeView here.
	 */
	tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
	
	/*
	 * If one item have stock we have to create a pixmap column as
	 * the first one.
	 */
	need_stock = is_item_with_this_tagattr(Attr, "stock") || 
		is_input_with_this_tagattr(Attr, "stock");
	
	for (n = 0; n < columns->n_lines; ++n){
		/*
		 * First the renderer or renderers
		 */
		if (need_stock && n == 0)
			pixmap_renderer = gtk_cell_renderer_pixbuf_new();
		renderer = gtk_cell_renderer_text_new ();
		
		/*
		 * Then a column and its header
		 */
		column = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title(column, columns->line[n]);	

		/*
		 * If need stock we add the pixmap
		 */
		if (need_stock && n == 0){
			gtk_tree_view_column_pack_start(column, 
					pixmap_renderer, FALSE);
			gtk_cell_layout_set_attributes(
					GTK_CELL_LAYOUT(column), 
					pixmap_renderer, 
					"stock_id", 0, NULL);
		}
		
		/*
		 * The text renderer always needed
		 */
		gtk_tree_view_column_pack_end(column, renderer, FALSE);
		if (need_stock)
			q = n + 1;
		else
			q = n;
		gtk_cell_layout_set_attributes(
				GTK_CELL_LAYOUT(column), 
				renderer, "text", q, NULL);

		/*
		 * Adding the new column
		 */
		gtk_tree_view_append_column (GTK_TREE_VIEW (tree), column);
	}
	
	return tree;
}
#endif

#if GTK_CHECK_VERSION(2,4,0)
static
GtkWidget *create_chooser(AttributeSet *Attr){
	GtkWidget *chooser;
	char *act;
	char *tagattr_value;
/*
	GTK_FILE_CHOOSER_ACTION_OPEN	
	Indicates open mode. The file chooser will only let the user
	pick an existing file.

	GTK_FILE_CHOOSER_ACTION_SAVE	
	Indicates save mode. The file chooser will let the user pick
	an existing file, or type in a new filename.

	GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER	
	Indicates an Open mode for selecting folders. The
	file chooser will let the user pick an existing folder.

	GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER	
	Indicates a mode for creating a new folder. The file
	chooser will let the user name an existing or new folder.
*/
	chooser = gtk_file_chooser_widget_new(GTK_FILE_CHOOSER_ACTION_OPEN); 
	if (attributeset_is_avail(Attr, ATTR_HEIGHT) &&
	    attributeset_is_avail(Attr, ATTR_WIDTH))
		gtk_widget_set_usize(chooser, 
				atoi(attributeset_get_first(Attr, ATTR_WIDTH)),
				atoi(attributeset_get_first(Attr, ATTR_HEIGHT)));
	
	if (attributeset_is_avail(Attr, ATTR_DEFAULT))
		gtk_file_chooser_set_current_folder(chooser, 
				attributeset_get_first(Attr, ATTR_DEFAULT));
	
	act = attributeset_get_first(Attr, ATTR_ACTION);
	while (act != NULL) {
		tagattr_value = attributeset_get_this_tagattr(
				Attr, ATTR_ACTION, "when");
		if (tagattr_value == NULL)
			tagattr_value = "file-activated";
		
		gtk_signal_connect(GTK_OBJECT(chooser),
				   tagattr_value,
				   GTK_SIGNAL_FUNC
				   (button_pressed), (gpointer) act);
		act = attributeset_get_next(Attr, ATTR_ACTION);
	}

	//gtk_file_chooser_set_preview_widget_active(chooser, TRUE);
	return chooser;
}
#endif

static
GtkWidget *create_tree(AttributeSet * Attr)
{
	char *headline;
	GtkWidget *tree;
	GtkTreeStore *store;
	/*
	 * We create a scrolled window to handle big trees.
	 */
	GtkWidget *ScrolledWindow = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW
				       (ScrolledWindow),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_ALWAYS);

	if (attributeset_is_avail(Attr, ATTR_HEIGHT) &&
	    attributeset_is_avail(Attr, ATTR_WIDTH))
		gtk_widget_set_usize(ScrolledWindow, 
				atoi(attributeset_get_first(Attr, ATTR_WIDTH)),
				atoi(attributeset_get_first(Attr, ATTR_HEIGHT)));
	else
		gtk_widget_set_usize(ScrolledWindow, 200, 100);

	/*
	 * We need a store, to hold data and a tree_view to show them.
	 */
	store = NULL;	
	if (attributeset_is_avail(Attr, ATTR_ITEM))
		store = create_store_from_attr(Attr);
	
	if (store == NULL && 
			attributeset_is_avail(Attr, ATTR_INPUT))
		store = create_store_from_input(Attr);

	if (store != NULL)
		tree = create_tree_view_from_attr(Attr, store);
	else
		tree = gtk_tree_view_new();

	gtk_container_add(GTK_CONTAINER(ScrolledWindow), tree);
	return ScrolledWindow;
}

#define TYPE_NOTHING 0
#define TYPE_LAB 1
#define TYPE_PIX 2
#define TYPE_LABPIX 3

static
GtkWidget *create_button(AttributeSet * Attr)
{
	GtkWidget *Button = NULL;
	GtkWidget *Icon = NULL;
	GtkWidget *Label = NULL;
	GtkWidget *Box = NULL;
	char *icon_stock_name = NULL;
	char *icon_file_name = NULL;
	char *str;
	char *act;
	int type;
	
#ifdef DEBUG
	fprintf(stderr, "%s: Start.\n", __func__);
	fflush(stderr);
#endif
	
	type = TYPE_NOTHING;
	if (!attributeset_is_avail(Attr, ATTR_INPUT) &&
		attributeset_is_avail(Attr, ATTR_LABEL))
		type = TYPE_LAB;
	
	if (attributeset_is_avail(Attr, ATTR_INPUT) &&
		!attributeset_is_avail(Attr, ATTR_LABEL))
		type = TYPE_PIX;
	
	if (attributeset_is_avail(Attr, ATTR_INPUT) &&
		attributeset_is_avail(Attr, ATTR_LABEL))
		type = TYPE_LABPIX;

	if (type == TYPE_PIX || type == TYPE_LABPIX){
		icon_file_name = attributeset_get_first(Attr, ATTR_INPUT) + 5;
		icon_stock_name = attributeset_get_this_tagattr(Attr, ATTR_INPUT, "stock");
		if (icon_stock_name != NULL){
			Icon = gtk_image_new_from_stock(icon_stock_name,
					GTK_ICON_SIZE_BUTTON);

#if GTK_CHECK_VERSION(2,4,0)
		}else{
			char *icon_name = attributeset_get_this_tagattr(Attr, ATTR_INPUT, "icon");
			
			if (icon_name != NULL){
				GtkIconTheme *icon_theme;
				GdkPixbuf *pixbuf;
				GError *error = NULL;
				
				icon_theme = gtk_icon_theme_get_default();
				pixbuf = gtk_icon_theme_load_icon (icon_theme,
                                   icon_name,
                                   20,
                                   0,
                                   &error);
				Icon = gtk_image_new_from_pixbuf(pixbuf);	
			}else{
				Icon = gtk_image_new_from_file(find_pixmap(icon_file_name));
			}
#endif
		}
	}

	attributeset_set_if_unset(Attr, ATTR_LABEL, "OK");
	attributeset_set_if_unset(Attr, ATTR_DEFAULT, "OK");
		
	switch (type) {
		case TYPE_NOTHING:
		case TYPE_LAB:
			Button = gtk_button_new_with_label(
					attributeset_get_first(Attr, 
						ATTR_LABEL));
			break;
		case TYPE_PIX:
			Button = gtk_button_new();
			gtk_container_add(GTK_CONTAINER(Button), Icon);
			break;
		case TYPE_LABPIX:
			Box = gtk_hbox_new(FALSE, 5);
			Button = gtk_button_new();
		    	Label = gtk_label_new(attributeset_get_first
				  (Attr, ATTR_LABEL));
			gtk_container_add(GTK_CONTAINER(Button), Box);
			gtk_box_pack_end(GTK_BOX(Box), Label, TRUE, TRUE, 0);
			gtk_box_pack_end(GTK_BOX(Box), Icon, TRUE, TRUE, 0);
	}
	
	/*
	 * If a button has no action, the action will be the exit
	 * default action.
	 */
	if (!attributeset_is_avail(Attr, ATTR_ACTION)) {
		str =
		    pip_malloc(strlen
			       (attributeset_get_first
				(Attr, ATTR_LABEL)) + 10, __func__);
		sprintf(str, "EXIT=\"%s\"\n",
			attributeset_get_first(Attr, ATTR_LABEL));
		attributeset_insert(Attr, ATTR_ACTION, str);
	}

	/*
	 ** The action attribute can hold arbitrary number of actions,
	 ** whose execution is a an important task when the button is
	 ** pressed by the user. 
	 */
	act = attributeset_get_first(Attr, ATTR_ACTION);
	while (act != NULL) {
		gtk_signal_connect(GTK_OBJECT(Button),
				   "clicked",
				   GTK_SIGNAL_FUNC
				   (button_pressed), (gpointer) act);
		act = attributeset_get_next(Attr, ATTR_ACTION);
	}

	if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
		gtk_widget_set_sensitive(Button, FALSE);

	return Button;
}

static
GtkWidget *create_pixmap(AttributeSet * Attr)
{
	GtkWidget *Pixmap;
	char *icon_file_name;
	char *icon_stock_name;
	
	icon_file_name = attributeset_get_first(Attr, ATTR_INPUT) + 5;
	icon_stock_name = attributeset_get_this_tagattr(Attr, 
				ATTR_INPUT, 
				"stock");
	// GTK_STOCK_OPEN
	if (icon_stock_name != NULL)
		Pixmap = gtk_image_new_from_stock(icon_stock_name,
					GTK_ICON_SIZE_DND);
	else
		Pixmap = gtk_image_new_from_file(find_pixmap(icon_file_name));

	if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
		gtk_widget_set_sensitive(Pixmap, FALSE);
	
	return Pixmap;
}

static
int instruction_execute_push(
		token Token, 
		AttributeSet *Attr,
		tag_attr *Tag_Attributes)
{
	int Widget_Type;
	static GtkWidget *LastRadioButton = NULL;
	GtkWidget *Widget;
	GtkWidget *OtherWidget;
	int n;
	char *attribute_value;

#ifdef EXECUTION_DEBUG
	fprintf(stderr, "%s(): start\n", __func__);
	fflush(stderr);
#endif
	Widget_Type = Token & WIDGET_TYPE;
	switch (Widget_Type) {
	case WIDGET_LABEL:
		Widget = create_label(Attr);
		push_widget(Widget, Widget_Type);
		break;
	case WIDGET_PIXMAP:
		Widget = create_pixmap(Attr);
		push_widget(Widget, Widget_Type);
		break;
	case WIDGET_ENTRY:
		/*
		 ** Creating the widget, and pushing it into the stack.
		 */
		Widget = gtk_entry_new();
		push_widget(Widget, Widget_Type);

		if (attributeset_is_avail(Attr, ATTR_DEFAULT))
			gtk_entry_set_text(GTK_ENTRY(Widget),
					   attributeset_get_first(Attr,
								  ATTR_DEFAULT));

		if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "password"))
			gtk_entry_set_visibility(GTK_ENTRY(Widget), FALSE);

		if (attributeset_is_avail(Attr, ATTR_HEIGHT) &&
		    attributeset_is_avail(Attr, ATTR_WIDTH))
			gtk_widget_set_usize(Widget,
					     atoi(attributeset_get_first
						  (Attr, ATTR_WIDTH)),
					     atoi(attributeset_get_first
						  (Attr, ATTR_HEIGHT)));
		/*
		 ** Entries can take actions.
		 */
		if (attributeset_is_avail(Attr, ATTR_ACTION)) {
			char *act;
			act = attributeset_get_first(Attr, ATTR_ACTION);
			while (act != NULL) {
				gtk_signal_connect(GTK_OBJECT(Widget),
						   "changed",
						   GTK_SIGNAL_FUNC
						   (entry_inserted),
						   (gpointer) act);
				act =
				    attributeset_get_next(Attr,
							  ATTR_ACTION);
			}
		}

		attribute_value = get_tag_attribute(Tag_Attributes, "max_length=");
		if (attribute_value != NULL){
			g_object_set(G_OBJECT(Widget),
					"max-length",
					atoi(attribute_value),
					NULL);
		}
		break;
	case WIDGET_CHOOSER:
#if GTK_CHECK_VERSION(2,4,0)
		Widget = create_chooser(Attr);
		push_widget(Widget, Widget_Type);
#else
		yyerror_simple("Chooser widget is not supported by"
				"this version of GTK+, you need at"
				"least GTK+ 2.4.0\n");
#endif
		break;
	case WIDGET_TREE:
#if GTK_CHECK_VERSION(2,4,0)
		Widget = create_tree(Attr);
		push_widget(Widget, Widget_Type);
#else
		yyerror_simple("Tree widget is not supported by"
				"this version of GTK+, you need at"
				"least GTK+ 2.4.0\n");
#endif
		break;
	case WIDGET_EDIT:
#if GTK_CHECK_VERSION(2,4,0)
		Widget = gtk_text_view_new();
//      gtk_text_view_get_buffer(GTK_TEXT_VIEW(s.widgets[0]));

		if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
			gtk_widget_set_sensitive(Widget, FALSE);

		{
			GtkWidget *Scrolledwindow;

			Scrolledwindow =
			    gtk_scrolled_window_new(NULL, NULL);
			gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW
						       (Scrolledwindow),
						       GTK_POLICY_AUTOMATIC,
						       GTK_POLICY_ALWAYS);

			if (attributeset_is_avail(Attr, ATTR_HEIGHT) &&
			    attributeset_is_avail(Attr, ATTR_WIDTH))
				gtk_widget_set_usize(Scrolledwindow,
						     atoi
						     (attributeset_get_first
						      (Attr, ATTR_WIDTH)),
						     atoi
						     (attributeset_get_first
						      (Attr, ATTR_HEIGHT))
				    );
			else
				gtk_widget_set_usize(Scrolledwindow, 200,
						     100);

			gtk_container_add(GTK_CONTAINER(Scrolledwindow),
					  Widget);

			push_widget(Scrolledwindow, WIDGET_SCROLLEDW);
		}		/*command group */
#else
		yyerror_simple("Editor widget is not supported by"
				"this version of GTK+, you need at"
				"least GTK+ 2.4.0\n");
#endif
		break;

	case WIDGET_COMBO:
		/* The combo box is quiet simple, however it can be filled with
		 ** a GList which contains a list of items.
		 */
		Widget = gtk_combo_new();

		if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
			gtk_widget_set_sensitive(Widget, FALSE);

		push_widget(Widget, Widget_Type);
		break;

	case WIDGET_OKBUTTON:
		Widget = gtk_button_new_from_stock(GTK_STOCK_OK);
		/*
		 ** The action attribute can hold arbitrary number of actions,
		 ** whose execution is a an important task when the button is
		 ** pressed by the user. 
		 */
		if (attributeset_is_avail(Attr, ATTR_ACTION)) {
			char *act;
			act = attributeset_get_first(Attr, ATTR_ACTION);
			while (act != NULL) {
				gtk_signal_connect(GTK_OBJECT(Widget),
						   "clicked",
						   GTK_SIGNAL_FUNC
						   (button_pressed),
						   (gpointer) act);
				act =
				    attributeset_get_next(Attr,
							  ATTR_ACTION);
			}
		} else {
			gtk_signal_connect(GTK_OBJECT(Widget), "clicked",
					   GTK_SIGNAL_FUNC(button_pressed),
					   (gpointer) "EXIT=\"OK\"\n");
		}

		if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
			gtk_widget_set_sensitive(Widget, FALSE);
		push_widget(Widget, Widget_Type);
		break;

	case WIDGET_CANCELBUTTON:
		Widget = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
		/*
		 ** The action attribute can hold arbitrary number of actions,
		 ** whose execution is a an important task when the button is
		 ** pressed by the user. 
		 */
		if (attributeset_is_avail(Attr, ATTR_ACTION)) {
			char *act;
			act = attributeset_get_first(Attr, ATTR_ACTION);
			while (act != NULL) {
				gtk_signal_connect(GTK_OBJECT(Widget),
						   "clicked",
						   GTK_SIGNAL_FUNC
						   (button_pressed),
						   (gpointer) act);
				act =
				    attributeset_get_next(Attr,
							  ATTR_ACTION);
			}
		} else {
			gtk_signal_connect(GTK_OBJECT(Widget), "clicked",
					   GTK_SIGNAL_FUNC(button_pressed),
					   (gpointer) "EXIT=\"Cancel\"\n");
		}

		if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
			gtk_widget_set_sensitive(Widget, FALSE);

		push_widget(Widget, Widget_Type);
		break;

	case WIDGET_HELPBUTTON:
		Widget = gtk_button_new_from_stock(GTK_STOCK_HELP);
		/*
		 ** The action attribute can hold arbitrary number of actions,
		 ** whose execution is a an important task when the button is
		 ** pressed by the user. 
		 */
		if (attributeset_is_avail(Attr, ATTR_ACTION)) {
			char *act;
			act = attributeset_get_first(Attr, ATTR_ACTION);
			while (act != NULL) {
				gtk_signal_connect(GTK_OBJECT(Widget),
						   "clicked",
						   GTK_SIGNAL_FUNC
						   (button_pressed),
						   (gpointer) act);
				act =
				    attributeset_get_next(Attr,
							  ATTR_ACTION);
			}
		} else {
			gtk_signal_connect(GTK_OBJECT(Widget), "clicked",
					   GTK_SIGNAL_FUNC(button_pressed),
					   (gpointer) "EXIT=\"Help\"\n");
		}

		if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
			gtk_widget_set_sensitive(Widget, FALSE);

		push_widget(Widget, Widget_Type);
		break;

	case WIDGET_YESBUTTON:
		Widget = gtk_button_new_from_stock(GTK_STOCK_YES);
		/*
		 ** The action attribute can hold arbitrary number of actions,
		 ** whose execution is a an important task when the button is
		 ** pressed by the user. 
		 */
		if (attributeset_is_avail(Attr, ATTR_ACTION)) {
			char *act;
			act = attributeset_get_first(Attr, ATTR_ACTION);
			while (act != NULL) {
				gtk_signal_connect(GTK_OBJECT(Widget),
						   "clicked",
						   GTK_SIGNAL_FUNC
						   (button_pressed),
						   (gpointer) act);
				act =
				    attributeset_get_next(Attr,
							  ATTR_ACTION);
			}
		} else {
			gtk_signal_connect(GTK_OBJECT(Widget), "clicked",
					   GTK_SIGNAL_FUNC(button_pressed),
					   (gpointer) "EXIT=\"Yes\"\n");
		}

		if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
			gtk_widget_set_sensitive(Widget, FALSE);

		push_widget(Widget, Widget_Type);
		break;

	case WIDGET_NOBUTTON:
		Widget = gtk_button_new_from_stock(GTK_STOCK_NO);
		/*
		 ** The action attribute can hold arbitrary number of 
		 ** actions, whose execution is a an important task 
		 ** when the button is pressed by the user. 
		 */
		if (attributeset_is_avail(Attr, ATTR_ACTION)) {
			char *act;
			act = attributeset_get_first(Attr, ATTR_ACTION);
			while (act != NULL) {
				gtk_signal_connect(GTK_OBJECT(Widget),
						   "clicked",
						   GTK_SIGNAL_FUNC
						   (button_pressed),
						   (gpointer) act);
				act =
				    attributeset_get_next(Attr,
							  ATTR_ACTION);
			}
		} else {
			gtk_signal_connect(GTK_OBJECT(Widget), "clicked",
					   GTK_SIGNAL_FUNC(button_pressed),
					   (gpointer) "EXIT=\"No\"\n");
		}

		if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
			gtk_widget_set_sensitive(Widget, FALSE);

		push_widget(Widget, Widget_Type);
		break;

	case WIDGET_BUTTON:
		Widget = create_button(Attr);
		push_widget(Widget, Widget_Type);
		break;
	case WIDGET_GVIM:
		Widget = create_gvim(Attr);
		push_widget(Widget, Widget_Type);
		break;
	case WIDGET_MENUBAR:
		Widget = create_menubar(Attr, pop());
		push_widget(Widget, Widget_Type);
		break;
	case WIDGET_MENU:
		attributeset_set_if_unset(Attr, ATTR_LABEL, "Unlabelled menu");
		Widget = create_menu(Attr, pop());
		push_widget(Widget, Widget_Type);
		break;
	case WIDGET_MENUITEM:
		attributeset_set_if_unset(Attr, ATTR_LABEL, "Unlabelled menu item");
		Widget = create_menuitem(Attr);
		push_widget(Widget, Widget_Type);
		break;
	case WIDGET_CHECKBOX:
		/*
		 **
		 */
		attributeset_set_if_unset(Attr, ATTR_LABEL,
					  "Uninitialized checkbox");
		/*
		 **
		 */
		Widget =
		    gtk_check_button_new_with_label(attributeset_get_first
						    (Attr, ATTR_LABEL));
		/*
		 **
		 */
		if (attributeset_cmp_left(Attr, ATTR_DEFAULT, "true") ||
		    attributeset_cmp_left(Attr, ATTR_DEFAULT, "yes"))
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
						     (Widget), TRUE);
		/*
		 ** Checkboxes also can take actions...
		 */
		if (attributeset_is_avail(Attr, ATTR_ACTION)) {
			char *act;
			act = attributeset_get_first(Attr, ATTR_ACTION);
			while (act != NULL) {
				gtk_signal_connect(GTK_OBJECT(Widget),
						   "toggled",
						   GTK_SIGNAL_FUNC
						   (button_toggled),
						   (gpointer) act);
				act =
				    attributeset_get_next(Attr,
							  ATTR_ACTION);
			}
		}

		if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
			gtk_widget_set_sensitive(Widget, FALSE);

		push_widget(Widget, Widget_Type);
		break;

	case WIDGET_RADIO:
		attributeset_set_if_unset(Attr, ATTR_LABEL, "Unnamed");
		/* 
		 ** It is a little strange hack with the radiobuttons. Not easy
		 ** to force them work together.
		 */
		if (LastRadioButton == NULL) {
			Widget = gtk_radio_button_new_with_label(NULL,
								 attributeset_get_first
								 (Attr,
								  ATTR_LABEL));
			LastRadioButton = Widget;
		} else {
			Widget =
			    gtk_radio_button_new_with_label_from_widget
			    (GTK_RADIO_BUTTON(LastRadioButton),
			     attributeset_get_first(Attr, ATTR_LABEL));
//        gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(Widget),
//                                      FALSE );
		}

		/*
		 ** Radio buttons can have a default value.
		 */
		if (attributeset_cmp_left(Attr, ATTR_DEFAULT, "true") ||
		    attributeset_cmp_left(Attr, ATTR_DEFAULT, "yes"))
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON
						     (Widget), TRUE);

		/*
		 ** Radiobuttons also can take actions...
		 */
		if (attributeset_is_avail(Attr, ATTR_ACTION)) {
			char *act;
			act = attributeset_get_first(Attr, ATTR_ACTION);
			while (act != NULL) {
				gtk_signal_connect(GTK_OBJECT(Widget),
						   "toggled",
						   GTK_SIGNAL_FUNC
						   (button_toggled),
						   (gpointer) act);
				act =
				    attributeset_get_next(Attr,
							  ATTR_ACTION);
			}
		}
		
		push_widget(Widget, Widget_Type);

		if (attributeset_cmp_left(Attr, ATTR_VISIBLE, "disabled"))
			gtk_widget_set_sensitive(Widget, FALSE);
		break;

	case WIDGET_LIST:
		Widget = gtk_list_new();
      /** When we have create a new LIST widget, we also have to make
       ** GtkScrolledWindow, because this is the only way we can 
       ** scroll in the list. We put the _scrolled window_ into the 
       ** stack, because this holds the list.
       */
		{
			GtkWidget *Scrolledwindow;
			Scrolledwindow =
			    gtk_scrolled_window_new(NULL, NULL);
			if (attributeset_is_avail(Attr, ATTR_HEIGHT)
			    && attributeset_is_avail(Attr, ATTR_WIDTH))
				gtk_widget_set_usize(Scrolledwindow,
						     atoi
						     (attributeset_get_first
						      (Attr, ATTR_WIDTH)),
						     atoi
						     (attributeset_get_first
						      (Attr, ATTR_HEIGHT))
				    );
			else
				gtk_widget_set_usize(Scrolledwindow, 300,
						     100);

			gtk_scrolled_window_add_with_viewport
			    (GTK_SCROLLED_WINDOW(Scrolledwindow), Widget);

			push_widget(Scrolledwindow, WIDGET_SCROLLEDW);
		}
		/*
		 ** The LIST widget can handle actions...
		 */
		if (attributeset_is_avail(Attr, ATTR_ACTION)) {
			char *act;
			act = attributeset_get_first(Attr, ATTR_ACTION);
			while (act != NULL) {
				gtk_signal_connect(GTK_OBJECT(Widget),
						   "selection_changed",
						   GTK_SIGNAL_FUNC
						   (list_selection_changed),
						   (gpointer) act);
				act =
				    attributeset_get_next(Attr,
							  ATTR_ACTION);
			}
		}

		break;

	case WIDGET_TABLE:
		{
			list_t *table_header;
			//int columns;
			if (attributeset_is_avail(Attr, ATTR_LABEL)) {
				table_header =
				    linecutter(attributeset_get_first
					       (Attr, ATTR_LABEL), '|');
				Widget =
				    gtk_clist_new_with_titles
				    (table_header->n_lines,
				     table_header->line);
			} else {
				Widget = gtk_clist_new(1);
			}
		}
		/*
		 ** We need connect the select_row to a function to store the
		 ** selected row.
		 */
		g_signal_connect(G_OBJECT(Widget), "select_row",
				 G_CALLBACK(table_selection_made),
				 strdup(attributeset_get_first
					(Attr, ATTR_VARIABLE)));
		/*
		 **
		 */
		{
			char *act;
			act = attributeset_get_first(Attr, ATTR_ACTION);
			while (act != NULL) {
				gtk_signal_connect(GTK_OBJECT(Widget),
						   "select_row",
						   G_CALLBACK
						   (table_selected),
						   strdup(act)
				    );
				act =
				    attributeset_get_next(Attr,
							  ATTR_ACTION);
			}
		}


      /** When create a new TABLE (clist in gtk) widget, we 
       ** also have to create a GtkScrolledWindow, because this is the 
       ** only way we can scroll in the clist. We put the _scrolled 
       ** window_ into the stack, because this holds the list.
       */
		{
			GtkWidget *Scrolledwindow;

			gtk_clist_set_shadow_type(GTK_CLIST(Widget),
						  GTK_SHADOW_OUT);
			Scrolledwindow =
			    gtk_scrolled_window_new(NULL, NULL);
			gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW
						       (Scrolledwindow),
						       GTK_POLICY_AUTOMATIC,
						       GTK_POLICY_ALWAYS);

			if (attributeset_is_avail(Attr, ATTR_HEIGHT) &&
			    attributeset_is_avail(Attr, ATTR_WIDTH))
				gtk_widget_set_usize(Scrolledwindow,
						     atoi
						     (attributeset_get_first
						      (Attr, ATTR_WIDTH)),
						     atoi
						     (attributeset_get_first
						      (Attr, ATTR_HEIGHT))
				    );
			else
				gtk_widget_set_usize(Scrolledwindow, 200,
						     100);

			gtk_container_add(GTK_CONTAINER(Scrolledwindow),
					  Widget);

			push_widget(Scrolledwindow, WIDGET_SCROLLEDW);
		}		/*command group */

		break;
	case WIDGET_NOTEBOOK:
		{
			stackelement s;
			GtkWidget *Label;
			list_t *lines = NULL;
			
			char *labels = get_tag_attribute(Tag_Attributes, "labels=");
			if (labels != NULL)
				lines = linecutter(strdup(labels), '|');
				
			s = pop();
			Widget = gtk_notebook_new();
			
			for (n = 0; n < s.nwidgets; ++n){
				if (lines != NULL &&
						lines->n_lines > n)
					Label = gtk_label_new(lines->line[n]);
				else
					Label = gtk_label_new("Unnamed");
				
				gtk_notebook_append_page( Widget, s.widgets[n], Label);
			}
			push_widget(Widget, Widget_Type);
			LastRadioButton = NULL;
		}
		break;
	case WIDGET_VBOX:
		{
			/*
			 ** Creating a box is not a simple process, because we have to
			 ** add the widgets to it. First we remove the widgets from the
			 ** top of the stack (only one stack element with more widgets,
			 ** thanks to the SUM instruction), then we add it to the box.
			 ** The last step is pushing the box back to the stack.
			 */
			stackelement s;
			s = pop();
			Widget = gtk_vbox_new(FALSE, 5);
			for (n = 0; n < s.nwidgets; ++n)
				if (s.widgettypes[n] == WIDGET_EDIT ||
				    s.widgettypes[n] == WIDGET_FRAME ||
				    s.widgettypes[n] == WIDGET_SCROLLEDW)
					gtk_box_pack_start(GTK_BOX(Widget),
							   s.widgets[n],
							   TRUE, TRUE, 0);
				else
					gtk_box_pack_start(GTK_BOX(Widget),
							   s.widgets[n],
							   FALSE, FALSE,
							   0);

			push_widget(Widget, Widget_Type);
			/*
			 ** The box widgets are holding the radiobuttons to groups, so
			 ** one radiobutton can turn off the others. 
			 */
			LastRadioButton = NULL;
		}
		break;
	case WIDGET_HBOX:
		{
			/*
			 ** The hbox is very similar to vbox...
			 */
			stackelement s;
			s = pop();
			Widget = gtk_hbox_new(FALSE, 5);
			for (n = s.nwidgets - 1; n >= 0; --n)
				if (s.widgettypes[n] == WIDGET_EDIT ||
				    s.widgettypes[n] == WIDGET_FRAME ||
				    s.widgettypes[n] == WIDGET_SCROLLEDW ||
				    s.widgettypes[n] == WIDGET_ENTRY)
					gtk_box_pack_end(GTK_BOX(Widget),
							 s.widgets[n],
							 TRUE, TRUE, 0);
				else
					gtk_box_pack_end(GTK_BOX(Widget),
							 s.widgets[n],
							 FALSE, FALSE, 0);

			push_widget(Widget, Widget_Type);
			/*
			 ** The box widgets holds the radiobuttons to groups, so
			 ** one radiobutton can turn off the others. 
			 */
			LastRadioButton = NULL;
		}
		break;

	case WIDGET_FRAME:
		{
			stackelement s;
			GtkWidget *vbox;
			s = pop();
			vbox = gtk_vbox_new(FALSE, 5);
			gtk_container_set_border_width(GTK_CONTAINER(vbox),
						       5);

			for (n = 0; n < s.nwidgets; ++n)
				if (s.widgettypes[n] == WIDGET_EDIT ||
				    s.widgettypes[n] == WIDGET_FRAME ||
				    s.widgettypes[n] == WIDGET_SCROLLEDW)
					gtk_box_pack_start(GTK_BOX(vbox),
							   s.widgets[n],
							   TRUE, TRUE, 0);
				else
					gtk_box_pack_start(GTK_BOX(vbox),
							   s.widgets[n],
							   FALSE, FALSE,
							   0);

			if (attributeset_is_avail(Attr, ATTR_LABEL))
				Widget =
				    gtk_frame_new(attributeset_get_first
						  (Attr, ATTR_LABEL));
			else
				Widget = gtk_frame_new(NULL);
			// gtk_container_set_border_width(GTK_CONTAINER(Widget), 5);
			gtk_container_add(GTK_CONTAINER(Widget), vbox);
		}
		push_widget(Widget, Widget_Type);
		/*
		 ** The frame widget holds the radiobuttons to groups, so
		 ** one radiobutton can turn off the others. 
		 */
		LastRadioButton = NULL;
		break;
	default:
		fprintf(stderr, "%s(): Unknown widget.\n", __func__);
		fflush(stderr);
	}
#ifdef EXECUTION_DEBUG
	fprintf(stderr, "%s(): Widget created.\n", __func__);
	fflush(stderr);
#endif
	/*
	 ** Now we create a new variable or refresh the old one for this
	 ** new widget.
	 */
	variables_new_with_widget(Attr, Widget, Widget_Type);
	variables_set_attributes(attributeset_get_first
				 (Attr, ATTR_VARIABLE), Attr);
	variables_set_parent(attributeset_get_first(Attr, ATTR_VARIABLE),
			     window);
#ifdef EXECUTION_DEBUG
	fprintf(stderr, "%s(): Variable created.\n", __func__);
	fflush(stderr);
#endif

	return 0;
}


/* The next few functions operates on stack
   Miert nem kapja meg az adatokat parameterben es foglalkozik
   a veremmel a hivo?
 */
stackelement _sum(stackelement a, stackelement b)
{
	int n;
	/* Let's copy the widgets from b to a */
	for (n = 0; n < b.nwidgets; ++n) {
		a.widgets[a.nwidgets + n] = b.widgets[n];
		a.widgettypes[a.nwidgets + n] = b.widgettypes[n];
	}
	a.nwidgets += b.nwidgets;
	return (a);
}

void _more_memory_needed(void)
{
	void *memory = NULL;

	if (program == NULL) {
		memory_counter = 128;
		memory = malloc(memory_counter * sizeof(instruction));
	} else {
		memory_counter *= 2;
		memory =
		    realloc(program, memory_counter * sizeof(instruction));
	}

	if (memory == NULL) {
		fprintf(stderr, "%s: not enought memory.\n", __func__);
		exit(EXIT_FAILURE);
	}
#ifdef DEBUG
	fprintf(stderr, "%s: Memory allocated: %d.\n",
		__func__, memory_counter);
#endif

	program = memory;
}

void instruction_new(instruction new)
{
#ifdef EXECUTION_DEBUG
	fprintf(stderr,
		"%s(): instruction_counter=%d, memory_counter=%d\n",
		__func__, instruction_counter, memory_counter);
	fflush(stderr);
#endif
	if (instruction_counter == memory_counter)
		_more_memory_needed();

	program[instruction_counter++] = new;
}

/* 
** Functions to store a token in the memory 
*/
int token_store(token command)
{
	instruction inst;
	inst.command = command;
	inst.argument = NULL;
	inst.tag_attributes = NULL;
	instruction_new(inst);
	return TRUE;
}

int token_store_attr(token command, tag_attr *attributes){
	instruction inst;
	inst.command = command;
	inst.argument = NULL;
	inst.tag_attributes = attributes;
	instruction_new(inst);

	return TRUE;
}

int token_store_with_argument_attr(token command,
				   const char *argument,
				   tag_attr *attributes){
	instruction inst;
	int sub_attribute;
	
	inst.tag_attributes = attributes;

#ifdef EXECUTION_DEBUG
	if (argument == NULL) {
		fprintf(stderr, "%s(): argument is NULL.\n", __func__);
		fflush(stderr);
	}
#endif
	if (argument == NULL)
		argument = "";

#ifdef EXECUTION_DEBUG
	printf("%s(): Command: %08x '%s'\n", __func__, command, argument);
#endif

	sub_attribute = (command & SUB_ATTRIBUTE) >> 24;
	inst.argument = pip_malloc(strlen(argument) + 32, __func__);
	switch (sub_attribute) {
	case 0:
		/* 
		 ** No sub_attribute, it is just a normal text.
		 */
		inst.argument = strdup(argument);
		sprintf(inst.argument, "%s", argument);
		break;
	case 1:
		/*
		 ** The text argument is a shell command.
		 */
		sprintf(inst.argument, "Command:%s", argument);
		break;
	case 2:
		/* 
		 ** The text argument is a file name.
		 */
		sprintf(inst.argument, "File:%s", argument);
		break;
	case 3:
		/* 
		 ** The text argument is a widget name which must be
		 ** cleared.
		 */
		sprintf(inst.argument, "Clear:%s", argument);
		break;
	case 4:
		/*
		**
		*/
		sprintf(inst.argument, "Append:%s", argument);
		break;
	case 10:
		/* 
		 ** The text argument is widget name which must be filled with
		 ** a file name from a file selection dialog.
		 */
		sprintf(inst.argument, "Fileselect:%s", argument);
		break;
	case 11:
		/* 
		 ** The text argument is widget name and we have to remove the
		 ** selected element from it.
		 */
		sprintf(inst.argument, "Removeselected:%s", argument);
		break;
	case 12:
		/* 
		 ** The text argument is widget name and we have to remove the
		 ** selected element from it.
		 */
		sprintf(inst.argument, "Refresh:%s", argument);
		break;
	case 13:
		/* 
		 ** The text argument is widget name and we have to
		 ** remove the selected element from it.
		 **  
		 */
		sprintf(inst.argument, "Launch:%s", argument);
		break;
	default:
		fprintf(stderr, "%s(): Warning: Unknown sub-attribute %d",
			__func__, sub_attribute);
		fflush(stderr);
	}
	inst.command = command;
	instruction_new(inst);
	return (0);
	
}

int token_store_with_argument(
		token command, 
		const char *argument)
{
	tag_attr *attributes = new_tag_attributeset(
			strdup("text"),
			strdup(argument));
	
	return token_store_with_argument_attr(
			command, 
			argument,
			attributes);
}
