/* This file is for showing dependency informations
 * Gnometoaster depends on a variety of other programs to accomplish it's work.
 * Those programs are referenced by the respective filenames of their
 * executables. While this scheme works perfectly well, it is sometimes
 * difficult for a normal user to find out how to resolve a dependency
 * as chances aren't bad he won't even know what a specific executable
 * Gnometoaster needs is actually doing. 
 * To assist the user in that matter, dependencies.h/.c will maintain
 * a small database containing informations about the client programs
 * Gnometoaster needs. Those informations will contain an URL where
 * a specific program can be found and other useful informations.
 * For a start, we'll pop up those informations whenever something goes
 * wrong. In the future, we might have a browseable dependency database
 * or a help shortcut bringing up the desired informations depending on the
 * screen context */
#ifndef DEPENDENCIES_H
#define DEPENDENCIES_H

#include "int.h"

typedef struct 
{
   /* Name of the distribution */
   char *product_name;
   /* Manufacturer or Organization */
   char *vendor;
   /* Last known Version Number */
   char *version_number;
   /* Download locations */
   char **urls;
   /* This could possibly contains a few informations about this
    * specific distribution. */
   char *description;
} dependencies_distrib;

typedef struct 
{
   /* the executable called to invoked the described program */
   char *executable;
   /* Full product name */
   char *product_name;
   /* Vendor, Author etc. */
   char *vendor;
   /* Last known version number */
   char *version_number;
   /* Generic Download locations. Should point to the source distribution
    * archive where available */
   char **urls;
   /* A list of distributions providing this package,
    * including additional informations about a distribution.
    * Maybe provide a simple installation mechanism here someday ? */
   dependencies_distrib **supported_distribs;
   /* a list of URL-Lists where distribution-specific packages of the current
    * program can be obtained.
    * Each distrib has a separate list of download-URLs */
   char ***distrib_specific_urls;
   /* a full-blown description of the program and what it does within
    * the context of Gnometoaster */
   char *description;
} dependencies_client;

/* retrieve informations about the given command.
 * returns a dependencies_client struct */
dependencies_client *dependencies_getinfo(const char *command);
/* show the informations stored in the dependencies_client struct
 * plus a message possibly telling the user what went wrong. */
void dependencies_showinfo(const char *message,			  
			   dependencies_client *);
#define dependencies_showdep(command,message) \
   dependencies_showinfo(message,dependencies_getinfo(command))
void dependencies_showmissingexec(const char *command);

#endif // DEPENDENCIES_H
