/* this file defines a generic way to copy streams
 * from and into any filedescriptor.
 * If desired, a process id can simultaneously be watched so's an error
 * condition will be detected in time */

#ifndef STREAMCOPY_H
#define STREAMCOPY_H

#include <gtk/gtk.h>

#include "config.h"

#include "datacopydlg.h"

#ifdef HAVE_PTHREADS
#include <pthread.h>
#endif

typedef void (*streamcopy_callback)(gpointer state,gpointer data);

/* the following structure is used to control the recording process */
typedef struct
{
   /* this is the filedescriptor the track is being copied to */
   int inpipe;
   /* this pipe connects to the track pipe currently processed. */
   int trackpipe;   
   /* those are used to control the state of the receiving process */
   int receiverpid;
   int receiverstatus;
   /* used by the single thread variant only. It's the IO watch tag
    * used to determine when to write the next block of data */
   int receiverinpipetag;
   /* the following var represents the progress within the current track */
   int trackread;
   /* this is the current track's size */
   int tracksize;
   /* becomes true when a fatal error occurs. This means that the copying
    * process is to be terminated immediately */
   int fatalerror;
   /* becomes true if the read/write thread is in padding mode, that is:
    * instead of taking it's data from stdin, it write a bunch of zero
    * bytes */
   int padding;
   /* the progress dialog, to be accessed by the main thread only */
   datacopydlg_dlginfo *dlg;
   
   /* this int is used to synchronize the timeouthandler of the main thread
    * with the writing thread.
    * if this flag becomes 1, the writing thread has left the stage 
    * and a new one can be initialized */
   int trackdone;
   /* this int holds data about the writing process, what went wrong and
    * so on. It's to be analysed by the timeout handler */
   int writingstate;
   /* read since last display update */
   int rsldu;
   
   /* this is the tag used to control the timeout handler */
   int timeouttag;
      
   /* options like padding requested, see below */
   int options;

   /* the callback definition */
   gpointer callback;
   gpointer data;
   
   /* those two definitions are used and compiled in pthread mode only. 
    * They are holding informations about the writing thread as well as
    * a mutex to grant exclusive access to this structure */
#ifdef HAVE_PTHREADS
   pthread_mutex_t semaphore;
   pthread_t *copythread;
#endif   
} streamcopy_state;

/* those are the option definitions to be used */

/* pad the track up to tracksize */
#define STREAMCOPY_OPT_PAD 1
/* cut the track to tracksize if necessary */
#define STREAMCOPY_OPT_CUT 2
/* close the source filedescriptor when done */
#define STREAMCOPY_OPT_CLOSESOURCE 4
/* close the destination filedescriptor when done */
#define STREAMCOPY_OPT_CLOSEDEST 8
/* wait for the destination process to exit.
 * this option will cause a hang if not used in conjunction with
 * STREAMCOPY_OPT_CLOSEDEST for all processes that don't terminate
 * automatically after a certain number of bytes.
 * In DEBUG mode, this will yield a warning */
#define STREAMCOPY_OPT_WAITEXIT 16
/* send a SIGTERM to the destination process when the user clicks 
 * "Cancel". This is for backends that won't leave the scene
 * voluntarily. */
#define STREAMCOPY_OPT_SIGTERMONUSEREXT 32

streamcopy_state *streamcopy_create(/* destination */
				    int inpipe,
				    /* source */
				    int trackpipe,
				    /* receiving process pid or -1 */
				    int receiverpid,
				    /* bytes to copy */
				    int tracksize,
				    /* dialog to use or NULL if none */
				    datacopydlg_dlginfo *dlg,
				    /* options field (Bitmask,
				     * use definitions above) */
				    int options,
				    streamcopy_callback callback,
				    /* this callback will be called
				     * on termination of the streamcopy
				     * process. it will be passed the
				     * gpointer data field */
				    gpointer data
				    );

/* cancel the streamcopy process referred to by 'tag'.
 * 
 * This will cause 
 *   - the callback handler to be called 
 *   - all internals to be deinitialized
 *   - the streamcopy_state structure to be freed eventually.
 *     (eventually means that it's gonna be freed once callback got it's
 *     call as we still need this structure for that) */
void streamcopy_cancel(streamcopy_state *tag);

/* the end of the input track was reached */
#define STREAMCOPY_STATE_ENDOFTRACK 0
/* child exited unexpectedly */
#define STREAMCOPY_STATE_BACKENDEXT 1
/* couldn't create thread */
#define STREAMCOPY_STATE_PTHREADERR 2
/* copy process was terminated by user */
#define STREAMCOPY_STATE_USERTERM 3

#endif // STREAMCOPY_H
