/*
 * /etc/network/interfaces parser
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#define APPNAME PACKAGE
#else
#warning No config.h found: using fallback values
#define APPNAME __FILE__
#define VERSION "unknown"
#endif

#include "IfaceParser.h"
#include "stringf.h"
#include "Regexp.h"
#include "Environment.h"

#include <map>

using namespace std;
using namespace stringf;

class Tokenizer
{
protected:
	std::string str;
	std::string::size_type s;
public:
	Tokenizer(const std::string& str) throw ()
		: str(str), s(0) {}

	std::string next()
	{
		// Skip leading spaces
		while (s < str.size() && isspace(str[s]))
			s++;

		if (s == str.size()) return string();
		
		string::size_type start = s;

		while (s < str.size() && !isspace(str[s]))
			s++;

		return str.substr(start, s - start);
	}
};

/* Parse the input from `input'
 * To make it simple, use regexps on input lines instead of implementing a real
 * parser.
 */
void IfaceParser::parse(FILE* input, ScanConsumer& sc)
	throw (Exception)
{
	ExtendedRegexp null_line("^[[:blank:]]*(#.*)?$");
	ExtendedRegexp iface_line(
		"^[[:blank:]]*iface[[:blank:]]+"
		"([^[:blank:]]+)[[:blank:]]+"
		"([^[:blank:]]+)[[:blank:]]+"
		"([^[:blank:]]+)[[:blank:]]*$", 4);
	ExtendedRegexp parm_line(
		"^[[:blank:]]*([^[:blank:]]+)[[:blank:]]+(.+)$", 2);
	ExtendedRegexp old_peer_line(
		"^[[:blank:]]*guessnet[[:blank:]]+"
		"peer[[:blank:]]+"
		"([0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9]+)[[:blank:]]+"
		"([0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2}:[0-9A-Za-z]{2})"
		"[[:blank:]]*$", 3);
	ExtendedRegexp old_script_line(
		"^[[:blank:]]*guessnet[[:blank:]]+"
		"script[[:blank:]]+(.+)$", 2);
	ExtendedRegexp old_default_line(
		"^[[:blank:]]*guessnet[[:blank:]]+"
		"default[[:blank:]]*$");
	ExtendedRegexp cable_line(
		"^[[:blank:]]*(guessnet[0-9]*[[:blank:]]+)?test[0-9]*(-| )missing-cable[[:blank:]]*([[:blank:]]+.+)?$");
	ExtendedRegexp pppoe_line(
		"^[[:blank:]]*(guessnet[0-9]*[[:blank:]]+)?test[0-9]*(-| )pppoe[[:blank:]]*([[:blank:]]+.+)?$");
	ExtendedRegexp peer_line(
		"^[[:blank:]]*(guessnet[0-9]*[[:blank:]]+)?test[0-9]*(-| )peer[[:blank:]]+(.+)$", 4);
	ExtendedRegexp script_line(
		"^[[:blank:]]*(guessnet[0-9]*[[:blank:]]+)?test[0-9]*(-| )command[[:blank:]]+(.+)$", 4);
	ExtendedRegexp wireless_line(
		"^[[:blank:]]*(guessnet[0-9]*[[:blank:]]+)?test[0-9]*(-| )wireless-(ap|id|scan)[[:blank:]]+(.+)$", 5);
	string profileName;

	string line;
	int linenum = 1;
	int found = 0;
	int c;
	while ((c = fgetc(input)) != EOF)
	{
		if (c != '\n')
			line += c;
		else
		{
			if (null_line.match(line))
			{
				//fprintf(stderr, "EMPTY\n");
			}
			else if (iface_line.match(line))
			{
				//string name(line, parts[1].rm_so, parts[1].rm_eo - parts[1].rm_so);
				//string net(line, parts[2].rm_so, parts[2].rm_eo - parts[2].rm_so);
				//string type(line, parts[3].rm_so, parts[3].rm_eo - parts[3].rm_so);

				//fprintf(stderr, "IFACE: %.*s/%.*s/%.*s\n", PFSTR(name), PFSTR(net), PFSTR(type));
				profileName = iface_line[1];
			}
			else if (old_peer_line.match(line))
			{
				//string ip(line, parts[1].rm_so, parts[1].rm_eo - parts[1].rm_so);
				//string mac(line, parts[2].rm_so, parts[2].rm_eo - parts[2].rm_so);
				//fprintf(stderr, "PEER: %.*s/%.*s\n", PFSTR(ip), PFSTR(mac));
				if (profileName.size())
				{
					struct ether_addr macAddr;
					parse_mac(&macAddr, old_peer_line[2]);

					IPAddress ipAddr(old_peer_line[1]);

					//debug("Will scan network %.*s for %.*s (%.*s)\n",
					//		PFSTR(profileName), PFSTR(ip), PFSTR(mac));
					//fprintf(stderr, "  GOT: %.*s: %.*s, %.*s\n",
							//PFSTR(profileName), PFSTR(ip), PFSTR(mac));

					sc.handleScan(new PeerScan(profileName, macAddr, ipAddr));
					found++;
				}
			}
			else if (cable_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new LinkBeatScan(profileName));
					found++;
				}
			}
#ifdef HAVE_PPPOE
			else if (pppoe_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new ScriptScan(profileName, string(PPPOE " -I ") + Environment::get().iface() + " -A"));
					found++;
				}
			}
#endif
			else if (peer_line.match(line))
			{
				//string ip(line, parts[1].rm_so, parts[1].rm_eo - parts[1].rm_so);
				//string mac(line, parts[2].rm_so, parts[2].rm_eo - parts[2].rm_so);
				//fprintf(stderr, "PEER: %.*s/%.*s\n", PFSTR(ip), PFSTR(mac));
				if (profileName.size())
				{
					string argstr = peer_line[3];

					// split in a map of key->val
					map<string, string> args;
					string key;
					string val;
					enum { SKEY, KEY, SVAL, VAL } state = KEY;
					for (string::const_iterator s = argstr.begin();
							s != argstr.end(); s++)
					{
						//debug("Read `%c', state: %d\n", *s, (int)state);
						if (isspace(*s))
							switch (state)
							{
								case SKEY: break;
								case KEY: state = SVAL; break;
								case SVAL: break;
								case VAL:
									state = SKEY;
									//debug("Found args: %.*s: %.*s\n", PFSTR(key), PFSTR(val));
									args.insert(make_pair(key, val));
									key = string();
									val = string();
									break;
							}
						else
							switch (state)
							{
								case SKEY: key += *s; state = KEY; break;
								case KEY: key += *s; break;
								case SVAL: val += *s; state = VAL; break;
								case VAL: val += *s; break;
							}
					}
					if (key.size() > 0 && val.size() > 0)
						args.insert(make_pair(key, val));

					map<string, string>::const_iterator ip = args.find("address");
					map<string, string>::const_iterator mac = args.find("mac");

					if (ip != args.end())
					{
						struct ether_addr macAddr;
						if (mac != args.end())
							parse_mac(&macAddr, mac->second);
						else {
							debug("Missing mac at line %d: only check for the IP\n", linenum);
							bzero(&macAddr, sizeof(struct ether_addr));
						}

						IPAddress ipAddr(ip->second);

						//debug("Will scan network %.*s for %.*s (%.*s)\n",
						//		PFSTR(profileName), PFSTR(ip), PFSTR(mac));
						//fprintf(stderr, "  GOT: %.*s: %.*s, %.*s\n",
								//PFSTR(profileName), PFSTR(ip), PFSTR(mac));

						sc.handleScan(new PeerScan(profileName, macAddr, ipAddr));
						found++;
					} else {
						warning("Missing address at line %d: skipping line\n", linenum);
					}
				}
			}
			else if (old_script_line.match(line))
			{
				//string cmd(line, parts[1].rm_so, parts[1].rm_eo - parts[1].rm_so);
				//fprintf(stderr, "TEST: %.*s\n", PFSTR(cmd));
				if (profileName.size())
				{
					//debug("Will use script %.*s to test %.*s\n",
					//	PFSTR(cmd), PFSTR(profileName));

					sc.handleScan(new ScriptScan(profileName, old_script_line[1]));
					found++;
				}
			}
			else if (wireless_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new ScriptScan(profileName, string(SCRIPTDIR "/test-wifi-") + script_line[3] + " " + script_line[4]));
					found++;
				}
			}
			else if (script_line.match(line))
			{
				if (profileName.size())
				{
					sc.handleScan(new ScriptScan(profileName, script_line[3]));
					found++;
				}
			}
			else if (old_default_line.match(line))
			{
				warning("line %d: Use of \"guessnet default\" lines is obsolete and will be discontinued in the future.  Use \"map default: profile\" in the \"mapping\" section instead.\n", linenum);
				//fprintf(stderr, "DEFAULT\n");
				if (profileName.size())
				{
					//debug("Will use tag %.*s as default\n", PFSTR(profileName));

					Environment::get().defprof(profileName);
				}
			}
			else if (parm_line.match(line))
			{
				//string name(line, parts[1].rm_so, parts[1].rm_eo - parts[1].rm_so);
				//string parms(line, parts[2].rm_so, parts[2].rm_eo - parts[2].rm_so);
				//fprintf(stderr, "PARM: %.*s/%.*s\n", PFSTR(name), PFSTR(parms));
			}
			else
			{
				warning("Parse error at line %d: line ignored\n", linenum);
			}
			line = string();
			linenum++;
		}
	}
	debug("%d candidates found in input\n", found);
}

// vim:set ts=4 sw=4:
