/*
 * Thread to capture packets from a network
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "NetWatcher.h"

#include "Thread.h"
#include "Mutex.h"

extern "C" {
#include <pcap.h>
#include <libnet.h>
}

#include <list>
#include <queue>

#include <netinet/in.h>	// ntohs, htons, ...

#include "Environment.h"


using namespace std;

class NetWatcherImpl : public Thread
{
protected:
	static const int captureSize = LIBNET_ETH_H + (LIBNET_ARP_H >? LIBNET_IPV4_H) + (LIBNET_UDP_H >? ICMP_ECHO) + 300;

	int _ref;
	string iface;
	pcap_t *pcap_interface;
	bool _canceled;

	Mutex listenersMutex;

	list<PacketListener*> listeners_arp;
	list<PacketListener*> listeners_ethernet;
	list<PacketListener*> listeners_dhcp;
	list<PacketListener*> listeners_icmp;

	struct libnet_link_int* link_interface;

	virtual void* main() throw ();

public:
	NetWatcherImpl(const string& iface) throw (SystemException, pcapException)
		: _ref(0), iface(iface), _canceled(false)
	{
		char errbuf[PCAP_ERRBUF_SIZE];

		// TODO: Try to use 1 from "promisc": maybe there won't be a need to setup an
		// address on the interface
		if (!(pcap_interface = pcap_open_live (
						(char*)iface.c_str(), captureSize, 1, 500, errbuf)))
			throw pcapException (errbuf, "initializing pcap packet capture library");
	}
	~NetWatcherImpl() throw ()
	{
		pcap_close(pcap_interface);
	}

	bool canceled() const throw () { return _canceled; }
	bool canceled(bool val) throw () { return _canceled = val; }

	/// Increment the reference count for this object
	void ref() throw () { ++_ref; }

	/// Decrement the reference count for this object, returning true when it
	/// reaches 0
	bool unref() throw () { return --_ref == 0; }

	void addARPListener(PacketListener* pl) throw ()
	{
		MutexLock lock(listenersMutex);
		listeners_arp.push_back(pl);
	}

	void addEthernetListener(PacketListener* pl) throw ()
	{
		MutexLock lock(listenersMutex);
		listeners_ethernet.push_back(pl);
	}

	void addDHCPListener(PacketListener* pl) throw ()
	{
		MutexLock lock(listenersMutex);
		listeners_dhcp.push_back(pl);
	}

	void addICMPListener(PacketListener* pl) throw ()
	{
		MutexLock lock(listenersMutex);
		listeners_icmp.push_back(pl);
	}
};

/*
static void memdump(const string& prefix, unsigned char* mem, int size) throw ()
{
	warning("%.*s", PFSTR(prefix));
	for (int i = 0; i < size; i++)
	{
		warning(" %02x", (int)mem[i]);
	}
	warning("\n");
}
*/

void* NetWatcherImpl::main() throw ()
{
	struct pcap_pkthdr pcap_header;

	// Let the signals be caught by some other process
	sigset_t sigs, oldsigs;
	sigfillset(&sigs);
	sigdelset(&sigs, SIGFPE);
	sigdelset(&sigs, SIGILL);
	sigdelset(&sigs, SIGSEGV);
	sigdelset(&sigs, SIGBUS);
	sigdelset(&sigs, SIGABRT);
	sigdelset(&sigs, SIGIOT);
	sigdelset(&sigs, SIGTRAP);
	sigdelset(&sigs, SIGSYS);
	pthread_sigmask(SIG_SETMASK, &sigs, &oldsigs);

	try {
		while (true)
		{
			unsigned char* packet = (u_char *)pcap_next (pcap_interface, &pcap_header);
			if (packet == 0)
				throw pcapException (pcap_geterr(pcap_interface), "getting a new packet from the network");

			NetBuffer pkt(packet, captureSize, false);

			MutexLock lock(listenersMutex);
			if (!listeners_ethernet.empty())
				for (list<PacketListener*>::iterator i = listeners_ethernet.begin();
						i != listeners_ethernet.end(); i++)
					(*i)->handleEthernet(pkt);

			const libnet_ethernet_hdr* packet_header = pkt.cast<struct libnet_ethernet_hdr>();
			if (ntohs (packet_header->ether_type) == ETHERTYPE_ARP)
			{
				if (!listeners_arp.empty())
				{
					// ARP packet
					NetBuffer arp = pkt.after(LIBNET_ETH_H);
					for (list<PacketListener*>::iterator i = listeners_arp.begin();
							i != listeners_arp.end(); i++)
						(*i)->handleARP(arp);
				}
			}
			else if (ntohs (packet_header->ether_type) == ETHERTYPE_IP)
			{
				if (!(listeners_dhcp.empty() || listeners_icmp.empty()))
				{
					// IPv4 packet
					NetBuffer ipv4 = pkt.after(LIBNET_ETH_H);

					const libnet_ipv4_hdr* ipv4_header = ipv4.cast<struct libnet_ipv4_hdr>();
/* If needed again, add a memdump method to NetBuffer
debug("--IP proto %d src %x dst %x len %d\n",
			(int)ipv4_header->ip_p,
			*(int*)&(ipv4_header->ip_src),
			*(int*)&(ipv4_header->ip_dst),
			(int)(ipv4_header->ip_hl*4));
memdump("IP: ", (unsigned char*)ipv4_header, (int)(ipv4_header->ip_hl*4) + LIBNET_UDP_H + 200);
*/

					if (ipv4_header->ip_p == IPPROTO_UDP && *(uint32_t*)&(ipv4_header->ip_dst) == 0xffffffff)
					{
						// UDP packet
						NetBuffer udp = ipv4.after(ipv4_header->ip_hl*4);
						const libnet_udp_hdr* udp_header = udp.cast<struct libnet_udp_hdr>();
						int sport = ntohs(udp_header->uh_sport);
						int dport = ntohs(udp_header->uh_dport);
//memdump("UDP: ", (unsigned char*)udp_header, LIBNET_UDP_H + 20);
//debug("---UDP %d->%d len %d\n", sport, dport, ntohs(udp_header->uh_ulen));

						if (sport == 67 && dport == 68)
						{
//debug("----DHCP\n");
							// DHCP packet
							NetBuffer dhcp = udp.after(LIBNET_UDP_H);
							for (list<PacketListener*>::iterator i = listeners_dhcp.begin();
									i != listeners_dhcp.end(); i++)
								(*i)->handleDHCP(dhcp);
						}
					} else if (ipv4_header->ip_p == IPPROTO_ICMP) {
						// ICMP packet
						NetBuffer icmp = ipv4.after(ipv4_header->ip_hl*4);
						for (list<PacketListener*>::iterator i = listeners_icmp.begin();
								i != listeners_icmp.end(); i++)
							(*i)->handleICMP(icmp);
					}
				}
			}
		}
	} catch (Exception& e) {
		error("%s: %.*s.  Quitting NetWatcher thread.\n", e.type(), PFSTR(e.desc()));
	}
	return 0;
}



NetWatcher::NetWatcher(const string& iface) throw (SystemException, pcapException)
{
	impl = new NetWatcherImpl(iface);
	impl->ref();
	impl->start();
}

NetWatcher::NetWatcher(const NetWatcher& ns) throw ()
{
	if (ns.impl)
		ns.impl->ref();
	impl = ns.impl;
}

NetWatcher::~NetWatcher() throw (SystemException)
{
	if (impl && impl->unref())
	{
		shutdown();
		delete impl;
	}
}

NetWatcher& NetWatcher::operator=(const NetWatcher& ns) throw (SystemException)
{
	if (ns.impl)
		ns.impl->ref();  // Do it early to correctly handle the case of x = x;
	if (impl && impl->unref())
	{
		shutdown();
		delete impl;
	}
	impl = ns.impl;
	return *this;
}

void NetWatcher::addARPListener(PacketListener* pl) throw ()
{
	return impl->addARPListener(pl);
}

void NetWatcher::addEthernetListener(PacketListener* pl) throw ()
{
	return impl->addEthernetListener(pl);
}

void NetWatcher::addDHCPListener(PacketListener* pl) throw ()
{
	return impl->addDHCPListener(pl);
}

void NetWatcher::shutdown() throw ()
{
	if (!impl->canceled())
		try {
			impl->cancel();
			impl->join();
			impl->canceled(true);
		} catch (SystemException& e) {
			warning("%s: %.*s when shutting down NetWatcher\n", e.type(), PFSTR(e.desc()));
		}
}

// vim:set ts=4 sw=4:
