//$Id: rcp-saver.cc,v 1.3 2004/03/21 17:55:38 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2002 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "rcp-saver.h"

#include <sstream>

#include "blob-res.h"
#include "dialog-res.h"
#include "menu-res.h"
#include "string-res.h"
#include "stringlist-res.h"
#include "bitmap-res.h"
#include "bitmapfamily-res.h"

#include "rcp-save-form.h" // TODO: change this to a Visitor

#include "config.h" // for VERSION

namespace Guikachu
{
    
    namespace IO
    {
        
        namespace RCP
        {
            class ResourceSaver: public ResourceVisitor
            {
                std::ostream &ostr;
                GnomeVFSURI  *parent_uri;
                
            public:
                ResourceSaver (std::ostream &ostr_, GnomeVFSURI *parent_uri_):
                    ostr (ostr_),
                    parent_uri (gnome_vfs_uri_ref (parent_uri_))
                    {};
                
                ~ResourceSaver ()
                    {
                        gnome_vfs_uri_unref (parent_uri);
                    }
                
                void save_app      (Resources::Application *app);
                
                void visit_resource (Resources::Dialog       *res);
                void visit_resource (Resources::String       *res);
                void visit_resource (Resources::StringList   *res);
                void visit_resource (Resources::Menu         *res);
                void visit_resource (Resources::Form         *res);
                void visit_resource (Resources::Blob         *res);
                void visit_resource (Resources::Bitmap       *res);
                void visit_resource (Resources::BitmapFamily *res);
            };
        }
    }
}

using namespace Guikachu;
using namespace Guikachu::IO;

IOResult RCPSaver::save (ResourceManager *manager, const std::string &uri)
{
    std::stringstream stream;

    time_t curr_date = time (0);
    const char* date_str = ctime (&curr_date);
    
    stream << "/*" << std::endl;
    stream << " * This file was created by Guikachu " << VERSION
	   << ". Do not edit it manually." << std::endl;
    stream << " * Creation date: " << date_str; // ctime ()'s return value ends in \n
    stream << " */" << std::endl << std::endl;;
    
    const std::set<Resource*> &resources = manager->get_resources ();
    
    GnomeVFSURI *orig_uri = gnome_vfs_uri_new (uri.c_str ());
    GnomeVFSURI *parent_uri = gnome_vfs_uri_get_parent (orig_uri);
    RCP::ResourceSaver resource_saver (stream, parent_uri);
    gnome_vfs_uri_unref (orig_uri);    
    gnome_vfs_uri_unref (parent_uri);    
    
    // Export blobs before anything else, to allow #includes and whatnot
    for (std::set<Resource*>::const_iterator i = resources.begin ();
         i != resources.end (); i++)
    {
        if ((*i)->get_type () == Resources::RESOURCE_BLOB)
            (*i)->apply_visitor (resource_saver);
    }
    
    // Export per-app data
    resource_saver.save_app (manager->get_application ());
    
    // Export the rest of the resources
    for (std::set<Resource*>::const_iterator i = resources.begin ();
	 i != resources.end (); i++)
    {
        if ((*i)->get_type () != Resources::RESOURCE_BLOB)
            (*i)->apply_visitor (resource_saver);
    }
    
    // Write buffer to URI
    GnomeVFSHandle *vfs_handle = 0;
    IOResult res = gnome_vfs_create (&vfs_handle, uri.c_str (),
				     GNOME_VFS_OPEN_WRITE, 0, 0644);
    if (res != GNOME_VFS_OK)
	return res;
    
    const std::string       stream_str = stream.str ();
    const char             *stream_start = stream_str.c_str ();
    GnomeVFSFileSize        stream_size = stream_str.length ();
    const GnomeVFSFileSize  chunk_size = 1024;
    
    for (GnomeVFSFileSize bytes_written = 0; stream_size; stream_start += bytes_written)
    {
        res = gnome_vfs_write (vfs_handle, stream_start, std::min (chunk_size, stream_size), &bytes_written);
        stream_size -= bytes_written;
        
        if (res != GNOME_VFS_OK)
            return gnome_vfs_close (vfs_handle);
    }

    return gnome_vfs_close (vfs_handle);
}

void RCP::ResourceSaver::save_app (Resources::Application *app)
{
    if (app->version != "")
	ostr << "VERSION ID 1 \"" << app->version << "\"" << std::endl;
    if (app->iconname != "")
	ostr << "APPLICATIONICONNAME ID 1 \"" << app->iconname << "\"" << std::endl;
    if (app->vendor != "")
	ostr << "APPLICATION ID 1 \"" << app->vendor << "\"" << std::endl;
    ostr << std::endl;
}

void RCP::ResourceSaver::visit_resource (Resources::Blob *res)
{
    // We use contents.get_val() instead of contents to avoid the
    // escaping of quote marks
    
    ostr << "/* Contents of blob '" << res->id << "' */" << std::endl;
    ostr << res->contents.get_val () << std::endl << std::endl;
}


void RCP::ResourceSaver::visit_resource (Resources::Dialog *res)
{
    ostr << "ALERT ID " << res->id << std::endl;

    switch (res->dialog_type)
    {
    case Resources::Dialog::TYPE_INFORMATION:
	ostr << "INFORMATION" << std::endl;
	break;
    case Resources::Dialog::TYPE_CONFIRMATION:
	ostr << "CONFIRMATION" << std::endl;
	break;
    case Resources::Dialog::TYPE_WARNING:
	ostr << "WARNING" << std::endl;
	break;
    case Resources::Dialog::TYPE_ERROR:
	ostr << "ERROR" << std::endl;
	break;
    }
    
    ostr << "DEFAULTBUTTON " << res->default_button << std::endl;
    
    if (res->help_id != "")
	ostr << "  HELPID " << res->help_id << std::endl;

    ostr << "BEGIN" << std::endl;

    ostr << "  TITLE \"" << res->title << "\"" << std::endl
	 << "  MESSAGE \"" << res->text << "\"" << std::endl;

    const std::vector<std::string> &buttons = res->buttons;
    if (buttons.size ())
    {
	ostr << "  BUTTONS ";

	for (std::vector<std::string>::const_iterator i = buttons.begin ();
	     i !=  buttons.end (); i++)
	    ostr << "\"" << escape_backslashes (*i) << "\" ";
	
	ostr << std::endl;
    }
    
    ostr << "END" << std::endl << std::endl;
}

void RCP::ResourceSaver::visit_resource (Resources::Menu *res)
{
    ostr << "MENU ID " << res->id << std::endl;

    ostr << "BEGIN" << std::endl;

    const Resources::Menu::MenuTree &submenus = res->get_submenus ();
    for (Resources::Menu::MenuTree::const_iterator i = submenus.begin ();
	 i != submenus.end (); i++)
    {
	ostr << "  PULLDOWN \"" << i->label << "\"" << std::endl;
	
	ostr << "  BEGIN" << std::endl;
	for (std::vector<Resources::Menu::MenuItem>::const_iterator j = i->items.begin ();
	     j != i->items.end (); j++)
	{
	    if (j->separator)
		ostr << "    MENUITEM SEPARATOR" << std::endl;
	    else
	    {
		ostr << "    MENUITEM \"" << j->label << "\" ID " << j->id;
		if (j->shortcut)
		    ostr << " \"" << j->shortcut << "\"";
		ostr << std::endl;
	    }
	}
	ostr << "  END" << std::endl;
    }
    
    ostr << "END" << std::endl << std::endl;
}

void RCP::ResourceSaver::visit_resource (Resources::String *res)
{
    ostr << "STRING ID " << res->id << " \"" << res->text << "\"" << std::endl
	 << std::endl;
}

void RCP::ResourceSaver::visit_resource (Resources::StringList *res)
{
    ostr << "STRINGTABLE ID " << res->id
	 << " \"" << res->prefix << "\"" << std::endl;

    ostr << " ";
    const std::vector<std::string> &strings = res->strings;
    for (std::vector<std::string>::const_iterator i = strings.begin ();
	 i != strings.end (); i++)
	ostr << " \"" << escape_backslashes (*i) << "\"";
    
    ostr << std::endl << std::endl;
}

void RCP::ResourceSaver::visit_resource (Resources::Form *res)
{
    RCPSaver_funcs::save_res_form (res, ostr);
}
    
void RCP::ResourceSaver::visit_resource (Resources::Bitmap *res)
{
    std::string resource_type;
    switch (res->bitmap_type)
    {
    case Resources::Bitmap::TYPE_MONO:
        resource_type = "BITMAP";
        break;
    case Resources::Bitmap::TYPE_GREY_4:
        resource_type = "BITMAPGREY";
        break;
    case Resources::Bitmap::TYPE_GREY_16:
        resource_type = "BITMAPGREY16";
        break;
    case Resources::Bitmap::TYPE_COLOR_16:
        resource_type = "BITMAPCOLOR16";
        break;
    case Resources::Bitmap::TYPE_COLOR_256:
        resource_type = "BITMAPCOLOR";
        break;
    case Resources::Bitmap::TYPE_COLOR_16K:
	resource_type = "BITMAPCOLOR16K";
	break;
    }

    std::string bitmap_filename = res->id () + ".bmp";
    GnomeVFSURI *bitmap_uri =
	gnome_vfs_uri_append_file_name (parent_uri, bitmap_filename.c_str ());
    g_return_if_fail (bitmap_uri);
    char *bitmap_uri_str = gnome_vfs_uri_to_string (bitmap_uri, GNOME_VFS_URI_HIDE_NONE);
    gnome_vfs_uri_unref (bitmap_uri);

    res->save_file_bmp (bitmap_uri_str);
    g_free (bitmap_uri_str);
    
    ostr << resource_type << " ID " << res->id << " \"" << bitmap_filename << "\"" << std::endl;
    ostr << std::endl;
}

namespace
{
    void save_bitmapfamily_image (std::ostream &ostr, GnomeVFSURI *parent_uri,
                                  Resources::BitmapFamily *res,
                                  Resources::Bitmap::BitmapType type, int bpp)
    {
        if (!res->get_image (type).isValid ())
            return;

        char *bitmap_filename_str = g_strdup_printf ("%s-%dbpp.bmp", res->id ().c_str (), bpp);
        std::string bitmap_filename (bitmap_filename_str);
        g_free (bitmap_filename_str);
        GnomeVFSURI *bitmap_uri =
            gnome_vfs_uri_append_file_name (parent_uri, bitmap_filename.c_str ());
        g_return_if_fail (bitmap_uri);
        
        char *bitmap_uri_str = gnome_vfs_uri_to_string (bitmap_uri, GNOME_VFS_URI_HIDE_NONE);
        gnome_vfs_uri_unref (bitmap_uri);

        res->save_file_bmp (type, bitmap_uri_str);
        g_free (bitmap_uri_str);

            
        ostr << "  BITMAP " << " \"" << bitmap_filename << "\" "
             << "BPP " << bpp << std::endl;
    }
    
} // anonymous namespace

void RCP::ResourceSaver::visit_resource (Resources::BitmapFamily *res)
{

    ostr << "BITMAPFAMILY ID " << res->id << std::endl
         << "COMPRESS" << std::endl
         << "BEGIN" << std::endl;
    
    save_bitmapfamily_image (ostr, parent_uri, res, Resources::Bitmap::TYPE_MONO,      1);
    save_bitmapfamily_image (ostr, parent_uri, res, Resources::Bitmap::TYPE_GREY_4,    2);
    save_bitmapfamily_image (ostr, parent_uri, res, Resources::Bitmap::TYPE_GREY_16,   4);
    save_bitmapfamily_image (ostr, parent_uri, res, Resources::Bitmap::TYPE_COLOR_16,  5);
    save_bitmapfamily_image (ostr, parent_uri, res, Resources::Bitmap::TYPE_COLOR_256, 8);
    save_bitmapfamily_image (ostr, parent_uri, res, Resources::Bitmap::TYPE_COLOR_16K, 16);

    ostr << "END" << std::endl << std::endl;
}
