//$Id: mainwin-io.cc,v 1.32 2004/03/24 17:03:20 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"
#include <libgnome/libgnome.h>

#include "mainwin.h"
#include "ui.h"

#include "resource-manager.h"
#include "session.h"
#include <gnome--/main.h>
#include <gnome--/messagebox.h>

#include <gtk--/optionmenu.h>
#include <gtk--/menu.h>
#include <gtk--/menuitem.h>

#include "io/rcp-saver.h"

#include "argumentsink.h"
#include <sigc++/retbind.h>

using namespace Guikachu;

namespace
{
std::string time_ago_string (time_t ago)
{
    std::string  ret_val;
    gchar       *buf = 0;
    
    int minutes = ago / 60;
    int hours = minutes / 60;
    int days = hours / 24;

    // Although we discriminate by pluralness ourselves, ngettext() is
    // still used, to help languages where there are multiple plurals
    // based on the actual number
    if (days > 1)
	buf = g_strdup_printf (
	    ngettext ("If you don't save, your work from the last %d day will be lost.",
		      "If you don't save, your work from the last %d days will be lost.",
		      days), days);
    else if (hours > 1)
	buf = g_strdup_printf (
	    ngettext ("If you don't save, your work from the last %d hour will be lost.",
		      "If you don't save, your work from the last %d hours will be lost.",
		      hours), hours);
    else if (minutes > 1)
	buf = g_strdup_printf (
	    ngettext ("If you don't save, your work from the last %d minute will be lost.",
		      "If you don't save, your work from the last %d minutes will be lost.",
		      minutes), minutes);
    else
	buf = g_strdup_printf (_("If you don't save, your work from "
				 "the last minute will be lost."));
    
    ret_val = buf;
    g_free (buf);
    return ret_val;
}

class OpenFileSel: public Gtk::FileSelection
{
    std::string mime_type;
public:
    OpenFileSel (Gtk::Window &parent);
    
    SigC::Signal2<void, std::string, std::string> open_clicked;

private:
    void close_clicked_cb ();
    void open_clicked_cb ();

    void filetype_change_cb (const char *mime_type);
};
    
} // anonymous namespace

OpenFileSel::OpenFileSel (Gtk::Window &parent):
    mime_type ("application/x-guikachu")
{
    Gtk::Box *box = get_action_area ();
    Gtk::HBox *ftype_hbox = new Gtk::HBox (false, 5);

    Gtk::OptionMenu *ftype_list = new Gtk::OptionMenu;
    Gtk::Menu *ftype_menu = new Gtk::Menu;

#define ADD_FILETYPE(label,mime)                           		\
    ftype_menu->items ().push_back (Gtk::Menu_Helpers::MenuElem ( 	\
        label, SigC::bind (SigC::slot (this, &OpenFileSel::filetype_change_cb), mime)));

    ADD_FILETYPE(_("Guikachu document"), "application/x-guikachu");
    ADD_FILETYPE(_("PilRC document"), "text/x-rcp");

#undef ADD_FILETYPE
    
    ftype_menu->show_all ();
    ftype_list->set_menu (*manage (ftype_menu));

    ftype_hbox->pack_end (*manage (ftype_list), false, true, 0);
    ftype_hbox->pack_end (*manage (new Gtk::Label (_("File Type:"))),
			    false, true, 0);
    
    ftype_hbox->show_all ();
    box->add (*manage (ftype_hbox));
    
    hide_fileop_buttons ();
    set_title (_("Select Guikachu file to open"));

    get_ok_button ()->clicked.connect (
	SigC::slot (this, &OpenFileSel::open_clicked_cb));
    get_cancel_button ()->clicked.connect (
	SigC::slot (this, &OpenFileSel::close_clicked_cb));
    delete_event.connect (
	SigC::retbind (
	    SigC::hide<void, GdkEventAny*> (
		SigC::slot (this, &OpenFileSel::close_clicked_cb)),
	    1));
    
    set_transient_for (parent);
    set_modal (true);
}

void OpenFileSel::close_clicked_cb ()
{
    delete this;
    Gtk::Main::quit ();
}

void OpenFileSel::open_clicked_cb ()
{
    open_clicked.emit (get_filename (), mime_type);

    close_clicked_cb ();
}

void OpenFileSel::filetype_change_cb (const char *mime_type_)
{
    mime_type = mime_type_;
}

bool GUI::MainWin::check_save ()
{
    if (!manager->is_dirty ())
    {
	if (in_exit)
	    Gnome::Main::quit ();
	return true;
    }
    
    int   response;
    int   time_since_save = time (0) - manager->get_modification_time ();
    char *filename_message;

    if (filename != "")
	filename_message = g_strdup_printf (
	    _("Save changes to document \"%s\"?"),
	    UI::visible_filename (filename).c_str ());
    else
	filename_message = g_strdup (
	    _("Save changes to the current document?"));

    std::string message = filename_message;
    message += "\n\n" + time_ago_string (time_since_save);

    g_free (filename_message);

    Gnome::Dialog *dialog = new Gnome::MessageBox (
	message, GNOME_MESSAGE_BOX_WARNING, std::vector<std::string>());

    static const int save_btn = 2;
    static const int discard_btn = 0;
    static const int cancel_btn = 1;
    
    dialog->append_button (_("Don't Save"), GNOME_STOCK_PIXMAP_TRASH);
    dialog->append_button (GNOME_STOCK_BUTTON_CANCEL);
    dialog->append_button (_("Save"), GNOME_STOCK_PIXMAP_SAVE);

    dialog->set_modal (true);
    dialog->set_default (save_btn);
    gnome_dialog_grab_focus (dialog->gtkobj (), save_btn);
    
    response = dialog->run_and_close ();

    if (response == cancel_btn || response == -1) // Cancel
	return false;

    if (response == discard_btn)
	if (in_exit)
	    Gnome::Main::quit ();
    
    if (response == save_btn)
    {
	save_cb ();
	//if (manager->is_dirty ())
	    return check_save ();
    }
	    
    return true;
}

void GUI::MainWin::new_cb ()
{
    // Check if file has changed
    if (!check_save ())
	return;

    Guikachu::Main::instance ()->new_doc ();
}

void GUI::MainWin::set_filename (const std::string &filename_)
{
    filename = filename_;
    update_title ();
}

void GUI::MainWin::save_cb ()
{
    if (filename == "")
	save_as_cb ();
    else
	Guikachu::Main::instance ()->save_doc (filename);
}

void GUI::MainWin::save_as_cb ()
{
    Gtk::FileSelection *fs = new Gtk::FileSelection ();

    fs->show_fileop_buttons ();
    fs->set_title (_("Save Guikachu file"));

    fs->get_ok_button ()->clicked.connect (SigC::bind (
	SigC::slot (this, &GUI::MainWin::filesel_save_cb), fs));
    fs->get_cancel_button ()->clicked.connect (SigC::bind (
	SigC::slot (this, &GUI::MainWin::filesel_close_cb), fs));
    fs->delete_event.connect (
	SigC::retbind (
	    SigC::hide<void, GdkEventAny*> (SigC::bind (
		SigC::slot (this, &GUI::MainWin::filesel_close_cb), fs)),
	    1));

    fs->set_transient_for (*this);
    fs->set_modal (true);
    fs->show ();
    Gtk::Main::run ();
}

void GUI::MainWin::load_cb ()
{
    OpenFileSel *fs = new OpenFileSel (*this);
    fs->open_clicked.connect (SigC::slot (this, &GUI::MainWin::open));

    fs->show ();
    Gtk::Main::run ();
}

void GUI::MainWin::export_cb ()
{
    Gtk::FileSelection *fs = new Gtk::FileSelection ();

    fs->show_fileop_buttons ();
    fs->set_title (_("Export to RCP"));
    
    fs->get_ok_button ()->clicked.connect (SigC::bind (
	SigC::slot (this, &GUI::MainWin::filesel_export_cb), fs));
    fs->get_cancel_button ()->clicked.connect (SigC::bind (
	SigC::slot (this, &GUI::MainWin::filesel_close_cb), fs));
    fs->delete_event.connect (
	SigC::retbind (
	    SigC::hide<void, GdkEventAny*> (SigC::bind (
		SigC::slot (this, &GUI::MainWin::filesel_close_cb), fs)),
	    1));

    fs->set_transient_for (*this);
    fs->set_modal (true);
    fs->show ();
    Gtk::Main::run ();
}

void GUI::MainWin::open (std::string filename, std::string mime_type)
{
    // Prompt the user about saving the current doc if it's changed
    if (!check_save ())
	return;

    Guikachu::Main::instance ()->load_doc (filename, mime_type);
}

/*********************************
 * File I/O dialog callbacks
 *********************************/
void GUI::MainWin::filesel_close_cb (Gtk::FileSelection *fs)
{
    delete fs;
    Gtk::Main::quit ();
}

void GUI::MainWin::filesel_save_cb (Gtk::FileSelection *fs)
{
    desired_filename = fs->get_filename ();
    filesel_close_cb (fs);

    if (*desired_filename.rbegin () == '/')
	desired_filename = "";

    if (desired_filename != "")
	Guikachu::Main::instance ()->save_doc (desired_filename);
}

void GUI::MainWin::filesel_export_cb (Gtk::FileSelection *fs)
{
    std::string filename = fs->get_filename ();
    filesel_close_cb (fs);

    IO::Saver    *rcp_saver = new IO::RCPSaver;
    IO::IOResult  res = rcp_saver->save (manager, filename);
    if (res != GNOME_VFS_OK)
    {
	const char *error_message = gnome_vfs_result_to_string (res);
	char *message = g_strdup_printf (
	    _("Error exporting to `%s':\n%s"),
	    g_basename (filename.c_str ()), error_message);

	UI::show_error (message);
	g_free (message);
    }
    
    delete rcp_saver;
}

int GUI::MainWin::delete_event_impl (GdkEventAny *e)
{
    in_exit = true;

    if (!check_save ())
	in_exit = false;

    return true;
}
