##############################################################################
# guml: GUI UML Management Love
#
# An abstract interface for UML instances.
#
# Copyright (C) 2005 Matthew Palmer
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation (version 2 of the License)
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##############################################################################               

import re
from ConfigParser import SafeConfigParser, NoSectionError
import time
import os
import gtk
import gobject
import vte
import sys
import string
import guml_debug
import md5

def list():
	"""Return the controller objects for all of the UMLs available in a
	   dictionary, where the key is the umid and the value is the
	   controller object for the UML."""

	namematch = re.compile('^([a-zA-Z0-9_-]+)\.conf$')
	
	conflist = {}
	for entry in os.listdir("%s/.guml" % os.getenv("HOME")):
		matches = namematch.match(entry)
		if matches:
			umid = matches.group(1)
			conflist[umid] = controller(umid)
	
	return conflist

class controller:
	def __init__(self, umid):
		guml_debug.prn("New controller")
		self.config = config(umid)
		self.running = False
		
	def start(self):
		if self.running:
			# I don't think so, Tim
			raise AlreadyRunning
		self.running = True
		self.console = console(self.config.command_line())
		self.console.connect('destroy', self.controller_dead)
		
		self.do_the_gui()

	def finished(self):
		"""Signal to the controller that the UML has finished."""
		self.running = False
		
	def hardhalt(self, widget, data = None):
		os.system("uml_mconsole %s halt >/dev/null" % self.config.umid)
		
	def ctrlaltdel(self, widget, data = None):
		os.system("uml_mconsole %s cad >/dev/null" % self.config.umid)

	def do_the_gui(self):
		textlabel = gtk.Label(self.config.name)
		self.label = gtk.EventBox()
		self.label.add(textlabel)
		self.label.connect('event', self.gimme_a_menu)
		self.label.set_visible_window(False)
		self.label.show()
		textlabel.show()

		self.menu = gtk.Menu()
		cad = gtk.MenuItem("CtrlAltDel")
		cad.connect('activate', self.ctrlaltdel)
		self.menu.append(cad)
		cad.show()
		hardhalt = gtk.MenuItem("Hard Halt")
		hardhalt.connect('activate', self.hardhalt)
		self.menu.append(hardhalt)
		hardhalt.show()

	def gimme_a_menu(self, widget, event):
		if event.type == gtk.gdk.BUTTON_PRESS and event.button == 3:
			self.menu.popup(None, None, None, event.button, event.time)

	def controller_dead(self, widget, data = None):
		self.running = False

class console(gtk.Notebook):
	def __init__(self, command_line):
		gtk.Notebook.__init__(self)
		self.append_page(uml_terminal(self, command_line), gtk.Label("Console"))
		self.show()
	
	def add_vt(self, pts, label):
		self.add_page_in_label_order(pts_terminal(self, pts), label)

	def add_page_in_label_order(self, widget, label_text):
		for i in range(self.get_n_pages()):
			nbwidget = self.get_nth_page(i)
			guml_debug.prn("Checking %s against %s" % (label_text, self.get_tab_label_text(nbwidget)))
			if label_text < self.get_tab_label_text(nbwidget):
				guml_debug.prn("Inserting")
				self.insert_page(widget, gtk.Label(label_text), i)
				return
		
		guml_debug.prn("Appending notebook page")
		# If we're here, then our new page belongs at the end of everything
		self.append_page(widget, gtk.Label(label_text))

	def close(self):
		guml_debug.prn("Closing console")
		self.destroy()

class terminal(gtk.HBox):
	def __init__(self, console, data):
		gtk.HBox.__init__(self)
		self.make_terminal()
		self.console = console
		self.vcre = re.compile("Virtual console (\d+) assigned device '(.*)'")

		self.run(data)

	def run(self, data):
		print "Pure virtual method called.  It's called subclassing, Matt, try it sometime"

	def make_terminal(self):
		self.terminal = vte.Terminal()
		self.terminal.set_emulation("xterm")
		self.terminal.set_font_from_string("Mono 8")
		self.terminal.set_scrollback_lines(300)
		self.terminal.set_backspace_binding('ascii-backspace')
		self.terminal.set_size(80, 25)
		
		scrollbar = gtk.VScrollbar()
		scrollbar.set_adjustment(self.terminal.get_adjustment())
		self.pack_start(self.terminal, expand=False)
		self.pack_start(scrollbar, expand=False)
		self.show_all()
		
	def close(self, widget, data = None):
		guml_debug.prn("Closing terminal")
		self.destroy()

class uml_terminal(terminal):
	def run(self, cmd):
		os.putenv('TERM', 'xterm')
		(infd, self.pipe) = os.popen4(' '.join(cmd), 't', 0)
		self.input_watcher = gobject.io_add_watch(self.pipe, gobject.IO_IN, self.handle_input)
		self.close_watcher = gobject.io_add_watch(self.pipe, gobject.IO_HUP, self.handle_close)
		self.line_buffer = ""

	def handle_input(self, fd, cond):
		c = fd.read(1)

		if c == "\n":
			self.scan_for_pts()
			self.line_buffer = ""
			self.terminal.feed("\r")
		else:
			self.line_buffer = self.line_buffer + c
		
		self.terminal.feed(c)
		
		return True

	def scan_for_pts(self):
		matches = self.vcre.match(self.line_buffer)
		if matches:
			guml_debug.prn("Creating VC #%s on %s" % (matches.group(1), matches.group(2)))
			self.console.add_vt(matches.group(2), "VC #%s" % matches.group(1))
		
	def handle_close(self, fd, cond):
		guml_debug.prn("UML terminal HUPed")
		gobject.source_remove(self.input_watcher)
		gobject.source_remove(self.close_watcher)
		self.pipe.close()
		self.destroy()
		self.console.close()
		
		return False

class pts_terminal(terminal):
	def run(self, pts):
		self.pipe = open(pts, 'r+', 0)
		self.input_watcher = gobject.io_add_watch(self.pipe, gobject.IO_IN, self.handle_input)
		self.close_watcher = gobject.io_add_watch(self.pipe, gobject.IO_HUP, self.handle_close)

	def handle_input(self, fd, cond):
		c = fd.read(1)

		# Strange newline handling in VTE necessitates this hack
		if c == "\n":
			self.terminal.feed("\r")
		
		self.terminal.feed(c)
		
		return True

	def handle_output(self, widget, str, len):
		self.pipe.write(str)
		
	def handle_close(self, fd, cond):
		guml_debug.prn("VC terminal HUPed")
		gobject.source_remove(self.input_watcher)
		gobject.source_remove(self.close_watcher)
		self.pipe.close()
		self.destroy()
		self.console.close()
		
		return False

	def make_terminal(self):
		terminal.make_terminal(self)
		self.terminal.connect('child-exited', self.close)
		self.terminal.connect('commit', self.handle_output)

class config:
	def __init__(self, umid):
		# Lo, verify, and he did give his attributes defaults, for such is the
		# path to Righteousness, lest thy AttributeErrors bite thee on the arse
		# and cause much wailing and gnashing of teeth.
		#            -- Book of Geeks, 4:2
		self.umid = umid
		self.disks = {}
		self.networks = {}
		self.name = ""
		self.mem = 32
		self.extraopts = ""

		cfg = self.readconfig(umid)

	def readconfig(self, umid):
		confdata = SafeConfigParser()
		confdata.read("%s/.guml/%s.conf" % (os.getenv("HOME"), umid))
	
		try:
			self.setgeneraloptions(self._tuples_to_dict(confdata.items("general")))
		except NoSectionError:
			print "No general section found for %s." % umid
			sys.exit(2)
		
		try:
			self.setdiskoptions(self._tuples_to_dict(confdata.items("disks")))
		except NoSectionError:
			# How the fsck do I leave an empty except block?
			xyzzy = 3.14159265
			
		for s in confdata.sections():
			if re.compile('^eth\d+$').match(s):
				self.setnetworkoptions(s, self._tuples_to_dict(confdata.items(s)))
		
	def setgeneraloptions(self, conf):
		try:
			self.name = conf['name']
		except KeyError:
			print "Fatal: failed to find a name for %s" % self.umid
			sys.exit(2)

		try:
			self.mem = conf['mem']
		except KeyError:
			self.mem = 32
		
		try:
			self.image = conf['image']
		except KeyError:
			self.image = ''
		
		try:
			self.extraopts = conf['extraopts']
		except KeyError:
			self.extraopts = ''

	def setnetworkoptions(self, devname, conf):
		data = { 'type': conf['type'],
		         'mac': conf['mac'] }
		
		try:
			data['socket'] = conf['socket']
		except KeyError:
			data['socket'] = '/var/run/uml-utilities/uml_switch.ctl'
			
		self.networks[devname] = data

	def setdiskoptions(self, conf):
		for k in conf.keys():
			if re.compile('^ubd\d+$').match(k):
				self.disks[k] = conf[k]

	def command_line(self):
		return ["/usr/bin/linux",
		        "mem=%sM" % self.mem,
		        "con=pts",
		        "con0=fd:0,fd:1",
		        "umid=%s" % self.umid] + self.networkcmdline() + self.blockdevscmdline() + [ self.extraopts ]

	def networkcmdline(self):
		cmdline = []
		
		for dev in self.networks.keys():
			cfg = self.networks[dev]
			
			if cfg['mac'] == 'random':
				idhash = md5.new()
				idhash.update(self.umid)
				hash = idhash.hexdigest()
				cfg['mac'] = 'fe:fd:%s:%s:%s:%s' % (hash[0:2], hash[2:4], hash[4:6], hash[6:8])

			cmdline.append("%s=%s,%s,,%s" % (dev, cfg['type'], cfg['mac'], cfg['socket']))
			
		return cmdline

	def blockdevscmdline(self):
		cmdline = []
		
		if len(self.disks) == 0:
			cmdline = [ "ubd0=%s" % self.image ]
		else:
			for k in self.disks.keys():
				cmdline.append("%s=%s" % (k, self.disks[k]))

		return cmdline
	
	def _tuples_to_dict(self, tuplelist):
		dict = {}
		
		for i in tuplelist:
			dict[i[0]] = i[1]

		return dict

class AlreadyRunning(Exception):
	def __init__(self, value = None):
		self.value = value
	
	def __str__(self):
		if self.value != None:
			return repr(self.value)
		else:
			return 'UML already running'
