/* 
 * Copyright (C) 2002-2005 
 * Emmanuel Saracco <esaracco@users.labs.libre-entreprise.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

#include "application.h"
#include "check.h"
#include "project.h"
#include "lists.h"
#include "report.h"
#include "html_parser.h"
#include "utils.h"
#include "url.h"
#include "timeout.h"

#include "bookmarks.h"


static xmlDocPtr uc_bookmarks_xml_doc = NULL;

static GList *uc_bookmarks_get_links (void);
static UCLinkProperties *parents[UC_MAX_DEPTH_LEVEL];
static xmlDocPtr uc_bookmarks_xml_parse_file (const gchar * docname);
static GList *uc_bookmarks_xml_parse_bookmark (guint depth, GList * list,
					       xmlDocPtr doc, xmlNodePtr cur);
static GList *uc_bookmarks_xml_parse_folder (guint depth, GList * list,
					     xmlDocPtr doc, xmlNodePtr cur);
static GList *uc_bookmarks_xml_get_tags (const gchar * docname);
static void uc_bookmarks_reassign_parents (GList * list,
					   UCLinkProperties * parent);

/**
 * uc_bookmarks_save_changes:
 *
 * Save bookmarks changes.
 */
gint
uc_bookmarks_save_changes (void)
{
  gchar *new_path = NULL;
  gint res = 0;

  if (!g_file_test (uc_project_get_bookmarks_file (), G_FILE_TEST_EXISTS))
    {
      gchar *filename = NULL;

      res =
	uc_application_dialog_yes_no_show (_
					   ("Should remote bookmarks be saved on your machine?"),
					   GTK_MESSAGE_QUESTION);
      if (res != GTK_RESPONSE_YES)
	return res;

      filename =
	uc_application_input_file_dialog (_("Save bookmarks file"),
					  _("Input filename"));
      if (!filename)
	return GTK_RESPONSE_CANCEL;

      xmlSaveFile (filename, uc_bookmarks_xml_doc);
      if ((!g_file_test (filename, G_FILE_TEST_EXISTS)))
	{
	  uc_application_dialog_show (_
				      ("The filename is not in a correct "
				       "format.\n"
				       "<b>Nothing has been done</>."),
				      GTK_MESSAGE_ERROR);
	  g_free (filename), filename = NULL;
	  return GTK_RESPONSE_CANCEL;
	}

      g_free (filename), filename = NULL;

      uc_project_set_save_bookmarks (FALSE);
    }
  else
    {
      res =
	uc_application_dialog_yes_no_show (_
					   ("Bookmarks changes will be "
					    "saved.\n"
					    "<i>Your old bookmarks file will "
					    "be renamed with \".uc\" "
					    "extension</i>.\n"
					    "<b>Do you want to save</b>?"),
					   GTK_MESSAGE_QUESTION);
      if (res != GTK_RESPONSE_YES)
	return res;

      new_path = g_strconcat (uc_project_get_bookmarks_file (), ".uc", NULL);

      if (!rename (uc_project_get_bookmarks_file (), new_path))
	{
	  xmlSaveFile (uc_project_get_bookmarks_file (),
		       uc_bookmarks_xml_doc);
	  uc_project_set_save_bookmarks (FALSE);
	}
      else
	uc_application_dialog_show (_
				    ("A error occured.\n"
				     "<b>Nothing has been done</b>."),
				    GTK_MESSAGE_ERROR);

      g_free (new_path), new_path = NULL;
    }

  return GTK_RESPONSE_YES;
}

/**
 * uc_bookmarks_delete_link:
 * @id: id of the #UCLinkProperties node to work with.
 * 
 * Remove a node on the treeview list.
 * The removal is virtual (to_delete = TRUE)
 */
void
uc_bookmarks_delete_link (const guint32 id)
{
  UCLinkProperties *prop = NULL;
  xmlNodePtr xml_node = NULL;
  GtkTreeIter iter;

  uc_project_set_save_bookmarks (TRUE);

  uc_application_treeview_get_selected_iter (treeview, &iter);
  prop = uc_lists_checked_links_lookup_by_uid (id);
  prop->to_delete = TRUE;

  /* !! HACK !!: don't know how to remove properly a node 
   * with libxml... any help? */
  xml_node = prop->xml_node;
  if (xml_node->prev == NULL ||
      ((UCLinkProperties *) xml_node)->link_type == LINK_TYPE_BOOKMARK_FOLDER)
    xml_node = (xml_node->next) ? xml_node->next : NULL;
  else
    xml_node->prev->next = (xml_node->next) ? xml_node->next : NULL;
  /* !! HACK !! */

  gtk_tree_store_remove (treestore, &iter);
}

/**
 * uc_bookmarks_get_links:
 * 
 * Get all the links of a given bookmarks
 * file.
 *
 * Returns: A #GList of #UCLinkPropeties node from bookmarks file.
 */
static GList *
uc_bookmarks_get_links (void)
{
  GList *list = NULL;

  list = uc_bookmarks_xml_get_tags (uc_project_get_bookmarks_file ());

  return list;
}

/**
 * uc_bookmarks_begin_check:
 * 
 * Begin the bookmarks check.
 */
void
uc_bookmarks_begin_check (void)
{
  guint timer_id = 0;
  gchar *tmp = NULL;
  GtkWidget *widget = NULL;

  uc_application_draw_main_frames ();
  uc_application_progress_dialog_show ();
  WSENS ("pd_dynamic_values_security_checks", FALSE);
  UC_UPDATE_UI;

  UC_GET_WIDGET ("urls_list", WGET ("main_window"), widget);
  treeview = GTK_TREE_VIEW (widget);

  tmp = g_strconcat ("gURLChecker ", UC_VERSION, " - ",
		     uc_project_get_bookmarks_file (), NULL);
  gtk_window_set_title (GTK_WINDOW (WGET ("main_window")), tmp);
  g_free (tmp), tmp = NULL;

  /* init timer */
  timer_id = g_timeout_add (1000, &uc_report_timer_callback, NULL);

  uc_project_set_check_is_current (TRUE);
  uc_project_set_current_host ("");
  uc_project_set_current_port (UC_URL_DEFAULT_PORT);
  uc_lists_checked_links_set (uc_bookmarks_get_links ());
  uc_project_set_check_is_current (FALSE);

  g_source_remove (timer_id), timer_id = 0;

  treestore = gtk_tree_store_new (N_COLUMNS,
				  G_TYPE_INT,
				  GDK_TYPE_PIXBUF,
				  GDK_TYPE_PIXBUF,
				  GDK_TYPE_PIXBUF,
				  GDK_TYPE_PIXBUF,
				  G_TYPE_STRING, G_TYPE_STRING,
				  G_TYPE_STRING);

  uc_check_display_list (uc_lists_checked_links_get (), 0);
  uc_application_build_url_treeview ();

  gtk_widget_hide (WGET ("progress_dialog"));

  /* Update some menu items */
  WSENS ("mwm_display", TRUE);
  WSENS ("mwm_delete_project", FALSE);
  WSENS ("mwm_refresh_all", FALSE);
  WSENS ("mwm_refresh_main_page", FALSE);
  WSENS ("mwm_find", !uc_project_get_speed_check ());
  WSENS ("mwm_display_security_alerts", FALSE);

  uc_project_set_save_bookmarks (TRUE);

  uc_check_cancel_set_value (FALSE);
}

/**
 * uc_bookmarks_format_is_xbel:
 * @file: The file to check.
 * 
 * Check the format of a given XBEL file.
 * 
 * Returns: TRUE if the given file is in XBEL format.
 */
gboolean
uc_bookmarks_format_is_xbel (const gchar * file)
{
  gchar *buffer = NULL;
  gboolean ret = FALSE;
  gsize length = 0;
  GError *error = NULL;

  g_file_get_contents (file, &buffer, &length, &error);
  ret =
    ((strstr (buffer, "<xbel ") != NULL) ||
     (strstr (buffer, "<XBEL ") != NULL) ||
     (strstr (buffer, " xbel>") != NULL) ||
     (strstr (buffer, " XBEL>") != NULL));
  g_free (buffer), buffer = NULL;

  return ret;
}

/**
 * uc_bookmarks_xml_parse_bookmark:
 * @depth: Depth of the node.
 * @list: The list to work with.
 * @doc: The XML doc to work with.
 * @cur: The XML cursor to work with.
 * 
 * Parse bookmark entity.
 *
 * Returns: A #GList of #UCLinkProperties nodes.
 */
static GList *
uc_bookmarks_xml_parse_bookmark (guint depth,
				 GList * list, xmlDocPtr doc, xmlNodePtr cur)
{
  xmlChar *key = NULL;
  xmlChar *value = NULL;
  xmlNodePtr parent = NULL;

  uc_report_display_update ();

  uc_check_wait ();

  value = xmlGetProp (cur, (const xmlChar *) "href");

  parent = cur;
  cur = cur->xmlChildrenNode;
  while (cur != NULL && !uc_check_cancel_get_value ())
    {
      if (!xmlStrcmp (cur->name, (const xmlChar *) "title"))
	{
	  UCLinkProperties *prop = NULL;
	  UCHTMLTag *tag = NULL;
	  gchar *hostname = NULL;
	  gboolean accept;

	  key = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);

	  tag = uc_html_parser_node_new ();
	  tag->depth = depth;
	  tag->label = g_strdup ((gchar *) key);
	  tag->value = g_strdup ((gchar *) value);
	  tag->type = LINK_TYPE_HREF;

	  uc_check_currentitem_init (NULL, "", "", tag, (gchar *) value,
				     (gchar *) key,
				     (const xmlNodePtr *) &parent);

	  hostname = uc_url_get_hostname ("", tag->value);

	  if (!uc_timeout_domains_is_blocked (hostname))
	    {
	      prop =
		uc_check_link_get_properties (depth, "", "", tag, NULL,
					      &accept, 0);
	      if (prop != NULL && accept)
		{
		  prop->xml_node = parent;
		  prop->normalized_url = g_strdup (prop->url);
		  list =
		    g_list_append (list,
				   uc_check_register_link (prop->url, prop));
		}
	      else
		(GList *) uc_lists_checked_links_node_free (NULL, prop);
	    }

	  xmlFree (key), key = NULL;
	  g_free (hostname), hostname = NULL;

	  break;
	}

      cur = cur->next;
    }

  xmlFree (value), value = NULL;

  return list;
}

/**
 * uc_bookmarks_xml_parse_folder:
 * @depth: Depth of the node.
 * @list: The list to work with.
 * @doc: The XML doc to work with.
 * @cur: The XML cursor to work with.
 * 
 * Parse folder entity.
 *
 * Returns: A #GList of #UCLinkProperties nodes.
 */
static GList *
uc_bookmarks_xml_parse_folder (guint depth,
			       GList * list, xmlDocPtr doc, xmlNodePtr cur)
{
  xmlChar *key = NULL;
  xmlNodePtr parent = NULL;

  uc_report_display_update ();

  if (depth == uc_project_get_depth_level ())
    return list;

  parent = cur;
  cur = cur->xmlChildrenNode;
  while (cur != NULL && !uc_check_cancel_get_value ())
    {
      if (!xmlStrcmp (cur->name, (const xmlChar *) "title"))
	{
	  UCLinkProperties *prop = NULL;

	  key = xmlNodeListGetString (doc, cur->xmlChildrenNode, 1);

	  prop = uc_check_link_properties_node_new ();
	  prop->xml_node = parent;
	  prop->depth_level = depth;
	  prop->label = g_strdup ((gchar *) key);
	  prop->link_value = g_strdup ("");
	  prop->link_icon =
	    gdk_pixbuf_new_from_file (UC_PIXMAPS_DIR "/link_type_folder.png",
				      NULL);
	  prop->link_type = LINK_TYPE_BOOKMARK_FOLDER;
	  prop->header = g_hash_table_new_full (g_str_hash, g_str_equal,
						g_free, g_free);

	  uc_check_link_already_checked_with_insert (prop, "");
	  list = g_list_append (list, uc_check_register_link ("", prop));

	  parents[depth] = prop;

	  xmlFree (key), key = NULL;
	}
      else if (!xmlStrcmp (cur->name, (const xmlChar *) "bookmark"))
	parents[depth]->childs =
	  uc_bookmarks_xml_parse_bookmark (depth + 1,
					   parents[depth]->childs, doc, cur);
      else if (!xmlStrcmp (cur->name, (const xmlChar *) "folder"))
	parents[depth]->childs =
	  uc_bookmarks_xml_parse_folder (depth + 1, parents[depth]->childs,
					 doc, cur);

      cur = cur->next;
    }

  return list;
}

/**
 * uc_bookmarks_xml_parse_file:
 * @docname: Path of the document to open.
 * 
 * Open xml file and parse it with DOM.
 * if there is a encoding problem, rebuild the given file by specifying
 * the encoding based on current locales and try to parse it again.
 *
 * Returns: A #xmlDocPtr reference.
 */
static xmlDocPtr
uc_bookmarks_xml_parse_file (const gchar * docname)
{
  xmlDocPtr doc = NULL;

  doc = xmlParseFile (docname);
  if (doc == NULL)
    {
      FILE *fd = NULL;
      gchar *buffer = NULL;
      gchar **tab = NULL;
      gchar *filename = NULL;
      guint32 i = 0;

      filename = g_strdup_printf ("%s/%s/%s.%u",
				  uc_project_get_working_path (),
				  uc_project_get_cache_name (),
				  strrchr (docname, '/'), getpid ());
      fd = fopen (filename, "w");
      g_assert (fd != NULL);

      g_file_get_contents (docname, &buffer, NULL, NULL);
      g_assert (buffer != NULL);

      tab = g_strsplit (buffer, "\n", 0);

      while (tab[i])
	{
	  if (strstr (tab[i], "<?xml ") != NULL)
	    {
	      g_free (tab[i]), tab[i] = NULL;

	      tab[i] =
		g_strdup_printf ("<?xml version=\"1.0\" encoding=\"%s\"?>",
				 uc_project_get_local_charset ());

	      break;
	    }
	  i++;
	}

      if (tab[i] == NULL)
	fprintf (fd,
		 "<?xml version=\"1.0\" encoding=\"%s\"?>\n"
		 "%s", uc_project_get_local_charset (), buffer);
      else
	{
	  g_free (buffer), buffer = NULL;
	  buffer = g_strjoinv ("\n", tab);
	  fprintf (fd, "%s", buffer);
	}

      fclose (fd);

      doc = xmlParseFile (filename);
      unlink (filename);

      g_free (filename), filename = NULL;
      g_strfreev (tab), tab = NULL;
      g_free (buffer), buffer = NULL;
    }

  return doc;
}

/**
 * uc_bookmarks_xml_get_tags:
 * @docname: Path of the document to open.
 * 
 * Retreive URLs from a given file.
 * 
 * Returns: A #GList of #UCLinkPropeties node from bookmarks file.
 */
static GList *
uc_bookmarks_xml_get_tags (const gchar * docname)
{
  GList *list = NULL;
  xmlNodePtr cur = NULL;

  if (uc_bookmarks_xml_doc != NULL)
    xmlFreeDoc (uc_bookmarks_xml_doc), uc_bookmarks_xml_doc = NULL;

  uc_bookmarks_xml_doc = uc_bookmarks_xml_parse_file (docname);
  if (uc_bookmarks_xml_doc == NULL)
    {
      uc_application_dialog_show (_("Either the given XML file is not well "
				    "formed or its encoding can not "
				    "be found...\n"
				    "<i>If you are checking a remote bookmarks "
				    "file, try to pump up the timeout "
				    "delay</i>.\n"
				    "<b>Check has been aborted</b>."),
				  GTK_MESSAGE_ERROR);

      return NULL;
    }

  cur = xmlDocGetRootElement (uc_bookmarks_xml_doc);
  if (cur == NULL)
    {
      uc_application_dialog_show (_("The given XML file is empty.\n"),
				  GTK_MESSAGE_WARNING);

      return NULL;
    }

  cur = cur->xmlChildrenNode;
  while (cur != NULL && !uc_check_cancel_get_value ())
    {
      if (!xmlStrcmp (cur->name, (const xmlChar *) "bookmark"))
	list =
	  uc_bookmarks_xml_parse_bookmark (0, list, uc_bookmarks_xml_doc,
					   cur);
      else if (!xmlStrcmp (cur->name, (const xmlChar *) "folder"))
	list =
	  uc_bookmarks_xml_parse_folder (0, list, uc_bookmarks_xml_doc, cur);

      cur = cur->next;
    }

  uc_bookmarks_reassign_parents (list, NULL);

  return list;
}

/**
 * uc_bookmarks_reassign_parents:
 * @list: Bookmarks list.
 * @parent: Parent node.
 *
 * Reorganize parents/childs in the URLs list. 
 */
static void
uc_bookmarks_reassign_parents (GList * list, UCLinkProperties * parent)
{
  GList *item = NULL;

  item = g_list_first (list);
  while (item != NULL)
    {
      UCLinkProperties *prop = (UCLinkProperties *) item->data;

      item = g_list_next (item);

      prop->parent = parent;

      if (prop->childs)
	uc_bookmarks_reassign_parents (prop->childs, prop);
    }
}
