# Network.pm - Network access library for Gutenbook
# http://www.gutenbook.org/
# Copyright (C) 2000  Lee "Lefty" Burgess
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# This library module contains code contributed by Glenn Cooper <cooper@fedex.pa.dec.com>

package PGB::Network;

use URI::URL;
use LWP::UserAgent;
use HTTP::Request;
use HTTP::Response;
use PGB;

use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

require Exporter;
require AutoLoader;

@ISA = qw(Exporter AutoLoader);
@EXPORT = qw(http_get derive_etext_path);
$VERSION = '0.01';

sub http_get( $ );
sub derive_etext_path( $$ );
sub conditional_proxy( $ );

sub http_get( $ ) {
  my ($http_file) = @_;
  my $uri = new URI::URL($http_file);
  my $ua = new LWP::UserAgent();
  my ($guzfile, $array);
  my ($req, $resp);
 
  BUG_ME(0, "Requesting $uri...");
  conditional_proxy($ua);

  $req = new HTTP::Request('GET', $uri);
  $resp = $ua->request($req);

  if ($resp->is_success) {
    BUG_ME(0, "Success!");
    
    # if we are getting a zip file, must uncompress it into memory
    if ($http_file =~ /\.zip$/) {
      BUG_ME(0, "NETWORK: Got zip file, uncompressing $http_file...");
      
      ## I hate doing this, but Compress::Zlib is not cutting it.
      ## It is supposed to work for zip'd/pkzip'd files/buffers?
      my ($fh, $output);
      my $tmpfile = $plist_hash->{"library_dir"} . "/gb_". time() . ".zip";
      $fh = new FileHandle($tmpfile, "w") || BUG_ME(1, "Cannot write to $tmpfile: $!");
      if (defined($fh)) {
	$fh->print($resp->content());
	$fh->close();
      }
      
      $output = `$unzip -p $tmpfile`;
      @{$array} = split(/\n/, $output);
      unlink($tmpfile) || BUG_ME(0, "Cannot remove $tmpfile: $!");
    }
    else {
      BUG_ME(0, "NETWORK: Got txt file $http_file...");
      @{$array} = split(/\n/, $resp->content);
    }
    
  }
  else {
    BUG_ME(0, "Denied!");
    @array = split(/\n/, $resp->error_as_HTML);
  }
  
  foreach (@{$array}) {
    $_ .= "\n";
  }
  
  return($array);
}

sub derive_etext_path( $$ ) {
  my ($key, $index_hash) = @_; 
  my $file = $index_hash->{$key}{'FILE'};
  my $year = $index_hash->{$key}{'YEAR'};
  my ($uri, $ua, $req, $resp, $http_dir);
  my ($etext, @array, @found);

  # try to construct the best possible pattern match
  # from the wildfile name
  if ($file !~ /txt|zip$/) {
    $file =~ s/\.x+$//;       # strip off the .xxx extension
    $file =~ s/x+$//;         # strip out trailing x's
    $file =~ s/x{2,}/\\d\+/;  # any 2 or more x's are most likely digits
  }
  
  if ($year < 1990) {
    $year = 1990;
  }
  $year = substr($year, 2, 2);
  $http_dir = $plist_hash->{"http_url"} . "etext$year/";
  
  $uri = new URI::URL($http_dir);
  $ua = new LWP::UserAgent();
  
  BUG_ME(0, "Requesting $uri...");
  conditional_proxy($ua);

  $req = new HTTP::Request('GET', $uri);
  $resp = $ua->request($req);
  
  if ($resp->is_success) {
    BUG_ME(0, "Success!");
    @array = split(/\n/, $resp->content);
  }
  else {
    BUG_ME(0, "Denied!");
    @array = split(/\n/, $resp->error_as_HTML);
  }
  
  BUG_ME(0, "Looking for $file...");  
  # TODO: maybe use grep here?
  foreach (@array) {
    if ($uri =~ /^ftp:/ && $_ =~ /($file(.+)?)$/i) {
      push(@found, $1);
    }
    if ($uri =~ /^http:/ && $_ =~ /\>($file(.+)?)\<\/A\>/i) {
      push(@found, $1);
    }
  }
  
  # arrange to get the version with the highest number
  # TODO: does this work all the time?
  @found = reverse(sort(@found));

  BUG_ME(0, "Found @found");

  if (@found) {
    my $zip = (grep(/\.zip$/, @found))[0];
    my $txt = (grep(/\.txt$/, @found))[0];
    
    # give preference to zip files for now
    $etext = $zip || $txt;
  }
  else {
    # TODO: make nice with the better error thowing!
    return("$http_dir" . "ETEXT_NOT_AVAILABLE");
  }
  
  return("$http_dir" . "$etext");
}

sub conditional_proxy( $ ) {
  my ($ua) = @_;
  
  if ($plist_hash->{"http_proxy"}) {
    $ua->proxy("http", $plist_hash->{"http_proxy"});
  }
  else {
    $ua->env_proxy();
  }
  return;
}

1;
