// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         Acquisition dialog
// ****************************************************************************

// Copyright 2008, 2009, 2010 Guy Voncken
//
// This file is part of guymager.
//
// guymager is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// guymager is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with guymager. If not, see <http://www.gnu.org/licenses/>.

#include <QtGui>
#include <QDesktopWidget>
#include <QX11Info>

#include "toolconstants.h"

#include "common.h"
#include "compileinfo.h"
#include "config.h"
#include "qtutil.h"
#include "dlgdirsel.h"
#include "devicelistmodel.h"
#include "dlgacquire.h"
#include "dlgacquire_private.h"
#include "main.h"

// -----------------------------
//           Constants
// -----------------------------

const char *DLGACQUIRE_DEFAULT_EMPTY_FILENAME   = "Out";  // Default image file name if - after removing special chars - the file name is empty.
const char *DLGACQUIRE_PROPERTY_SENDER_LINEEDIT = "SenderLineEdit";

// -----------------------------
//           Classes
// -----------------------------

class t_DlgAcquireLocal
{
   public:
      bool              Clone;
      QRadioButton    *pRadioButtonFormatDD;
      QRadioButton    *pRadioButtonFormatEWF;
      QRadioButton    *pRadioButtonFormatAFF;

      QList<QWidget*>   EwfWidgetsList;                  // Used to comfortably enable / disable EWF related entry fields and labels

      QTableWidget    *pDeviceTable;                     // Only created and displayed when cloning a device
      int               DeviceTableLinuxDeviceCol;

      QPushButton     *pButtonOk;
      QPushButton     *pButtonCancel;

      t_pDevice        pDevice;
};

static t_File::Format DlgAcquireLastUsedFormat = t_File::NotSet;  // For remembering the format that was used for the last acquisition

// -----------------------------
//         Unit conversion
// -----------------------------

const QStringList UnitStrings = QStringList() << "MiB" << "GiB" << "TiB" << "PiB" << "EiB";

static int UnitStringToIndex (QString &Str)
{
   if      (Str.contains ('M', Qt::CaseInsensitive)) return 0;
   else if (Str.contains ('G', Qt::CaseInsensitive)) return 1;
   else if (Str.contains ('T', Qt::CaseInsensitive)) return 2;
   else if (Str.contains ('P', Qt::CaseInsensitive)) return 3;
   else if (Str.contains ('E', Qt::CaseInsensitive)) return 4;
   LOG_ERROR ("Unknown Unit %s", QSTR_TO_PSZ(Str))
   return 0;
}

static unsigned long long UnitIndexToMultiplier (int Index)
{
   unsigned long long Mul = 1024*1024;

   if ((Index < 0) || (Index > 4))
        LOG_ERROR ("Unit index out of range: %d", Index)
   else Mul <<= (Index*10);

   return Mul;
}

// -----------------------------
//    Field utility functions
// -----------------------------

static t_pCfgDlgAcquireField DlgAcquireGetField (const QString &Name)
{
   t_pCfgDlgAcquireFields pDlgAcquireFields;
   t_pCfgDlgAcquireField  pDlgAcquireField;
   int i;

   CHK_EXIT (CfgGetDlgAcquireFields (&pDlgAcquireFields))

   for (i=0; i<pDlgAcquireFields->count(); i++)
   {
      pDlgAcquireField = pDlgAcquireFields->at(i);
      if (pDlgAcquireField->FieldName.compare(Name, Qt::CaseInsensitive)==0)
         return pDlgAcquireField;
   }
   LOG_ERROR ("DlgAcquire field %s not found", QSTR_TO_PSZ(Name))
   return NULL;
}


static t_pDlgAcquireLineEdit DlgAcquireGetLineEdit (const QString &Name)
{
   return DlgAcquireGetField(Name)->pLineEdit;
}


static APIRET DlgAcquireResolveSpecialSequences (t_pDevice pDevice, bool ConsiderFields, const QString &In, QString &Out)
{
   t_pCfgDlgAcquireFields pDlgAcquireFields;
   t_pCfgDlgAcquireField  pDlgAcquireField;
   QDateTime               Now = QDateTime::currentDateTime();
   int                     i;
   QString                 Search;

   // Replace date/time values
   // ------------------------
   Out = In;
   Out.replace ("%d%"   , Now.toString("d"   ));  // Special sequences
   Out.replace ("%dd%"  , Now.toString("dd"  ));
   Out.replace ("%ddd%" , Now.toString("ddd" ));
   Out.replace ("%dddd%", Now.toString("dddd"));
   Out.replace ("%M%"   , Now.toString("M"   ));
   Out.replace ("%MM%"  , Now.toString("MM"  ));
   Out.replace ("%MMM%" , Now.toString("MMM" ));
   Out.replace ("%MMMM%", Now.toString("MMMM"));
   Out.replace ("%yy%"  , Now.toString("yy"  ));
   Out.replace ("%yyyy%", Now.toString("yyyy"));
   Out.replace ("%h%"   , Now.toString("h"   ));
   Out.replace ("%hh%"  , Now.toString("hh"  ));
   Out.replace ("%m%"   , Now.toString("m"   ));
   Out.replace ("%mm%"  , Now.toString("mm"  ));
   Out.replace ("%s%"   , Now.toString("s"   ));
   Out.replace ("%ss%"  , Now.toString("ss"  ));
   Out.replace ("%AP%"  , Now.toString("AP"  ));
   Out.replace ("%ap%"  , Now.toString("ap"  ));

   // Replace device values
   // ---------------------
   QString SizeHuman = t_Device::GetSizeHumanFrac (pDevice, true, 0, 10000).toString();
   SizeHuman.replace (MainGetpNumberLocale()->groupSeparator(), "");

   Out.replace ("%serial%" , pDevice->SerialNumber);
   Out.replace ("%model%"  , pDevice->Model       );
   Out.replace ("%size%"   , SizeHuman            );
   Out.replace ("%version%", pCompileInfoVersion  );

   // Replace field values
   // --------------------
   if (ConsiderFields)
   {
      CHK_EXIT (CfgGetDlgAcquireFields (&pDlgAcquireFields))
      for (i=0; i<pDlgAcquireFields->count(); i++)
      {
         pDlgAcquireField = pDlgAcquireFields->at(i);
         if (pDlgAcquireField->pLineEdit)
         {
            Search = "%" + pDlgAcquireField->FieldName + "%";
            Out.replace (Search, pDlgAcquireField->pLineEdit->text());
         }
      }
   }

   return NO_ERROR;
}


static APIRET DlgAcquireGetFieldValue (t_pDevice pDevice, bool Clone, t_pCfgDlgAcquireField pDlgAcquireField, QString &Set)
{
   t_CfgEntryMode EntryMode;

   if (Clone && ((pDlgAcquireField->FieldName == CFG_DLGACQUIRE_DEST_IMAGEDIRECTORY) ||
                 (pDlgAcquireField->FieldName == CFG_DLGACQUIRE_DEST_IMAGEFILENAME )))
   {
      Set = QString();
      return NO_ERROR;
   }

   if (Clone)
        EntryMode = pDlgAcquireField->EntryModeClone;
   else EntryMode = pDlgAcquireField->EntryModeImage;

   switch (EntryMode)
   {
      case CFG_ENTRYMODE_HIDE:
      case CFG_ENTRYMODE_SHOWLAST:
         Set = pDlgAcquireField->LastEnteredValue;
         if (Set.isNull())
         {
            Set = pDlgAcquireField->DefaultValue;
            CHK (DlgAcquireResolveSpecialSequences (pDevice, false, pDlgAcquireField->DefaultValue, Set))
         }
         break;

      case CFG_ENTRYMODE_SHOWDEFAULT:
         Set = pDlgAcquireField->DefaultValue;
         CHK (DlgAcquireResolveSpecialSequences (pDevice, false, pDlgAcquireField->DefaultValue, Set))
         break;

      default:
         CHK (ERROR_DLGACQUIRE_INVALID_ENTRYMODE)
   }

   return NO_ERROR;
}

// -----------------------------
//         t_DlgAcquire
// -----------------------------

static bool DlgAcquireStrToBool (const QString &Set)
{
   return !Set.isEmpty() && (Set.compare ("NO"         , Qt::CaseInsensitive) != 0)  &&
          !Set.isNull () && (Set.compare ("OFF"        , Qt::CaseInsensitive) != 0)  &&
                            (Set.compare ("FALSE"      , Qt::CaseInsensitive) != 0)  &&
                            (Set.compare ("0"          , Qt::CaseInsensitive) != 0)  &&
                            (Set.compare ("DISABLED"   , Qt::CaseInsensitive) != 0)  &&
                            (Set.compare ("DEACTIVATED", Qt::CaseInsensitive) != 0);
}

t_CfgEntryMode t_DlgAcquire::EntryMode (t_pCfgDlgAcquireField pField)
{
   return pOwn->Clone ? pField->EntryModeClone : pField->EntryModeImage;
}

APIRET t_DlgAcquire::AddField (t_pDevice pDevice, t_pCfgDlgAcquireField pField, QGridLayout *pLayout, int *pRow, int *pCol)
{
   t_pDlgAcquireLineEdit  pLineEdit     = NULL;
   QCheckBox             *pCheckBox     = NULL;
   QComboBox             *pComboBox     = NULL;
   QLabel                *pLabel        = NULL;
   QWidget               *pEntryWidget  = NULL;
   QPushButton           *pButtonBrowse = NULL;
   QString                 Set;
   QSize                   MinButtonSize;
   bool                    Ok;

   // Create field
   // ------------
   if (pField->HashField || pField->FieldName == CFG_DLGACQUIRE_SPLITFILESWITCH)
   {
      pEntryWidget = pCheckBox = new QCheckBox (tr(QSTR_TO_PSZ(pField->FieldName)), this);
   }
   else if (pField->FieldName == CFG_DLGACQUIRE_SPLITFILEUNIT)
   {
      pEntryWidget = pComboBox = new QComboBox (this);
      pComboBox->addItems (UnitStrings);
   }
   else
   {
      pEntryWidget = pLineEdit = new t_DlgAcquireLineEdit (this, pField->FieldName);
      pLabel       = new QLabel (tr(QSTR_TO_PSZ(pField->FieldName)), this);
   }

   if ((pField->DirField) && (EntryMode(pField) != CFG_ENTRYMODE_HIDE))
      pButtonBrowse = new QPushButton (tr("...", "The directory browse button"), this);

   // Position field
   // --------------
   pField->pLineEdit = pLineEdit;
   pField->pCheckBox = pCheckBox;
   pField->pComboBox = pComboBox;
   pField->pLabel    = pLabel;
   if (EntryMode(pField) == CFG_ENTRYMODE_HIDE)
   {
      if (pLabel)
         pLabel->hide();
      pEntryWidget->hide();
   }
   else
   {
      if (pField->FieldName == CFG_DLGACQUIRE_SPLITFILESIZE)
      {
         pLayout->addWidget (pLabel   , *pRow, *pCol  );
         pLayout->addWidget (pLineEdit, *pRow, *pCol+1);
      }
      else
      {
         if (pButtonBrowse)
         {
            pLayout->addWidget (pLabel       , *pRow, 0);
            pLayout->addWidget (pButtonBrowse, *pRow, 1);
            MinButtonSize = pButtonBrowse->minimumSize();
            MinButtonSize.setWidth(20);
            pButtonBrowse->setSizePolicy (QSizePolicy(QSizePolicy::Ignored, QSizePolicy::Preferred));   // Allow to shrink horizontal size below preferred minimum
            pButtonBrowse->setMinimumSize(MinButtonSize);
         }
         else if (pLabel)
         {
            pLayout->addWidget (pLabel, *pRow, 0, 1, 2);        // if there's no button then use the first 2 columns for the label
         }
         if (pLineEdit)
         {
            pLayout->addWidget (pLineEdit, *pRow, 2);
            if (pField->EwfField)
            {
               pOwn->EwfWidgetsList.append (pLabel   );
               pOwn->EwfWidgetsList.append (pLineEdit);
            }
            (*pRow)++;
         }
         if (pCheckBox)
         {
	    if (pField->FieldName == CFG_DLGACQUIRE_SPLITFILESWITCH)
	    {
               pLayout->addWidget (pCheckBox, *pRow, *pCol, 1, 2);
               (*pRow)++;
	    }
            else if (pField->FieldName.startsWith (CFG_DLGACQUIRE_HASHCALC_FIELDID))
            {
               pLayout->addWidget (pCheckBox, 0, *pCol);  // Put all HashCalc field in the first row
               (*pCol)++;
            }
            else
            {
               pLayout->addWidget (pCheckBox, (*pRow)+1, 0, 1, 2); // Put each verification field in an own row below
               (*pRow)++;
            }
         }
         if (pComboBox)
         {
            pLayout->addWidget (pComboBox, *pRow, 3);
         }
      }
   }

   // Set field value
   // ---------------
   CHK_EXIT (DlgAcquireGetFieldValue (pDevice, pOwn->Clone, pField, Set))
   if (pField->DirField)
   {
      pLineEdit->setReadOnly (true);
      if (!Set.endsWith ("/"))
         Set += "/";
   }
   if (pLineEdit)
   {
      pLineEdit->setText (Set);
      CHK_QT_EXIT (connect (pLineEdit, SIGNAL (SignalTextEdited (t_DlgAcquireLineEdit *, const QString &)),
                                 this, SLOT   (SlotTextEdited   (t_DlgAcquireLineEdit *, const QString &))))
   }

   if (pField->DstField)
      CHK_QT_EXIT (connect (pLineEdit, SIGNAL (textChanged(const QString &)), this, SLOT(UpdateDialogState(const QString &))))

   if (pCheckBox)
      pCheckBox->setChecked (DlgAcquireStrToBool(Set));

   if (pComboBox)
   {
      pComboBox->setCurrentIndex(Set.toInt(&Ok));
      if (!Ok)
         LOG_ERROR ("Invalid set value for ComboBox: %s", QSTR_TO_PSZ(Set));
   }
   if (pButtonBrowse)
   {
      (void) pButtonBrowse->setProperty (DLGACQUIRE_PROPERTY_SENDER_LINEEDIT, qVariantFromValue ((void *)pLineEdit));
      CHK_QT_EXIT (connect (pButtonBrowse, SIGNAL (released()), this, SLOT(SlotBrowse())))
   }

   return NO_ERROR;
}

APIRET t_DlgAcquire::InsertDeviceTableRow (QTableWidget *pTable, int Row, t_pDeviceList pDeviceList, t_pDevice pDevSrc, t_pDevice pDevDst)
{
   QTableWidgetItem *pItem;
   int                Col;
   QVariant           SizeHuman;
   QString            Remark;
   bool               Selectable = false;

   if (Row == 0)
   {
      Col = 0;
      pTable->setHorizontalHeaderItem (Col++, new QTableWidgetItem(t_DeviceListModel::tr("Serial\nnr."  , "Column of device table")));
      pOwn->DeviceTableLinuxDeviceCol = Col;
      pTable->setHorizontalHeaderItem (Col++, new QTableWidgetItem(t_DeviceListModel::tr("Linux\ndevice", "Column of device table")));
      pTable->setHorizontalHeaderItem (Col++, new QTableWidgetItem(t_DeviceListModel::tr("Model"        , "Column of device table")));
      pTable->setHorizontalHeaderItem (Col++, new QTableWidgetItem(t_DeviceListModel::tr("Size"         , "Column of device table")));
      pTable->setHorizontalHeaderItem (Col++, new QTableWidgetItem(t_DeviceListModel::tr("Remarks"      , "Column of device table")));
   }

   Col = 0;
   SizeHuman = t_Device::GetSizeHuman (pDevDst);

   bool InUse = (pDevDst->State != t_Device::Idle    ) &&
                (pDevDst->State != t_Device::Finished) &&
                (pDevDst->State != t_Device::Aborted );

   if      (pDevDst->Local)                               Remark = tr("Local device, cannot be written");
   else if (pDevDst == pDevSrc)                           Remark = tr("Device to be cloned");
   else if (pDevDst->Size < pDevSrc->Size)                Remark = tr("Too small");
   else if (InUse)                                        Remark = tr("In use");
   else if (pDeviceList->UsedAsCloneDestination(pDevDst)) Remark = tr("Used in another clone operation");
   else                                                 { Remark = tr("Ok for cloning"); Selectable = true;}

   #define SET_ITEM(Str)                                            \
      pItem = new QTableWidgetItem (Str);                           \
      pTable->setItem (Row, Col++, pItem);                          \
      pItem->setFlags (Selectable ? (Qt::ItemIsSelectable | Qt::ItemIsEnabled) : Qt::ItemFlags(0));

   SET_ITEM (pDevDst->SerialNumber);
   SET_ITEM (pDevDst->LinuxDevice );
   SET_ITEM (pDevDst->Model       );
   SET_ITEM (SizeHuman.toString() );
   SET_ITEM (Remark               );
   #undef SET_ITEM

   return NO_ERROR;
}


APIRET t_DlgAcquire::CreateDeviceTable (t_pDeviceList pDeviceList, t_pDevice pDeviceSrc, QTableWidget **ppTable)
{
   QTableWidget *pTable;
   t_pDevice     pDev;
   int            i=0;
   int            ColWidth=0;
   int            ExtraWidth;
   int            ScreenWidth;
   int            Width=0;

   pTable = new QTableWidget(0, 5, this);
   *ppTable = pTable;
   pOwn->pDeviceTable = pTable;

   pTable->setSelectionBehavior (QAbstractItemView::SelectRows     );
   pTable->setSelectionMode     (QAbstractItemView::SingleSelection);

   pTable->setRowCount(pDeviceList->count());
   for (i=0; i<pDeviceList->count(); i++)
   {
      pDev = pDeviceList->at (i);
      CHK (InsertDeviceTableRow (pTable, i, pDeviceList, pDeviceSrc, pDev))
   }
   ExtraWidth = pTable->columnWidth(0);  // Somehow, the table widget always gets this value larger when calling setMinimumWidth...
   pTable->resizeColumnsToContents();
   pTable->resizeRowsToContents   ();
   pTable->verticalHeader  ()->hide ();
//   pTable->horizontalHeader()->setClickable          (true);
//   pTable->horizontalHeader()->setSortIndicatorShown (true);
   pTable->setHorizontalScrollMode (QAbstractItemView::ScrollPerPixel);
   pTable->setVerticalScrollMode   (QAbstractItemView::ScrollPerPixel);

   for (i=0; i<pTable->columnCount(); i++)
      ColWidth += pTable->columnWidth(i);

   ScreenWidth = QApplication::desktop()->availableGeometry().width();
   Width = GETMIN ((int)(0.8 * ScreenWidth), ColWidth - ExtraWidth);   // Do not use more than 80% of the screen width
   pTable->setMinimumWidth  (Width);
   pTable->setMinimumHeight (130);

   CHK_QT (connect (pTable, SIGNAL(itemSelectionChanged ()), this, SLOT(SlotDeviceTableSelectionChanged())))

   return NO_ERROR;
}

t_DlgAcquire::t_DlgAcquire ()
{
   CHK_EXIT (ERROR_DLGACQUIRE_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 pOwn not initialised


class t_DlgAcquireLayoutScroller: public QScrollArea // This class makes layouts scrollable.
{
   public:
      t_DlgAcquireLayoutScroller (QWidget *pParent, QLayout *pLayout)
         : QScrollArea (pParent)
      {
         setFrameStyle      (QFrame::NoFrame);  // We do not want any frames
         setLineWidth       (0);                // nor margins
         setMidLineWidth    (0);
         setContentsMargins (0,0,0,0);
         setWidgetResizable (true);      // necessary to have the layout inside the ScrollArea follow the dialog's resizing

         QWidget   *pWidget = new QWidget (this);  // we need an intermediate widget, as QScrollArea only accepts
         setWidget (pWidget);                      // a widget and does ont allow for setting a layout directly
         pWidget->setLayout (pLayout);             // We then put the layout in the widget

         setHorizontalScrollBarPolicy (Qt::ScrollBarAlwaysOff); // Only scroll vertically
      }

     ~t_DlgAcquireLayoutScroller ()
      {
      }

      virtual QSize sizeHint () const
      {
         QWidget *pWidget;
         QLayout *pLayout;

         pWidget = widget();
         if (pWidget)
         {
            pLayout = pWidget->layout();
            if (pLayout)                       // Give the layout's minimal size as size hint. So, when putting the whole
               return pLayout->minimumSize (); // scroll area in a layout, it should be displayed just big enough to have
         }                                     // the scroll bar hidden.
         return QScrollArea::sizeHint();
      }

      virtual QSize minimumSizeHint () const                      // In hor. direction, the area must never be smaller then
      {                                                           // the minimum required by the layout inside (as we do not
         return QSize (sizeHint().width(),                        // want a hor. scrollbar and thus switched it off, see
                       QScrollArea::minimumSizeHint().height());  // constructor above). Vertically, the default minimum
      }                                                           // of QScrollArea is used (whatever this might be).
};


t_DlgAcquire::t_DlgAcquire (t_pDevice pDevice, bool Clone, t_pDeviceList pDeviceList, QWidget *pParent, Qt::WFlags Flags)
   :QDialog (pParent, Flags)
{
   static bool                  Initialised = false;
   t_pCfgDlgAcquireFields      pDlgAcquireFields;
   t_pCfgDlgAcquireField       pDlgAcquireField;
   t_DlgAcquireLayoutScroller *pLayoutScroller;
   QVBoxLayout                *pTopLayout;
   QVBoxLayout                *pLayout;
   QLabel                     *pLabel;
   QString                      DefaultFilename;
   QString                      Path;
   QString                      Str, StrAlt;
   QString                      ButtonTextDD;
   QString                      ButtonTextEWF;
   QString                      ButtonTextAFF;
   QString                      Set;
   int                          Row, Col;
   int                          i;

   if (!Initialised)
   {
      Initialised = true;
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DLGACQUIRE_CONSTRUCTOR_NOT_SUPPORTED))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DLGACQUIRE_UNKNOWN_FILEDIALOG_SIZE))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DLGACQUIRE_INVALID_ENTRYMODE))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DLGACQUIRE_INVALID_FORMAT))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DLGACQUIRE_INVALID_SELECTION))

      pDlgAcquireField = DlgAcquireGetField (CFG_DLGACQUIRE_SPLITFILEUNIT);                                 // This initialisation needs to be done only once, as the user writes text (MiB, GiB, ...) 
      pDlgAcquireField->DefaultValue = QString::number (UnitStringToIndex(pDlgAcquireField->DefaultValue)); // in the configuration file, but we internally work with the ComboBox index numbers
   }

   if (Clone)
        setWindowTitle (tr ("Clone %1"           , "Dialog title, %1 is the device (for instance /dev/hdc)") .arg(pDevice->LinuxDevice));
   else setWindowTitle (tr ("Acquire image of %1", "Dialog title, %1 is the device (for instance /dev/hdc)") .arg(pDevice->LinuxDevice));

   pOwn    = new t_DlgAcquireLocal;
   pOwn->pDevice      = pDevice;
   pOwn->Clone        = Clone;
   pOwn->pDeviceTable = NULL;

   // Organisation of the whole dialog
   //     Dialog
   //        TopLayout
   //           ScrollArea    - t_DlgAcquireLayoutScroller
   //              Widget     - see t_DlgAcquireLayoutScroller
   //                 Layout  - see t_DlgAcquireLayoutScroller
   //           LayoutButtons

   pTopLayout = new QVBoxLayout(this);
   pTopLayout->setSpacing(0);  // We do not want extra margins everywhere (see t_DlgAcquireScrollArea as well)
   pTopLayout->setMargin (0);

   pLayout = new QVBoxLayout();
   pLayout->setSizeConstraint (QLayout::SetMinAndMaxSize);
   pLayoutScroller = new t_DlgAcquireLayoutScroller (this, pLayout);
   pTopLayout->addWidget(pLayoutScroller);

   CHK_EXIT (CfgGetDlgAcquireFields (&pDlgAcquireFields))

//   pLayout->addWidget (new QLabel(("<b>" + tr("Acquisition parameters for %1") + "</b>") .arg(pDevice->LinuxDevice) , this));

   if (!Clone)
   {
      // Format box (with EWF fields)
      // ----------------------------
      QGroupBox   *pGroupBoxFormat    = new QGroupBox (tr("File format"), this);
      QVBoxLayout *pLayoutGroupBox    = new QVBoxLayout;
      QHBoxLayout *pLayoutFormatSplit = new QHBoxLayout;
      QGridLayout *pLayoutFormat      = new QGridLayout;
      QGridLayout *pLayoutSplit       = new QGridLayout;
      QGridLayout *pLayoutEwf         = new QGridLayout;
      QFrame      *pSeparator         = new QFrame(this); // vertical separator line between format and split fields
      pSeparator->setFrameShape (QFrame::VLine);

      pLayout           ->addWidget (pGroupBoxFormat);
      pGroupBoxFormat   ->setLayout (pLayoutGroupBox);
      pLayoutGroupBox   ->addLayout (pLayoutFormatSplit);
      pLayoutFormatSplit->addLayout (pLayoutFormat);
      pLayoutFormatSplit->addWidget (pSeparator);
      pLayoutFormatSplit->addLayout (pLayoutSplit);
      pLayoutGroupBox   ->addLayout (pLayoutEwf);

      CHK_EXIT (t_File::GetFormatDescription (t_File::DD , false, 0, Str))            ButtonTextDD  = "&" + Str;
      CHK_EXIT (t_File::GetFormatDescription (t_File::EWF, false, 0, Str))            ButtonTextEWF = "&" + Str;
      CHK_EXIT (t_File::GetFormatDescription (t_File::AFF, false, 0, Str))            ButtonTextAFF = "&" + Str;
      CHK_EXIT (t_File::GetFormatExtension   (t_File::DD , false, 0, NULL, &Str))
      CHK_EXIT (t_File::GetFormatExtension   (t_File::DD , false, 3, NULL, &StrAlt))  ButtonTextDD += " " + tr("(file extension %1 or %2)") .arg (Str) .arg (StrAlt);
      CHK_EXIT (t_File::GetFormatExtension   (t_File::EWF, false, 0, NULL, &Str))     ButtonTextEWF+= " " + tr("(file extension %1)") .arg (Str);
      CHK_EXIT (t_File::GetFormatExtension   (t_File::AFF, false, 0, NULL, &Str))     ButtonTextAFF+= " " + tr("(file extension %1)") .arg (Str);

      pOwn->pRadioButtonFormatDD  = new QRadioButton (ButtonTextDD , this);
      pOwn->pRadioButtonFormatEWF = new QRadioButton (ButtonTextEWF, this);
      pOwn->pRadioButtonFormatAFF = new QRadioButton (ButtonTextAFF, this);

      pLayoutFormat->addWidget  (pOwn->pRadioButtonFormatDD );
      pLayoutFormat->addWidget  (pOwn->pRadioButtonFormatEWF);
      pLayoutFormat->addWidget  (pOwn->pRadioButtonFormatAFF);

      // Split file controls
      // -------------------
      t_pCfgDlgAcquireField pFieldSplitFileSwitch = DlgAcquireGetField ("SplitFileSwitch");
      t_pCfgDlgAcquireField pFieldSplitFileSize   = DlgAcquireGetField ("SplitFileSize"  );
      t_pCfgDlgAcquireField pFieldSplitFileUnit   = DlgAcquireGetField ("SplitFileUnit"  );

      if ((EntryMode(pFieldSplitFileSwitch) == CFG_ENTRYMODE_HIDE) &&
          (EntryMode(pFieldSplitFileSize  ) == CFG_ENTRYMODE_HIDE) &&
          (EntryMode(pFieldSplitFileUnit  ) == CFG_ENTRYMODE_HIDE))
         pSeparator->hide();
      Row = 0;
      Col = 1;
      CHK_EXIT (AddField (pDevice, pFieldSplitFileSwitch, pLayoutSplit, &Row, &Col))
      CHK_EXIT (AddField (pDevice, pFieldSplitFileSize  , pLayoutSplit, &Row, &Col))
      CHK_EXIT (AddField (pDevice, pFieldSplitFileUnit  , pLayoutSplit, &Row, &Col))

      Row = Col = 0;
      for (i=0; i<pDlgAcquireFields->count(); i++)
      {
         pDlgAcquireField = pDlgAcquireFields->at(i);
         if (pDlgAcquireField->EwfField)
            CHK_EXIT (AddField (pDevice, pDlgAcquireField, pLayoutEwf, &Row, &Col))
      }
      pLayoutEwf->setColumnMinimumWidth (0, 20);
   }

   // Destination box
   // ---------------

   QGroupBox   *pGroupBoxDest = new QGroupBox(tr("Destination"), this);
   QGridLayout *pLayoutDest   = new QGridLayout ();
   pGroupBoxDest->setLayout (pLayoutDest);
   pLayout->addWidget (pGroupBoxDest);
   Row = Col = 0;

   if (Clone)
   {
      CHK_EXIT (CreateDeviceTable (pDeviceList, pDevice, &pOwn->pDeviceTable))
      pLayoutDest->addWidget (pOwn->pDeviceTable, Row++, 0, 1, 3);
   }

   if (CONFIG(WriteToDevNull))
   {
      pLabel = new QLabel(QString("<font color='red'>") + tr("Configuration flag WriteToDevNull is set!") + "</font>", this);
      pLayoutDest->addWidget (pLabel, Row++, 0, 1, 3);
   }
   //lint -restore

   for (i=0; i<pDlgAcquireFields->count(); i++)
   {
      pDlgAcquireField = pDlgAcquireFields->at(i);
      if (pDlgAcquireField->DstField)
         CHK_EXIT (AddField (pDevice, pDlgAcquireField, pLayoutDest, &Row, &Col))
   }

   // Hash box
   // --------
   Row = Col = 0;
   QGroupBox   *pGroupBoxHash = new QGroupBox(tr("Hash calculation / verification"), this);
   QGridLayout *pLayoutHash   = new QGridLayout ();
   pGroupBoxHash->setLayout (pLayoutHash);
   pLayout->addWidget (pGroupBoxHash);

   for (i=0; i<pDlgAcquireFields->count(); i++)
   {
      pDlgAcquireField = pDlgAcquireFields->at(i);
      if (pDlgAcquireField->HashField)
      {
         CHK_EXIT (AddField (pDevice, pDlgAcquireField, pLayoutHash, &Row, &Col))
         if (pDlgAcquireField->FieldName.startsWith (CFG_DLGACQUIRE_HASHCALC_FIELDID))
            CHK_QT_EXIT (connect (pDlgAcquireField->pCheckBox, SIGNAL (stateChanged(int)), this, SLOT(UpdateHashState(int))))
      }
   }

   // Dialog buttons
   // --------------
   QHBoxLayout *pLayoutButtons = new QHBoxLayout (); // The new layout would normally use the margin of its parent layout
   pLayoutButtons->setMargin  (pLayout->margin ());  // (TopLayout), but those have been set to set zero. Instead, we tell
   pLayoutButtons->setSpacing (pLayout->spacing());  // the button layout top use the same margins as the main layout in
   pTopLayout->addLayout (pLayoutButtons);           // order to have everything appear nicely and the same way on the screen).

   pOwn->pButtonOk     = new QPushButton (QObject::tr("Ok"    ), this);
   pOwn->pButtonCancel = new QPushButton (QObject::tr("Cancel"), this);
   pLayoutButtons->addWidget (pOwn->pButtonOk    );
   pLayoutButtons->addWidget (pOwn->pButtonCancel);
   pOwn->pButtonOk->setDefault (true);

   // Set other defaults
   // ------------------
   if (!Clone)
   {
      if (DlgAcquireLastUsedFormat == t_File::NotSet)
         DlgAcquireLastUsedFormat = (t_File::Format) CONFIG (DefaultFormat);

      switch (DlgAcquireLastUsedFormat)
      {
         case t_File::DD : pOwn->pRadioButtonFormatDD ->setChecked (true); break;
         case t_File::EWF: pOwn->pRadioButtonFormatEWF->setChecked (true); break;
         case t_File::AFF: pOwn->pRadioButtonFormatAFF->setChecked (true); break;
         default         : CHK_EXIT (ERROR_DLGACQUIRE_INVALID_FORMAT)
      }
   }

   UpdateHashState  ();
   UpdateDialogState();

   // Adjust size
   // -----------
   // Qt limits dialogs to 2/3 of the screen height and width. That's why we adjust it with
   //  the following code if the dialog is smaller then its size hint.
   adjustSize();  // Let Qt calculate the size (limited to 2/3 of the screen)
   QRect DialogRect = geometry ();
   
   if (DialogRect.height() < sizeHint().height())
   {
      QRect ScreenRect = QApplication::desktop()->screenGeometry(x11Info().screen());     
      int   NewHeight  = GETMIN (ScreenRect.height()*0.9, sizeHint().height());
      
      DialogRect.setX      ((ScreenRect.width () - DialogRect.width())/2);
      DialogRect.setY      ((ScreenRect.height() - NewHeight         )/2);
      DialogRect.setHeight (NewHeight);  // Must be done after setY!
      setGeometry(DialogRect);
   }
   
   // Connections
   // -----------
   if (!Clone)
   {
      CHK_QT_EXIT (connect (pOwn->pRadioButtonFormatDD , SIGNAL (released()), this, SLOT(UpdateFieldState())))
      CHK_QT_EXIT (connect (pOwn->pRadioButtonFormatEWF, SIGNAL (released()), this, SLOT(UpdateFieldState())))
      CHK_QT_EXIT (connect (pOwn->pRadioButtonFormatAFF, SIGNAL (released()), this, SLOT(UpdateFieldState())))
      CHK_QT_EXIT (connect (DlgAcquireGetField ("SplitFileSwitch")->pCheckBox, SIGNAL (released()), this, SLOT(UpdateFileSplitState())))
      UpdateFieldState();
   }
   CHK_QT_EXIT (connect (pOwn->pButtonOk    , SIGNAL (released()), this, SLOT(SlotAccept())))
   CHK_QT_EXIT (connect (pOwn->pButtonCancel, SIGNAL (released()), this, SLOT(reject    ())))
}

void t_DlgAcquire::UpdateHashState (int /*State*/)
{
   t_pCfgDlgAcquireFields pDlgAcquireFields;
   t_pCfgDlgAcquireField  pDlgAcquireField;
   bool                    Hash = false;
   int                     i;

   CHK_EXIT (CfgGetDlgAcquireFields (&pDlgAcquireFields))
   for (i=0; i<pDlgAcquireFields->count(); i++)
   {
      pDlgAcquireField = pDlgAcquireFields->at(i);
      if (pDlgAcquireField->FieldName.startsWith (CFG_DLGACQUIRE_HASHCALC_FIELDID))
         Hash = Hash || pDlgAcquireField->pCheckBox->isChecked();
   }

   for (i=0; i<pDlgAcquireFields->count(); i++)
   {
      pDlgAcquireField = pDlgAcquireFields->at(i);
      if (pDlgAcquireField->FieldName.startsWith (CFG_DLGACQUIRE_HASHVERIFY_FIELDID))
      {
         if (!Hash)
            pDlgAcquireField->pCheckBox->setChecked (false);
         pDlgAcquireField->pCheckBox->setEnabled (Hash);
      }
   }
}

void t_DlgAcquire::UpdateDialogState (const QString & /*NewText*/)
{
   t_pCfgDlgAcquireFields pFields;
   t_pCfgDlgAcquireField  pField;
   bool                    Enabled=true;
   int                     i;

   CHK_EXIT (CfgGetDlgAcquireFields (&pFields))

   for (i=0; i<pFields->count() && Enabled; i++)
   {
      pField = pFields->at(i);
      if (pField->DstField)
         Enabled = Enabled && !pField->pLineEdit->text().isEmpty();
   }

   pOwn->pButtonOk->setEnabled (Enabled);
}

void t_DlgAcquire::SlotDeviceTableSelectionChanged (void)
{
   t_pDlgAcquireLineEdit     pLineEditDirectory;
   t_pDlgAcquireLineEdit     pLineEditFilename;
   QTableWidgetItem         *pItem;
   QList<QTableWidgetItem*>   SelectedItemsList = pOwn->pDeviceTable->selectedItems();
   QFileInfo                  FileInfo;
   int                        Row;
   int                        Selected;
   QString                    LinuxDevice;
   QString                    Directory;

   pLineEditDirectory = DlgAcquireGetField (CFG_DLGACQUIRE_DEST_IMAGEDIRECTORY)->pLineEdit;
   pLineEditFilename  = DlgAcquireGetField (CFG_DLGACQUIRE_DEST_IMAGEFILENAME )->pLineEdit;

   Selected = SelectedItemsList.count();
   if (Selected == 0)
   {
      pLineEditDirectory->setText (QString());
      pLineEditFilename ->setText (QString());
   }
   else
   {
      if (Selected != pOwn->pDeviceTable->columnCount())
      {
         LOG_INFO ("%d devices in device tabel for clone destination", Selected)
         CHK_EXIT (ERROR_DLGACQUIRE_INVALID_SELECTION)
      }
      Row   = SelectedItemsList[0]->row();
      pItem = pOwn->pDeviceTable->item (Row, pOwn->DeviceTableLinuxDeviceCol);
      LinuxDevice = pItem->text();
      FileInfo.setFile (LinuxDevice);
      Directory = FileInfo.dir().absolutePath();
      if (!Directory.endsWith ("/"))
         Directory += "/";

      pLineEditDirectory->setText (Directory);
      pLineEditFilename ->setText (FileInfo.fileName());
   }

   // No need to call UpdateDialogState, as the LineEdit widget will emit a textChanged signal which is connected to UpdateDialogState.
}

void t_DlgAcquire::UpdateFieldState (void)
{
   QWidget *pWidget;
   bool      Enabled;
   int       i;

   // EWF fields
   // ----------
   Enabled = (pOwn->pRadioButtonFormatEWF->isChecked() ||
              pOwn->pRadioButtonFormatAFF->isChecked());
   for (i = 0;
        i < pOwn->EwfWidgetsList.count();
        i++)
   {
      pWidget = pOwn->EwfWidgetsList.at(i);
      pWidget->setEnabled (Enabled);
   }

   // Split file fields
   // -----------------
   static bool OldSplitFileSwitch = false;  // should be replaced

   t_pCfgDlgAcquireField pFieldSplitFileSwitch = DlgAcquireGetField (CFG_DLGACQUIRE_SPLITFILESWITCH);
   t_pCfgDlgAcquireField pFieldSplitFileSize   = DlgAcquireGetField (CFG_DLGACQUIRE_SPLITFILESIZE  );
   t_pCfgDlgAcquireField pFieldSplitFileUnit   = DlgAcquireGetField (CFG_DLGACQUIRE_SPLITFILEUNIT  );
   if (pOwn->pRadioButtonFormatDD->isChecked())
   {
      pFieldSplitFileSwitch->pCheckBox->setChecked (OldSplitFileSwitch);
      pFieldSplitFileSwitch->pCheckBox->setEnabled (true);
   }
   else if (pOwn->pRadioButtonFormatEWF->isChecked())
   {
      if (pFieldSplitFileSwitch->pCheckBox->isEnabled())
         OldSplitFileSwitch = pFieldSplitFileSwitch->pCheckBox->isChecked();
      pFieldSplitFileSwitch->pCheckBox->setChecked (true);
      pFieldSplitFileSwitch->pCheckBox->setEnabled (false);
   }
   else if (pOwn->pRadioButtonFormatAFF->isChecked())
   {
      if (pFieldSplitFileSwitch->pCheckBox->isEnabled())
         OldSplitFileSwitch = pFieldSplitFileSwitch->pCheckBox->isChecked();
      pFieldSplitFileSwitch->pCheckBox->setChecked (false);
      pFieldSplitFileSwitch->pCheckBox->setEnabled (false);
   }
   Enabled = pFieldSplitFileSwitch->pCheckBox->isChecked();
   pFieldSplitFileSize->pLineEdit->setEnabled (Enabled);
   pFieldSplitFileSize->pLabel   ->setEnabled (Enabled);
   pFieldSplitFileUnit->pComboBox->setEnabled (Enabled);
}

void t_DlgAcquire::UpdateFileSplitState (void)
{
   t_pCfgDlgAcquireField pFieldSplitFileSwitch = DlgAcquireGetField (CFG_DLGACQUIRE_SPLITFILESWITCH);
   t_pCfgDlgAcquireField pFieldSplitFileSize   = DlgAcquireGetField (CFG_DLGACQUIRE_SPLITFILESIZE  );
   t_pCfgDlgAcquireField pFieldSplitFileUnit   = DlgAcquireGetField (CFG_DLGACQUIRE_SPLITFILEUNIT  );

   bool Enabled = pFieldSplitFileSwitch->pCheckBox->isChecked();
   pFieldSplitFileSize->pLineEdit->setEnabled (Enabled);
   pFieldSplitFileSize->pLabel   ->setEnabled (Enabled);
   pFieldSplitFileUnit->pComboBox->setEnabled (Enabled);
}

void t_DlgAcquire::SlotBrowse (void)
{
   t_pDlgAcquireLineEdit pLineEdit;
   QString                Path;
   bool                   OkPressed;

   pLineEdit = (t_pDlgAcquireLineEdit) sender()->property (DLGACQUIRE_PROPERTY_SENDER_LINEEDIT).value<void *>();

   if (CONFIG(UseFileDialogFromQt))
   {
      Path = QFileDialog::getExistingDirectory(this, tr("Select destination directory", "Dialog title"), pLineEdit->text(), QFileDialog::ShowDirsOnly);
      OkPressed = !Path.isNull();
   }
   else
   {
      Path = pLineEdit->text();
      t_DlgDirSel Dlg (&Path, this);

      switch (CONFIG(FileDialogSize))
      {
         case CFG_STARTUPSIZE_STANDARD  :                       break;
         case CFG_STARTUPSIZE_FULLSCREEN: Dlg.showFullScreen(); break;
         case CFG_STARTUPSIZE_MAXIMIZED : Dlg.showMaximized (); break;
         case CFG_STARTUPSIZE_MANUAL    : CHK_EXIT (QtUtilSetGeometryCentered (&Dlg, CONFIG(FileDialogSizeManualDx), CONFIG(FileDialogSizeManualDy))); break;
         default                        : CHK_EXIT (ERROR_DLGACQUIRE_UNKNOWN_FILEDIALOG_SIZE)
      }
      OkPressed = (Dlg.exec() == QDialog::Accepted);
   }

   if (OkPressed)
   {
      if (!Path.endsWith ("/"))
         Path += "/";
      pLineEdit->setText (Path); // setText doesn't emit a textEdited signal (that's completely right  in order to prevent endless
      pLineEdit->TextUpdated (); // update signals) and so we have the LineEdit emit a SignalTextEdit, thus triggering field updates.
   }
}

void t_DlgAcquire::SlotTextEdited (t_pDlgAcquireLineEdit pLineEdit, const QString &/*NewVal*/)
{
   t_pCfgDlgAcquireRules pDlgAcquireRules;
   t_pCfgDlgAcquireRule  pDlgAcquireRule;
   t_pDlgAcquireLineEdit pLineEditDest;
   int                    i;
   QString                Set;

//   printf ("\nNewVal in %s: %s", QSTR_TO_PSZ(pLineEdit->Name), QSTR_TO_PSZ(NewVal));

   CHK_EXIT (CfgGetDlgAcquireRules (&pDlgAcquireRules))

   for (i=0; i<pDlgAcquireRules->count(); i++)
   {
      pDlgAcquireRule = pDlgAcquireRules->at(i);
      if (pDlgAcquireRule->TriggerFieldName.compare (pLineEdit->Name, Qt::CaseInsensitive) == 0)
      {
         pLineEditDest = DlgAcquireGetLineEdit (pDlgAcquireRule->DestFieldName);

         CHK_EXIT (DlgAcquireResolveSpecialSequences (pOwn->pDevice, true, pDlgAcquireRule->Value, Set))
         pLineEditDest->setText(Set);
      }
   }
}

static APIRET DlgAcquireCheckFilename (const QString &In, bool &Clean, QString &Out)
{
   unsigned char Ch;
   int           i;
   bool          Ok;
   QString       SpecialChars = CONFIG (SpecialFilenameChars);

   Clean = true;
   Out   = "";
   for (i=0; i<In.length(); i++)
   {
      Ch = In[i].toAscii();
      Ok = ((Ch >= '0') && (Ch <= '9')) ||
           ((Ch >= 'a') && (Ch <= 'z')) ||
           ((Ch >= 'A') && (Ch <= 'Z')) ||
            (Ch == '_') || SpecialChars.contains(Ch);
      if (Ok)
           Out += Ch;
      else Clean = false;
   }
   if (Out.isEmpty())
      Out = DLGACQUIRE_DEFAULT_EMPTY_FILENAME;

   return NO_ERROR;
}


APIRET t_DlgAcquire::CheckWriteAccess (const QString &Path, const QString &Filename, bool &Ok)
{
   FILE       *pFile;
   const char *pStr = "Test file created by Guymager for checking write access.\r\nYou may delete this file";
   const int    StrLen = strlen (pStr);
   QString      TestFileName = Path + Filename + ".test";
   int          wr;
   bool         DirOk;

   // Create directory if it doesn't exist
   // ------------------------------------
   QDir Dir(Path);

   if (Dir.exists())
   {
      LOG_INFO ("Directory %s exists", QSTR_TO_PSZ(Path))
   }
   else
   {
      LOG_INFO ("Directory %s doesn't exist, trying to create it", QSTR_TO_PSZ(Path))
      DirOk = Dir.mkpath(Path);
      if (DirOk)
         DirOk = Dir.exists();

      if (DirOk)
           LOG_INFO ("Directory %s created successfully", QSTR_TO_PSZ(Path))
      else LOG_INFO ("Error while creating directory %s", QSTR_TO_PSZ(Path))
   }

   LOG_INFO ("Trying write access to test file %s", QSTR_TO_PSZ(TestFileName))
   pFile = fopen (QSTR_TO_PSZ(TestFileName), "w");
   Ok = (pFile != NULL);
   if (!Ok)
   {
      LOG_INFO ("Opening test file %s failed", QSTR_TO_PSZ(TestFileName))
   }
   else
   {
      if ((wr = fprintf (pFile, "%s", pStr)) != StrLen)
      {
         Ok = false;
         LOG_INFO ("Writing to test file %s failed, %d of %d bytes written", QSTR_TO_PSZ(TestFileName), wr, StrLen)
      }

      if (fclose (pFile) != 0)
      {
         Ok = false;
         LOG_INFO ("Closing test file %s failed", QSTR_TO_PSZ(TestFileName))
      }

      QDir          Dir(Path);                                                                                           // Check if the test file really exists with that
      QFileInfoList FileInfoList = Dir.entryInfoList (QStringList(Filename + ".test"), QDir::Files | QDir::NoSymLinks);  // name (Otherwise, strange things may happen with
      if (FileInfoList.isEmpty())                                                                                        // the : character, as it used for alternate data
      {                                                                                                                  // streams in some DOS-based OSs.
         Ok = false;
         LOG_INFO ("The test file %s doesn't exist with the correct name", QSTR_TO_PSZ(TestFileName))
      }

      if (!QFile::remove (TestFileName))
      {
         Ok = false;
         LOG_INFO ("Removing the test file %s failed", QSTR_TO_PSZ(TestFileName))
      }
   }

   if (Ok)
        LOG_INFO ("Write access test succeeded")
   else QMessageBox::information (this, tr ("Access denied", "Dialog title"),
                                        tr ("Guymager cannot write to the directory"
                                            "\n\t%1"
                                            "\nThis may be due to insufficient access rights or unsupported filename characters. Please choose another directory.")
                                            .arg(Path), QMessageBox::Ok);
   return NO_ERROR;
}


void t_DlgAcquire::SlotAccept (void)
{
   t_Device::t_Acquisition      Acquisition;
   t_pDlgAcquireLineEdit       pLineEditDestImageFilename;
   t_pDlgAcquireLineEdit       pLineEditDestInfoFilename;
   QMessageBox::StandardButton  Button;
   QString                      ImageFilenameCorrected;
   QString                      InfoFilenameCorrected;
   QString                      Message;
   bool                         ImageFilenameClean;
   bool                         InfoFilenameClean;

   // Check split file size
   // ---------------------
   if (!pOwn->Clone)
   {
      if (DlgAcquireGetField(CFG_DLGACQUIRE_SPLITFILESWITCH)->pCheckBox->isChecked())
      {
         QString            StrValue;
         double             NumValue;
         unsigned long long SplitSize;
         bool               Ok;

         StrValue = DlgAcquireGetField(CFG_DLGACQUIRE_SPLITFILESIZE)->pLineEdit->text();
         NumValue = StrValue.toDouble(&Ok);
         if (!Ok)
         {
            QMessageBox::information (this, tr ("Incorrect value", "Dialog title"),
                                            tr ("The split file size \"%1\" is not valid. Only positive numbers can be entered.").arg(StrValue), QMessageBox::Ok);
            return;
         }
         SplitSize = NumValue * UnitIndexToMultiplier (DlgAcquireGetField(CFG_DLGACQUIRE_SPLITFILEUNIT)->pComboBox->currentIndex());
         #if (LIBEWF_VERSION != 20100226)
            #error "Please check EWF documentation for newer Encase formats and adjust following code"
         #endif
         if (pOwn->pRadioButtonFormatEWF->isChecked())
         {
            if (CONFIG(EwfFormat) == LIBEWF_FORMAT_ENCASE6)
            {
               if ((SplitSize > EWF_MAX_SEGMENT_SIZE_EXT) || (SplitSize < EWF_MIN_SEGMENT_SIZE))
               {
                  QMessageBox::information (this, tr ("Incorrect value", "Dialog title"),
                                                  tr ("The split size for the selected image format must be in the range %1MiB - %2EiB.")
                                                  .arg(EWF_MIN_SEGMENT_SIZE    / (1024*1024))
                                                  .arg(EWF_MAX_SEGMENT_SIZE_EXT/ (double)BYTES_PER_EiB, 0, 'f', 2), QMessageBox::Ok);
                  return;
               }
            }
            else
            {
               if ((NumValue > EWF_MAX_SEGMENT_SIZE) || (NumValue < EWF_MIN_SEGMENT_SIZE))
               {
                  QMessageBox::information (this, tr ("Incorrect value", "Dialog title"),
                                                  tr ("The split size for the configured EWF format must be in the range %1 - %2 MiB."
                                                      "For bigger files, switch to ""Encase6"" format or later (see Guymager configuration file, parameter EwfFormat).")
                                                  .arg(EWF_MIN_SEGMENT_SIZE/(1024*1024))
                                                  .arg(EWF_MAX_SEGMENT_SIZE/(1024*1024)), QMessageBox::Ok);
                  return;
               }
            }
         }
      }
   }

   // Check for strange characters in filenames
   // -----------------------------------------
   pLineEditDestImageFilename = DlgAcquireGetField (CFG_DLGACQUIRE_DEST_IMAGEFILENAME)->pLineEdit;
   pLineEditDestInfoFilename  = DlgAcquireGetField (CFG_DLGACQUIRE_DEST_INFOFILENAME )->pLineEdit;

   if (pOwn->Clone)
   {
      ImageFilenameClean     = true;
      ImageFilenameCorrected = pLineEditDestImageFilename->text();
   }
   else
   {
      CHK_EXIT (DlgAcquireCheckFilename (pLineEditDestImageFilename->text(), ImageFilenameClean, ImageFilenameCorrected))
   }
   CHK_EXIT (DlgAcquireCheckFilename (pLineEditDestInfoFilename->text(), InfoFilenameClean, InfoFilenameCorrected))

   if (!ImageFilenameClean || !InfoFilenameClean)
   {
      LOG_INFO ("Unallowed characters in filenames found")
      if (pOwn->Clone)
           Message = tr ("The info filename contains special characters which are not allowed. Guymager suggests the following changes:"
                         "\n\t%1"
                         "\nDo you accept these changes?") .arg(InfoFilenameCorrected);
      else Message = tr ("The filenames contain special characters which are not allowed. Guymager suggests the following changes:"
                         "\n\tImage filename: %1"
                         "\n\tInfo filename: %2"
                         "\nDo you accept these changes?") .arg(ImageFilenameCorrected) .arg(InfoFilenameCorrected);

      Button = QMessageBox::question (this, tr ("Special characters", "Dialog title"), Message, QMessageBox::Yes | QMessageBox::No, QMessageBox::Yes);
      if (Button == QMessageBox::Yes)
      {
         LOG_INFO ("User accepts filename changes")
         pLineEditDestImageFilename->setText(ImageFilenameCorrected);
         pLineEditDestInfoFilename ->setText(InfoFilenameCorrected );
      }
      else
      {
         LOG_INFO ("User rejects filename changes")
         return;
      }
   }

   // Check if file can be created on destination path
   // ------------------------------------------------
   bool Ok;

   CHK_EXIT (GetParameters (Acquisition, false))

   if (!pOwn->Clone)
   {
      CHK_EXIT (t_DlgAcquire::CheckWriteAccess (Acquisition.InfoPath, Acquisition.InfoFilename, Ok))
      if (Ok && (Acquisition.InfoPath != Acquisition.ImagePath))
         CHK_EXIT (t_DlgAcquire::CheckWriteAccess (Acquisition.ImagePath, Acquisition.ImageFilename, Ok))
      if (!Ok)
         return;
   }

   // Check if image file already exists
   // ----------------------------------

      QDir    Dir (Acquisition.ImagePath);
      QString ExtensionImage;
      bool    Empty;
      QString NameFilter;
      int     DdSplitDecimals=0;
      
      if (pOwn->Clone)
      {
         Empty = true;
      }
      else
      {
         DdSplitDecimals = t_File::GetDdSplitNrDecimals (pOwn->pDevice->Size, Acquisition.SplitFileSize);
         CHK_EXIT (t_File::GetFormatExtension (Acquisition.Format, Acquisition.Clone, DdSplitDecimals, &ExtensionImage))
         NameFilter = Acquisition.ImageFilename + ExtensionImage;
         Empty = Dir.entryInfoList (QStringList(NameFilter), QDir::Files, QDir::Name).isEmpty();
      }
      if (Empty)
      {
         Dir.setPath (Acquisition.InfoPath);
         NameFilter = Acquisition.InfoFilename + t_File::pExtensionInfo;
         Empty = Dir.entryInfoList (QStringList(NameFilter), QDir::Files, QDir::Name).isEmpty();
      }

      if (!Empty)
      {
         LOG_INFO ("Image file / info files already exist")
         Button = QMessageBox::question (this, tr ("Files exist", "Dialog title"),
                                               pOwn->Clone ? tr ("The info file already exists. Do you want to overwrite it?") :
                                                             tr ("The image or info files already exist. Do you want to overwrite them?"),
                                               QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
         if (Button == QMessageBox::Yes)
         {
            LOG_INFO ("User accepts to overwrite existing image/info files")
         }
         else
         {
            LOG_INFO ("User rejects to overwrite existing image/info files")
            return;
         }
      }
   accept();
}


APIRET t_DlgAcquire::GetParameters (t_Device::t_Acquisition &Acquisition, bool RememberLastUsedValues)
{
   t_pCfgDlgAcquireField pDlgAcquireField;
   bool                   SplitFileSwitch;
   QString                SplitFileSize = "0";

   Acquisition.Clone = pOwn->Clone;
   if (pOwn->Clone)
   {
      Acquisition.Format = t_File::DD;
   }
   else
   {
      if      (pOwn->pRadioButtonFormatDD ->isChecked()) Acquisition.Format = t_File::DD;
      else if (pOwn->pRadioButtonFormatEWF->isChecked()) Acquisition.Format = t_File::EWF;
      else                                               Acquisition.Format = t_File::AFF;
      if (RememberLastUsedValues)
         DlgAcquireLastUsedFormat = Acquisition.Format;
   }

   #define COPY_LINEENTRY(Acq, FieldName, Remember)      \
   {                                                     \
      pDlgAcquireField = DlgAcquireGetField (FieldName); \
      Acq = pDlgAcquireField->pLineEdit->text();         \
      if (Remember)                                      \
         pDlgAcquireField->LastEnteredValue = Acq;       \
   }

   #define COPY_CHECKBOX(Acq, FieldName)                 \
   {                                                     \
      pDlgAcquireField = DlgAcquireGetField (FieldName); \
      Acq = pDlgAcquireField->pCheckBox->isChecked();    \
      if (RememberLastUsedValues)                        \
         pDlgAcquireField->LastEnteredValue = Acq  ? "1" : "0"; \
   }

   #define COPY_COMBOBOX(Acq, FieldName, Remember)       \
   {                                                     \
      pDlgAcquireField = DlgAcquireGetField (FieldName); \
      Acq = pDlgAcquireField->pComboBox->currentIndex(); \
      if (Remember)                                      \
         pDlgAcquireField->LastEnteredValue = QString::number (Acq); \
   }

   if (!pOwn->Clone)
   {
      Acquisition.SplitFileSize = 0;
      COPY_LINEENTRY (Acquisition.CaseNumber    , CFG_DLGACQUIRE_EWF_CASENUMBER     , RememberLastUsedValues)
      COPY_LINEENTRY (Acquisition.EvidenceNumber, CFG_DLGACQUIRE_EWF_EVIDENCENUMBER , RememberLastUsedValues)
      COPY_LINEENTRY (Acquisition.Examiner      , CFG_DLGACQUIRE_EWF_EXAMINER       , RememberLastUsedValues)
      COPY_LINEENTRY (Acquisition.Description   , CFG_DLGACQUIRE_EWF_DESCRIPTION    , RememberLastUsedValues)
      COPY_LINEENTRY (Acquisition.Notes         , CFG_DLGACQUIRE_EWF_NOTES          , RememberLastUsedValues)
      COPY_CHECKBOX (SplitFileSwitch, CFG_DLGACQUIRE_SPLITFILESWITCH )
      if (SplitFileSwitch)
      {
         int UnitIndex;
         COPY_LINEENTRY (SplitFileSize, CFG_DLGACQUIRE_SPLITFILESIZE, RememberLastUsedValues)
         COPY_COMBOBOX  (UnitIndex    , CFG_DLGACQUIRE_SPLITFILEUNIT, RememberLastUsedValues)
         Acquisition.SplitFileSize = SplitFileSize.toULongLong() * UnitIndexToMultiplier(UnitIndex);
      }
   }
   COPY_LINEENTRY (Acquisition.ImagePath     , CFG_DLGACQUIRE_DEST_IMAGEDIRECTORY, pOwn->Clone ? false : RememberLastUsedValues)
   COPY_LINEENTRY (Acquisition.ImageFilename , CFG_DLGACQUIRE_DEST_IMAGEFILENAME , pOwn->Clone ? false : RememberLastUsedValues)
   COPY_LINEENTRY (Acquisition.InfoPath      , CFG_DLGACQUIRE_DEST_INFODIRECTORY , RememberLastUsedValues)
   COPY_LINEENTRY (Acquisition.InfoFilename  , CFG_DLGACQUIRE_DEST_INFOFILENAME  , RememberLastUsedValues)

   COPY_CHECKBOX (Acquisition.CalcMD5   , CFG_DLGACQUIRE_HASH_CALC_MD5   )
   COPY_CHECKBOX (Acquisition.CalcSHA256, CFG_DLGACQUIRE_HASH_CALC_SHA256)
   COPY_CHECKBOX (Acquisition.VerifySrc , CFG_DLGACQUIRE_HASH_VERIFY_SRC )
   COPY_CHECKBOX (Acquisition.VerifyDst , CFG_DLGACQUIRE_HASH_VERIFY_DST )

   #undef COPY_LINEENTRY
   #undef COPY_CHECKBOX

   return NO_ERROR;
}

t_DlgAcquire::~t_DlgAcquire ()
{
   delete pOwn;
}

