// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         The DeviceList represents the central data structure of
//                  the application. Its contents are displayed in the main
//                  window.
// ****************************************************************************

// Copyright 2008, 2009, 2010, 2011 Guy Voncken
//
// This file is part of guymager.
//
// guymager is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// guymager is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with guymager. If not, see <http://www.gnu.org/licenses/>.

#include <float.h>

#include <QString>
#include <QList>

#include "toolconstants.h"

#include "common.h"
#include "qtutil.h"
#include "config.h"
#include "mainwindow.h"
#include "main.h"

// ---------------------------
//         t_Device
// ---------------------------

static const int DEVICE_MIN_RUNNING_SECONDS = 20;  // The minmum number of acquisition seconds before any estimation can be made

void t_Device::Initialise (void)
{
   static bool Initialised = false;

   if (!Initialised)
   {
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_SERNR_MATCH_MODEL_MISMATCH ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_SERNR_MATCH_LENGTH_MISMATCH))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_BAD_STATE                  ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_BAD_ABORTREASON            ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_NOT_CLEAN                  ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_INVALID_MEDIAINFOSTATE     ))
      Initialised = true;
   }

//   this->SerialNumber      = QString();
//   this->LinuxDevice       = QString();
//   this->Model             = QString();
   this->Local                = false;
   this->SectorSize           = 0;
   this->SectorSizePhys       = 0;
   this->Size                 = 0;
   this->SpecialDevice        = false;
   this->Removable            = false;

   this->State                = Idle;
//   this->Acquisition.Message       = QString();
   this->AbortRequest         = false;
   this->AbortReason          = None;
   this->AbortCount           = 0;
   this->DeleteAfterAbort     = false;
   this->pFifoMemory          = NULL;
   this->pFifoRead            = NULL;
   this->pThreadRead          = NULL;
   this->FileDescSrc          = t_Device::FileDescEmpty;
   this->CurrentReadPos       = 0;
//   this->BadSectors
//   this->BadSectorsVerify
   this->FallbackMode         = false;
   this->StartTimestamp       = QDateTime();
   this->StartTimestampVerify = QDateTime();
   this->StopTimestamp        = QDateTime();

   this->pThreadWrite         = NULL;
   this->CurrentWritePos      = 0;
   this->CurrentVerifyPosSrc  = 0;
   this->CurrentVerifyPosDst  = 0;
   this->pFifoWrite           = NULL;

   this->pThreadHash          = NULL;
   this->pFifoHashIn          = NULL;
   this->pFifoHashOut         = NULL;
   memset (&this->MD5Digest            , 0, sizeof(this->MD5Digest            ));
   memset (&this->MD5DigestVerifySrc   , 0, sizeof(this->MD5DigestVerifySrc   ));
   memset (&this->MD5DigestVerifyDst   , 0, sizeof(this->MD5DigestVerifyDst   ));
   memset (&this->SHA256Digest         , 0, sizeof(this->SHA256Digest         ));
   memset (&this->SHA256DigestVerifySrc, 0, sizeof(this->SHA256DigestVerifySrc));
   memset (&this->SHA256DigestVerifyDst, 0, sizeof(this->SHA256DigestVerifyDst));
// this->ImageFileHashes

   this->pFifoCompressIn      = NULL;
   this->pFifoCompressOut     = NULL;
   this->FifoMaxBlocks        = 0;
   this->FifoBlockSize        = 0;

//   this->Acquisition.Path          = QString();
//   this->Acquisition.ImageFilename = QString();
//   this->Acquisition.InfoFilename  = QString();
   this->Acquisition.Clone         = false;
   this->Acquisition.Format        = t_File::NotSet;
   this->Acquisition.CalcMD5       = false;
   this->Acquisition.CalcSHA256    = false;
   this->Acquisition.VerifySrc     = false;
   this->Acquisition.VerifyDst     = false;
   this->Acquisition.SplitFileSize = 0;
//   this->Acquisition.CaseNumber    = QString();
//   this->Acquisition.EvidenceNumber= QString();
//   this->Acquisition.Examiner      = QString();
//   this->Acquisition.Description   = QString();
//   this->Acquisition.Notes         = QString();

   this->AddStateInfo.CanBeAcquired = true   ;
   this->AddStateInfo.Color         = COLOR_DEFAULT;
// this->AddStateInfo.Info          = QString();
   
   this->Info.SetDevice(this);

   this->PrevTimestamp = QTime();
   this->PrevSpeed     = 0.0;
   this->Checked       = false;
   this->AddedNow      = false;
}

void t_Device::InitialiseDeviceSpecific (const QString &SerialNumber, const QString &LinuxDevice, const QString &Model,
                                         quint64 SectorSize, quint64 SectorSizePhys, quint64 Size)
{                                        //lint !e578: Declaration of symbol 'Xxx' hides symbol 't_Device::Xxx'
   this->SerialNumber   = SerialNumber;
   this->LinuxDevice    = LinuxDevice;
//   this->LinuxDevice    = "/data/virtualbox/knoppicilin.iso";
   this->Model          = Model;

   this->SectorSize     = SectorSize;
   this->SectorSizePhys = SectorSizePhys;
   this->Size           = Size;

//   #define FAKE_DEVICE_SIZE
   #ifdef FAKE_DEVICE_SIZE
   #warning "Debug code for device size faking still enabled"
       this->Size = GETMIN ( 10*1024, Size);               // 10K
      // this->Size = GETMIN ( 20971520ULL, Size);           // 20 MiB
      // this->Size = GETMIN ( 1024*1024*1024ULL, Size);     //  1 GB
      // this->Size = GETMIN (10737418240ULL, Size);         // 10 GiB
      // this->Size = 3ULL*1024ULL*1024ULL*1024ULL*1024ULL;  //  3 TB
      printf ("\nSize of all devices faked to %llu bytes", this->Size);
   #endif
}

t_Device::t_Device (const QString &SerialNumber, const QString &LinuxDevice, const QString &Model,
                    quint64 SectorSize, quint64 SectorSizePhys, quint64 Size)
{                   //lint !e578: Declaration of symbol 'Xxx' hides symbol 't_Device::Xxx'
   Initialise ();
   InitialiseDeviceSpecific (SerialNumber, LinuxDevice, Model, SectorSize, SectorSizePhys, Size);
}

t_Device::t_Device (const QString &SerialNumber, const PedDevice *pPedDev)
{                   //lint !e578: Declaration of symbol 'Xxx' hides symbol 't_Device::Xxx'
   Initialise ();
   InitialiseDeviceSpecific (SerialNumber, pPedDev->path, pPedDev->model,
                                           (quint64) pPedDev->sector_size, (quint64) pPedDev->phys_sector_size,
                                           (quint64) pPedDev->length * (quint64) pPedDev->sector_size);
}

t_Device::~t_Device()
{
   if (pFifoRead    || pThreadRead  ||
       pFifoHashIn  || pThreadHash  ||
       pFifoHashOut || pThreadWrite ||
       pFifoWrite   || (FileDescSrc != t_Device::FileDescEmpty))
      CHK_EXIT (ERROR_DEVICE_NOT_CLEAN)
}

bool t_Device::HasHashThread (void) const
{
   return CONFIG (UseSeparateHashThread) && (Acquisition.CalcMD5 ||
                                             Acquisition.CalcSHA256);
}

bool t_Device::HasCompressionThreads (void) const
{
   return (CONFIG (CompressionThreads) > 0) && ((Acquisition.Format == t_File::EWF ) ||
                                                (Acquisition.Format == t_File::AAFF) ||
                                                (Acquisition.Format == t_File::AEWF));
}


//lint -save -e818 pDevice could have declared as pointing to const

QVariant t_Device::GetSerialNumber (t_pDevice pDevice)
{
   return pDevice->SerialNumber;
}

QVariant t_Device::GetAddStateInfo (t_pDevice pDevice)
{
   return pDevice->AddStateInfo.Info;
}

QVariant t_Device::GetLinuxDevice (t_pDevice pDevice)
{
   return pDevice->LinuxDevice;
}

QVariant t_Device::GetModel (t_pDevice pDevice)
{
   return pDevice->Model;
}

QVariant t_Device::GetState (t_pDevice pDevice)
{
   QString StrState;

   switch (pDevice->State)
   {
      case Idle         : if (pDevice->Local)
                          {
                             StrState = t_MainWindow::tr("Local device");
                          }
                          else 
                          {
                             if (MainGetDeviceList()->UsedAsCloneDestination(pDevice))
                             {
                                StrState = t_MainWindow::tr("Used in clone operation");
                             }
                             else
                             {
                                StrState = t_MainWindow::tr("Idle");
                                if (CONFIG (CommandGetAddStateInfo)[0])
                                   StrState += " - " + pDevice->AddStateInfo.Info;
                             }
                          }
                          break;
      case Acquire      : if (pDevice->AbortRequest)
                               StrState = t_MainWindow::tr("Aborting..."        );
                          else StrState = t_MainWindow::tr("Acquisition running");

                          if (pDevice->AbortRequest)
                          {
                             switch (pDevice->AbortCount)
                             {
                                case  0:
                                case  1: break;
                                case  2: StrState += " please be patient"; break;
                                case  3: StrState += " just keep cool"   ; break;
                                case  4: StrState += " I said KEEP COOL!"; break;
                                case  5: StrState += " you start annoying me"; break;
                                case  6: StrState += " just shut up"; break;
                                case  7: StrState += " SHUT UP!"; break;
                                case  8: StrState += " you want a segmentation fault?"; break;
                                case  9: StrState += " you can have one if you want"; break;
                                case 10: StrState += " I warn you"; break;
                                case 11: StrState += " last warning"; break;
                                case 12: StrState += " ultimate warning"; break;
                                case 13: StrState += " one more time and I'll seg fault"; break;
                                case 14: static bool JokeOver = false;
                                         if (!JokeOver)
                                         {
                                            JokeOver = true;
                                            printf ("\n%c[1;37;41m", ASCII_ESC);  // white on red
                                            printf ("Signal no. 11 received: Segmentation fault");
                                            printf ("%c[0m"      , ASCII_ESC);  // standard
                                            (void) QtUtilSleep (4000);
                                            printf ("     Just kidding  :-)");
                                         }
                                         StrState += " just kidding :-)";
                                default: break;
                             }
                          }
                          break;
      case Verify       : StrState = t_MainWindow::tr("Verification running"); break;
      case AcquirePaused: StrState = t_MainWindow::tr("Device disconnected, acquisition paused" ); break;
      case VerifyPaused : StrState = t_MainWindow::tr("Device disconnected, verification paused"); break;
      case Cleanup      : StrState = t_MainWindow::tr("Cleanup" ); break;
      case Finished     : if ((pDevice->Acquisition.VerifySrc) || (pDevice->Acquisition.VerifyDst))
                          {
                             bool Match = true;
                             if (pDevice->Acquisition.VerifySrc) Match = Match && HashMD5Match   (&pDevice->MD5Digest   , &pDevice->MD5DigestVerifySrc   )
                                                                               && HashSHA256Match(&pDevice->SHA256Digest, &pDevice->SHA256DigestVerifySrc);
                             if (pDevice->Acquisition.VerifyDst) Match = Match && HashMD5Match   (&pDevice->MD5Digest   , &pDevice->MD5DigestVerifyDst   )
                                                                               && HashSHA256Match(&pDevice->SHA256Digest, &pDevice->SHA256DigestVerifyDst);
                             if (Match)
                                  StrState = t_MainWindow::tr("Finished - Verified & ok"      );
                             else StrState = t_MainWindow::tr("Finished - Verification failed");
                          }
                          else
                          {
                             StrState = t_MainWindow::tr("Finished");
                          }
                          break;
      case Aborted : switch (pDevice->AbortReason)
                     {
                        case t_Device::None                  : StrState = t_MainWindow::tr("Aborted - Error: Reason is 'none'" ); break;
                        case t_Device::UserRequest           : StrState = t_MainWindow::tr("Aborted by user" );                   break;
                        case t_Device::ThreadWriteWriteError : StrState = t_MainWindow::tr("Aborted - Image file write error" );  break;
                        case t_Device::ThreadWriteVerifyError: StrState = t_MainWindow::tr("Aborted - Image file verify error" ); break;
                        case t_Device::ThreadReadFileError   : StrState = t_MainWindow::tr("Aborted - Device read error" );       break;
                        case t_Device::InfoWriteError        : StrState = t_MainWindow::tr("Aborted - Info file write error");    break;
                        case t_Device::AcquisitionStartFailed: StrState = t_MainWindow::tr("Aborted - Acquisition start failure");break;
                        default                              : CHK_EXIT (ERROR_DEVICE_BAD_ABORTREASON)
                     }
                     break;
      default      : CHK_EXIT (ERROR_DEVICE_BAD_STATE)
   }
   if (pDevice->State != Aborted)
   {
      QString Msg;
      CHK_EXIT (pDevice->GetMessage (Msg))
      if (!Msg.isEmpty())
         StrState += " - " + Msg;
   }

   return StrState;
}

QVariant t_Device::GetSectorSize (t_pDevice pDevice)
{
   return pDevice->SectorSize;
}

QVariant t_Device::GetSectorSizePhys (t_pDevice pDevice)
{
   return pDevice->SectorSizePhys;
}

QVariant t_Device::GetSize (t_pDevice pDevice)
{
   return pDevice->Size;
}


QVariant t_Device::GetSizeHumanFrac (t_pDevice pDevice, bool SI, int FracLen, int UnitThreshold)
{
   QString      SizeHuman;
   const char *pBaseUnit = "Byte";
   const char *pUnit;
   double       Sz;
   double       Divisor;

   if (SI)
        Divisor = 1000.0;
   else Divisor = 1024.0;

   if (UnitThreshold == AutoUnitThreshold)
      UnitThreshold = (int) Divisor;

   Sz = pDevice->Size;
   pUnit = pBaseUnit;
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "kB" : "KiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "MB" : "MiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "GB" : "GiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "TB" : "TiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "PB" : "PiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "EB" : "EiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "ZB" : "ZiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "YB" : "YiB"; }

   if (FracLen == AutoFracLen)
   {
      if (pUnit == pBaseUnit) FracLen = 0; // no frac if unit is bytes
      else if (Sz >= 100)     FracLen = 0;
      else if (Sz >= 10 )     FracLen = 1;
      else                    FracLen = 2;
   }
   SizeHuman = MainGetpNumberLocale()->toString (Sz, 'f', FracLen) + pUnit;

   return SizeHuman;
}

QVariant t_Device::GetSizeHuman (t_pDevice pDevice)
{
   return GetSizeHumanFrac (pDevice, true, 1);
}

static QString DeviceMediaInfoStr (t_MediaInfo::t_MediaState State)
{
   QString Str;
   switch (State)
   {
      case t_MediaInfo::Unknown: Str = t_MainWindow::tr("Unknown", "Media info string for informing about hidden areas (HPA/DCO)"); break;
      case t_MediaInfo::No     : Str = t_MainWindow::tr("No"     , "Media info string for informing about hidden areas (HPA/DCO)"); break;
      case t_MediaInfo::Yes    : Str = t_MainWindow::tr("Yes"    , "Media info string for informing about hidden areas (HPA/DCO)"); break;
      default: CHK_EXIT (ERROR_DEVICE_INVALID_MEDIAINFOSTATE)
   }
   return Str;
}

QVariant t_Device::GetHiddenAreas (t_pDevice pDevice)
{
   QString Str;

   if      ((pDevice->MediaInfo.HasHPA == t_MediaInfo::No     ) && (pDevice->MediaInfo.HasDCO == t_MediaInfo::No     )) Str = t_MainWindow::tr("none"   , "column hidden areas");
   else if ((pDevice->MediaInfo.HasHPA == t_MediaInfo::Unknown) && (pDevice->MediaInfo.HasDCO == t_MediaInfo::Unknown)) Str = t_MainWindow::tr("unknown", "column hidden areas");
   else
   {
      Str  = "HPA:"    + DeviceMediaInfoStr(pDevice->MediaInfo.HasHPA);
      Str += " / DCO:" + DeviceMediaInfoStr(pDevice->MediaInfo.HasDCO);
   }

   return Str;
}

QVariant t_Device::GetBadSectorCount (t_pDevice pDevice)
{
   if (!pDevice->StartTimestamp.isNull())
        return pDevice->GetBadSectorCount (false);
   else return "";
}

static void DeviceGetProgress (t_pDevice pDevice, quint64 *pCurrent=NULL, quint64 *pTotal=NULL)
{
   quint64 Current, Total;

   if ((pDevice->Acquisition.VerifySrc) ||
       (pDevice->Acquisition.VerifyDst))
   {
      Total = 2 * pDevice->Size;
      if (pDevice->StartTimestampVerify.isNull())
           Current = pDevice->GetCurrentWritePos ();
      else Current = pDevice->GetCurrentVerifyPos() + pDevice->Size;
   }
   else
   {
      Total   = pDevice->Size;
      Current = pDevice->GetCurrentWritePos();
   }
   if (pTotal  ) *pTotal   = Total;
   if (pCurrent) *pCurrent = Current;
}

QVariant t_Device::GetProgress (t_pDevice pDevice)
{
   quint64 Current, Total;

   if (!pDevice->StartTimestamp.isNull())
   {
      DeviceGetProgress (pDevice, &Current, &Total);
      return (double) Current / Total;
   }

   return -DBL_MAX;
}

QVariant t_Device::GetCurrentSpeed (t_pDevice pDevice)
{
   QString Result;
   QTime   Now;
   double  MBs;
   double  Speed;
   int     MilliSeconds;
   quint64 CurrentPos;

   if ((pDevice->State == Acquire) ||  // Don't display anything if no acquisition is running
       (pDevice->State == Verify ))
   {
      Now = QTime::currentTime();
      DeviceGetProgress (pDevice, &CurrentPos);
      if (!pDevice->PrevTimestamp.isNull())
      {
         MilliSeconds = pDevice->PrevTimestamp.msecsTo (Now); // As this fn is called within 1s-interval, time_t would not be precise enough
         if (MilliSeconds > 0)
         {
            MBs   = (double) (CurrentPos - pDevice->PrevPos) / BYTES_PER_MEGABYTE;
            Speed = (1000.0*MBs) / MilliSeconds;
            pDevice->PrevSpeed = (pDevice->PrevSpeed + Speed) / 2.0;
         }
         Result = MainGetpNumberLocale()->toString (pDevice->PrevSpeed, 'f', 2);
      }
      pDevice->PrevTimestamp = Now;
      pDevice->PrevPos       = CurrentPos;
   }

   return Result;
}

QVariant t_Device::GetAverageSpeed (t_pDevice pDevice)
{
   QLocale Locale;
   QString Result;
   double  MBs;
   int     Seconds;
   quint64 CurrentPos;

   if (pDevice->StartTimestamp.isNull())
      return QString();

   if (pDevice->StopTimestamp.isNull()) // As long as the stop timestamp is Null, the acquisition is still running
        Seconds = pDevice->StartTimestamp.secsTo (QDateTime::currentDateTime());
   else Seconds = pDevice->StartTimestamp.secsTo (pDevice->StopTimestamp);

   if ((pDevice->StopTimestamp.isNull()) && (Seconds < DEVICE_MIN_RUNNING_SECONDS))
      return "--";

   DeviceGetProgress (pDevice, &CurrentPos);
   MBs = ((double) CurrentPos) / BYTES_PER_MEGABYTE;
   Result = Locale.toString (MBs/Seconds, 'f', 2);

   return Result;
}

QVariant t_Device::GetRemaining (t_pDevice pDevice)
{
   QString Result;
   char    Buff[10];
   int     TotalSeconds;
   time_t  Now;
   int     hh, mm, ss;
   quint64 Current, Total;

   if (!pDevice->AbortRequest && ((pDevice->State == Acquire) || // Don't display anything if no acquisition is running
                                  (pDevice->State == Verify )))
   {
      time (&Now);
      TotalSeconds = pDevice->StartTimestamp.secsTo (QDateTime::currentDateTime());
      if (TotalSeconds < DEVICE_MIN_RUNNING_SECONDS)
      {
         Result = "--";
      }
      else
      {
         DeviceGetProgress (pDevice, &Current, &Total);
         ss  = (int) ((double)Total / Current * TotalSeconds); // Estimated total time
         ss -= TotalSeconds;                                   // Estimated remaining time
         hh = ss / SECONDS_PER_HOUR;   ss %= SECONDS_PER_HOUR;
         mm = ss / SECONDS_PER_MINUTE; ss %= SECONDS_PER_MINUTE;
         snprintf (Buff, sizeof(Buff), "%02d:%02d:%02d", hh, mm, ss);
         Result = Buff;
      }
   }

   return Result;
}

static inline int DeviceFifoUsage (t_pFifo pFifo)
{
   int Usage;
   CHK_EXIT (pFifo->Usage(Usage))
   return Usage;
}

QVariant t_Device::GetFifoStatus (t_pDevice pDevice)
{
   QString Result;

   if ((pDevice->State == Acquire) ||  // Don't display anything if no acquisition is running
       (pDevice->State == Verify ))
   {
      int FifoUsageWrite = 0;                                   // The write thread gets a lot of dummy blocks into its Fifo (from SlotThreadCompressFinished), but 
                                                                // only 1 of them is taken out (then, the write already exits its loop). If image verification is on
      if (pDevice->State != Verify)                             // then these dummy blocks remain the in the write fifo, leading to an ugly fifo usage display on the 
         FifoUsageWrite = DeviceFifoUsage(pDevice->pFifoWrite); // GUI. As I do not like to add code for remove dummy blocks (and waiting until they all have been 
                                                                // received etc.) I decided to zero them artificially here.
      if (!pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
      {
         Result = QString ("r %1 w") .arg (FifoUsageWrite);
      }
      else if (!pDevice->HasHashThread() && pDevice->HasCompressionThreads())
      {
         Result = QString ("r %1 c %2 w") .arg (DeviceFifoUsage(pDevice->pFifoRead      ))
                                          .arg (FifoUsageWrite);
      }
      else if (pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
      {
         Result = QString ("r %1 m %2 w") .arg (DeviceFifoUsage(pDevice->pFifoRead   ))
                                          .arg (FifoUsageWrite);
      }
      else if (pDevice->HasHashThread() && pDevice->HasCompressionThreads())
      {
         Result = QString ("r %1 h %2 c %3 w") .arg (DeviceFifoUsage(pDevice->pFifoRead       ))
                                               .arg (DeviceFifoUsage(pDevice->pFifoHashOut    ))
                                               .arg (FifoUsageWrite);
      }
   }

   return Result;
}

//lint -restore

const char *t_Device::StateStr (void)
{
   const char *pStr;
   switch (State)
   {
      case Idle         : pStr = "Idle";            break;
      case Acquire      : pStr = "Acquire";         break;
      case AcquirePaused: pStr = "AcquirePaused";   break;
      case Verify       : pStr = "Verify";          break;
      case VerifyPaused : pStr = "VerifyPaused";    break;
      case Cleanup      : pStr = "Cleanup";         break;
      case Finished     : pStr = "Finished";        break;
      case Aborted      : pStr = "Aborte";          break;
      default           : pStr = "Unknown";
   }

   return pStr;
}

// ---------------------------
//        t_DeviceList
// ---------------------------

t_DeviceList::t_DeviceList(void)
   :QList<t_pDevice>()
{
   static bool Initialised = false;

   if (!Initialised)
   {
      Initialised = true;
      qRegisterMetaType<t_pDeviceList>("t_pDeviceList");
      qRegisterMetaType<t_pDevice>    ("t_pDevice"    );
   }
}

t_DeviceList::~t_DeviceList()
{
   int i;

   for (i=0; i<count(); i++)
      delete at(i);
}

t_pDevice t_DeviceList::AppendNew (const QString &SerialNumber, const PedDevice *pPedDev)
{
   t_pDevice pDev;

   pDev = new t_Device (SerialNumber, pPedDev);
   append (pDev);

   return pDev;
}

t_pDevice t_DeviceList::AppendNew (const QString &SerialNumber, const QString &LinuxDevice, const QString &Model,
                                   quint64 SectorSize, quint64 SectorSizePhys, quint64 Size)
{
   t_pDevice pDev;

   pDev = new t_Device (SerialNumber, LinuxDevice, Model, SectorSize, SectorSizePhys, Size);
   append (pDev);

   return pDev;
}

APIRET t_DeviceList::MatchDevice (t_pcDevice pDevCmp, t_pDevice &pDevMatch)
{
   bool Found = false;
   int  i;

   for (i=0; (i<count()) && !Found; i++)
   {
      pDevMatch = at(i);
      if (pDevCmp->SpecialDevice)  // Special device, we may only compare the LinuxDevice string
      {
         Found = (pDevMatch->LinuxDevice == pDevCmp->LinuxDevice);
      }         
      else  // Not a special device, we can use normal matching
      {
// Lines below commented out, as somebody might change assignement of the devices starting with the names shown below after Guymager has done a device scan
//         if ((pDevMatch->LinuxDevice.startsWidth("/dev/loop") ||  // These are devices generated internally by the loop device driver,
//             (pDevMatch->LinuxDevice.startsWidth("/dev/dm-" ) ||  // LVM or DM. we can be sure that they have unique names, so let's
//             (pDevMatch->LinuxDevice.startsWidth("/dev/md"  ))    // avoid all the problems with empty serial numbers and so on.
//         {
//            Found = (pDevMatch->LinuxDevice == pDevCmp->LinuxDevice);
//         }
//         else
              if (pDevMatch->SerialNumber.isEmpty())
         {
            Found = (pDevCmp->SerialNumber.isEmpty() && (pDevMatch->State != t_Device::AcquirePaused) &&
                                                        (pDevMatch->State != t_Device::VerifyPaused ) &&
                                                        (pDevMatch->Model == pDevCmp->Model         ) &&
                                                        (pDevMatch->Size  == pDevCmp->Size          ));
         }
         else
         {
//         Found = (pDevMatch->SerialNumber == pDevCmp->SerialNumber);
//         if (Found)
//         {                                                                                            // If the serial numbers match, the whole
//            if (pDevMatch->Model != pDevCmp->Model) return ERROR_DEVICE_SERNR_MATCH_MODEL_MISMATCH;   // rest must match as well. Otherwise,
//            if (pDevMatch->Size  != pDevCmp->Size ) return ERROR_DEVICE_SERNR_MATCH_LENGTH_MISMATCH;  // something very strange is going on...
//         }

// The comparision above cannot be used, because there are certain devices, that behave as if 2 devices are connected when
// plugged in - with the same serial number!! Example: I once had a memory stick from Intuix that reported 2 devices: The
// memory stick itself and a CD-Rom containing device drivers for some strange OS. Both devices had the same serial number. 
// That's why the check now also includes the size and the model.

            Found = ((pDevMatch->SerialNumber == pDevCmp->SerialNumber) &&
                     (pDevMatch->Size         == pDevCmp->Size )        &&
                     (pDevMatch->Model        == pDevCmp->Model));
         }
      }
   }

   if (!Found)
      pDevMatch = NULL;

   return NO_ERROR;
}

bool t_DeviceList::UsedAsCloneDestination (t_pcDevice pDevChk)
{
   t_pcDevice pDevCmp;
   int         i;

   for (i=0; i<count(); i++)
   {
      pDevCmp = at(i);
      if ( (pDevCmp->State != t_Device::Idle    ) &&
           (pDevCmp->State != t_Device::Finished) &&
           (pDevCmp->State != t_Device::Aborted ) &&
           (pDevCmp->Acquisition.Clone          ) &&
          ((pDevCmp->Acquisition.ImagePath + pDevCmp->Acquisition.ImageFilename) == pDevChk->LinuxDevice))
         return true;
   }
   return false;
}

