/*  GWhere
 *  Copyright (C) 2000  Sbastien LECACHEUR
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*!
 * @header	gwcategory.h
 *		GWhere<br>
 *		Copyright (C) 2000  Sbastien LECACHEUR<br><br>
 *		This Distributed under the terms of the <a href="http://www.gnu.org/copyleft/gpl.html">GNU General Public Licence</a><br><br>
 *		This component was developed for the GWhere project.<br>
 *		It's an object for category representation. His structure gets many informations as file name, description, ...<br><br>
 * @abstract	This is the category data structure.
 * @author	Sbastien LECACHEUR <zero@gwhere.org>
 * @version	1.0 2001/08/21
 * @see		gwsupport.h
 * @see		glib.h
 * @link		GWhere Web Site <http://www.gwhere.org>
 * @link		GTK+ Web Site <http://www.gtk.org>
 * @ref		gwcategory.c
 * @author	Author <zero@gwhere.org>
 */


#ifndef GW_DATA_CATEGORY_H
#define GW_DATA_CATEGORY_H


#include <glib.h>


/*!
 * @struct	category
 * @abstract	Category data structure
 * @discussion	Category data is an opaque data structure containing information about a category.
 */
struct category;


/*!
 * @function	category_new
 * @abstract	Creates a new category.
 * @discussion	This function creates a new category. It's an allocation function.
 * @result	a pointer to the category structure.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
struct category * category_new ( void);


/*!
 * @function	category_set_index
 * @abstract	Sets the category's index.
 * @discussion	This function sets the category's index. This index must be unique.<br>
 * @param	p		the category to set the name
 * @param	index		the index number
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_set_index ( struct category *p, gint index);


/*!
 * @function	category_set_name
 * @abstract	Sets the category's name.
 * @discussion	This function sets the category's name.<br>
 *		The name may be <i>null</i>. It's dangerous!!
 * @param	p		the category to set the name
 * @param	name		the name
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_set_name ( struct category *p, gchar *name);


/*!
 * @function	category_set_description
 * @abstract	Sets the category's description.
 * @discussion	This function sets the category's description.<br>
 * @param	p		the category to set the description
 * @param	description	the description
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_set_description ( struct category *p, gchar *description);


/*!
 * @function	category_set_number_catalog
 * @abstract	Sets the category's number catalogs.
 * @discussion	This function sets the number of catalogs which use this category.<br>
 * @param	p		the category to set the number of catalogs
 * @param	num		the number of catalogs
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_set_number_catalog ( struct category *p, gint num);


/*!
 * @function	category_set_number_disk
 * @abstract	Sets the category's number disks.
 * @discussion	This function sets the number of disks which use this category.<br>
 * @param	p		the category to set the number of disks
 * @param	num		the number of disks
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_set_number_disk ( struct category *p, gint num);


/*!
 * @function	category_set_number_folder
 * @abstract	Sets the category's number folders.
 * @discussion	This function sets the number of folders which use this category.<br>
 * @param	p		the category to set the number of folders
 * @param	num		the number of folders
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_set_number_folder ( struct category *p, gint num);


/*!
 * @function	category_set_number_file
 * @abstract	Sets the category's number files.
 * @discussion	This function sets the number of files which use this category.<br>
 * @param	p		the category to set the number of files
 * @param	num		the number of files
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_set_number_file ( struct category *p, gint num);


/*!
 * @function	category_get_index
 * @abstract	Gets the category's index.
 * @discussion	This function gets the category's index.<br>
 * @param	p		the category to get index
 * @result	the category index.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_get_index ( struct category *p);


/*!
 * @function	category_get_name
 * @abstract	Gets the category's name.
 * @discussion	This function gets the category's name.<br>
 * @param	p		the category to get name
 * @result	the category name.<br>
 *		Returns <i>""</i> if the category name is <i>NULL</i>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * category_get_name ( struct category *p);


/*!
 * @function	category_get_description
 * @abstract	Gets the category's description.
 * @discussion	This function gets the category's description.<br>
 * @param	p		the category to get description
 * @result	the category description.<br>
 *		Returns <i>""</i> if the category description is <i>NULL</i>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
gchar * category_get_description ( struct category *p);


/*!
 * @function	category_get_number_catalog
 * @abstract	Gets the category's number catalogs.
 * @discussion	This function gets the number of catalogs which use this category.<br>
 * @param	p		the category to get number of catalogs
 * @result	the number of catalogs.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_get_number_catalog ( struct category *p);


/*!
 * @function	category_get_number_disk
 * @abstract	Gets the category's number disks.
 * @discussion	This function gets the number of disks which use this category.<br>
 * @param	p		the category to get number of disks
 * @result	the number of disks.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_get_number_disk ( struct category *p);


/*!
 * @function	category_get_number_folder
 * @abstract	Gets the category's number folders.
 * @discussion	This function gets the number of folders which use this category.<br>
 * @param	p		the category to get number of folders
 * @result	the number of folders.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_get_number_folder ( struct category *p);


/*!
 * @function	category_get_number_file
 * @abstract	Gets the category's number files.
 * @discussion	This function gets the number of files which use this category.<br>
 * @param	p		the category to get number of files
 * @result	the number of files.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_get_number_file ( struct category *p);


/*!
 * @function	category_add_catalog
 * @abstract	Adds a catalog to the category
 * @discussion	This function adds a catalog that use this category. This function increments the category's number of catalogs.<br>
 * @param	p		the category to add a catalog
 * @result	the number of catalogs.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_add_catalog ( struct category *p);


/*!
 * @function	category_add_disk
 * @abstract	Adds a disk to the category
 * @discussion	This function adds a disk that use this category. This function increments the category's number of disks.<br>
 * @param	p		the category to add a disk
 * @result	the number of disks.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_add_disk ( struct category *p);


/*!
 * @function	category_add_folder
 * @abstract	Adds a folder to the category
 * @discussion	This function adds a folder that use this category. This function increments the category's number of folders.<br>
 * @param	p		the category to add a folder
 * @result	the number of folders.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_add_folder ( struct category *p);


/*!
 * @function	category_add_file
 * @abstract	Adds a file to the category
 * @discussion	This function adds a file that use this category. This function increments the category's number of files.<br>
 * @param	p		the category to add a file
 * @result	the number of files.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_add_file ( struct category *p);


/*!
 * @function	category_remove_catalog
 * @abstract	Removes a catalog to the category
 * @discussion	This function removes a catalog that use this category. This function decriment the category's number of catalogs.<br>
 * @param	p		the category to remove a catalog
 * @result	the number of catalogs.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_remove_catalog ( struct category *p);


/*!
 * @function	category_remove_disk
 * @abstract	Removes a disk to the category
 * @discussion	This function removes a disk that use this category. This function decriment the category's number of disks.<br>
 * @param	p		the category to remove a disk
 * @result	the number of disks.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_remove_disk ( struct category *p);


/*!
 * @function	category_remove_folder
 * @abstract	Removes a folder to the category
 * @discussion	This function removes a folder that use this category. This function decriment the category's number of folders.<br>
 * @param	p		the category to remove a folder
 * @result	the number of folders.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_remove_folder ( struct category *p);


/*!
 * @function	category_remove_file
 * @abstract	Removes a file to the category
 * @discussion	This function removes a file that use this category. This function decriment the category's number of files.<br>
 * @param	p		the category to remove a file
 * @result	the number of files.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_remove_file ( struct category *p);


/*!
 * @function	category_can_removed
 * @abstract	Gets if the category can be removed.
 * @discussion	This function gets if the category can be removed. The category can be removed if nothing use this category<br>
 * @param	p		the category to know if it can be removed
 * @result	is can be removed.<br>
 *		Returns <i>FALSE</i> when an error occured.<br>
 */
gboolean category_can_removed ( struct category *p);


/*!
 * @function	category_free
 * @abstract	Frees datas structure.
 * @discussion	This function frees the category's datas.
 * @param	p		the category to free
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint category_free ( struct category *p);


#endif
