/*  GWhere
 *  Copyright (C) 2000  Sbastien LECACHEUR
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*!
 * @header	gwdirdata.h
 *		GWhere<br>
 *		Copyright (C) 2000  Sbastien LECACHEUR<br><br>
 *		This Distributed under the terms of the <a href="http://www.gnu.org/copyleft/gpl.html">GNU General Public Licence</a><br><br>
 *		This component was developed for the GWhere project.<br>
 *		It's an object for directory representation. His structure gets many informations and the list of his components.<br><br>
 * @abstract	This is the dirdata data structure.
 * @author	Sbastien LECACHEUR <zero@gwhere.org>
 * @version	1.0 2001/08/21
 * @see		gwsupport.h
 * @see		glib.h
 * @see		gwfileinfo.h
 * @link		GWhere Web Site <http://www.gwhere.org>
 * @link		GTK+ Web Site <http://www.gtk.org>
 * @ref		gwdirdata.c
 * @author	Author <zero@gwhere.org>
 */


#ifndef GW_DATA_DIR_DATA_H
#define GW_DATA_DIR_DATA_H


#include <glib.h>
#include "gwfileinfo.h"


/*!
 * @struct	dir_data
 * @abstract	Directory data structure
 * @discussion	Directory data is an opaque data structure containing information about directory properties and all contained sub items.
 */
struct dir_data;


/*!
 * @function	dir_data_new
 * @abstract	Creates a new dir data.
 * @discussion	This function creates a new dir data. It's an allocation function.
 * @result	a pointer to the dir data structure.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
struct dir_data * dir_data_new ( void);


/*!
 * @function	dir_data_set_info
 * @abstract	Sets the dir data's informations.
 * @discussion	This function sets the directory's information. Directory informations is a <i>struct file_info</i>.<br>
 * @param	p		the dir data to set the informations
 * @param	info		the directory informations
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_data_set_info ( struct dir_data *p, struct file_info *info);


/*!
 * @function	dir_data_set_child
 * @abstract	Sets the dir data's children.
 * @discussion	This function sets the directory's children. The list of children is the list of file and subfolders in the directory.<br>
 * @param	p		the dir data to set the children
 * @param	child		the list of children
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_data_set_child ( struct dir_data *p, GList *child);


/*!
 * @function	dir_data_append_child
 * @abstract	Appends in the dir data's children a new child.
 * @discussion	This function appends in the dir dat's children list a new child.<br>
 * @param	p		the dir data to append a child
 * @param	child		the child
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_data_append_child ( struct dir_data *p, struct file_info *child);


/*!
 * @function	dir_data_remove_child
 * @abstract	Removes and frees in the dir data's children a child.
 * @discussion	This function removes and frees in the dir dat's children list a child.<br>
 * @param	p		the dir data to removes and frees a child
 * @param	child		the child
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gint dir_data_remove_child ( struct dir_data *p, struct file_info *child);


/*!
 * @function	dir_data_get_info
 * @abstract	Gets dir data's informations.
 * @discussion	This function gets the dir data's informations.<br>
 * @param	p		the dir data to get informations
 * @result	the file informations.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
struct file_info * dir_data_get_info ( struct dir_data *p);


/*!
 * @function	dir_data_get_child
 * @abstract	Gets dir data's list of children.
 * @discussion	This function gets the dir data's list of children. This list is the list of files and subfolders in the directory.<br>
 * @param	p		the dir data to get list of children
 * @result	the list of children.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
GList * dir_data_get_child ( struct dir_data *p);


/*!
 * @function	dir_data_get_child_named
 * @abstract	Gets special dir data's child.
 * @discussion	This function gets the dir data's child with a special name.<br>
 * @param	p		the dir data to get the child
 * @result	the child.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
GList * dir_data_get_child_named ( struct dir_data *p, gchar *name);


/*!
 * @function	dir_data_get_num_file
 * @abstract	Gets the number of file in the dir data.
 * @discussion	This function gets the number of files and subfolders in the dir data.<br>
 * @param	p		the dir data to get the number of file
 * @result	the number of files and subfolders.<br>
 *		Returns <i>NULL</i> when an error occured.<br>
 */
int dir_data_get_num_file ( struct dir_data *p);


/*!
 * @function	dir_data_free
 * @abstract	Frees datas structure.
 * @discussion	This function frees the dir data's datas.
 * @param	p		the dir data to free
 * @result	an error code.<br>
 *		Returns <i>0</i> when it's done.<br>
 *		Returns <i>-1</i> when an error occured.<br>
 */
gboolean dir_data_free ( struct dir_data *p);


#endif
