/*
 *  $Id: xyzexport.c 25063 2022-10-03 08:45:08Z yeti-dn $
 *  Copyright (C) 2015-2022 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/* NB: Magic is in rawxyz. */

/**
 * [FILE-MAGIC-MISSING]
 * Export only.
 **/

#include "config.h"
#include <locale.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwyutils.h>
#include <libgwymodule/gwymodule-file.h>
#include <libprocess/datafield.h>
#include <libprocess/gwyprocesstypes.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwydgetutils.h>
#include <app/gwyapp.h>

#include "err.h"

#define EXTENSION ".xyz"

enum {
    PARAM_ADD_COMMENT,
    PARAM_DECIMAL_DOT,
    PARAM_PRECISION,
    PARAM_MASKING,

    INFO_CHANNEL,
};

typedef struct {
    gboolean needs_decimal_dot;
    guint decimal_dot_len;
    gchar *decimal_dot;
} DecimalDotInfo;

typedef struct {
    GwyParams *params;
    DecimalDotInfo decinfo;
    GwyAppPage pageno;
    gboolean have_mask;
} ModuleArgs;

static gboolean         module_register            (void);
static gint             xyzexport_detect           (const GwyFileDetectInfo *fileinfo,
                                                    gboolean only_name);
static gboolean         xyzexport_export           (GwyContainer *data,
                                                    const gchar *filename,
                                                    GwyRunType mode,
                                                    GError **error);
static GwyDialogOutcome run_gui                    (ModuleArgs *args,
                                                    const gchar *title);
static gboolean         xyzexport_export_data_field(GwyDataField *dfield,
                                                    GwyDataField *mfield,
                                                    FILE *fh,
                                                    ModuleArgs *args);
static gboolean         xyzexport_export_surface   (GwySurface *surface,
                                                    FILE *fh,
                                                    ModuleArgs *args);
static void             fill_decimal_dot_info      (DecimalDotInfo *info);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Exports data as simple XYZ text file."),
    "Yeti <yeti@gwyddion.net>",
    "3.0",
    "David Nečas (Yeti)",
    "2015",
};

GWY_MODULE_QUERY2(module_info, xyzexport)

static gboolean
module_register(void)
{
    gwy_file_func_register("xyzexport",
                           N_("XYZ text data (.xyz)"),
                           (GwyFileDetectFunc)&xyzexport_detect,
                           NULL,
                           NULL,
                           (GwyFileSaveFunc)&xyzexport_export);

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_file_func_current());
    gwy_param_def_add_boolean(paramdef, PARAM_ADD_COMMENT, "add-comment", _("Add _informational comment header"),
                              FALSE);
    gwy_param_def_add_boolean(paramdef, PARAM_DECIMAL_DOT, "decimal-dot", _("Use _dot as decimal separator"), TRUE);
    gwy_param_def_add_int(paramdef, PARAM_PRECISION, "precision", _("_Precision"), 0, 16, 5);
    gwy_param_def_add_enum(paramdef, PARAM_MASKING, "Masking", NULL, GWY_TYPE_MASKING_TYPE, GWY_MASK_IGNORE);

    return paramdef;
}

static gint
xyzexport_detect(const GwyFileDetectInfo *fileinfo,
                 G_GNUC_UNUSED gboolean only_name)
{
    return g_str_has_suffix(fileinfo->name_lowercase, EXTENSION) ? 20 : 0;
}

static gboolean
xyzexport_export(G_GNUC_UNUSED GwyContainer *data,
                 const gchar *filename,
                 GwyRunType mode,
                 GError **error)
{
    ModuleArgs args;
    GwyDataField *dfield, *mfield;
    GwySurface *surface;
    gint fid, sid;
    const guchar *title = NULL;
    GwyDialogOutcome outcome;
    FILE *fh = NULL;
    gboolean ok = FALSE;

    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &fid,
                                     GWY_APP_MASK_FIELD, &mfield,
                                     GWY_APP_SURFACE, &surface,
                                     GWY_APP_SURFACE_ID, &sid,
                                     GWY_APP_PAGE, &args.pageno,
                                     0);

    /* Ensure at most one is set.  We produce an error if no exportable data type is available or both types are
     * available but neither is active. When only one is available or one is active we assume that is what the user
     * wants to export. */
    if (dfield && surface) {
        if (args.pageno != GWY_PAGE_CHANNELS)
            dfield = NULL;
        if (args.pageno != GWY_PAGE_XYZS)
            surface = NULL;
    }
    if (!dfield && !surface) {
        err_NO_CHANNEL_EXPORT(error);
        return FALSE;
    }

    if (dfield) {
        args.pageno = GWY_PAGE_CHANNELS;
    }
    if (surface) {
        args.pageno = GWY_PAGE_XYZS;
        mfield = NULL;
    }

    args.params = gwy_params_new_from_settings(define_module_params());
    fill_decimal_dot_info(&args.decinfo);
    if (!args.decinfo.needs_decimal_dot)
        gwy_params_set_boolean(args.params, PARAM_DECIMAL_DOT, TRUE);

    if (dfield)
        gwy_container_gis_string(data, gwy_app_get_data_title_key_for_id(fid), &title);
    if (surface)
        gwy_container_gis_string(data, gwy_app_get_surface_title_key_for_id(sid), &title);

    if (!title)
        title = _("Untitled");

    if (mode == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args, title);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL) {
            err_CANCELLED(error);
            goto fail;
        }
    }

    if (!(fh = gwy_fopen(filename, "w"))) {
        err_OPEN_WRITE(error);
        goto fail;
    }

    if (gwy_params_get_boolean(args.params, PARAM_ADD_COMMENT)) {
        GwySIUnit *xyunit = NULL, *zunit = NULL;
        gchar *s;

        if (args.pageno == GWY_PAGE_CHANNELS) {
            gwy_fprintf(fh, "# %s %s\n", _("Channel:"), title);
            xyunit = gwy_data_field_get_si_unit_xy(dfield);
            zunit = gwy_data_field_get_si_unit_z(dfield);
        }
        else if (args.pageno == GWY_PAGE_XYZS) {
            gwy_fprintf(fh, "# %s %s\n", _("XYZ data:"), title);
            xyunit = gwy_surface_get_si_unit_xy(surface);
            zunit = gwy_surface_get_si_unit_z(surface);
        }

        s = gwy_si_unit_get_string(xyunit, GWY_SI_UNIT_FORMAT_VFMARKUP);
        gwy_fprintf(fh, "# %s %s\n", _("Lateral units:"), s);
        g_free(s);

        s = gwy_si_unit_get_string(zunit, GWY_SI_UNIT_FORMAT_VFMARKUP);
        gwy_fprintf(fh, "# %s %s\n", _("Value units:"), s);
        g_free(s);
    }

    if (args.pageno == GWY_PAGE_CHANNELS) {
        if (!xyzexport_export_data_field(dfield, mfield, fh, &args)) {
            err_WRITE(error);
            goto fail;
        }
    }
    else if (args.pageno == GWY_PAGE_XYZS) {
        if (!xyzexport_export_surface(surface, fh, &args)) {
            err_WRITE(error);
            goto fail;
        }
    }

    ok = TRUE;

fail:
    g_object_unref(args.params);
    if (fh)
        fclose(fh);
    if (!ok)
        g_unlink(filename);

    return ok;
}

static GwyDialogOutcome
run_gui(ModuleArgs *args, const gchar *title)
{
    GwyDialog *dialog;
    GwyParamTable *table;
    gchar *desc = NULL;

    dialog = GWY_DIALOG(gwy_dialog_new(_("Export XYZ")));
    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    if (args->pageno == GWY_PAGE_CHANNELS)
        desc = _("Channel");
    else if (args->pageno == GWY_PAGE_XYZS)
        desc = _("XYZ data");

    table = gwy_param_table_new(args->params);
    gwy_param_table_append_info(table, INFO_CHANNEL, desc);
    gwy_param_table_info_set_valuestr(table, INFO_CHANNEL, title);
    gwy_param_table_append_header(table, -1, _("Options"));
    gwy_param_table_append_checkbox(table, PARAM_DECIMAL_DOT);
    gwy_param_table_set_sensitive(table, PARAM_DECIMAL_DOT, args->decinfo.needs_decimal_dot);
    gwy_param_table_append_checkbox(table, PARAM_ADD_COMMENT);
    gwy_param_table_append_slider(table, PARAM_PRECISION);
    gwy_param_table_slider_set_mapping(table, PARAM_PRECISION, GWY_SCALE_MAPPING_LINEAR);
    if (args->have_mask)
        gwy_param_table_append_radio(table, PARAM_MASKING);
    gwy_dialog_add_content(dialog, gwy_param_table_widget(table), FALSE, FALSE, 0);
    gwy_dialog_add_param_table(dialog, table);

    return gwy_dialog_run(dialog);
}

static inline gint
print_with_decimal_dot(FILE *fh,
                       gchar *formatted_number,
                       const gchar *decimal_dot,
                       guint decimal_dot_len)
{
    gchar *pos = strstr(formatted_number, decimal_dot);

    if (!pos)
        return fputs(formatted_number, fh);
    else {
        pos[0] = '.';
        if (decimal_dot_len == 1)
            return fputs(formatted_number, fh);
        else {
            gint l1, l2;

            pos[1] = '\0';
            if ((l1 = fputs(formatted_number, fh)) == EOF)
                return EOF;
            if ((l2 = fputs(pos + decimal_dot_len, fh)) == EOF)
                return EOF;
            return l1 + l2;
        }
    }
}

static gboolean
xyzexport_export_data_field(GwyDataField *dfield, GwyDataField *mfield,
                            FILE *fh, ModuleArgs *args)
{
    gboolean force_decimal_dot = (args->decinfo.needs_decimal_dot
                                  && gwy_params_get_boolean(args->params, PARAM_DECIMAL_DOT));
    gint precision = gwy_params_get_int(args->params, PARAM_PRECISION);
    GwyMaskingType masking = gwy_params_get_masking(args->params, PARAM_MASKING, &mfield);
    gint ddlen = args->decinfo.decimal_dot_len;
    const gchar *ddot = args->decinfo.decimal_dot;
    gdouble dx, dy, xoff, yoff;
    const gdouble *d, *m;
    gint xres, yres, i, j;
    gchar buf[40];

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    dx = gwy_data_field_get_dx(dfield);
    dy = gwy_data_field_get_dy(dfield);
    xoff = gwy_data_field_get_xoffset(dfield);
    yoff = gwy_data_field_get_yoffset(dfield);

    d = gwy_data_field_get_data_const(dfield);
    m = mfield ? gwy_data_field_get_data_const(mfield) : NULL;

    if (force_decimal_dot) {
        for (i = 0; i < yres; i++) {
            gdouble y = dy*(i + 0.5) + yoff;
            for (j = 0; j < xres; j++) {
                gdouble x = dx*(j + 0.5) + xoff;
                gdouble z = d[i*xres + j];

                if (m) {
                    if (masking == GWY_MASK_EXCLUDE && m[i*xres + j] > 0.5)
                        continue;
                    if (masking == GWY_MASK_INCLUDE && m[i*xres + j] < 0.5)
                        continue;
                }

                g_snprintf(buf, sizeof(buf), "%.*g\t", precision, x);
                if (print_with_decimal_dot(fh, buf, ddot, ddlen) == EOF)
                    return FALSE;
                g_snprintf(buf, sizeof(buf), "%.*g\t", precision, y);
                if (print_with_decimal_dot(fh, buf, ddot, ddlen) == EOF)
                    return FALSE;
                g_snprintf(buf, sizeof(buf), "%.*g\n", precision, z);
                if (print_with_decimal_dot(fh, buf, ddot, ddlen) == EOF)
                    return FALSE;
            }
        }
    }
    else {
        for (i = 0; i < yres; i++) {
            gdouble y = dy*(i + 0.5) + yoff;
            for (j = 0; j < xres; j++) {
                gdouble x = dx*(j + 0.5) + xoff;
                gdouble z = d[i*xres + j];

                if (m) {
                    if (masking == GWY_MASK_EXCLUDE && m[i*xres + j] > 0.5)
                        continue;
                    if (masking == GWY_MASK_INCLUDE && m[i*xres + j] < 0.5)
                        continue;
                }

                if (gwy_fprintf(fh, "%.*g\t%.*g\t%.*g\n", precision, x, precision, y, precision, z) < 3)
                    return FALSE;
            }
        }
    }

    return TRUE;
}

static gboolean
xyzexport_export_surface(GwySurface *surface,
                         FILE *fh, ModuleArgs *args)
{
    gboolean force_decimal_dot = (args->decinfo.needs_decimal_dot
                                  && gwy_params_get_boolean(args->params, PARAM_DECIMAL_DOT));
    gint precision = gwy_params_get_int(args->params, PARAM_PRECISION);
    gint ddlen = args->decinfo.decimal_dot_len;
    const gchar *ddot = args->decinfo.decimal_dot;
    const GwyXYZ *data;
    guint n, i;
    gchar buf[40];

    data = surface->data;
    n = surface->n;

    if (force_decimal_dot) {
        for (i = 0; i < n; i++) {
            g_snprintf(buf, sizeof(buf), "%.*g\t", precision, data[i].x);
            if (print_with_decimal_dot(fh, buf, ddot, ddlen) == EOF)
                return FALSE;
            g_snprintf(buf, sizeof(buf), "%.*g\t", precision, data[i].y);
            if (print_with_decimal_dot(fh, buf, ddot, ddlen) == EOF)
                return FALSE;
            g_snprintf(buf, sizeof(buf), "%.*g\n", precision, data[i].z);
            if (print_with_decimal_dot(fh, buf, ddot, ddlen) == EOF)
                return FALSE;
        }
    }
    else {
        for (i = 0; i < n; i++) {
            if (gwy_fprintf(fh, "%.*g\t%.*g\t%.*g\n",
                            precision, data[i].x, precision, data[i].y, precision, data[i].z) < 3)
                return FALSE;
        }
    }

    return TRUE;
}

static void
fill_decimal_dot_info(DecimalDotInfo *info)
{
    struct lconv *locale_data;
    guint len;

    locale_data = localeconv();
    info->decimal_dot = g_strdup(locale_data->decimal_point);
    if (!info->decimal_dot || !(len = strlen(info->decimal_dot))) {
        g_warning("Cannot get decimal dot information from localeconv().");
        g_free(info->decimal_dot);
        info->decimal_dot = g_strdup(".");
        len = 1;
    }
    info->decimal_dot_len = len;
    info->needs_decimal_dot = !gwy_strequal(info->decimal_dot, ".");
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
