/*
 * Copyright (C) 2001-2004 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: lirc.c,v 1.12 2005/10/14 17:59:58 dsalt Exp $
 *
 * lirc (infrared control) code, originally based on oxine
 */

#include "globals.h"
#include "defs.h"

#include <string.h>
#include <pthread.h>
#include <stdlib.h>
#include <stdio.h>

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <glib.h>
#include <gtk/gtk.h>

#ifdef HAVE_LIRC
#include <lirc/lirc_client.h>
#endif

#include "engine.h"

static pthread_t             gxine_lirc_thread;
static struct lirc_config   *lircconfig;
static int                   gxine_lirc_fd;

/* lirc thread
  primary config file for lirc is ~/.gxine/lircrc
  secondary is ~/.lircrc
  prog name = gxine
  possible config = lots :-)
*/

#ifdef HAVE_LIRC
static int gxine_lirc_tryload (const char *file)
{
  struct stat st;
  char *fname = g_strconcat (g_get_home_dir(), file, NULL);

  if (stat (fname, &st))
  {
    if (errno == ENOENT)
    {
      free (fname);
      return 1;
    }
    fprintf (stderr, _("lirc: problem with ~/%s: %s\n"), file, strerror (errno));
    free (fname);
    return 1;
  }

  if (lirc_readconfig (fname, &lircconfig, NULL) != 0)
  {
    fprintf (stderr, _("lirc: lirc_readconfig of %s failed\n"), fname);
    free (fname);
    return 1;
  }

  free (fname);
  return 0;
}

static void gxine_lirc_cleanup (void *data)
{
  lirc_deinit ();
}

/*
static gboolean gxine_lirc_exec (char **data)
{
  engine_exec (data[0], NULL, NULL, data[1]);
  free (data[1]);
  free (data);
  return FALSE;
}
*/

static void *gxine_lirc_run (void *data)
{
  char *config;
  gboolean first = TRUE;

  for (;;)
  {
    gxine_lirc_fd = -1;
    while (gxine_lirc_fd == -1)
    {
      if ((gxine_lirc_fd = lirc_init ("gxine", 0)) != -1)
      {
        if (!first)
          logprintf ("lirc: reconnected\n");
        break;
      }

      if (!first)
      {
        sleep (1);
        continue;
      }

      fprintf (stderr,
	       _("lirc: cannot initialise - disabling remote control\n"
	         "lirc: maybe lircd isn't running or you can't connect to the socket?\n"));
      pthread_exit(NULL);
      return NULL;
    }

    first = FALSE;
    pthread_cleanup_push (gxine_lirc_cleanup, NULL);

    /* read ~/.gxine/lircrc then ~/.lircrc - definitions are not overridden */
    if ((gxine_lirc_tryload ("/.gxine/lircrc")
	 && gxine_lirc_tryload ("/.lircrc"))
	|| !lircconfig || !lircconfig->first)
    {
      /* if neither file has been read successfully or there's nothing for
       * gxine, exit now */
      puts (_("lirc: no configuration found - disabling remote control"));
      pthread_exit (NULL);
      return NULL;
    }

    for (;;)
    {
      char *code;
      pthread_testcancel();

      if (lirc_nextcode (&code) == 0)
      {
        pthread_testcancel();

        if (code != NULL)
        {
	  int ret;
          while ((ret = lirc_code2char (lircconfig, code, &config) == 0)
		 && config != NULL)
	  {
	    char **ptr = malloc (2 * sizeof (char *));
	    ptr[0] = config;
	    ptr[1] = g_strdup_printf (_("LIRC binding %s.%s"),
				      lircconfig->current_mode, code);
	    g_async_queue_push (js_queue, ptr);
	    GdkEventClient event;
	    event.type = GDK_CLIENT_EVENT;
            event.window = app->window;
	    event.send_event = TRUE;
	    event.message_type = GDK_NONE;
	    event.data_format = 8;
	    strcpy (event.data.b, "Javascript");
	    gdk_event_put ((GdkEvent *) &event);
	  }

	  free(code);
        }
      }
      else
      {
	/* lircd has exited? try reinitialising in case it's been restarted */
	logprintf ("lirc: socket read error\n");
        pthread_testcancel();
        break;
      }
    }

    pthread_cleanup_pop (1);
  }
}
#endif /* HAVE_LIRC */

void gxine_lirc_init (void)
{
#ifdef HAVE_LIRC
  pthread_create (&gxine_lirc_thread, NULL, gxine_lirc_run, NULL);
#endif
}

void gxine_lirc_quit (void)
{
#ifdef HAVE_LIRC
  pthread_cancel (gxine_lirc_thread);
#endif
}
